/*!
 * \file design.c
 * \brief Combo6 design / component related functions
 * \author Jaroslav Kysela <perex@perex.cz>
 * \date 2003, 2004
 *
 * Copyright (C) 2003, 2004  CESNET
 *
 * The design and component functions work with the FPGA designs.
 * For more details, please, look to \ref design page.
 */
/*
 * LICENSE TERMS
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name of the Company nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * ALTERNATIVELY, provided that this notice is retained in full, this
 * product may be distributed under the terms of the GNU General Public
 * License (GPL) version 2 or later, in which case the provisions
 * of the GPL apply INSTEAD OF those given above.
 *
 * This software is provided ``as is'', and any express or implied
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose are disclaimed.
 * In no event shall the company or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 *
 * $Id$
 *
 */
/*! \page design Design management
 *
 * The design management functions take care about loading of the selected
 * FPGA designs and building of the component tree.
 *
 * The #cs_design_load function loads a selected FPGA designs and builds
 * the component tree. For the component tree creation, there must be a valid
 * XML file with design description available.
 *
 * See #cs_design_load, #cs_design_reload, #cs_design_free, #cs_design_id.
 *
 * \section components Component description
 *
 * Components are stored in tree. Each component is identified with
 * name (it describes component functionality - example "CAM" or "LUP").
 * Also, all components are versioned - see #cs_cver_t. Major version
 * number specifies the component behaviour. Minor version number specifies
 * minor (small) updates to behaviour. Minor updates must be backward
 * compatible with the initial minor version. The component index means
 * component instance number on same level in the component tree.
 *
 * See #cs_component_find, #cs_component_sfind for component searching.
 *
 * See #cs_component_alloc, #cs_component_free, #cs_component_append and
 * #cs_component_remove for manual building.
 */

#include <err.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#ifndef __USE_BSD
#define __USE_BSD
#endif
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <commlbr.h>
#include <combo6.h>
#include "combosix.h"
#include "cs_local.h"

__RCSID("$Id$");

/*!
 * \def CS_DEFAULT_XML_INDEX_PATH
 * \brief Default path for design XML files
 */
#ifndef CS_DEFAULT_XML_INDEX_PATH
#define CS_DEFAULT_XML_INDEX_PATH "/usr/share/mcs/index"
#endif

int
cs_boot_get_id(cs_device_t *d);

/**
 * \brief Get verbosity level from the environment
 * \return Verbosity level
 * \retval 0 No verbosity
 */
static int
cs_design_verbose(void)
{
	const char *s = getenv("LIBCOMBO_DESIGN_VERBOSE");

	if (s == NULL)
		return 0;
	if (isdigit((int)*s))
		return atol(s);
	return 1;
}

/**
 * \brief Get component version
 * \param c Component structure
 * \return Component version (see #CS_CVER_MAJOR and #CS_CVER_MINOR macros)
 */
u_int32_t
cs_component_version(cs_component_t *c)
{
	return c->version;
}

/**
 * \brief Allocate a component
 * \param d Combo6 device structure
 * \param res Returned allocated component structure
 * \param name Component ASCII name
 * \param idx Component index
 * \param ver Component version
 * \param space Component space
 * \return zero on success or a negative error code
 *
 * \note Allocated component is not put automatically into the
 * component tree. Use #cs_component_append function for this
 * action.
 */
int
cs_component_alloc(cs_device_t *d ATTRIBUTE_UNUSED, cs_component_t **res,
	const char *name, cs_cidx_t idx,
	cs_cver_t ver, cs_space_t *space)
{
	*res = (cs_component_t *)calloc (1, sizeof (cs_component_t));
	if (*res == NULL)
		return -ENOMEM;
	(*res)->name = strdup(name);
	if ((*res)->name == NULL) {
		free(*res);
		*res = NULL;
		return -ENOMEM;
	}
	(*res)->index = idx;
	(*res)->version = ver;
	(*res)->space = space;

	return 0;
}

/**
 * \brief Free a component
 * \param d Combo6 device structure
 * \param c Component structure
 * \return zero on success otherwise a negative error code
 */
int
cs_component_free(cs_device_t *d, cs_component_t **c)
{
	if ((*c)->upriv_free)
		(*c)->upriv_free((*c)->upriv_data);
	if ((*c)->priv_free)
		(*c)->priv_free((*c)->priv_data);
	if ((*c)->name)
		free((void *)(*c)->name);
	if ((*c)->space)
		cs_space_unmap(d, &(*c)->space);
	free(*c);
	*c = NULL;
	return 0;
}

/**
 * \brief Free a component and its children
 * \param d Combo6 device structure
 * \param c Component structure
 * \return zero on success otherwise a negative error code
 */
static int
cs_component_free_subtree(cs_device_t *d, cs_component_t **c)
{
	cs_component_t *look, *next;
	int xerr;

	for (look = (*c)->child; look != NULL; look = next) {
		next = look->next;
		xerr = cs_component_free_subtree(d, &look);
		if (xerr < 0)
			return xerr;
	}

	return cs_component_free(d, c);
}

/**
 * \brief Remove a component from tree and free it
 * \param d Combo6 device structure
 * \param c Component structure
 * \return zero on success otherwise a negative error code
 */
int
cs_component_remove(cs_device_t *d, cs_component_t **c)
{
	cs_component_t **look;

	look = ((*c)->parent != NULL) ? &(*c)->parent->child : &d->first;

	while (*look != NULL) {
		if (*look == *c) {
			*look = (*look)->next;
			return cs_component_free_subtree(d, c);
		}
		look = &(*look)->next;
	}

	return -ENOENT;
}

/**
 * \brief Call a callback function for a component and its children
 * \param d Combo6 device structure
 * \param c Component structure
 * \param callback Callback function
 * \param callback_data Data passed to the callback function
 * \return zero on success otherwise a negative error code
 *
 * The callback function should not change the tree's structure (adding
 * or removing entries).
 */
int
cs_component_for_subtree(cs_device_t *d,
	cs_component_t *c,
	int (*callback)(cs_device_t *, cs_component_t *, const void *),
	const void *callback_data)
{
	cs_component_t *look;
	int xerr;

	xerr = callback(d, c, callback_data);
	if (xerr < 0)
		return xerr;
	for (look = c->child; look != NULL; look = look->next) {
		xerr = cs_component_for_subtree(d, look, callback, callback_data);
		if (xerr < 0)
			return xerr;
	}
	return 0;
}

/**
 * \brief Call a callback function for all components in tree
 * \param d Combo6 device structure
 * \param callback Callback function
 * \param callback_data Data passed to the callback function
 * \return zero on success otherwise a negative error code
 *
 * The callback function should not change the tree's structure (adding
 * or removing entries).
 */
int
cs_component_for_all(cs_device_t *d,
	int (*callback)(cs_device_t *, cs_component_t *, const void *),
	const void *callback_data)
{
	cs_component_t *look;
	int xerr;

	for (look = d->first; look != NULL; look = look->next) {
		xerr = cs_component_for_subtree(d, look, callback, callback_data);
		if (xerr < 0)
			return xerr;
	}
	return 0;
}

/**
 * \brief Free all components associated to active design
 * \param d Combo6 device structure
 * \return zero on success otherwise a negative error code
 */
int
cs_design_free(cs_device_t *d)
{
	cs_component_t *next;

	while (d->first) {
		next = d->first->next;
		cs_component_free_subtree(d, &d->first);
		d->first = next;
	}
	if (d->name != NULL) {
		free(d->name);
		d->name = NULL;
	}
	d->ver_major = d->ver_minor = 0;
	d->timestamp = (time_t)0;

	return 0;
}

/**
 * \brief Append a component to tree
 * \param d Combo6 device structure
 * \param parent Component's parent
 * \param c Component to insert
 * \return zero on success otherwise a negative error code
 */
int
cs_component_append(cs_device_t *d, cs_component_t *parent, cs_component_t *c)
{
	cs_component_t **look;

	c->parent = parent;
	look = (parent != NULL) ? &parent->child : &d->first;

	while (*look != NULL)
		look = &(*look)->next;
	*look = c;

	return 0;
}

/**
 * \brief Allocate a new component and insert it into a component tree
 * \param d Combo6 device structure
 * \param c Component to insert
 * \param parent Component's parent
 * \param name Component name
 * \param idx Component index
 * \param ver Component version
 * \param ofs Component offset in the firmware
 * \param size Component size
 * \return zero on success or a negative error code
 */
static int
append_component(cs_device_t *d, cs_component_t **c, cs_component_t *parent,
	const char *name, cs_cidx_t idx, cs_cver_t ver, cs_addr_t ofs,
	cs_addr_t size)
{
	cs_space_t *s;
	cs_component_t *res;
	int xerr;

	if (c)
		*c = NULL;

	if (parent != NULL)
		ofs += parent->space->sp_base;

	xerr = cs_space_map(d, &s, CS_SPACE_FPGA, size, ofs, 0);
	if (xerr < 0) {
		fprintf(stderr, "Unable to map space %s at 0x%x [0x%x]\n", name, ofs, size);
		return xerr;
	}

	xerr = cs_component_alloc(d, &res, name, idx, ver, s);
	if (xerr < 0) {
		cs_space_unmap(d, &s);
		return xerr;
	}

	xerr = cs_component_append(d, parent, res);
	if (xerr < 0) {
		cs_component_free(d, &res);
		return xerr;
	}

	if (c)
		*c = res;
	return 0;
}

static int cs_include_addresses(cs_device_t *d, cs_boot_v1_t *boot)
{
	static const struct {
		const char *name;
		unsigned int len;
		u_int32_t id;
	} to_include[] = {
		{ "IBUF", 4, COMBO_DI_IBUF },
		{ "OBUF", 4, COMBO_DI_OBUF },
		{ NULL, 0, 0 }
	};
	struct combo_design_info *addrs;
	cs_component_t *c;
	unsigned int a, nr_addrs = 0;

	for (c = d->first; c; c = c->next) {
		for (a = 0; to_include[a].name; a++) {
			if (!strncmp(c->name, to_include[a].name,
						to_include[a].len))
					break;
		}
		if (!to_include[a].name)
			continue;
		nr_addrs++;
	}

	if (!nr_addrs)
		return 0;

	addrs = calloc(nr_addrs, sizeof(*boot->next));
	if (!addrs)
		return -ENOMEM;

	boot->addr = 1025;
	boot->data = (u_int8_t *)addrs;
	boot->size = nr_addrs * sizeof(*addrs);

	for (c = d->first; c; c = c->next) {
		for (a = 0; to_include[a].name; a++) {
			if (!strncmp(c->name, to_include[a].name,
						to_include[a].len))
					break;
		}
		if (!to_include[a].name)
			continue;
		addrs->id = to_include[a].id;
		addrs->idx = c->index;
		addrs->addr = c->space->sp_base;
		addrs->size = c->space->sp_size;
		addrs++;
	}

	return 0;
}

/**
 * \brief Boot a MCS design
 * \param d Combo6 device structure
 * \param mcs MCS structure
 * \return zero on success or a negative error code
 */
static int
cs_design_load_mcs(cs_device_t *d, cs_design_mcs_t *mcs)
{
	int i, count = 0, xerr = 0, verbose, must_have = 1;
	cs_boot_v1_t *boot, *ppcboot;
	FILE *file;

	verbose = cs_design_verbose();

	/* CV2 family cards */
	if (d->dv_board != NULL && !strcmp (d->dv_board, "combo")) {
		/* now we have FPGAs only on mother cards in CV2 family */
		must_have = 1;

	/* Condition for C6 and C6X cards */
	} else if (d->dv_if0_card != NULL && strcmp(d->dv_if0_card, "unknown")) {
		if (!strcmp(d->dv_if0_card, "xfp") ||
			!strcmp(d->dv_if0_card, "sfpro") ||
			!strcmp(d->dv_if0_card, "xfp2") ||
			!strcmp(d->dv_if0_card, "xfp2.0") ||
			!strcmp(d->dv_if0_card, "xfp2.2")
			)
			must_have |= 1 << 8;
		else
			must_have |= (1 << 8) | (1 << 9);
	}

	for (i = 0; i < CS_MCS_SIZE; i++)
		if (mcs->mcs[i].filename != NULL)
			count++;
	if (count == 0)
		return -EINVAL;
	/* + 1, since we may need one in insert_addresses */
	boot = calloc(count + 1, sizeof(cs_boot_v1_t));
	if (boot == NULL)
		return -ENOMEM;
	for (i = 0; i < count; i++) {
		boot[i].addr = mcs->mcs[i].addr;
		if (boot[i].addr < 31)
			must_have &= ~(1 << boot[i].addr);
		if (boot[i].addr == 1024) {
			xerr = cs_load_pcippc(mcs->mcs[i].filename, &ppcboot);
			if (xerr < 0) {
				fprintf(stderr, "Unable to load PPC file '%s': %s\n",
					mcs->mcs[i].filename, strerror(-xerr));
				goto __err;
			}
			boot[i].addr = ppcboot->addr;
			boot[i].data = ppcboot->data;
			boot[i].size = ppcboot->size;
			free(ppcboot);
		} else {
			file = fopen(mcs->mcs[i].filename, "r");
			if (file == NULL) {
				fprintf(stderr, "Unable to open MCS file '%s': %s\n",
					mcs->mcs[i].filename, strerror(errno));
				xerr = -ENOENT;
				goto __err;
			}
			xerr = cs_mcs_decode(file, &boot[i].data, &boot[i].size);
			fclose(file);
			if (xerr < 0)
				goto __err;
		}
		if (verbose)
			fprintf(stderr, "booting %i[%i]: filename %s, size = 0x%x\n", i,
				mcs->mcs[i].addr, mcs->mcs[i].filename, boot[i].size);
		/* do not link the last one */
		if (i + 1 < count)
			boot[i].next = &boot[i + 1];
	}
	if (must_have) {
		xerr = -EINVAL;
		for (i = 0; i < 32; i++)
			if (must_have & (1 << i))
				fprintf(stderr, "mcs file with address %i was not found, aborting\n", i);
		goto __err;
	}

	if (!cs_include_addresses(d, &boot[count]))
		boot[count - 1].next = &boot[count];

	xerr = cs_boot_v1(d, CS_BOOT_V1_STANDARD, boot);
	if (xerr >= 0 && verbose)
		fprintf(stderr, "booting finished successfully\n");
      __err:
	for (i = 0; i < count; i++) {
		if (boot[i].data)
			free(boot[i].data);
	}
	free(boot);
	return xerr;
}

/**
 * \brief Parse informations about a new component from XML
 * \param d Combo6 device structure
 * \param tag XML tag with informations about the component
 * \param base Offset of a firmware
 * \param parent Component's parent
 * \return zero on success or a negative error code
 */
static int
cs_design_xml_component(cs_device_t *d, cs_xml_tag_t *tag, u_int32_t base,
		cs_component_t *parent)
{
	int xerr = 0;
	char *name = NULL, *ver;
	const char *start;
	size_t size;
	u_int32_t cbase, csize;
	unsigned int major, minor, index;
	cs_xml_tag_t *freq_tag;
	char *freq_str = NULL;
	float *freq = NULL;
	cs_component_t *c;

	if (!cs_xml_attr_match(tag, "cversion", "1.0"))
		return -EINVAL;
	if (cs_xml_tag_attr(tag, &start, &size, "version") < 0)
		return -EINVAL;
	if (cs_xml_get_string(&ver, start, size) < 0)
		return -ENOMEM;
	major = minor = 0;
	if (ver != NULL) {
		sscanf(ver, "%u.%u", &major, &minor);
		free(ver);
	}
	if (cs_xml_tag_attr(tag, &start, &size, "name") < 0)
		return -EINVAL;
	if (cs_xml_get_string(&name, start, size) < 0)
		return -ENOMEM;
	if (cs_xml_tag_attr(tag, &start, &size, "base") < 0 ||
		cs_xml_get_uint32(&cbase, start, size) < 0 ||
		cs_xml_tag_attr(tag, &start, &size, "size") < 0 ||
		cs_xml_get_uint32(&csize, start, size) < 0 ||
		cs_xml_tag_attr(tag, &start, &size, "index") < 0 ||
		cs_xml_get_uint32(&index, start, size) < 0
		) {
		xerr = -EINVAL;
		goto __err;
	}

	/* if frequency tag set (for MAN component) */
	if (cs_xml_tag_find(cs_xml_tag_child(tag), &freq_tag, "param", 0) == 0) {
		if (cs_xml_tag_body(freq_tag, &start, &size) < 0 ||
			cs_xml_get_string(&freq_str, start, size) < 0
			) {
			xerr = -EINVAL;
			goto __err;
		}

		freq = (float*)malloc(sizeof(float));
		if (freq == NULL) {
			xerr = -ENOMEM;
			goto __err;
		}
		if (sscanf(freq_str, "%f", freq) != 1) {
			xerr = -EINVAL;
			goto __err;
		}
	}

	xerr = append_component(d, &c, parent, name, index,
		CS_CVER(major, minor), base + cbase, csize);

	if (xerr < 0)
		goto __err;

	/* update component MAN with detected frequency MTU */
	if (freq) {
		cs_component_upriv_set(c, freq, free);
		freq = NULL;
	}

	for (tag = cs_xml_tag_child(tag); tag != NULL; tag = cs_xml_tag_next(tag)) {
		if (cs_xml_name_match(tag, "component")) {
			xerr = cs_design_xml_component(d, tag, 0, c);
			if (xerr < 0) {
				cs_component_remove(d, &c);
				goto __err;
			}
		}
	}

      __err:
	if (name)
		free(name);
	if (freq_str)
		free(freq_str);
	if (freq)
		free(freq);
	return xerr;
}

/**
 * \brief Check XML's version and encoding
 * \param xml XML tree
 * \return zero on success or a negative error code
 */
static int
cs_design_xml_check(cs_xml_t *xml)
{
	if (!cs_xml_is_version(xml, "1.0") ||
		!cs_xml_is_encoding(xml, "ASCII")
		)
		return -EINVAL;
	return 0;
}

/**
 * \brief Parse a XML tree and create a new component tree
 * \param d Combo6 device structure
 * \param root_tag XML tree's root tag
 * \param reprogram Whether the processed design is going to be booted or not
 * \return zero on success or a negative error code
 */
static int
cs_design_xml_parse(cs_device_t *d, cs_xml_tag_t *root_tag, int reprogram)
{
	cs_xml_tag_t *design_tag, *master, *tag, *toplevel;
	const char *start, *mchip;
	char *tmp;
	size_t size;
	u_int32_t base, utctimestamp;
	int xerr, verbose;
	u_int32_t old_id;
	char old_strid[CS_DESIGN_STRID_SIZE];

	verbose = cs_design_verbose();

	mchip = cs_hw_mchip(d);
	if (mchip == NULL || strcmp(mchip, "unknown") == 0) {
		if (verbose & 2)
			fprintf(stderr, "Unknown chip of specified device.\n");
		return -EINVAL;
	}

	if (cs_xml_tag_find(root_tag, &design_tag, "combo6design", 0) < 0 ||
		!cs_xml_attr_match(design_tag, "dversion", "1.0") ||
		(master = cs_xml_tag_child(design_tag)) == NULL
		)
		return -EINVAL;

	if (cs_xml_tag_attr(design_tag, &start, &size, "version") < 0)
		return -EINVAL;
	if (cs_xml_get_string(&tmp, start, size) < 0)
		return -ENOMEM;

	d->ver_major = d->ver_minor = 0;
	if (tmp != NULL) {
		sscanf(tmp, "%u.%u", &d->ver_major, &d->ver_minor);
		free(tmp);
	}

	if (cs_xml_tag_find(master, &tag, "utctimestamp", 0) == 0) {
		if (cs_xml_tag_body(tag, &start, &size) < 0 ||
			cs_xml_get_uint32(&utctimestamp, start, size) < 0
			)
			return -EINVAL;
		d->timestamp = (time_t)utctimestamp;
	}
	if (cs_xml_tag_find(master, &tag, "comment", 0) == 0) {
		if (cs_xml_tag_body(tag, &start, &size) < 0 ||
			cs_xml_get_string(&d->name, start, size) < 0
			)
			return -EINVAL;
	}

	/* locate the right type of chip in design.xml */
	for (toplevel = master; toplevel != NULL; toplevel = cs_xml_tag_next(toplevel)) {
		if (cs_xml_name_match(toplevel, "firmware") &&
			cs_xml_attr_match(toplevel, "location", "0") &&
			cs_xml_attr_match(toplevel, "chip", mchip)
			)
			break;
	}

	if (toplevel == NULL) {
		if (verbose & 2)
			fprintf(stderr, "No appropriate firmware for your card"
				" (%s) was found in design.xml.\n", mchip);
		return -EINVAL;
	}

	old_id = d->dv_id;
	strcpy(old_strid, (char*)d->dv_strid);
	if (cs_xml_tag_attr(toplevel, &start, &size, "id") < 0 ||
		cs_xml_get_uint32(&d->dv_id, start, size) || d->dv_id == 0
		)
		return -EINVAL;
	if (cs_xml_tag_attr(toplevel, &start, &size, "strid") == 0) {
		if (cs_xml_get_string(&tmp, start, size) < 0)
			return -EINVAL;
		strncpy((char *)d->dv_strid, tmp, CS_DESIGN_STRID_SIZE);
		if (d->dv_strid[CS_DESIGN_STRID_SIZE - 1] != '\0') {
			if (verbose & 2)
				fprintf(stderr, "Design ID \"%s\" is too long.\n", tmp);
			free(tmp);
			d->dv_strid[CS_DESIGN_STRID_SIZE - 1] = '\0';
			return -EINVAL;
		}
		free(tmp);
	} else
		d->dv_strid[0] = '\0';

	if (!reprogram && (old_id != d->dv_id || strcmp(old_strid, (char *)d->dv_strid))) {
		if (verbose & 2)
			fprintf(stderr,
				"Design ID does not match 0x%x/%s != 0x%x/%s\n",
				old_id, old_strid, d->dv_id, d->dv_strid);
		return -EINVAL;
	}
	base = 0;
	if (cs_xml_tag_attr(toplevel, &start, &size, "base") == 0 &&
		cs_xml_get_uint32(&base, start, size) < 0
		)
		return -EINVAL;
	for (tag = cs_xml_tag_child(toplevel); tag != NULL; tag = cs_xml_tag_next(tag)) {
		if (cs_xml_name_match(tag, "component")) {
			xerr = cs_design_xml_component(d, tag, base, NULL);
			if (xerr < 0)
				return xerr;
		}
	}

	return 0;
}

/**
 * \brief Print a combination of strings and characters on stderr
          (\"\a prefix: \a array [\a extra]\")
 * \param prefix First string
 * \param array Array of characters
 * \param size Number of characters
 * \param extra Optional second string
 */
static void
cs_design_dump_str(const char *prefix, const char *array, size_t size,
	const char *extra)
{
	fprintf(stderr, "%s: ", prefix);
	while (size--)
		fprintf(stderr, "%c", *array++);
	if (extra)
		fprintf(stderr, " [%s]", extra);
	fprintf(stderr, "\n");
}

/**
 * \brief Check the specified XML file and boot an appropriate MCS design
 * \param d Combo6 device structure
 * \param root_tag XML tree's root tag
 * \param mcs MCS structure
 * \param filename Path to XML file
 * \return zero on success or a negative error code
 */
static int
cs_design_xml_program(cs_device_t *d, cs_xml_tag_t *root_tag,
	cs_design_mcs_t *mcs, const char *filename)
{
	cs_xml_tag_t *design_tag, *master, *tag, *toplevel;
	const char *mchip, *chip, *iface, *tmp, *tmp1, *start;
	size_t mchipsize, chipsize, ifacesize, size;
	u_int32_t location;
	int i, len, xerr, verbose;

	verbose = cs_design_verbose();

	if (verbose & 2)
		fprintf(stderr, "cs_design_xml_program from %s\n", filename);

	memset(mcs, 0, sizeof(mcs));
	mchip = cs_hw_mchip(d);
	if (mchip == NULL || strcmp(mchip, "unknown") == 0) {
		if (verbose & 2)
			fprintf(stderr, "unknown chip of specified device.\n");
		return -EINVAL;
	}
	chip = d->dv_if0_chip;
	iface = d->dv_if0_card;
	mchipsize = strlen(mchip);
	chipsize = strlen(chip);
	ifacesize = strlen(iface);

	/* TODO: Modify for CV2 card family */
	if (verbose) {
		fprintf(stderr, "board: %s\n", d->dv_board);
		cs_design_dump_str("mchip", mchip, mchipsize, NULL);
		cs_design_dump_str("chip", chip, chipsize, d->dv_if0_chip);
		cs_design_dump_str("iface", iface, ifacesize, d->dv_if0_card);
	}

	if (cs_xml_tag_find(root_tag, &design_tag, "combo6design", 0) < 0 ||
		!cs_xml_attr_match(design_tag, "dversion", "1.0") ||
		(master = cs_xml_tag_child(design_tag)) == NULL
		) {
		if (verbose & 2)
			fprintf(stderr, "combo6design tag missing, dversion is not 1.0 "
				"or there is no child tag in design.xml\n");
		return -EINVAL;
	}

	i = 0;
	for (toplevel = master; (toplevel != NULL) && (i < CS_MCS_SIZE); toplevel = cs_xml_tag_next(toplevel)) {
		if (!cs_xml_name_match(toplevel, "firmware"))
			continue;
		if (!cs_xml_attr_match(toplevel, "fversion", "1.0")) {
			if (verbose & 2)
				fprintf(stderr, "fversion is not 1.0 in design.xml\n");
			xerr = -EINVAL;
			goto __end;
		}
		if (cs_xml_tag_attr(toplevel, &start, &size, "chip") < 0) {
			if (verbose & 2)
				fprintf(stderr, "firmware chip could not be detected in design.xml\n");
			xerr = -EINVAL;
			goto __end;
		}
		if (memcmp(start, "pcippc", 6) == 0) {
			if (strcmp(d->dv_board, "combo6x") != 0)
				continue;
			location = 1024;
			if (verbose & 2)
				fprintf(stderr, "detected pcippc firmware\n");
			goto __fwdata;
		} else {
			if (cs_xml_tag_attr(toplevel, &start, &size, "location") < 0 ||
				cs_xml_get_uint32(&location, start, size) < 0 ||
				location > 127
				) {
				if (verbose & 2)
					fprintf(stderr, "wrong number of location in design.xml\n");
				xerr = -EINVAL;
				goto __end;
			}
		}
		if (verbose & 2)
			fprintf(stderr, "detected firmware: location 0x%x\n", location);
		cs_xml_tag_attr(toplevel, &start, &size, "chip");
		if (location > 0) {
			if (verbose & 2)
				cs_design_dump_str("detected firmware chip", start, size, NULL);
			if (chipsize != size || memcmp(start, chip, size))
				continue;
			if (cs_xml_tag_attr(toplevel, &start, &size, "interface") < 0) {
				if (verbose & 2)
					fprintf(stderr, "firmware interface could not be detected in design.xml\n");
				xerr = -EINVAL;
				goto __end;
			}
			if (verbose & 2)
				cs_design_dump_str("detected firmware interface", start, size, NULL);
			if (ifacesize != size || memcmp(start, iface, size))
				continue;
		} else {
			if (mchipsize != size || memcmp(start, mchip, size))
				continue;
			if (verbose & 2)
				cs_design_dump_str("detected firmware chip (main)", start, size, NULL);
		}
	      __fwdata:
		if (cs_xml_tag_find(cs_xml_tag_child(toplevel), &tag, "fwdata", 0) < 0) {
			if (verbose & 2)
				fprintf(stderr, "fwdata could not be detected in design.xml\n");
			xerr = -EINVAL;
			goto __end;
		}
		mcs->mcs[i].addr = location;
		if (cs_xml_tag_attr(tag, &start, &size, "url") < 0) {
			xerr = -EINVAL;
			goto __end;
		}
		if (!memcmp(start, "file://", 7)) {
			if (start[7] != '/' && (tmp1 = strrchr(filename, '/')) != NULL) {
				len = (tmp1 - filename) + 1;
				tmp = (const char*)malloc(len + (size - 7) + 1);
				if (tmp == NULL) {
					xerr = -ENOMEM;
					goto __end;
				}
				memcpy((char *)tmp, filename, len);
				memcpy((char *)tmp + len, start + 7, size - 7);
				((char *)tmp)[len + size - 7] = '\0';
			} else {
				tmp = (const char *)malloc((size - 7) + 1);
				if (tmp == NULL) {
					xerr = -ENOMEM;
					goto __end;
				}
				memcpy((char *)tmp, start + 7, size - 7);
				((char *)tmp)[size - 7] = '\0';
			}
			mcs->mcs[i].filename = tmp;
			if (verbose & 2)
				fprintf(stderr, "detected mcs filename '%s'\n", tmp);
		}
		i++;
	}

	if (i == 0) {
		fprintf(stderr, "XML file without firmware - unable to load\n");
		return -EINVAL;
	}

	xerr = cs_design_load_mcs(d, mcs);
      __end:
	while (i > 0) {
		if (mcs->mcs[i - 1].filename) {
			free((char *)mcs->mcs[i - 1].filename);
			mcs->mcs[i - 1].filename = NULL;
		}
		i--;
	}
	return xerr;
}

/**
 * \brief Load design from a XML file and optionaly boot it
 * \param d Combo6 device name
 * \param filename Path to the XML file
 * \param reprogram Whether to boot processed design or not
 * \return zero on success or a negative error code
 */
static int
cs_design_load_xml(cs_device_t *d, const char *filename, int reprogram)
{
	int xerr = 0;
	cs_xml_t *xml = NULL;
	cs_xml_tag_t *root_tag;
	cs_design_mcs_t *mcs;

	if (!reprogram && d->dv_id == 0)
		return -EINVAL;
	xerr = cs_xml_open(&xml, &root_tag, filename);
	if (xerr < 0)
		return xerr;
	xerr = cs_design_xml_check(xml);
	if (xerr < 0)
		goto __end;
	xerr = cs_design_xml_parse(d, root_tag, reprogram);
	if (xerr < 0)
		goto __end;
	if (reprogram) {
		mcs = (cs_design_mcs_t *)malloc(sizeof(*mcs));
		if (mcs == NULL) {
			xerr = -ENOMEM;
			goto __end;
		}
		xerr = cs_design_xml_program(d, root_tag, mcs, filename);
		free(mcs);
	}
      __end:
	if (xml)
		cs_xml_close(&xml);
	if (xerr < 0) {
		cs_design_free(d);
		cs_boot_get_id(d);
		return xerr;
	}
	return 0;
}

/**
 * \brief Test if design is active inside hardware
 * \param d Combo6 device name
 * \retval zero design is not present and has to be loaded
 * \retval one design is present
 * \retval negative error code
 */
int
cs_design_present(cs_device_t *d)
{
	return (d->dv_id != 0);
}

/**
 * \brief Load design into hardware and prepare component tree
 * \param d Combo6 device name
 * \param type Design type
 * \param arg Argument
 * \return zero on success or a negative error code
 *
 * When the \a type is
 * \arg #CS_DESIGN_XML, the passed XML file is loaded and parsed. These files
 * contain component tree description and also information which MCS files are
 * used to boot the FPGA chips. After the appropriate firmware section is found,
 * information about its components is loaded into the memory and a component
 * tree is created. Then the MCS file is located an booted into the hardware.
 *
 * \arg #CS_DESIGN_NONE, nothing is loaded into hardware. First the 
 * identification from active design is read, which is used for the appropriate
 * XML file look up. If the file is found, it's parsed and the component tree
 * is created.
 *
 * \arg #CS_DESIGN_MCS, the passed MCS structure is parsed and selected MCS
 * files are booted. If the booting succeeds, process continues as if the
 * #CS_DESIGN_NONE was selected (check design ID and create the component tree).
 *
 *
 * \note Note that in the XML files, the "file: URL" attribude (specifies the
 * location of the corresponding MCS file) can be also relative (two '/' chars
 * are used) or absolute (three '/' chars). The code detects the present chips
 * using information from CPLD so only MCS files following the CPLD contents are
 * used.
 *
 * \note The XML files are expected to be in directories
 * BASE_COMBO6_DIR/index/0x{hex_id_representation} for old designs and
 * BASE_COMBO6_DIR/index/{string identification}/0x{hex_id_representation}
 * for newer designs (with ID component). BASE_COMBO6_DIR is usually
 * /usr/local/combo6. Default index directory is defined by
 * #CS_DEFAULT_XML_INDEX_PATH, but it can be overruled with
 * CS_XML_INDEX_PATH environment variable.
 */
int
cs_design_load(cs_device_t *d, cs_design_t type, const cs_design_data_t *arg)
{
	char *filename, *path;
	int xerr;

	/* cannot build new component structure */
	/* free components before */
	if (d->first)
		return -EBUSY;

	switch (type) {
	case CS_DESIGN_NONE:
		if (arg != NULL)
			return -EINVAL;
		break;
	case CS_DESIGN_MCS:
		xerr = cs_design_load_mcs(d, (cs_design_mcs_t *)arg);
		if (xerr < 0)
			return xerr;
		break;
	case CS_DESIGN_XML:
		return cs_design_load_xml(d, (const char *)arg, 1);
	default:
		return -EINVAL;
	}

	if (d->dv_id == 0)
		return -EINVAL;

	path = getenv("CS_XML_INDEX_PATH");
	if (path == NULL)
		path = (char*)CS_DEFAULT_XML_INDEX_PATH;

	filename = (char*)malloc(strlen(path) + strlen((char *)d->dv_strid) + 16);
	if (filename == NULL) {
		d->dv_id = 0;
		d->dv_strid[0] = '\0';
		return -ENOMEM;
	}
	if (d->dv_strid[0] != '\0')
		sprintf(filename, "%s/%s/0x%08x", path, d->dv_strid, d->dv_id);
	else
		sprintf(filename, "%s/0x%08x", path, d->dv_id);
	xerr = cs_design_load_xml(d, filename, 0);
	if (xerr < 0) {
		fprintf(stderr, "Unable to load XML file '%s': %s\n", filename, strerror(-xerr));
		d->dv_id = 0;
		d->dv_strid[0] = '\0';
	}
	free(filename);

	return xerr;
}

/**
 * \brief Load design information without booting
 * \param d Combo6 device name
 * \param type Design type
 * \param arg Argument
 * \return zero on success or a negative error code
 *
 * This function is similar to #cs_design_load except that only the component
 * tree is created. The booting is omitted. Only #CS_DESIGN_NONE or
 * #CS_DESIGN_XML type is allowed.
 */
int
cs_design_reload(cs_device_t *d, cs_design_t type, const cs_design_data_t *arg)
{
	/* cannot build new component structure */
	/* free components before */
	if (d->first)
		return -EBUSY;

	switch (type) {
	case CS_DESIGN_NONE:
		return cs_design_load(d, type, arg);
	case CS_DESIGN_MCS:
		return -EINVAL;
	case CS_DESIGN_XML:
		return cs_design_load_xml(d, (const char *)arg, 0);
	default:
		return -EINVAL;
	}
}

/**
 * \brief Return the design identification
 * \param d Combo6 device
 * \return Allocated string or NULL (means error)
 *
 * The returned string looks like:
 *  'name="Scampi PH1",version="0.01",built=(time_t)123456,id32=0xabcdef10'
 */
char *
cs_design_id(cs_device_t *d)
{
	char *res;
	int len;

	if (d->first == NULL)
		return NULL;
	len = (d->name == NULL) ? 0 : strlen(d->name);
	res = (char*)malloc(len + 128);
	if (res == NULL)
		return NULL;
	sprintf(res, "name=\"%s\",version=\"%u.%02u\",built=(time_t)%u,id32=0x%x\n",
		d->name == NULL ? "" : d->name,
		d->ver_major, d->ver_minor,
		(unsigned int)d->timestamp,
		d->dv_id);
	return res;
}

/**
 * \brief Print component information on stdout
 * \param c Component structure
 * \param parent_base Parent base address
 * \param space Number of spaces to indent
 */
static void
cs_design_dump_component(cs_component_t *c, u_int32_t parent_base, int space)
{
	cs_component_t *look;
	int i;

	for (i = 0; i < space; i++)
		putchar(' ');
	printf("<component cversion=\"1.0\" version=\"%u.%02u\" name=\"%s\" "
		"index=\"%u\" base=\"0x%08x\" size=\"0x%08x\">\n",
		(unsigned int)CS_CVER_MAJOR(c->version),
		(unsigned int)CS_CVER_MINOR(c->version),
		c->name, c->index,
		c->space->sp_base - parent_base, c->space->sp_size);
	for (look = c->child; look != NULL; look = look->next)
		cs_design_dump_component(look, c->space->sp_base, space + 2);
	for (i = 0; i < space; i++)
		putchar(' ');
	printf("</component>\n");
}

/**
 * \brief Dump the current design structure to stdout
 * \param d Combo6 device
 */
void
cs_design_dump(cs_device_t *d)
{
	cs_component_t *c;

	if (d->first == NULL)
		return;
	printf("<?xml version=\"1.0\" encoding=\"ASCII\">\n"
		"<combo6design dversion=\"1.0\" version=\"%u.%02u\">\n",
		d->ver_major, d->ver_minor);
	if (d->name)
		printf("  <comment>%s</comment>\n", d->name);
	if (d->timestamp > 0)
		printf("  <utctimestamp>%u</utctimestamp>\n", (unsigned int)d->timestamp);
	printf("  <author>libcombo</author>\n");
	printf("  <firmware fversion=\"1.0\" id=\"0x%08x\"", d->dv_id);
	if (d->dv_strid[0] != '\0')
		printf(" strid=\"%s\"", d->dv_strid);
	printf(" chip=\"%s\" location=\"0\" base=\"0\">\n", cs_hw_mchip(d));
	for (c = d->first; c != NULL; c = c->next)
		cs_design_dump_component(c, 0, 4);
	printf("  </firmware>\n");
	printf("</combo6design>\n");
}

/**
 * \brief Find a space inside active design
 * \param d Combo6 device structure
 * \param result Returned component
 * \param parent Parent component or NULL (root space)
 * \param name Component name in ASCII
 * \param index Component index (from zero)
 * \return zero on success otherwise a negative error code
 */
int
cs_component_find(cs_device_t *d, cs_component_t **result,
	cs_component_t *parent, const char *name, cs_cidx_t index)
{
	cs_component_t *look;

	look = (parent != NULL) ? parent->child : d->first;
	while (look != NULL) {
		if (look->name && !strcmp(look->name, name) && look->index == index) {
			*result = look;
			return 0;
		}
		look = look->next;
	}

	return -ENOENT;
}

/**
 * \brief Find a component space inside active design
 * \param d Combo6 device structure
 * \param parent Parent component or NULL (root space)
 * \param cinfo Basic component information used as a search key
 * \param result Returned space
 * \return zero on success otherwise a negative error code
 */
int
cs_component_find_space(cs_device_t *d, cs_component_t *parent,
	cs_component_info_t *cinfo, cs_space_t **result)
{
	int xerr;
	cs_component_t *comp;

	xerr = cs_component_find(d, &comp, parent, cinfo->name, cinfo->index);
	if (xerr >= 0)
		xerr = cs_component_space(comp, result);

	return xerr;
}

/**
 * \brief Find multiple component spaces inside active design
 * \param d Combo6 device structure
 * \param parent Parent component or NULL (root space)
 * \param cinfo Array of basic component information used as a search key
 * \param result Array of returned spaces (must be already allocated)
 * \param count How many components to find
 * \return zero on success otherwise a negative error code
 */
int
cs_component_find_space_multiple(cs_device_t *d, cs_component_t *parent,
	cs_component_info_t *cinfo, cs_space_t **result, u_int32_t count)
{
	u_int32_t i;
	int xerr;

	for (i = 0; i < count; i++) {
		xerr = cs_component_find_space(d, parent, cinfo + i, result + i);
		if (xerr < 0)
			return xerr;
	}

	return 0;
}

/**
 * \brief Find a space inside active design (simple version)
 * \param d Combo6 device structure
 * \param result Returned component
 * \param id Component ASCII ID (see bellow)
 * \return zero on success otherwise a negative error code
 *
 * Example ID: LUP-0.CAM-0
 * Example ID: A-0.B-0.C-0.D-0.E-0
 */
int
cs_component_sfind(cs_device_t *d, cs_component_t **result, const char *id)
{
	cs_component_t *c = NULL;
	const char *end;
	char name[32];
	cs_cidx_t idx;
	int xerr;

	while (*id != '\0') {
		end = strchr(id, '-');
		if (end == NULL)
			return -EINVAL;
		if ((unsigned int)(end - id) > sizeof(name) - 1)
			return -ENOMEM;
		strncpy(name, id, end - id);
		name[end - id] = '\0';
		if (!isdigit((int)*(++end)))
			return -EINVAL;
		idx = atoi(end);
		while (isdigit((int)*end))
			end++;
		if (*(id = end) == '.')
			id++;
		xerr = cs_component_find(d, &c, c, name, idx);
		if (xerr < 0)
			return xerr;
	}
	*result = c;
	return 0;
}

/**
 * \brief Get component space
 * \param c Component structure
 * \param space Returned assigned space
 * \return zero on success otherwise a negative error code
 */
int
cs_component_space(cs_component_t *c, cs_space_t **space)
{
	*space = c->space;
	if (*space == NULL)
		return -EINVAL;
	return 0;
}

/**
 * \brief Assign an user private value to component
 * \param c Component structure
 * \param upriv User private data
 * \param ufree User private data free callback
 * \return zero on success otherwise a negative error code
 */
int
cs_component_upriv_set(cs_component_t *c, void *upriv, void (*ufree)(void *))
{
	c->upriv_data = upriv;
	c->upriv_free = ufree;
	return 0;
}

/**
 * \brief Get user private value assigned to component
 * \param c Component structure
 * \return user private data
 */
void *
cs_component_upriv_get(cs_component_t *c)
{
	return c->upriv_data;
}

/**
 * \brief Get number of components with specified name
 * \param d Combo6 device structure
 * \param parent Parent component or NULL (root space)
 * \param name Component name
 * \param number Number of founded components
 * \return zero on success otherwise a negative error code
 */
int
cs_component_number(cs_device_t *d, cs_component_t *parent, const char *name, unsigned int *number)
{
	cs_component_t *look;
	*number = 0;
	
	look = (parent != NULL) ? parent->child : d->first;
	while (look != NULL) {
		if (look->name && !strcmp(look->name, name)) {
			(*number)++;
		}
		look = look->next;
	}

	return 0;
}

