/*
 * combo6proc.h: Interface to the Combo6 Linux core.
 * Copyright (C) 2004 CESNET
 * Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#ifndef __COMBO6PROC_H
#define __COMBO6PROC_H

#include <linux/poll.h>

/* buffer for information */
struct combo6_info_buffer {
	char *buffer;		/* pointer to begin of buffer */
	char *curr;		/* current position in buffer */
	unsigned long size;	/* current size */
	unsigned long len;	/* total length of buffer */
	int stop;		/* stop flag */
	int error;		/* error code */
};

struct combo6_info_entry;

struct combo6_info_entry_text {
	unsigned long read_size;
	void (*read) (struct combo6_info_entry *entry,
			struct combo6_info_buffer *buffer);
};

struct combo6_info_entry {
	const char *name;
	mode_t mode;
	long size;
	struct combo6_info_entry_text text;
	struct combo6_info_entry *parent;
	struct combo6 *card;
	struct module *module;
	void *private_data;
	void (*private_free)(struct combo6_info_entry *entry);
	struct proc_dir_entry *p;
	struct mutex access;
};

#ifdef CONFIG_PROC_FS

int combo6_iprintf(struct combo6_info_buffer *buffer, char *fmt, ...)
	__attribute__ ((format (printf, 2, 3)));
int combo6_info_init(void);
int combo6_info_done(void);

struct combo6_info_entry *combo6_info_create_module_entry(struct module *module,
		const char *name, struct combo6_info_entry *parent);
struct combo6_info_entry *combo6_info_create_card_entry(struct combo6 *card,
		const char *name, struct combo6_info_entry *parent);
void combo6_info_free_entry(struct combo6_info_entry *entry);
int combo6_info_store_text(struct combo6_info_entry *entry);
int combo6_info_restore_text(struct combo6_info_entry *entry);

int combo6_info_card_create(struct combo6 *card);
int combo6_info_card_free(struct combo6 *card);
int combo6_info_register(struct combo6_info_entry *entry);
int combo6_info_unregister(struct combo6_info_entry *entry);

/* for card drivers */
int combo6_card_proc_new(struct combo6 *card, const char *name,
		struct combo6_info_entry **entryp);

static inline void combo6_info_set_text_ops(struct combo6_info_entry *entry,
		void *private_data, long read_size,
		void (*read)(struct combo6_info_entry *,
			struct combo6_info_buffer *))
{
	entry->private_data = private_data;
	entry->text.read_size = read_size;
	entry->text.read = read;
}

#else

static inline int combo6_iprintf(struct combo6_info_buffer *buffer,
		char *fmt, ...) { return 0; }
static inline int combo6_info_init(void) { return 0; }
static inline int combo6_info_done(void) { return 0; }

static inline struct combo6_info_entry *combo6_info_create_module_entry(
		struct module *module, const char *name,
		struct combo6_info_entry *parent) { return NULL; }
static inline struct combo6_info_entry *combo6_info_create_card_entry(
		struct combo6 *card, const char *name,
		struct combo6_info_entry *parent) { return NULL; }
static inline void combo6_info_free_entry(struct combo6_info_entry *entry) { }

static inline int combo6_info_card_create(struct combo6 *card) { return 0; }
static inline int combo6_info_card_free(struct combo6 *card) { return 0; }
static inline int combo6_info_register(struct combo6_info_entry *entry)
{ return 0; }
static inline int combo6_info_unregister(struct combo6_info_entry *entry)
{ return 0; }

static inline struct proc_dir_entry *combo6_create_proc_entry(const char *name,
		mode_t mode, struct proc_dir_entry *parent) { return NULL; }
static inline void combo6_remove_proc_entry(struct proc_dir_entry *parent,
		struct proc_dir_entry *de) { }
static inline int combo6_card_proc_new(struct combo6 *card, const char *name,
		struct combo6_info_entry **entryp) { return 0; }
static inline void combo6_info_set_text_ops(struct combo6_info_entry *entry,
		void *private_data, long read_size,
		void (*read)(struct combo6_info_entry *,
			struct combo6_info_buffer *)) { }

#endif

#endif /* __COMBO6PROC_H */
