/*
 * combo6.h: Interface to the Combo6 Linux ioctl() facility.
 * Copyright (C) 2003 CESNET
 * Author(s): Jachym Holecek <freza@liberouter.org>,
 *	      Jaroslav Kysela <perex@perex.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#ifndef __COMBO6IO_H
#define __COMBO6IO_H

#include <linux/types.h>
#include <linux/ioctl.h>
#include <linux/i2c.h>

#define COMBOCTL_MSG_SYSMON	1
#define COMBOCTL_MSG_LINK	2

/**
 * struct comboctl_message - messages passed through /dev/comboctl
 *
 * @card: source of the message (-1 if unknown)
 * @message: the message itself
 */
struct comboctl_message {
	int card;
	int message;
	union {
		char data[56];
	} m;
};

#define COMBOCTL_BASE		'd'
#define COMBOCTL_IOC_VERSION	_IOR(COMBOCTL_BASE, 0, __u8)

/*
 * Ioctl 'target' areas.
 */

#define COMBO6_CPLD_BUS		0
#define COMBO6_PLX_REGS		1 /* Offset must be % 4, sizeof 4, count 1 */
#define COMBO6_BRIDGE_REGS	1 /* more "universal" name than PLX regs */
#define COMBO6_FPGA_BUS	2
#define COMBO6_PLX_EEPROM	3
#define COMBO6_FIRMWARE		4 /* Write-only */
#define COMBO6_PCI_REGS		5

/**
 * struct combo6_ioc_arg - argument for COMBO6_READ/WRITE ioctls
 * @target: COMBO6_PLX_REGS etc.
 * @size: size of one data element
 * @count: number of entities of len sizeof
 *
 * XXX Note that this structure is *NOT* 32/64 bit binary compatible.
 */
struct combo6_ioc_arg {
	int		target;
	int		size;
	u_int64_t	offset;
	u_int64_t	count;
	void		*data;
};

#define COMBO6_BOOT_IMAGES	8

/**
 * struct combo6_boot - argument for COMBO6_IOC_BOOT ioctl
 * @d: images to boot, each contains unit to load to, data size and the data
 *
 * XXX Note that this structure is *NOT* 32/64 bit binary compatible.
 */
struct combo6_boot {
	struct {
		u_int32_t	unit;
		u_int32_t	size;
		u_int8_t	*data;
	} d[COMBO6_BOOT_IMAGES];
};

#define COMBO_DI_IBUF	0
#define COMBO_DI_OBUF	1
#define COMBO_DI_MAX	1

/**
 * struct combo_design_info - for passing information about designs
 *
 * It has two intentions:
 * - to be passed as a chain to combo6_boot.d.data
 * - to ask driver about addresses and sizes of components
 *
 * For the latter case (via COMBO6_IOC_DINFO ioctl), id+idx are IN,
 * addr+size are OUT.
 *
 * @id: one of COMBO_DI_*
 * @idx: index of the component, if applicable (0 otherwise)
 * @addr: address of the component
 * @size: size of the component
 */
struct combo_design_info {
	__u16 id;
	__u16 idx;
	__u32 addr;
	__u32 size;
};

/**
 * struct combo6_eppc_id - info about powerppc, part of combo6_info
 * @h_magic: PPCESNET
 * @h_format: 32-bit format ID
 * @h_text: Cesnet PCI Controller PowerPC Code
 * @i_version: code version
 * @i_text: code text id
 */
struct combo6_eppc_id {
	u_int8_t	h_magic[8];
	u_int32_t	h_format;
	u_int8_t	h_text[116];
	u_int32_t	i_version;
	u_int8_t	i_text[124];
} __attribute__((packed));

#define COMBO6_NAMELEN		16
#define COMBO6_ADDON_CARDS_MAX  2

#define COMBO6_OK_FPGA0		(1<<0)		/* main FPGA is booted */
#define COMBO6_OK_PCRPPC	(1<<1)		/* PCR PPC is booted */

/**
 * struct combo6_info - argument for COMBO6_IOC_INFO ioctl
 * @ci_win_size: FPGA window size
 * @ci_id_sw: 32-bit sw firmware identification
 * @ci_id_hw: 32-bit hw firmware identification
 * @ci_id_txt: firmware identification - null terminated string
 * @ci_fw_ok: firmware ok flags
 * @ci_pcippc: info for PCI bridge PowerPC program
 */
struct combo6_info {
	u_int32_t	ci_win_size;
	char		ci_board_name[COMBO6_NAMELEN];
	char		ci_board_subtype[COMBO6_NAMELEN];
	u_int32_t	ci_id_sw;
	u_int32_t	ci_id_hw;
	char		ci_id_txt[64];
	u_int32_t	ci_fw_ok;
	struct combo6_eppc_id ci_pcippc;
	struct {
		char	card[COMBO6_NAMELEN];
		char	chip[COMBO6_NAMELEN];
	} ci_addon[COMBO6_ADDON_CARDS_MAX];
};

#define COMBO6_I2C_CARD		0U
#define COMBO6_I2C_ADDON(x)	(8U + (x))

/**
 * struct combo6_i2c - argument for COMBO6_IOC_I2C_OP ioctl
 *
 * The behaviour is the same as the one used by i2c-dev (/dev/i2c-*).
 *
 * Example of i2c_msg structure which sets userzone to 0 (see cryptochip man):
 * u8 set_uz[] = { AT88SC_SYSWR_SET_UZ, 0, 0 };
 * struct i2c_msg msg = {
 *	.addr = AT88SC_CMD_SYS_WRITE,
 *	.flags = 0,
 *	.buf = set_uz,
 *	.len = 3,
 * };
 *
 * @controller: which i2c controller to communicate with (COMBO6_I2C_* constant)
 * @nmsg: count of structures in msg
 * @msg: see i2c_msg documentation
 * @filler: placeholder to achieve 32/64-bit compatibility (do not use)
 */
struct combo6_i2c {
	__u32 controller;
	__u32 nmsg;
	union {
		struct i2c_msg __user *msg;
		__u64 filler;
	};
};

#define COMBO6_PTM_OP_INC	0
#define COMBO6_PTM_OP_RT	1
#define COMBO6_PTM_OP_TS	2
#define COMBO6_PTM_OP_PS	3

/**
 * struct combo6_ptm - argument for COMBO6_IOC_PTM_READ/WRITE ioctls
 * @op: COMBO6_PTM_OP_INC etc.
 */
struct combo6_ptm {
	u_int32_t	op;
	u_int32_t	regs[3];
};

/*
 * Ioctl definitions
 */
#define COMBO6_BASE		'c'
#define COMBO6_READ		_IOW(COMBO6_BASE, 0, struct combo6_ioc_arg)
#define COMBO6_WRITE		_IOW(COMBO6_BASE, 1, struct combo6_ioc_arg)
#define COMBO6_IOC_BOOT		_IOWR(COMBO6_BASE, 3, struct combo6_boot)
#define COMBO6_IOC_BUS_DETACH	_IO(COMBO6_BASE, 4)
#define COMBO6_IOC_BUS_ATTACH   _IO(COMBO6_BASE, 5)
#define COMBO6_IOC_INFO		_IOR(COMBO6_BASE, 6, struct combo6_info)
#define COMBO6_IOC_DINFO	_IOWR(COMBO6_BASE, 7, struct combo_design_info)
#define COMBO6_IOC_FWSEL_READ	_IOR(COMBO6_BASE, 192, int)
#define COMBO6_IOC_FWSEL_WRITE	_IOW(COMBO6_BASE, 193, int)
#define COMBO6_IOC_I2C_OP	_IOWR(COMBO6_BASE, 249, struct combo6_i2c)
#define COMBO6_IOC_PTM_READ	_IOWR(COMBO6_BASE, 250, struct combo6_ptm)
#define COMBO6_IOC_PTM_WRITE	_IOW(COMBO6_BASE, 251, struct combo6_ptm)

/*
 * mmap areas
 */
#define COMBO6_MMAP_FPGA	0x00000000	/* FPGA space */

#endif /* __COMBO6IO_H */
