/*
 * combov2.h: Interface to the Combo6 version 2 core
 * Copyright (C) 2006-2009 CESNET
 * Author(s): Jaroslav Kysela <perex@perex.cz>
 *	      Jiri Slaby <jirislaby@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#ifndef __COMBO6V2_H
#define __COMBO6V2_H

#include <linux/types.h>

#include <asm/io.h>

#include "combo6k.h"

/* Firmware revision info. Minor versions preserve API. */
#define PCR_IDENT		0x000000		/* ro */
#define  PCR_ID_IDENT(val)	(((val) >> 16) & 0xffff)
#define  PCR_ID_MAJOR(val)	(((val) >> 8) & 0xff)
#define  PCR_ID_MINOR(val)	((val) & 0xff)

/* PCR_ID_IDENT, or "what's behind the PCI core?". XXX better name */
#define PCR_IDENT_FIRST	0x6d05


/* Firmware build time. Nifty. */
#define PCR_BIRTH		0x000004		/* ro */
#define  BIRTH_MINUTE(val)	((val) & 0xff)		/* 0..59 */
#define  BIRTH_HOUR(val)	(((val) >> 8) & 0xff)	/* 0..24 */
#define  BIRTH_DAY(val)		(((val) >> 16) & 0xff)	/* 1..31 */
#define  BIRTH_MONTH(val)	(((val) >> 24) & 0x0f)	/* 1..12 */
#define  BIRTH_YEAR(val)	(((val) >> 28) & 0x0f)	/* Since 2000 */

#define PCR_FWCONTROL		0x000010	/* rw */
#define  PCR_FWC_BOOT_DELAYED	0x80000000
#define  PCR_FWC_BOOT		0x40000000
#define  PCR_FWC_REC_ERR		0x00400000
#define  PCR_FWC_REC_STAT	0x00200000
#define  PCR_FWC_BOOT_SRC	0x00100000
#define  PCR_FWC_BOOT_DESIGN_S	16
#define  PCR_FWC_BOOT_DESIGN_M	0x000f0000
#define  PCR_FWC_RESET		0x00000001
#define PCR_FWDELAY		0x000014
/* 125 MHz clock, but 0x1000000 granularity (low bits masked) */
#define  PCR_FWD_MS(ms)		roundup((ms) * 125000, 0x1000000)
#define PCR_FWSTAT		0x000018
#define  PCR_FWS_IF_MISSING(x)	(0x01 << ((x) * 8))
#define  PCR_FWS_IF_JTAG(x)	(0x02 << ((x) * 8))
#define  PCR_FWS_IF_BOOTOK(x)	(0x04 << ((x) * 8))

#define PCR_I2C_LXT_CTL		0x000020
#define PCR_I2C_LXT_CMD		0x000024
#define PCR_I2C_IF1_CTL		0x000030
#define PCR_I2C_IF1_CMD		0x000034
#define PCR_I2C_IF2_CTL		0x000038
#define PCR_I2C_IF2_CMD		0x00003c

#define  PCR_I2C_CMD_TX(cmd, tx)	(((tx) << 8) | cmd)
#define  PCR_I2C_CMD_RX(x)	(((x) >> 8) & 0xff)
#define  PCR_I2C_CMD_STAT(x)	((x) & 0xff)

#define  PCR_I2C_CMD_IACK	0x01
#define  PCR_I2C_CMD_ACK		0x08
#define  PCR_I2C_CMD_WR		0x10
#define  PCR_I2C_CMD_RD		0x20
#define  PCR_I2C_CMD_STO		0x40
#define  PCR_I2C_CMD_STA		0x80

#define  PCR_I2C_STAT_IF		0x01
#define  PCR_I2C_STAT_TIP	0x02
#define  PCR_I2C_STAT_AL		0x20
#define  PCR_I2C_STAT_BUSY	0x40
#define  PCR_I2C_STAT_RXACK	0x80

#define PCR_MEM_CMDSTAT		0x000100
#define  PCR_MEM_CMD(cmd, mem, addr)	(((cmd) << 28) | ((mem) << 27) | (addr))

#define  PCR_MEM_CMD_READ	0x0
#define  PCR_MEM_CMD_WRITE	0x1
#define  PCR_MEM_CMD_FLASH	0x4 /* erase sector on FLASH */
#define  PCR_MEM_CMD_INIT_PSRAM	0xF

#define  PCR_MEM_CMD_MEM_PSRAM	0x0
#define  PCR_MEM_CMD_MEM_FLASH	0x1

#define  PCR_MEM_STAT_READY	0x1

#define PCR_MEM_WR		0x000104
#define PCR_MEM_RD		0x000108

#define PCR_IF_CTL(x)		(0x000200 + (x) * 8)
#define  PCR_IF_CTL_BOOT	0x01
#define  PCR_IF_CTL_DRDY	0x02
#define PCR_IF_DATA(x)		(0x000204 + (x) * 8)

/*
 * FPGA space defines
 */

#define COMBOV2_ID_NEG		0x00
#define COMBOV2_ID_SW		0x04
#define COMBOV2_ID_HW		0x08
#define COMBOV2_ID_TEXT		0x20
#define COMBOV2_ID_CHANNELS	0x40
#define COMBOV2_ID_ID_VER	0x48
#define  COMBOV2_ID_VER_MAJ(x)	((x) >> 8)
#define  COMBOV2_ID_VER_MIN(x)	((x) & 0xff)
#define COMBOV2_ID_NETCOPE_VER	0x4A
#define  COMBOV2_NC_VER_MAJ(x)	((x) >> 8)
#define  COMBOV2_NC_VER_MIN(x)	((x) & 0xff)
#define COMBOV2_ID_IRQSTAT	0x54
#define  COMBOV2_IRQS_RX	0x00000001
#define  COMBOV2_IRQS_TX	0x00000002
#define  COMBOV2_IRQS_SYSMON	0x00000004
#define  COMBOV2_IRQS_LINK_CHG	0x00000008
#define COMBOV2_ID_IRQMASKAND	0x78
#define COMBOV2_ID_IRQMASKOR	0x7C

/* combo6->u.three.caps bits */
#define COMBOV2_CAP_MRRS4096	0x00000001 /* max read request size 4096 is OK*/
#define COMBOV2_CAP_NCVER	0x00000002 /* netcope version in ID component */
#define COMBOV2_CAP_IRQSTAT	0x00000004 /* irq status register present */
#define COMBOV2_CAP_GICS	0x00000008 /* HW with GICS, new address space */

#define COMBOV2_BOOTDELAY	500	/* in ms */

/* I2C User zone 0 structure */
struct combov2_i2c_0 {
	u8 id;
	u8 rev;
	u8 ctype;
	u8 cstype;
	u8 hwrev;
	u8 sn[16];
	u8 spgrade;
} __attribute__((packed));

/*
 * initialization parameters
 */
struct combov2_init {
        int invalidmemsize;
        int bootprotect;
        int bootdelay;
        int usemode;
};

/*
 * initialization functions
 */
int combov2_probe(struct pci_dev *pci,
                  const struct pci_device_id *id,
                  struct combov2_init *init,
                  struct module *this_module);
void combov2_remove(struct pci_dev *pci);

/*
 * h/w access primitives
 */
static inline void combo6_br_writeb(struct combo6 *combo6, u32 reg, u8 val)
{
	writeb(val, combo6->mem_virt + reg);
}

static inline u8 combo6_br_readb(struct combo6 *combo6, u32 reg)
{
	return readb(combo6->mem_virt + reg);
}

static inline void combo6_br_writel(struct combo6 *combo6, u32 reg, u32 val)
{
	writel(val, combo6->mem_virt + reg);
}

static inline u32 combo6_br_readl(struct combo6 *combo6, u32 reg)
{
	return readl(combo6->mem_virt + reg);
}

static inline void combo6_cpld_writeb(struct combo6 *combo6, u32 reg, u8 val)
{
	writeb(val, combo6->cmem_virt + reg);
}

static inline u8 combo6_cpld_readb(struct combo6 *combo6, u32 reg)
{
	return readb(combo6->cmem_virt + reg);
}

static inline void combo6_cpld_writel(struct combo6 *combo6, u32 reg, u32 val)
{
	writel(val, combo6->cmem_virt + reg);
}

static inline u32 combo6_cpld_readl(struct combo6 *combo6, u32 reg)
{
	return readl(combo6->cmem_virt + reg);
}

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
extern int combov2_i2c_init(struct combo6 *combo6);
extern void combov2_i2c_exit(struct combo6 *combo6);

extern int combov2_i2c_read_id(struct combo6 *combo6);
extern int combov2_i2c_read_zone(struct combo6 *combo, unsigned int adapi,
		u8 zone, void *buf, unsigned int len);
extern int combov2_i2c_do_op(struct combo6 *combo6,
		struct combo6_i2c __user *op);
#else
static inline int combov2_i2c_init(struct combo6 *combo6) { return 0; }
static inline void combov2_i2c_exit(struct combo6 *combo6) { }

static inline int combov2_i2c_read_id(struct combo6 *combo6)
{
	return -EOPNOTSUPP;
}
static inline int combov2_i2c_read_zone(struct combo6 *combo,
		unsigned int adapi, u8 zone, void *buf, unsigned int len)
{
	return -EOPNOTSUPP;
}
static inline int combov2_i2c_do_op(struct combo6 *combo6,
		struct combo6_i2c __user *op) { return -ENOSYS; }
#endif

#endif /* __COMBO6V2_H */

