/*
 * szedata 2 module
 *
 * Copyright (c) 2008 CESNET
 * Copyright (c) 2007-2008 Jiri Slaby <jirislaby@gmail.com>
 *
 * Licensed under GPLv2
 */

#include <linux/bitops.h>
#include <linux/cdev.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/poll.h>
#include <linux/types.h>

#include "szedata2k.h"
#include "sze2.h"


static struct class *szedata2_class;
static DECLARE_BITMAP(szedata2_minors, SZEDATA2_MAX_MINORS);
static DEFINE_MUTEX(szedata2_lock);

static void szedata2_adjust_timeout(struct szedata2 *sd,
		unsigned int space, unsigned int area_i,
		struct szedata2_area *area)
{
	unsigned long to;

	area->irqs++;

	if (time_after_eq(jiffies, area->to_reset)) {
		/* but first check whether there is irq often enough */
		if (area->irqs >= 10)
			goto reset;			/* 1) normal op */
		to = area->timeout - sd->to_step;	/* 2) irqs too low */
	} else if (area->irqs < 1000)
		return;					/* 3) normal op */
	else {
		to = area->timeout + sd->to_step;	/* 4) irqs too high */
	}

	to = clamp(to, sd->to_min, sd->to_max);

	if (to != area->timeout) {
		pr_debug("%s: %u.%u changing %lu->%lu irqs=%u\n",
				__func__, space, area_i, area->timeout,
				to, area->irqs);
		area->timeout = to;
		sd->set_timeout(sd, space, area_i, to);
	}
reset:
    pr_debug("%s: %u.%u resetting irqs=%u\n", __func__, space, area_i,
			area->irqs);
	area->irqs = 1;
	area->to_reset = jiffies + msecs_to_jiffies(100);
}

static void szedata2_intr_dir(struct szedata2 *sd, unsigned long *bitmap,
		unsigned int space)
{
	struct szedata2_ring *ring = &sd->ring[space];
	struct szedata2_app *app;
	unsigned int bit, rx = space == SZE2_MMIO_RX;
	long head;

	for_each_bit(bit, bitmap, ring->areas * 2) {
		/* odd bits are timeout */
		unsigned int a = bit / 2;
		struct szedata2_area *area = &ring->area[a];

		/*
		 * 1) restart intrs. Must be 1st, before we read ptr. Otherwise
		 * if ptr increased, data stopped and we set intr, we would
		 * lost an interrupt.
		 */
		if(!(sd->state & SZE2_NOADJUSTTIMER)) {
			szedata2_adjust_timeout(sd, space, a, area);
			sd->set_intr(sd, space, a, area->poll_thresh);
		}

		/* 2) update the pointer */
		if (rx) {
			if(sd->state & SZE2_READPTRONLOCK)
			{
				head = sd->get_ptr(sd, space, a, 0);
				if(head == area->u.rx.head) {
					//printk("Empty interrupt\n");
					sd->set_intr(sd, space, a, area->poll_thresh);
				}
				area->u.rx.head = head;
			}
			else
				area->u.rx.head = sd->get_ptr(sd, space, a, 0);
		}
		else
			if(!(sd->state & SZE2_READPTRONLOCK))
				area->u.tx.tail = sd->get_ptr(sd, space, a, 1);
		wmb();

		/* 3) wake apps */
		read_lock(&sd->app_list_lock);
		list_for_each_entry(app, &sd->app_list, list) {
			unsigned long started = app->started[space];
			if (!test_bit(a, &started))
				continue;
			set_bit(rx ? SZE2_WAKE_REASON_RX : SZE2_WAKE_REASON_TX,
					&app->wake_reason);
			wake_up_interruptible(&app->poll_wait);
		}
		read_unlock(&sd->app_list_lock);
	}
}

void szedata2_intr(struct szedata2 *sd, unsigned long status, unsigned long rx,
		unsigned long tx)
{
	szedata2_intr_dir(sd, &rx, SZE2_MMIO_RX);
	szedata2_intr_dir(sd, &tx, SZE2_MMIO_TX);
}
EXPORT_SYMBOL(szedata2_intr);

static void szedata2_missed_irq(unsigned long data)
{
	struct szedata2 *sd = (struct szedata2 *)data;

	szedata2_intr(sd, 0, ~0, ~0);
}

static int szedata2_start_hw(struct szedata2 *sd, unsigned int space,
		unsigned int area_i)
{
	struct szedata2_ring *r = &sd->ring[space];
	struct szedata2_area *area = &r->area[area_i];
	int ret;

	/* already started? */
	if (area->start_count++)
		return 0;

	if (sd->start) {
		ret = sd->start(sd, space, area_i);
		if (ret)
			goto err;
	}

	area->timeout = sd->to_min;
	area->irqs = 1;
	area->to_reset = jiffies + msecs_to_jiffies(100);
	if(!(sd->state & SZE2_NOADJUSTTIMER))
		sd->set_timeout(sd, space, area_i, area->timeout);
	sd->set_intr(sd, space, area_i, area->poll_thresh);

	if (space == SZE2_MMIO_RX) {
		area->u.rx.head = sd->get_ptr(sd, space, area_i, 0);
		area->u.rx.tail = sd->get_ptr(sd, space, area_i, 1);
	} else {
		area->u.tx.head = sd->get_ptr(sd, space, area_i, 0);
		area->u.tx.tail = sd->get_ptr(sd, space, area_i, 1);
	}

	return 0;
err:
	area->start_count--;
	return ret;
}

int szedata2_start_devices(struct szedata2_app *app)
{
	struct szedata2 *sd = app->sd;
	unsigned int a, b, started = 0;
	int ret = 0;

	if (!app->status_page->areas[SZE2_MMIO_RX] &&
			!app->status_page->areas[SZE2_MMIO_TX])
		return -EINVAL;

	mutex_lock(&szedata2_lock);
	for (a = SZE2_MMIO_RX; a <= SZE2_MMIO_TX; a++) {
		struct szedata2_ring *r = &sd->ring[a];
		unsigned long areas = app->status_page->areas[a];
		if (app->started[a] || !areas)
			continue;
		for_each_bit(b, &areas, r->areas) {
			ret = szedata2_start_hw(sd, a, b);
			if (ret)
				continue;
			started++;

			spin_lock(&sd->ptr_lock);
			set_bit(b, &app->started[a]);
			if (a == SZE2_MMIO_RX) {
				app->ptrs[b].rx.head =
				app->ptrs[b].rx.tail = r->area[b].u.rx.tail;
			}
			write_lock_irq(&sd->app_list_lock);
			if (list_empty(&app->list))
				list_add(&app->list, &sd->app_list);
			write_unlock_irq(&sd->app_list_lock);
			spin_unlock(&sd->ptr_lock);
		}
	}
	mutex_unlock(&szedata2_lock);

	if (!started)
		return ret ? : -ENODEV;

	return 0;
}

int szedata2_rxlock_data(struct szedata2_app *app, unsigned long areas)
{
	struct szedata2 *sd = app->sd;
	struct szedata2_ring *r = &sd->ring[SZE2_MMIO_RX];
	struct sze2_adesc *adesc;
	struct szedata2_area *area;
	unsigned long uninitialized_var(head), uninitialized_var(tail);
	unsigned long uninitialized_var(max_h), uninitialized_var(max_t);
	u32 uninitialized_var(max_a);
	u32 s = 0, max_s = 0, a;

	areas &= app->status_page->areas[SZE2_MMIO_RX];
	if (!areas)
		return -EINVAL;
	clear_bit(SZE2_WAKE_REASON_RX, &app->wake_reason);
	rmb();
	for_each_bit(a, &areas, r->areas) {
		area = &r->area[a];
		head = area->u.rx.head;
		if(sd->state & SZE2_READPTRONLOCK)
			head = sd->get_ptr(sd, 0, a, 0);
		tail = app->ptrs[a].rx.tail;
		s = szedata2_tail_head_size(head, tail, area->asize);
		if (s > max_s) {
			max_s = s;
			max_a = a;
			max_t = tail;
			max_h = head;
		}
	}
	if (a >= r->areas && max_s) {
		s = max_s;
		a = max_a;
		tail = max_t;
		head = max_h;
	} else if (!s) {
		a = 0; /* points to zeroed space */
		goto end;
	}

	area = &r->area[a];
	adesc = &app->status_page->adesc[a + 1][SZE2_MMIO_RX];
	adesc->flags = 0;
	adesc->offset = area->roffset + tail;
	adesc->size = s;
	if (s > area->asize - tail) {
		adesc->size -= head;
		adesc->flags |= SZE2_ADESC_FRAG |
				(a + 1 + r->areas) << SZE2_ADESC_NEXT_S;
		adesc = &app->status_page->adesc[a + 1 +
				r->areas][SZE2_MMIO_RX];
		adesc->offset = area->roffset;
		adesc->size = head;
		adesc->flags = 0;
	}

	app->ptrs[a].rx.head = head;
	wmb(); /* we read the value in ISR */
	a++;
end:
	return a;
}

static void szedata2_recompute_rxtail(struct szedata2 *sd, unsigned int area_i)
{
	struct szedata2_app *app;
	struct szedata2_ring *ring = &sd->ring[SZE2_MMIO_RX];
	struct szedata2_area *area = &ring->area[area_i];
	unsigned long rxtail, uninitialized_var(mintail);
	size_t min, m;

	prefetch(area);

	min = ULONG_MAX;
	spin_lock(&sd->ptr_lock);
	rxtail = area->u.rx.tail;

	read_lock(&sd->app_list_lock);
	list_for_each_entry(app, &sd->app_list, list) {
		if (!test_bit(area_i, &app->started[SZE2_MMIO_RX]))
			continue;
		m = szedata2_tail_head_size(app->ptrs[area_i].rx.tail,
				rxtail, area->asize);
		if (m >= min)
			continue;
		min = m;
		mintail = app->ptrs[area_i].rx.tail;
	}
	read_unlock(&sd->app_list_lock);

	if (min != ULONG_MAX && mintail != rxtail) {
		area->u.rx.tail = mintail;
		sd->set_ptr(sd, SZE2_MMIO_RX, area_i, mintail);
	}
	spin_unlock(&sd->ptr_lock);
}

void szedata2_rxunlock_data(struct szedata2_app *app)
{
	struct szedata2 *sd = app->sd;
	unsigned int a;

	for (a = 0; a < sd->ring[SZE2_MMIO_RX].areas; a++) {
		if (!app->status_page->adesc[a + 1][SZE2_MMIO_RX].size)
			continue;

		app->status_page->adesc[a + 1][SZE2_MMIO_RX].size = 0;

		app->ptrs[a].rx.tail = app->ptrs[a].rx.head;
		szedata2_recompute_rxtail(sd, a);
	}
}

/*
 * BIG FAT FIXME Another application may see others' data
 * per application ring buffer needed
 */
int szedata2_txlock_data(struct szedata2_app *app, struct sze2_tx_lock *tx)
{
	struct szedata2 *sd = app->sd;
	struct szedata2_ring *r = &sd->ring[SZE2_MMIO_TX];
	struct sze2_adesc *adesc;
	struct szedata2_area *area;
	unsigned long head;
	size_t s, asize;

	if (!tx->size || tx->area >= r->areas ||
			!test_bit(tx->area, &app->started[SZE2_MMIO_TX]))
		return -EINVAL;

	if (test_and_set_bit(tx->area, sd->tx_locked))
		return -EBUSY;

	clear_bit(SZE2_WAKE_REASON_TX, &app->wake_reason);
	rmb();
	area = &r->area[tx->area];
	asize = area->asize;
	head = area->u.tx.head;
	if(sd->state & SZE2_READPTRONLOCK)
		area->u.tx.tail = sd->get_ptr(sd, 1, tx->area, 1);
	s = szedata2_head_tail_size(head, area->u.tx.tail, asize);
	s = min_t(u32, tx->size, min_t(size_t, s, asize / 4));

	adesc = &app->status_page->adesc[tx->area][SZE2_MMIO_TX];

	if (!s) {
		clear_bit(tx->area, sd->tx_locked);
		if(sd->state & SZE2_READPTRONLOCK)
			sd->set_intr(sd, 0, tx->area, 0);
		goto end;
	}

	app->ptrs[tx->area].tx.size = s;
	adesc->offset = area->roffset + head;

	if (s > asize - head) {
		adesc->size = asize - head;
		s -= adesc->size;
		adesc->flags |= SZE2_ADESC_FRAG |
				(tx->area + r->areas) << SZE2_ADESC_NEXT_S;
		adesc = &app->status_page->adesc[tx->area +
				r->areas][SZE2_MMIO_TX];
		adesc->offset = area->roffset;
	}
	adesc->flags = 0;
end:
	adesc->size = s;
	return 0;
}

int szedata2_txunlock_data(struct szedata2_app *app,
		struct sze2_tx_unlock *tx)
{
	struct szedata2 *sd = app->sd;
	struct szedata2_ring *r = &sd->ring[SZE2_MMIO_TX];
	struct szedata2_area *area;
	unsigned long head;

	if (tx->area >= r->areas)
		return -EINVAL;

	area = &r->area[tx->area];
	prefetch(area);

	if (!app->status_page->adesc[tx->area][SZE2_MMIO_TX].size)
		return -EINVAL;

	app->status_page->adesc[tx->area][SZE2_MMIO_TX].size = 0;
	app->write_size[tx->area] = 0;
	head = area->u.tx.head;
	head += ALIGN(min_t(size_t, tx->size, app->ptrs[tx->area].tx.size), 8);
	head &= area->asize - 1;
	sd->set_ptr(sd, SZE2_MMIO_TX, tx->area, head);
	area->u.tx.head = head;
	wmb();

	clear_bit(tx->area, sd->tx_locked);

	return 0;
}

static void szedata2_stop_hw(struct szedata2 *sd, unsigned int space,
		unsigned int area_i)
{
	struct szedata2_ring *r = &sd->ring[space];
	struct szedata2_area *area = &r->area[area_i];

	/* stop now (i.e. the last one)? */
	if (--area->start_count)
		return;

	if (sd->stop)
		sd->stop(sd, space, area_i);
}

static void __szedata2_stop_devices(struct szedata2_app *app)
{
	struct szedata2 *sd = app->sd;
	unsigned int a, b;

	write_lock_irq(&sd->app_list_lock);
	if (!list_empty(&app->list))
		list_del_init(&app->list);
	write_unlock_irq(&sd->app_list_lock);
	for (a = 0; a < sd->ring[SZE2_MMIO_RX].areas; a++)
		szedata2_recompute_rxtail(sd, a);
	for (a = 0; a < sd->ring[SZE2_MMIO_TX].areas; a++)
		if (app->status_page->adesc[a][SZE2_MMIO_TX].size)
			clear_bit(a, sd->tx_locked);

	for (a = SZE2_MMIO_RX; a <= SZE2_MMIO_TX; a++) {
		struct szedata2_ring *r = &sd->ring[a];
		unsigned long *started = &app->started[a];

		for_each_bit(b, started, r->areas)
			szedata2_stop_hw(sd, a, b);

		*started = 0;
	}
}

void szedata2_stop_devices(struct szedata2_app *app)
{
	mutex_lock(&szedata2_lock);
	__szedata2_stop_devices(app);
	mutex_unlock(&szedata2_lock);
}

struct szedata2_app *szedata2_open(struct szedata2 *sd, unsigned int idx)
{
	struct szedata2_app *app;
	unsigned long off;
	unsigned int a;
	size_t ptrs_size;
	int ret;

	if (mutex_lock_interruptible(&szedata2_lock))
		return ERR_PTR(-ERESTARTSYS);
	if (!test_bit(idx, szedata2_minors) ||
			!(sd->state & SZE2_REGISTERED) ||
			!try_module_get(sd->owner)) {
		ret = -ENODEV;
		goto err_unlock;
	}

	ret = -ENOMEM;
	ptrs_size = sizeof(*app->ptrs) * max(sd->ring[SZE2_MMIO_RX].areas,
				sd->ring[SZE2_MMIO_TX].areas);
	app = kzalloc(sizeof(*app) + ptrs_size, GFP_KERNEL);
	if (app == NULL) {
		printk(KERN_ERR "szedata2: can't alloc app desc\n");
		goto err_dec;
	}
	app->status_page = vmalloc_32_user(PAGE_SIZE);
	if (app->status_page == NULL) {
		printk(KERN_ERR "szedata2: can't alloc app status page\n");
		goto err_freeapp;
	}
	app->write_size = vmalloc_32_user(PAGE_SIZE);
	if (app->write_size == NULL) {
		printk(KERN_ERR "szedata2: can't alloc write size page\n");
		goto err_freeapp;
	}
	app->sd = sd;
	init_waitqueue_head(&app->poll_wait);
	INIT_LIST_HEAD(&app->list);

	if (sd->open_count == 0 && sd->open) {
		ret = sd->open(sd);
		if (ret)
			goto err_freeapp;
	}

	app->status_page->magic = 0xde4d0201;
	off = PAGE_ALIGN(sizeof(*app->status_page)) + PAGE_SIZE;
	for (a = 0; a < SZE2_MMIO_MAX; a++) {
		struct szedata2_ring *r = &sd->ring[a];
		if (a <= SZE2_MMIO_TX)
			app->status_page->areas_available[a] = r->areas;
		app->status_page->offsets[a] = off;
		app->status_page->sizes[a] = r->blk_size * r->blk_count;
		off += app->status_page->sizes[a];
	}

	sd->open_count++;

	mutex_unlock(&szedata2_lock);

	return app;
err_freeapp:
	vfree(app->status_page);
	vfree(app->write_size);
	kfree(app);
err_dec:
	module_put(sd->owner);
err_unlock:
	mutex_unlock(&szedata2_lock);
	return ERR_PTR(ret);
}

void szedata2_close(struct szedata2_app *app)
{
	struct szedata2 *sd = app->sd;
	unsigned int a;

	mutex_lock(&szedata2_lock);

	for (a = 0; a < sd->ring[SZE2_MMIO_TX].areas; a++) {
		struct sze2_tx_unlock unlock;
		if (app->write_size[a]) {
			unlock.area = a;
			unlock.size = app->write_size[a];
			if (printk_ratelimit())
				printk(KERN_INFO "%s: remaining %u bytes "
						"at interface %d\n", __func__,
						unlock.size, a);
			szedata2_txunlock_data(app, &unlock);
		}
	}

	__szedata2_stop_devices(app);

	vfree(app->write_size);
	vfree(app->status_page);
	kfree(app);

	if (--sd->open_count == 0 && sd->close)
		sd->close(sd);

	module_put(sd->owner);

	mutex_unlock(&szedata2_lock);
}

int szedata2_subscribe_area(struct szedata2_app *app,
		struct sze2_subscribe_area *sub)
{
	struct szedata2 *sd = app->sd;
	unsigned int a, b, mask;

	mutex_lock(&szedata2_lock);
	for (a = SZE2_MMIO_RX; a <= SZE2_MMIO_TX; a++) {
		if (app->started[a])
			continue;
		mask = 0;
		for (b = 0; b < sd->ring[a].areas; b++)
			if (sd->ring[a].area[b].asize)
				mask |= BIT(b);
		sub->areas[a] &= mask;
		app->status_page->areas[a] = sub->areas[a];
	}
	mutex_unlock(&szedata2_lock);

	return 0;
}

unsigned int szedata2_poll(struct szedata2_app *app)
{
	struct szedata2 *sd = app->sd;
	unsigned int mask = POLLERR;

	mutex_lock(&szedata2_lock);
	if (!(sd->state & SZE2_REGISTERED))
		goto unlock;
	if (test_and_clear_bit(SZE2_WAKE_REASON_ERR, &app->wake_reason))
		goto unlock;
	if (!app->started[SZE2_MMIO_RX] && !app->started[SZE2_MMIO_TX])
		goto unlock;

	mask = 0;
	if (test_and_clear_bit(SZE2_WAKE_REASON_RX, &app->wake_reason))
		mask |= POLLIN | POLLRDNORM;
	if (test_and_clear_bit(SZE2_WAKE_REASON_TX, &app->wake_reason))
		mask |= POLLOUT | POLLWRNORM;
unlock:
	mutex_unlock(&szedata2_lock);

	return mask;
}

/**
 * szedata2_alloc_dma - alloc DMAable space (lowlevel)
 * @dev: device which will use this DMA area
 * @count: count of members to allocate
 * @size: size of the members
 * @return: array of szedata2_block structure with allocated addresses
 */
struct szedata2_block *szedata2_alloc_dma(struct device *dev,
		unsigned int count, size_t size)
{
	struct szedata2_block *ret;
	unsigned int a;

	ret = vmalloc(count * sizeof(struct szedata2_block));
	if (ret == NULL) {
		ret = ERR_PTR(-ENOMEM);
		goto end;
	}

	pr_debug("allocating %u * %lu size\n", count, (unsigned long)size);
	for (a = 0; a < count; a++) {
		ret[a].virt = dma_alloc_coherent(dev, size, &ret[a].phys,
				GFP_KERNEL);
		if (ret[a].virt == NULL) {
			ret = ERR_PTR(-ENOMEM);
			goto err_ring_pg;
		}
		clear_page(ret[a].virt); /* might go to US! */
	}
end:
	return ret;
err_ring_pg:
	for (; a > 0; a--)
		dma_free_coherent(dev, size, ret[a - 1].virt, ret[a - 1].phys);
	goto end;
}
EXPORT_SYMBOL(szedata2_alloc_dma);

/**
 * szedata2_free_dma - free DMAable space (lowlevel)
 * @dev: device which used this DMA area
 * @blks: array returned by szedata2_alloc_dma
 * @count: count of members in @blks
 * @size: size of members in @blks
 */
void szedata2_free_dma(struct device *dev, struct szedata2_block *blks,
		unsigned int count, size_t size)
{
	unsigned int a;

	for (a = 0; a < count; a++)
		dma_free_coherent(dev, size, blks[a].virt, blks[a].phys);

	vfree(blks);
}
EXPORT_SYMBOL(szedata2_free_dma);

/**
 * szedata2_alloc_dmaspace - allocate DMAable ring buffers
 * @sd: szedata2 structure
 * @space: RX/TX
 * @areas: number of ring buffers to allocate (@count array size)
 * @count: array of block counts
 * @size: block size (must be PAGE_SIZE aligned)
 *
 * Parent device must be set (on szedata2_alloc) and reference to it is raised
 * by one here.
 */
int szedata2_alloc_dmaspace(struct szedata2 *sd, unsigned int space,
		unsigned int areas, unsigned int *count, size_t size,
		unsigned int poll_divisor)
{
	struct szedata2_ring *ring = &sd->ring[space];
	unsigned long roffset;
	unsigned int a;
	int ret;

	if (!areas)
		return 0;

	if (WARN_ON(!sd || IS_ERR(sd) || ring->blk_count ||
				(size & ~PAGE_MASK)))
		return -EINVAL;

	if (sd->parent == NULL)
		return -EINVAL;

	for (a = 0; a < areas; a++)
		ring->blk_count += count[a];

	if (!ring->blk_count)
		return 0;

	ring->area = kzalloc(areas * sizeof(*ring->area), GFP_KERNEL);
	if (!ring->area) {
		ret = -ENOMEM;
		goto err_zero;
	}

	get_device(sd->parent);
	ring->ring = szedata2_alloc_dma(sd->parent, ring->blk_count, size);
	if (IS_ERR(ring->ring)) {
		ret = PTR_ERR(ring->ring);
		goto err_put;
	}

	ring->blk_size = size;
	ring->areas = areas;

	roffset = 0;
	for (a = 0; a < areas; a++) {
		struct szedata2_area *area = &ring->area[a];
		area->asize = size * count[a];
		area->poll_thresh = area->asize / poll_divisor;
		area->roffset = roffset;
		roffset += area->asize;
	}

	return 0;
err_put:
	put_device(sd->parent);
	kfree(ring->area);
err_zero:
	ring->blk_count = 0;
	return ret;
}
EXPORT_SYMBOL(szedata2_alloc_dmaspace);

/**
 * szedata2_free_dmaspace - free DMAable space
 * @sd: szedata2 structure
 * @space: RX/TX
 */
void szedata2_free_dmaspace(struct szedata2 *sd, unsigned int space)
{
	struct szedata2_ring *ring = &sd->ring[space];

	if (!ring->blk_count)
		return;

	pr_debug("freeing %u * %lu memory\n", ring->blk_count,
			(unsigned long)ring->blk_size);
	szedata2_free_dma(sd->parent, ring->ring, ring->blk_count,
			ring->blk_size);

	ring->blk_count = 0;
	ring->areas = 0;
	put_device(sd->parent);
	kfree(ring->area);
}
EXPORT_SYMBOL(szedata2_free_dmaspace);

/**
 * szedata2_get_addr - get virtual and physical address of ring buffer
 * @sd: szedata2 structure
 * @space: RX/TX
 * @area: ring buffer index
 * @virt: pointer to where to store the demanded virtual address
 * @phys: pointer to where to store the demanded physical address
 * @pos: offset of demanded position
 * @len: demanded length
 *
 * Length of space to the next page boundary is returned (if pos is &-ed with
 * PAGE_MASK, then the returned length should be PAGE_SIZE).
 */
size_t szedata2_get_addr(struct szedata2 *sd, unsigned int space,
		unsigned int area, void **virt, dma_addr_t *phys,
		unsigned long pos, size_t len)
{
	struct szedata2_block *blk;
	struct szedata2_ring *ring = &sd->ring[space];
	size_t blk_size = ring->blk_size;
	size_t rest = pos & (blk_size - 1);

	blk = &ring->ring[(pos + ring->area[area].roffset) / blk_size];

	*virt = blk->virt + rest;
	*phys = blk->phys + rest;

	return min(len, blk_size - rest);
}
EXPORT_SYMBOL(szedata2_get_addr);

#if defined(CONFIG_DEVICE_CREATE4) || defined(CONFIG_DEVICE_CREATE5)
static ssize_t szedata2_ring_info_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct szedata2 *sd = dev_get_drvdata(dev);
	const char *sbuf = buf;
	unsigned int ring, area;

	buf += sprintf(buf, "ring areas blk_size blk_count area_sizes...\n");
	for (ring = 0; ring < SZE2_MMIO_MAX; ring++) {
		const struct szedata2_ring *r = &sd->ring[ring];
		switch (ring) {
		case SZE2_MMIO_RX: buf += sprintf(buf, "  RX"); break;
		case SZE2_MMIO_TX: buf += sprintf(buf, "  TX"); break;
		default: buf += sprintf(buf, "%4u", ring); break;
		}
		buf += sprintf(buf, " %5u %8zu %9u",
				r->areas, r->blk_size, r->blk_count);
		for (area = 0; area < r->areas; area++)
			buf += sprintf(buf, " %u=%10zu", area,
					r->area[area].asize);
		buf += sprintf(buf, "\n");
	}

	return buf - sbuf;
}
static DEVICE_ATTR(ring_info, 0444, szedata2_ring_info_show, NULL);

static ssize_t szedata2_area_info_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	struct szedata2 *sd = dev_get_drvdata(dev);
	const char *sbuf = buf;
	unsigned int ring, area;

	for (ring = SZE2_MMIO_RX; ring <= SZE2_MMIO_TX; ring++) {
		const struct szedata2_ring *r = &sd->ring[ring];
		unsigned int rx = ring == SZE2_MMIO_RX;

		buf += sprintf(buf, "%cX\n", rx ? 'R' : 'T');

		for (area = 0; area < r->areas; area++)
			buf += sprintf(buf, "|AREA%.4u         ", area);
		buf += sprintf(buf, "|\n");
		for (area = 0; area < r->areas; area++)
			buf += sprintf(buf, "|head     tail    ");
		buf += sprintf(buf, "|\n");
		for (area = 0; area < r->areas; area++) {
			const struct szedata2_area *a = &r->area[area];
			unsigned long head, tail;

			if (rx) {
				head = a->u.rx.head;
				tail = a->u.rx.tail;
			} else {
				head = a->u.tx.head;
				tail = a->u.tx.tail;
			}

			buf += sprintf(buf, "|%.8lx %.8lx", head, tail);
		}
		buf += sprintf(buf, "|\n");
		for (area = 0; area < r->areas; area++)
			buf += sprintf(buf, "|timeout  irqs/s  ");
		buf += sprintf(buf, "|\n");
		for (area = 0; area < r->areas; area++) {
			const struct szedata2_area *a = &r->area[area];

			buf += sprintf(buf, "|%.8lx %.8x", a->timeout, a->irqs);
		}
		buf += sprintf(buf, "|\n");
	}

	return buf - sbuf;
}
static DEVICE_ATTR(area_info, 0444, szedata2_area_info_show, NULL);

static void szedata2_create_devfiles(struct szedata2 *sd)
{
	if (device_create_file(sd->dev, &dev_attr_ring_info))
		printk(KERN_ERR "szedata2: can't create ring_info file\n");
	if (device_create_file(sd->dev, &dev_attr_area_info))
		printk(KERN_ERR "szedata2: can't create area_info file\n");
}

static void szedata2_remove_devfiles(struct szedata2 *sd)
{
	device_remove_file(sd->dev, &dev_attr_area_info);
	device_remove_file(sd->dev, &dev_attr_ring_info);
}
#else
static inline void szedata2_create_devfiles(struct szedata2 *sd) { }
static inline void szedata2_remove_devfiles(struct szedata2 *sd) { }
#endif

/**
 * szedata2_alloc - allocate new szedata2 structure
 * @private_size: extra size to allocate; pointer to it is set to .private
 * @parent: device which this device is bound to
 *
 * You usually want to call this when new device comes. The you pass this to
 * register after you allocate some spaces and further initialization.
 *
 * Parent is needed when you plan to allocate DMA space through this module.
 *
 * Pointer to new device is returned, or ERR_PTR encoded retval, respectively.
 */
struct szedata2 *szedata2_alloc(unsigned int private_size,
		struct device *parent)
{
	struct szedata2 *sd;
	int ret;

	sd = kzalloc(sizeof(*sd) + private_size, GFP_KERNEL);
	if (sd == NULL) {
		ret = -ENOMEM;
		goto err;
	}

	if (private_size)
		sd->private = sd + 1;
	sd->parent = parent;
	INIT_LIST_HEAD(&sd->app_list);
	setup_timer(&sd->missed_irq, szedata2_missed_irq, (unsigned long)sd);
	spin_lock_init(&sd->ptr_lock);
	rwlock_init(&sd->app_list_lock);
	sd->cdev.owner = THIS_MODULE;

	return sd;
err:
	return ERR_PTR(ret);
}
EXPORT_SYMBOL(szedata2_alloc);

/**
 * szedata2_register - registerr allocate szedata2 device
 * @sd: structure allocated by szedata2_alloc
 *
 * Call this after you setup (owner, hooks, ...) the structure and, if you
 * plan to allocate DMA now, after you do so (on open alternatively).
 *
 * Returns 0 on success, otherwise -errno.
 */
int szedata2_register(struct szedata2 *sd)
{
	int ret, minor;

	if (WARN_ON(!sd || IS_ERR(sd)))
		return -EINVAL;

	if (!sd->owner)
		return -EINVAL;

	mutex_lock(&szedata2_lock);
	if (!sd->to_min || !sd->to_max || sd->state & SZE2_REGISTERED) {
		ret = -EINVAL;
		goto unlock;
	}
	minor = find_first_zero_bit(szedata2_minors, SZEDATA2_MAX_MINORS);
	if (minor >= SZEDATA2_MAX_MINORS) {
		printk(KERN_ERR "szedata2: can't add another device\n");
		ret = -EIO;
		goto unlock;
	}

	ret = szedata2_char_add(sd, minor);
	if (ret)
		goto unlock;

	sd->dev = device_create(szedata2_class, sd->parent, sd->cdev.dev, sd,
			"szedataII%u", minor);
	if (IS_ERR(sd->dev)) {
		printk(KERN_ERR "szedata2: can't create device\n");
		szedata2_char_remove(sd);
		ret = PTR_ERR(sd->dev);
		goto unlock;
	}

	__set_bit(minor, szedata2_minors);

	sd->state |= SZE2_REGISTERED;
	mutex_unlock(&szedata2_lock);

	szedata2_create_devfiles(sd);

	pr_debug("szedata2: registered minor %d\n", minor);

	return 0;
unlock:
	mutex_unlock(&szedata2_lock);
	return ret;
}
EXPORT_SYMBOL(szedata2_register);

/**
 * szedata2_destroy - cleanup szedata2 structure
 * @sd: structure to cleanup
 *
 * It's legal to destroy both registered and unregister structure.
 */
void szedata2_destroy(struct szedata2 *sd)
{
	struct szedata2_app *app;
	unsigned int a;

	if (!sd || IS_ERR(sd))
		return;
	mutex_lock(&szedata2_lock);
	if (sd->state & SZE2_REGISTERED) {
		szedata2_remove_devfiles(sd);
		sd->state &= ~SZE2_REGISTERED; /* racy with register test */
		read_lock(&sd->app_list_lock);
		list_for_each_entry(app, &sd->app_list, list) {
			set_bit(SZE2_WAKE_REASON_ERR, &app->wake_reason);
			wake_up_interruptible(&app->poll_wait);
		}
		read_unlock(&sd->app_list_lock);
		pr_debug("szedata2: unregistering minor %d\n",
				MINOR(sd->cdev.dev));

		while (sd->open_count) { /* wait for openers */
			mutex_unlock(&szedata2_lock);
			msleep(100);
			mutex_lock(&szedata2_lock);
		}

		del_timer_sync(&sd->missed_irq);
		__clear_bit(MINOR(sd->cdev.dev), szedata2_minors);
		device_destroy(szedata2_class, sd->cdev.dev);
		szedata2_char_remove(sd);
	}
	for (a = 0; a < SZE2_MMIO_MAX; a++)
		szedata2_free_dmaspace(sd, a);
	if (sd->destroy)
		sd->destroy(sd);
	mutex_unlock(&szedata2_lock);
	kfree(sd);
}
EXPORT_SYMBOL(szedata2_destroy);

/**
 * szedata2_set_timeout - sets clamp values for adaptive timeout computation
 * @sd: structure allocated by szedata2_alloc
 * @min: the lowest possible value
 * @min: the highest possible value
 *
 * This has to be called after szedata2_alloc and before szedata2_register.
 */
int szedata2_set_timeout(struct szedata2 *sd, unsigned long min,
		unsigned long max)
{
	int ret = -EINVAL;

	if (min > max || !min || !max)
		return ret;

	mutex_lock(&szedata2_lock);
	if (!(sd->state & SZE2_REGISTERED)) {
		sd->to_min = min;
		sd->to_max = max;
		sd->to_step = max(min / 100, 1UL);
		ret = 0;
	}
	mutex_unlock(&szedata2_lock);
	return ret;
}
EXPORT_SYMBOL(szedata2_set_timeout);

static __init int szedata_init(void)
{
	int ret;

	szedata2_class = class_create(THIS_MODULE, "szedata2");
	if (IS_ERR(szedata2_class)) {
		printk(KERN_ERR "szedata2: can't create a szedata2 class\n");
		ret = PTR_ERR(szedata2_class);
		goto err;
	}

	ret = szedata2_char_init();
	if (ret) {
		printk(KERN_ERR "szedata2: can't init char subsys\n");
		goto err_cls;
	}

	return 0;

err_cls:
	class_destroy(szedata2_class);
err:
	return ret;
}

static __exit void szedata_exit(void)
{
	szedata2_char_exit();
	class_destroy(szedata2_class);
}

module_init(szedata_init);
module_exit(szedata_exit);

MODULE_LICENSE("GPL");
MODULE_VERSION("0.1");
MODULE_AUTHOR("Jiri Slaby <jirislaby@gmail.com>");
