/*
 *  szecore.c: szedata linux driver, midlevel file
 *  Copyright (c) 2003-2008 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/mutex.h>
#include <linux/poll.h>
#include <linux/uaccess.h>

#include "szedata.h"
#include "szedatak.h"

#define SZEDATA_MAX_DEVICES 8

MODULE_LICENSE("GPL");
MODULE_AUTHOR("CESNET; Jaroslav Kysela <perex@perex.cz>");
MODULE_DESCRIPTION("SZEDATA Linux Driver");

static int max_apps = 4;
static int check_zero0, szedata_major;
static int poll_threshold = 1;
static int poll_timeout = 200;

module_param(max_apps, int, 0444);
MODULE_PARM_DESC(max_apps, "Maximum concurrent applications for one device "
		"(default = 4).");
module_param(check_zero0, int, 0644);
MODULE_PARM_DESC(check_zero0, "DEBUG: Check if first 32-bit word is zero from "
		"the block.");
module_param(poll_threshold, int, 0644);
MODULE_PARM_DESC(poll_threshold, "Poll threshold value (default = 1)");
module_param(poll_timeout, int, 0644);
MODULE_PARM_DESC(poll_timeout, "Poll timeout value in ms (default = 200ms)");

static struct szedata_device *szedata_devices[SZEDATA_MAX_DEVICES];
static DEFINE_MUTEX(open_lock);
static u32 szedata_pci_alloc_size;
static struct class *szedata_class;

static void szedata_unlock_all(struct szedata_instance *instance);
static void szedata_detach_all(struct szedata_instance *instance);
static int szedata_stop(struct szedata_instance *instance);

/*
 *  szedata driver interface
 */

#if 0 /* alloc test */
static void szedata_debug_alloc(struct szedata_device *dev, const char *prefix)
{
	struct list_head *pos;
	int lused = 0, pused = 0, dused = 0, pfree = 0;
	list_for_each(pos, &dev->locked_pool)
		lused++;
	list_for_each(pos, &dev->used_pool)
		pused++;
	list_for_each(pos, &dev->driver_pool)
		dused++;
	list_for_each(pos, &dev->free_pool)
		pfree++;
	printk(KERN_DEBUG "%s: lused = %i, pused = %i, dused = %i, pfree = %i, "
			"alloc = %i\n", prefix, lused, pused, dused, pfree,
			dev->alloc_blocks);
}
#else
static inline void szedata_debug_alloc(struct szedata_device *dev,
		const char *prefix)
{
}
#endif

int szedata_device_alloc(struct szedata_device **r_dev, int ports, long blocks,
		long block_size, long private_size)
{
	struct szedata_device *dev;
	unsigned int instance_size = max_apps * sizeof(struct szedata_instance);
	int i, size;

	size = sizeof(struct szedata_device) + instance_size + private_size;
	dev = kmalloc(size, GFP_KERNEL);
	if (dev == NULL)
		return -ENOMEM;
	memset(dev, 0, size);
	size = PAGE_ALIGN(blocks *
			sizeof(struct szedata_mmap_block_descriptor));
	dev->desc = vmalloc_user(size);
	if (dev->desc == NULL) {
		kfree(dev);
		return -ENOMEM;
	}
	size = (PAGE_ALIGN(block_size) / PAGE_SIZE) * blocks * sizeof(void *);
	dev->mmap_page_table = vmalloc(size);
	if (dev->mmap_page_table == NULL) {
		vfree(dev->desc);
		kfree(dev);
		return -ENOMEM;
	}
	memset(dev->mmap_page_table, 0, size);
	mutex_lock(&open_lock);
	for (i = 0; i < SZEDATA_MAX_DEVICES; i++)
		if (szedata_devices[i] == NULL) {
			szedata_devices[i] = dev;
			dev->device = i;
			break;
		}
	mutex_unlock(&open_lock);
	if (i >= SZEDATA_MAX_DEVICES) {
		vfree(dev->mmap_page_table);
		vfree(dev->desc);
		kfree(dev);
		return -EBUSY;
	}
	dev->max_apps = max_apps;
	dev->used_ports = ports;
	dev->device = i;
	spin_lock_init(&dev->lock);
	INIT_LIST_HEAD(&dev->free_pool);
	INIT_LIST_HEAD(&dev->used_pool);
	INIT_LIST_HEAD(&dev->locked_pool);
	INIT_LIST_HEAD(&dev->driver_pool);
	INIT_LIST_HEAD(&dev->alloc_pages);
	for (i = 0; i < max_apps; i++) {
		INIT_LIST_HEAD(&dev->appinst[i].apps);
		INIT_LIST_HEAD(&dev->appinst[i].apps_locked);
	}
	if (private_size > 0)
		dev->private_data = (void *)(((char *)(dev + 1)) +
				instance_size);
	*r_dev = dev;
	return 0;
}
EXPORT_SYMBOL(szedata_device_alloc);

int szedata_device_free(struct szedata_device *dev)
{
	mutex_lock(&open_lock);
	szedata_devices[dev->device] = NULL;
	mutex_unlock(&open_lock);
	vfree(dev->mmap_page_table);
	vfree(dev->desc);
	kfree(dev);
	return 0;
}
EXPORT_SYMBOL(szedata_device_free);

static struct szedata_block *szedata_device_alloc_block(
		struct szedata_device *dev)
{
	struct szedata_block *block;
	int i;

	block = kmalloc(sizeof(struct szedata_block) +
			sizeof(struct list_head) * max_apps, GFP_KERNEL);
	if (block == NULL)
		return NULL;
	memset(block, 0, sizeof(*block));
	block->desc = &dev->desc[dev->desc_init++];
	for (i = 0; i < max_apps; i++)
		INIT_LIST_HEAD(&block->apps[i]);
	list_add_tail(&block->list, &dev->free_pool);
	return block;
}

struct szedata_pci_block {
	struct list_head list;
	char *virt;
	dma_addr_t phys;
	long size;
};

void szedata_device_free_pci(struct szedata_device *dev, struct pci_dev *pci)
{
	struct szedata_block *block, *b;
	struct szedata_pci_block *pblock, *pb;

	if (!list_empty(&dev->driver_pool))
		printk(KERN_ERR "szedata: free memory - driver pool is not "
				"empty!\n");
	if (!list_empty(&dev->used_pool))
		printk(KERN_ERR "szedata: free memory - used pool is not "
				"empty!\n");
	if (!list_empty(&dev->locked_pool))
		printk(KERN_ERR "szedata: free memory - locked pool is not "
				"empty!\n");
	list_for_each_entry_safe(block, b, &dev->free_pool, list) {
		list_del(&block->list);
		kfree(block);
	}
	list_for_each_entry_safe(pblock, pb, &dev->alloc_pages, list) {
		list_del(&pblock->list);
		dma_free_coherent(&pci->dev, pblock->size, pblock->virt,
				pblock->phys);
		szedata_pci_alloc_size -= pblock->size;
		kfree(pblock);
	}
	dev->alloc_blocks = 0;
	dev->alloc_block_size = 0;
	dev->alloc_mmap_pages = 0;
}
EXPORT_SYMBOL(szedata_device_free_pci);

int szedata_device_alloc_pci(struct szedata_device *dev, struct pci_dev *pci,
			    long blocks, long block_size)
{
	long size, size1, block = 0;
	unsigned long mmap_page = 0, ofs;
	char *res;
	dma_addr_t dma_addr;
	struct szedata_block *pkt;
	struct szedata_pci_block *pblock;

	dev->desc_init = 0;
	size = 128 * 1024;
	while (block < blocks && size >= block_size) {
		do {
			res = dma_alloc_coherent(&pci->dev, size, &dma_addr,
					GFP_KERNEL);
			if (res == NULL)
				break;
			szedata_pci_alloc_size += size;
			pblock = kmalloc(sizeof(struct szedata_pci_block),
					GFP_KERNEL);
			if (pblock == NULL) {
				szedata_pci_alloc_size -= size;
				dma_free_coherent(&pci->dev, size, res,
						dma_addr);
				goto __nomem;
			}
			memset(res, 0, size);
			pblock->virt = res;
			pblock->phys = dma_addr;
			pblock->size = size;
			list_add_tail(&pblock->list, &dev->alloc_pages);
			size1 = size;
			ofs = 0;
			do {
				pkt = szedata_device_alloc_block(dev);
				if (pkt == NULL)
					goto __nomem;
				pkt->virt = res + ofs;
				pkt->phys = dma_addr + ofs;
				pkt->desc->block = mmap_page * PAGE_SIZE + ofs;
				ofs += block_size;
				size1 -= block_size;
				block++;
			} while (block < blocks && size1 >= block_size);
			size1 = size;
			ofs = 0;
			do {
				dev->mmap_page_table[mmap_page] = res + ofs;
				ofs += PAGE_SIZE;
				size1 -= PAGE_SIZE;
				mmap_page++;
			} while (size1 > 0);
		} while (block < blocks);
		size >>= 1;
	}
__nomem:
	if (block < blocks) {
		szedata_device_free_pci(dev, pci);
		return -ENOMEM;
	}
	dev->alloc_blocks = blocks;
	dev->alloc_block_size = block_size;
	dev->alloc_mmap_pages = mmap_page * PAGE_SIZE;
	return 0;
}
EXPORT_SYMBOL(szedata_device_alloc_pci);

int szedata_device_register(struct szedata_device *dev, struct module *module)
{
	mutex_lock(&open_lock);
	dev->owner = module;
	dev->registered = 1;
	szedata_info_device_create(dev);
	device_create(szedata_class, NULL, MKDEV(szedata_major, dev->device),
			NULL, "szedata%d", dev->device);
	mutex_unlock(&open_lock);
	return 0;
}
EXPORT_SYMBOL(szedata_device_register);

int szedata_device_unregister(struct szedata_device *dev)
{
	mutex_lock(&open_lock);
	dev->registered = 0;
	device_destroy(szedata_class, MKDEV(szedata_major, dev->device));
	szedata_info_device_free(dev);
	mutex_unlock(&open_lock);
	return szedata_device_free(dev);
}
EXPORT_SYMBOL(szedata_device_unregister);

struct szedata_block *szedata_device_get_block(struct szedata_device *dev,
						u_int32_t app_list,
						u_int32_t port)
{
	struct list_head *pos;
	struct szedata_block *block;
	int i;

	app_list &= dev->running;
	if (list_empty(&dev->free_pool)) {
		if (list_empty(&dev->used_pool)) {
			dev->alloc_failed++;
			return NULL;
		}
		pos = dev->used_pool.next;
		block = list_entry(pos, struct szedata_block, list);
		for (i = 0; i < max_apps; i++) {
			if (!list_empty(&block->apps[i])) {
				list_del_init(&block->apps[i]);
				if (dev->appinst[i].apps_count-- == 1) {
					del_timer(&dev->appinst[i].timer);
					dev->appinst[i].poll_timeouted = 0;
				}
				block->app_count--;
			}
		}
		if (block->app_count != 0)
			printk(KERN_ERR "szedata: oops, block->app_count is "
					"not zero in get_block?\n");
		dev->alloc_over++;
		for (i = 0; i < max_apps; i++)
			if (app_list & (1 << i))
				dev->appinst[i].istats->s[port].overwritten++;
	} else {
		pos = dev->free_pool.next;
		block = list_entry(pos, struct szedata_block, list);
	}
	list_move(pos, &dev->driver_pool);
	if (!dev->get_timestamp)
		block->desc->timestamp = 0;
	return block;
}
EXPORT_SYMBOL(szedata_device_get_block);

void szedata_device_put_block(struct szedata_device *dev,
			      struct szedata_block *block,
			      u_int32_t app_list)
{
	unsigned int i;
	struct szedata_instance *instance;
	struct szedata_interface_stats *stats;

	app_list &= dev->running;
	if (app_list == 0) {
		szedata_device_detach_block(dev, block);
		return;
	}
	block->app_count = 0;
	list_move_tail(&block->list, &dev->used_pool);
	for (i = 0; i < max_apps; i++) {
		if ((app_list & (1 << i)) == 0)
			continue;
		instance = &dev->appinst[i];
		if ((instance->active_ports &
					(1 << block->desc->if_index)) == 0)
			continue;
		list_add_tail(&block->apps[i], &instance->apps);
		if (instance->apps_count++ == 0) {
			instance->timer.expires = jiffies +
				instance->poll_timeout;
			add_timer(&instance->timer);
		}
		if (instance->apps_peak_count < instance->apps_count)
			instance->apps_peak_count = instance->apps_count;
		block->app_count++;
		if (instance->poll_threshold <= instance->apps_count)
			wake_up(&instance->dma_sleep);
		/* process statistics here */
		stats = &instance->istats->s[block->desc->if_index];
		stats->bytes += block->desc->size;
		stats->count++;
		if (block->desc->status == SZEDATA_BLOCK_STATUS_RXERR)
			stats->crc_errors++;
	}
	if (check_zero0 && *((u32 *)block->virt) == 0)
		printk(KERN_INFO "szedata: first 32-bit word is zero!\n");
}
EXPORT_SYMBOL(szedata_device_put_block);

void szedata_device_detach_block(struct szedata_device *dev,
		struct szedata_block *block)
{
	list_move(&block->list, &dev->free_pool);
	block->app_count = 0;
}
EXPORT_SYMBOL(szedata_device_detach_block);

void szedata_length_error(struct szedata_device *dev,
			  u_int32_t app_list, u_int32_t ifc, u_int32_t len)
{
	unsigned int i;
	struct szedata_instance *instance;
	struct szedata_interface_stats *stats;

	app_list &= dev->running;
	for (i = 0; i < max_apps; i++)
		if (app_list & (1 << i)) {
			instance = &dev->appinst[i];
			stats = &instance->istats->s[ifc];
			stats->length_errors++;
			stats->bytes += len;
			stats->count++;
		}
}
EXPORT_SYMBOL(szedata_length_error);

void szedata_block_dropped(struct szedata_device *dev,
			   u_int32_t app_list, u_int32_t ifc, u_int32_t len)
{
	unsigned int i;
	struct szedata_instance *instance;
	struct szedata_interface_stats *stats;

	app_list &= dev->running;
	for (i = 0; i < max_apps; i++)
		if (app_list & (1 << i)) {
			instance = &dev->appinst[i];
			stats = &instance->istats->s[ifc];
			stats->dropped++;
			stats->bytes += len;
			stats->count++;
		}
}
EXPORT_SYMBOL(szedata_block_dropped);

/*
 *  /dev/szedata/X devices
 */

static void szedata_timeout(unsigned long arg)
{
	struct szedata_instance *instance = (struct szedata_instance *)arg;

	instance->poll_timeouted = 1;
	wake_up(&instance->dma_sleep);

}

static inline struct szedata_instance *szedata_find_instance(
		struct szedata_device *dev, int instance)
{
	if (dev->active_apps & (1 << instance))
		return &dev->appinst[instance];
	return NULL;
}

static void szedata_free_instance(struct szedata_instance *instance)
{
	vfree(instance->app_lock);
	instance->app_lock = NULL;
	vfree(instance->istats);
	instance->istats = NULL;
	vfree(instance->status);
	instance->status = NULL;
}

static int szedata_open(struct inode *inode, struct file *file)
{
	unsigned int minor = ciminor(inode);
	struct szedata_device *dev;
	struct szedata_instance *instance;
	int retval, idx;

	if (minor >= SZEDATA_MAX_DEVICES)
		return -ENODEV;
	mutex_lock(&open_lock);
	dev = szedata_devices[minor];
	if (dev == NULL || !try_module_get(dev->owner)) {
		mutex_unlock(&open_lock);
		return -ENODEV;
	}
	if (!dev->registered) {
		module_put(dev->owner);
		mutex_unlock(&open_lock);
		return -ENODEV;
	}
	if (dev->instances >= max_apps) {
		module_put(dev->owner);
		mutex_unlock(&open_lock);
		return -ENODEV;
	}
	dev->instances++;
	if (dev->instances == 1) {
		retval = dev->open(dev);
		if (retval < 0) {
			dev->instances--;
			module_put(dev->owner);
			mutex_unlock(&open_lock);
			return retval;
		}
	}

	for (idx = 0; idx < max_apps; idx++) {
		if (szedata_find_instance(dev, idx) == NULL)
			break;
	}

	instance = &dev->appinst[idx];
	init_waitqueue_head(&instance->dma_sleep);
	instance->app_lock = vmalloc_user(PAGE_ALIGN(
				sizeof(struct szedata_mmap_app_lock)));
	instance->istats = vmalloc_user(PAGE_ALIGN(
				sizeof(struct szedata_mmap_block_istats)));
	instance->status = vmalloc_user(PAGE_ALIGN(
				sizeof(struct szedata_mmap_block_status)));
	if (instance->app_lock == NULL ||
	    instance->istats == NULL ||
	    instance->status == NULL) {
		dev->instances--;
		module_put(dev->owner);
		szedata_free_instance(instance);
		mutex_unlock(&open_lock);
		return -ENOMEM;
	}
	instance->apps_peak_count = 0;
	instance->poll_threshold = poll_threshold;
	instance->poll_timeouted = 0;
	instance->poll_timeout = (HZ * poll_timeout) / 1000;
	init_timer(&instance->timer);
	instance->timer.expires = jiffies + instance->poll_timeout;
	instance->timer.function = szedata_timeout;
	instance->timer.data = (unsigned long)instance;
	instance->active_ports = 0;
	instance->inum = idx;
	instance->dev = dev;
	dev->active_apps |= 1 << idx;
	mutex_unlock(&open_lock);
	file->private_data = instance;
	szedata_debug_alloc(dev, "open");
	return 0;
}

static int szedata_release(struct inode *inode, struct file *file)
{
	struct szedata_instance *instance;
	struct szedata_device *dev;

	instance = (struct szedata_instance *)file->private_data;
	if (instance != NULL) {
		dev = instance->dev;
		szedata_stop(instance);
		mutex_lock(&open_lock);
		szedata_device_lock_irq(dev);
		dev->active_apps &= ~(1 << instance->inum);
		szedata_unlock_all(instance);
		szedata_detach_all(instance);
		szedata_debug_alloc(dev, "release");
		szedata_device_unlock_irq(dev);
		dev->instances--;
		if (dev->instances == 0)
			dev->close(dev);
		module_put(dev->owner);
		szedata_free_instance(instance);
		mutex_unlock(&open_lock);
	}
	return 0;
}

static int szedata_get_interface(struct szedata_instance *instance,
		struct szedata_ioctl_get_interface __user *arg)
{
	struct szedata_ioctl_get_interface intf;
	struct szedata_device *dev;
	int retval;

	if (copy_from_user(&intf, arg, sizeof(intf)))
		return -EFAULT;
	mutex_lock(&open_lock);
	dev = instance->dev;
	if (dev != NULL)
		retval = dev->port_info(dev, &intf);
	else
		retval = -ENODEV;
	mutex_unlock(&open_lock);
	if (retval == 0 && copy_to_user(arg, &intf, sizeof(intf)))
		return -EFAULT;
	return retval;
}

static int szedata_subscribe_interface1(struct szedata_instance *instance,
		struct szedata_ioctl_subscribe_interface *sintf)
{
	struct szedata_device *dev;
	u_int32_t active_ports = 0;
	int i;

	dev = instance->dev;
	for (i = 0; i < SZEDATA_MAX_INTERFACES; i++) {
		if (sintf->if_index[i] == 0xff)
			break;
		if (sintf->if_index[i] >= dev->used_ports)
			return -EINVAL;
		active_ports |= 1 << sintf->if_index[i];
	}
	szedata_device_lock_irq(dev);
	instance->active_ports |= active_ports;
	if (sintf->poll_threshold < 1)
		sintf->poll_threshold = 1;
	else if (sintf->poll_threshold > (dev->alloc_blocks / 2))
		sintf->poll_threshold = dev->alloc_blocks / 2;
	instance->poll_threshold = sintf->poll_threshold;
	szedata_device_unlock_irq(dev);
	return 0;
}

static int szedata_subscribe_interface(struct szedata_instance *instance,
		struct szedata_ioctl_subscribe_interface __user *arg)
{
	struct szedata_ioctl_subscribe_interface sintf;

	if (copy_from_user(&sintf, arg, sizeof(sintf)))
		return -EFAULT;
	return szedata_subscribe_interface1(instance, &sintf);
}

static int szedata_subscribe_interface_old(struct szedata_instance *instance,
		struct szedata_ioctl_subscribe_interface_old __user *arg)
{
	struct szedata_ioctl_subscribe_interface_old sointf;
	struct szedata_ioctl_subscribe_interface sintf;

	if (copy_from_user(&sointf, arg, sizeof(sointf)))
		return -EFAULT;
	memcpy(&sintf.if_index, &sointf.if_index, SZEDATA_MAX_INTERFACES);
	sintf.poll_threshold = 1;
	return szedata_subscribe_interface1(instance, &sintf);
}

static int szedata_unsubscribe_interface(struct szedata_instance *instance,
		struct szedata_ioctl_subscribe_interface __user *arg)
{
	struct szedata_device *dev;
	struct szedata_ioctl_subscribe_interface sintf;
	u_int32_t active_ports = 0;
	int i;

	if (copy_from_user(&sintf, arg, sizeof(sintf)))
		return -EFAULT;
	dev = instance->dev;
	for (i = 0; i < SZEDATA_MAX_INTERFACES; i++) {
		if (sintf.if_index[i] == 0xff)
			break;
		if (sintf.if_index[i] >= dev->used_ports)
			return -EINVAL;
		active_ports |= 1 << sintf.if_index[i];
	}
	szedata_device_lock_irq(dev);
	instance->active_ports &= ~active_ports;
	szedata_device_unlock_irq(dev);
	return 0;
}

static int szedata_rbuffer(struct szedata_instance *instance,
		struct szedata_ioctl_rbuffer __user *arg)
{
	struct szedata_device *dev;
	struct szedata_ioctl_rbuffer rbuf;

	dev = instance->dev;
	rbuf.blocks = dev->alloc_blocks;
	rbuf.block_size = dev->alloc_block_size;
	rbuf.dma_size = dev->alloc_mmap_pages;
	if (copy_to_user(arg, &rbuf, sizeof(rbuf)))
		return -EFAULT;
	return 0;
}

static int szedata_start(struct szedata_instance *instance)
{
	struct szedata_device *dev;
	int err = 0;

	dev = instance->dev;
	szedata_device_lock_irq(dev);
	if (dev->running == 0) {
		err = dev->start(dev);
		if (err >= 0)
			dev->running |= 1 << instance->inum;
	} else {
		dev->running |= 1 << instance->inum;
	}
	szedata_device_unlock_irq(dev);
	return err;
}

static int szedata_stop(struct szedata_instance *instance)
{
	struct szedata_device *dev;
	int err = 0;

	dev = instance->dev;
	szedata_device_lock_irq(dev);
	if (dev->running) {
		dev->running &= ~(1 << instance->inum);
		if (dev->running == 0) {
			err = dev->stop(dev);
			if (err < 0)
				dev->running |= 1 << instance->inum;
		}
	}
	szedata_device_unlock_irq(dev);
	return err;
}

static int szedata_lock_one(struct szedata_instance *instance, int idx)
{
	struct szedata_device *dev = instance->dev;
	struct szedata_block *block;
	unsigned int inum = instance->inum;

	if (list_empty(&instance->apps))
		return -ENOENT;
	block = list_entry(instance->apps.next, struct szedata_block,
			apps[inum]);
	list_move_tail(&block->apps[inum], &instance->apps_locked);
	if (instance->apps_count-- == 1) {
		del_timer_sync(&instance->timer);
		/* for sure, timer callback might be called in the middle of
		   processing */
		instance->poll_timeouted = 0;
	}
	block->app_count--;
	if (block->locked_count++ == 0)
		list_move(&block->list, &dev->locked_pool);
	instance->app_lock->blk[idx] = (char *)block->desc - (char *)dev->desc;
	instance->apps_slocked++;
	return 0;
}

static void szedata_unlock_all(struct szedata_instance *instance)
{
	struct szedata_device *dev = instance->dev;
	struct szedata_block *block, *b;
	unsigned int inum = instance->inum;
	int i = 0;

	instance->app_lock->count = 0;
	list_for_each_entry_safe(block, b, &instance->apps_locked, apps[inum]) {
		if (i >= SZEDATA_MAX_PKTLOCK)
			panic("Oops, szedata - trying to unlock more than %i "
					"blocks!\n", i);
		list_del_init(&block->apps[inum]);
		if (--block->locked_count == 0)
			list_move(&block->list, block->app_count ?
					&dev->used_pool : &dev->free_pool);
		instance->app_lock->blk[i++] = 0;
		instance->apps_sunlocked++;
	}
}

static void szedata_detach_all(struct szedata_instance *instance)
{
	struct szedata_device *dev = instance->dev;
	struct szedata_block *block;
	unsigned int inum = instance->inum;

	del_timer_sync(&instance->timer);
	while (!list_empty(&instance->apps)) {
		block = list_entry(instance->apps.next, struct szedata_block,
				apps[inum]);
		list_del_init(&block->apps[inum]);
		instance->apps_count--;
		if (block->app_count == 0) {
			printk(KERN_ERR "szedata: oops, block apps_count "
					"zero?\n");
		} else {
			if (--block->app_count == 0 && block->locked_count == 0)
				list_move(&block->list, &dev->free_pool);
		}
	}
	if (instance->apps_count != 0) {
		printk(KERN_ERR "szedata: oops, apps_count is not zero after "
				"detach!\n");
		instance->apps_count = 0;
	}
}

static int szedata_lock_next(struct szedata_instance *instance)
{
	struct szedata_device *dev;
	int retval;

	dev = instance->dev;
	szedata_device_lock_irq(dev);
	instance->poll_timeouted = 0;
	szedata_unlock_all(instance);
	retval = szedata_lock_one(instance, 0);
	if (retval >= 0)
		instance->app_lock->count = 1;
	szedata_device_unlock_irq(dev);
	return retval;
}

static int szedata_lock_multi(struct szedata_instance *instance,
		u_int32_t blocks)
{
	struct szedata_device *dev;
	int retval = 0, i;

	if (blocks > SZEDATA_MAX_PKTLOCK)
		return -EINVAL;
	dev = instance->dev;
	szedata_device_lock_irq(dev);
	instance->poll_timeouted = 0;
	szedata_unlock_all(instance);
	for (i = 0; i < blocks; i++) {
		retval = szedata_lock_one(instance, i);
		if (retval < 0) {
			if (i > 0)
				retval = 0;
			break;
		}
	}
	if (i > 0 && retval >= 0)
		instance->app_lock->count = i;
	szedata_device_unlock_irq(dev);
	return retval;
}

static int szedata_unlock(struct szedata_instance *instance)
{
	struct szedata_device *dev;

	dev = instance->dev;
	szedata_device_lock_irq(dev);
	szedata_unlock_all(instance);
	szedata_device_unlock_irq(dev);
	return 0;
}

static int szedata_get_stats(struct szedata_instance *instance,
		struct szedata_ioctl_stats __user *arg)
{
	struct szedata_device *dev;
	struct szedata_ioctl_stats stats;

	dev = instance->dev;
	szedata_device_lock_irq(dev);
	if (dev->get_timestamp)
		dev->get_timestamp(&stats.timestamp);
	else
		stats.timestamp = 0;
	memcpy(&stats.s[0], instance->istats,
			sizeof(struct szedata_interface_stats) *
			dev->used_ports);
	memset(instance->istats, 0, sizeof(struct szedata_interface_stats) *
			dev->used_ports);
	szedata_device_unlock_irq(dev);
	if (copy_to_user(arg, &stats, sizeof(stats)))
		return -EFAULT;
	return 0;
}

static long szedata_ioctl(struct file *file, unsigned int cmd,
		unsigned long arg)
{
	static DEFINE_MUTEX(ioctl_lock);
	struct szedata_instance *instance = file->private_data;
	void __user *argp = (void __user *)arg;
	int ret = 0;

	pr_debug("szedata_ioctl: 0x%x\n", cmd);
	mutex_lock(&ioctl_lock);
	switch (cmd) {
	case SZEDATA_IOCTL_VERSION:
		if (put_user(SZEDATA_CUR_VERSION, (int __user *)argp))
			ret = -EFAULT;
		break;
	case SZEDATA_IOCTL_APP_NUMBER:
		if (put_user(instance->inum, (int __user *)argp))
			ret = -EFAULT;
		break;
	case SZEDATA_IOCTL_INTERFACES:
		if (put_user(instance->dev->used_ports, (int __user *)argp))
			ret = -EFAULT;
		break;
	case SZEDATA_IOCTL_GET_INTERFACE:
		ret = szedata_get_interface(instance, argp);
		break;
	case SZEDATA_IOCTL_SUBSCRIBE_INTERFACE:
		ret = szedata_subscribe_interface(instance, argp);
		break;
	case SZEDATA_IOCTL_SUBSCRIBE_INTERFACEO:
		ret = szedata_subscribe_interface_old(instance, argp);
		break;
	case SZEDATA_IOCTL_UNSUBSCRIBE_INTERFACE:
		ret = szedata_unsubscribe_interface(instance, argp);
		break;
	case SZEDATA_IOCTL_RBUFFER:
		ret = szedata_rbuffer(instance, argp);
		break;
	case SZEDATA_IOCTL_START:
		ret = szedata_start(instance);
		break;
	case SZEDATA_IOCTL_STOP:
		ret = szedata_stop(instance);
		break;
	case SZEDATA_IOCTL_LOCK_NEXT:
		ret = szedata_lock_next(instance);
		break;
	case SZEDATA_IOCTL_LOCK_MULTI: {
		u_int32_t val;
		if (get_user(val, (u_int32_t __user *)argp)) {
			ret = -EFAULT;
			break;
		}
		ret = szedata_lock_multi(instance, val);
		break;
	}
	case SZEDATA_IOCTL_UNLOCK:
		ret = szedata_unlock(instance);
		break;
	case SZEDATA_IOCTL_GET_STATS:
		ret = szedata_get_stats(instance, argp);
		break;
	default:
		ret = -ENOTTY;
	}
	mutex_unlock(&ioctl_lock);

	return ret;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 18)

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 23)
static int szedata_mmap_pkt_fault(struct vm_area_struct *vma,
		struct vm_fault *vmf)
{
	struct szedata_device *dev = vma->vm_private_data;
	unsigned long offset = vmf->pgoff << PAGE_SHIFT;
	struct page *page;

	offset -= SZEDATA_MMAP_OFFSET_PKT;
	if (offset > dev->alloc_mmap_pages - PAGE_SIZE)
		return VM_FAULT_SIGBUS;
	page = virt_to_page(dev->mmap_page_table[offset / PAGE_SIZE]);
	get_page(page);
	vmf->page = page;

	return 0;
}

static struct vm_operations_struct szedata_vm_ops_pkt = {
	.fault = szedata_mmap_pkt_fault
};

static int szedata_pkt_mmap(struct szedata_device *device,
		struct vm_area_struct *vma)
{
	unsigned long size = vma->vm_end - vma->vm_start;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;

	if (size > PAGE_ALIGN(device->alloc_mmap_pages))
		return -EINVAL;
	if (offset - SZEDATA_MMAP_OFFSET_PKT >
			PAGE_ALIGN(device->alloc_mmap_pages) - size)
		return -EINVAL;

	vma->vm_ops = &szedata_vm_ops_pkt;
	vma->vm_private_data = device;

	return 0;
}
#endif /* >= 2.6.23 */

static int szedata_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct szedata_instance *instance = file->private_data;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;

	if ((vma->vm_flags & (VM_WRITE|VM_READ)) != VM_READ)
		return -EINVAL;

	if (offset >= SZEDATA_MMAP_OFFSET_PKT)
		return szedata_pkt_mmap(instance->dev, vma);
	else if (offset >= SZEDATA_MMAP_OFFSET_PKTDSC)
		return remap_vmalloc_range(vma, instance->dev->desc,
			(offset - SZEDATA_MMAP_OFFSET_PKTDSC) >> PAGE_SHIFT);
	else if (offset >= SZEDATA_MMAP_OFFSET_STATUS)
		return remap_vmalloc_range(vma, instance->status,
			(offset - SZEDATA_MMAP_OFFSET_STATUS) >> PAGE_SHIFT);
	else if (offset >= SZEDATA_MMAP_OFFSET_ISTATS)
		return remap_vmalloc_range(vma, instance->istats,
			(offset - SZEDATA_MMAP_OFFSET_ISTATS) >> PAGE_SHIFT);
	else
		return remap_vmalloc_range(vma, instance->app_lock,
			vma->vm_pgoff);
}
#endif /* >= 2.6.18 */

static unsigned int szedata_poll(struct file *file, poll_table * wait)
{
	struct szedata_instance *instance = file->private_data;
	unsigned int mask;

	poll_wait(file, &instance->dma_sleep, wait);
	szedata_device_lock_irq(instance->dev);
	if (instance->poll_threshold <= instance->apps_count ||
	    (instance->poll_timeouted && instance->apps_count)) {
		mask = POLLIN | POLLRDNORM;
	} else if ((instance->dev->running & (1 << instance->inum)) == 0) {
		mask = POLLIN | POLLRDNORM | POLLERR;
	} else {
		mask = 0;
	}
	szedata_device_unlock_irq(instance->dev);
	return mask;
}

static struct file_operations szedata_fops = {
	.owner		= THIS_MODULE,
	.open		= szedata_open,
	.release	= szedata_release,
	.unlocked_ioctl	= szedata_ioctl,
	.mmap		= szedata_mmap,
	.poll		= szedata_poll
};

static int __init szedata_init(void)
{
	if (szedata_info_init() < 0) {
		printk(KERN_ERR "szedata info init error\n");
		return -EIO;
	}
	szedata_major = register_chrdev(0, "szedata", &szedata_fops);
	if (szedata_major < 0) {
		szedata_info_done();
		printk(KERN_ERR "unable to register chardev\n");
		return szedata_major;
	}

	szedata_class = class_create(THIS_MODULE, "szedata");
	if (IS_ERR(szedata_class)) {
		unregister_chrdev(szedata_major, "szedata");
		szedata_info_done();
		return PTR_ERR(szedata_class);
	}

	return 0;
}

static void __exit szedata_exit(void)
{
	szedata_info_done();
	if (szedata_pci_alloc_size != 0)
		printk(KERN_ERR "szedata: PCI allocation problem! size "
				"left = %u\n", szedata_pci_alloc_size);
	unregister_chrdev(szedata_major, "szedata");
	class_destroy(szedata_class);
}

module_init(szedata_init)
module_exit(szedata_exit)
