/*
 * cv3.c: Driver for the ComboV3 card
 * Copyright (c) 2013-2014 CESNET
 * Author(s): Martin Spinler <spinler@cesnet.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/module.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/time.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/mutex.h>
#include <linux/nmi.h>
#include <linux/types.h>

#include <linux/mtd/concat.h>
#include <linux/mtd/map.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/mtdram.h>
#include <linux/mtd/partitions.h>

#include "combo6.h"
#include "combo6k.h"
#include "cv3.h"

MODULE_AUTHOR("CESNET; Martin Spinler <spinler@cesnet.cz>");
MODULE_DESCRIPTION("ComboV3 Linux Driver");
MODULE_LICENSE("GPL");

static int use64b = 0;
static int synchack = 1;
static int createmtd = 0;

static struct pci_device_id combov3_ids[] __devinitdata = {
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xfb40 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0700 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0701 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0702 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0703 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfb40,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0704 },

	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfbc1,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xfbc1 },

	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfbc0,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0800 },
	{ .vendor = PCI_VENDOR_ID_CESNET, .device = 0xfbc1,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0x0800 },
	{ 0, }
};

MODULE_DEVICE_TABLE(pci, combov3_ids);

static struct pci_driver driver = {
	.name = "ComboV3",
	.id_table = combov3_ids,
	.probe = combov3_probe,
	.remove = __devexit_p(combov3_remove),
};

struct combov3_nr_name {
	unsigned char nr;
	const char *name;
};

static irqreturn_t combov3_interrupt(int irq, void *dev_id);
static void combov3_read_id(struct combo6 *combo6, const struct pci_device_id *id);
static int combov3_ioctl(struct combo6 *combo6, struct inode *inode,
						struct file *file, unsigned int cmd, unsigned long arg);

static void combov3_reload_firmware(struct combo6* combo, int fwnum);

static const struct combov3_nr_name *combov3_find_name(unsigned char nr,
		const struct combov3_nr_name *nr_name)
{
	while (nr_name->name) {
		if (nr_name->nr == nr)
			break;
		nr_name++;
	}
	return nr_name;
}

static const char *combov3_get_info(struct combo6 *combo6, char *buf,
		enum combo6_info_type info, unsigned long data)
{
	static const struct combov3_nr_name ctype[] = {
		{ 0x00, "Unknown" },
		{ 0x01, "FB8XG" },
		{ 0x02, "FB1CG" },
		{ }
	};

	static const struct combov3_nr_name cstype[] = {
		{ 0x00, "Unknown" },
		{ 0x01, "VXT690" },
		{ 0x02, "VHT580" },
		{ }
	};
	
	switch (info) {
	case COMBO6_INFO_TYPE:
		return combov3_find_name(combo6->u.cv3.ctype,
				ctype)->name;
	case COMBO6_INFO_SUBTYPE:
		return combov3_find_name(combo6->u.cv3.cstype,
				cstype)->name;
	case COMBO6_INFO_PORTS:
		sprintf(buf, "%d", data ?
				(combo6->addon_interfaces & 0xff00 ) >> 8:
				(combo6->addon_interfaces & 0x00ff ) >> 0);
		return buf;
	case COMBO6_INFO_SERNO:
		return combo6->u.cv3.serno[0];
	case COMBO6_INFO_SPGRADE:
		sprintf(buf, "%u", combo6->u.cv3.spgrade);
		return buf;
	default:
		return NULL;
	}
}

static void combov3_proc_info(struct combo6 *combo6,
		struct combo6_info_buffer *buffer)
{
	/*u32 ver, date;

	ver = combo6->pcibr_version;
	date = combo6->pcibr_date;*/
	combo6_iprintf(buffer, "Caps     : 0x%08x\n"
		"ID ver.  : 0x%04x\n"
		"NetCope  : 0x%04x\n",
		combo6->u.cv3.caps,
		combo6->u.cv3.id_ver,
		combo6->u.cv3.netcope_ver);
}

static void combov3_go_release(struct combo6 *combo6)
{
	if(combo6->u.cv3.fwnum != -1) {
		combov3_reload_firmware(combo6, combo6->u.cv3.fwnum);
		combo6->u.cv3.fwnum = -1;
	}
}

static void combov3_tuneup(struct pci_dev *pdev)
{
	struct pci_dev *bus = pdev->bus->self;
	int ret, bus_ecap;
	u16 bus_payload, devctl, dev_allows;

	int exp_cap;

	exp_cap = pci_find_capability(pdev, PCI_CAP_ID_EXP);
	if (!exp_cap)
		dev_err(&pdev->dev, "can't find PCI express capability on the "
				"card\n");

	ret = pcie_set_readrq(pdev, 4096);
	if (ret)
		dev_err(&pdev->dev, "can't set read request size\n");

	if (!exp_cap)
		return;
	/*
	 * first we check bus payload size, then device capabilities and
	 * choose the lower
	 */
	bus_ecap = pci_find_capability(bus, PCI_CAP_ID_EXP);
	if (!bus_ecap) {
		dev_err(&pdev->dev, "can't find PCI express capability on the "
				"parent bus\n");
		return;
	}
	pci_read_config_word(bus, bus_ecap + PCI_EXP_DEVCTL, &bus_payload);
	bus_payload &= PCI_EXP_DEVCTL_PAYLOAD;

	pci_read_config_word(pdev, exp_cap + PCI_EXP_DEVCTL, &devctl);
	pci_read_config_word(pdev, exp_cap + PCI_EXP_DEVCAP, &dev_allows);
	dev_allows &= PCI_EXP_DEVCAP_PAYLOAD;
	dev_allows <<= 12;
	dev_allows = min_t(u16, bus_payload, dev_allows);

	/* it's already there, bail out */
	if (dev_allows == (devctl & PCI_EXP_DEVCTL_PAYLOAD))
		return;
	devctl &= ~PCI_EXP_DEVCTL_PAYLOAD;
	devctl |= dev_allows;
	pci_write_config_word(pdev, exp_cap + PCI_EXP_DEVCTL, devctl);
}

inline int combov3_flash_wait(struct combo6*combo)
{
	int i = 0;
	int ret;
	if(synchack)
		combo6_fpga_readl(combo, COMBOV3_BOOT_BASE);

	while(1) {
		ret = combo6_fpga_readl(combo, COMBOV3_BOOT_BASE);
		if(ret & 0x00010000)
			break;
		if(++i > 50) {
			dev_err(&combo->pci->dev, "Flash is not ready. "
				"Is a BootFPGA unit in design?\n");
			break;
		}
	}
	return ret;
}

static map_word combov3_flash_read(struct map_info *map, unsigned long ofs)
{
	struct combo6* combo = (struct combo6*)map->map_priv_1;
	int data;
	map_word tmp;
	tmp.x[0] = 0;
	ofs = ofs/2;

	combo6_fpga_writel(combo, COMBOV3_BOOT_BASE, 0);
	combo6_fpga_writel(combo, COMBOV3_BOOT_BASE+4, (ofs & 0x01FFFFFF) | 0x10000000);
	data = combov3_flash_wait(combo) & 0x000FFFF;
	tmp.x[0] = data;
	return tmp;
}

static void combov3_flash_write(struct map_info *map, map_word d, unsigned long adr)
{
	struct combo6* combo = (struct combo6*)map->map_priv_1;
	adr = adr/2;

	combo6_fpga_writel(combo, COMBOV3_BOOT_BASE, d.x[0] & 0xFFFF);
	combo6_fpga_writel(combo, COMBOV3_BOOT_BASE+4, (adr & 0x01FFFFFF) | 0x20000000);
	combov3_flash_wait(combo);
}

static void combov3_flash_copy_from(struct map_info *map, void *to,
	unsigned long from, ssize_t len)
{
	int i;
	map_word tmp;
	for(i = 0; i < len/2; i++)
	{
		tmp = combov3_flash_read(map, from + i*2);
		((uint16_t*)to)[i] = tmp.x[0];
	}
}

int __devinit combov3_probe(struct pci_dev *pci,
		  const struct pci_device_id *id)
{
	static const struct combo6_ops ops = {
		.ioctl = combov3_ioctl,
		.go_release = combov3_go_release,
		.get_info = combov3_get_info,
		.proc_info = combov3_proc_info,
	};

	struct combo6 *combo6;
	struct map_info * map_flash0 = 0;
	int ret;

	if(id->device == 0xfbc0) {
		ret = pci_enable_device(pci);
		ret = pci_set_dma_mask(pci, DMA_BIT_MASK(64));
		if(use64b)
			ret = pci_set_consistent_dma_mask(pci, DMA_BIT_MASK(64));
		pci_set_master(pci);
		combov3_tuneup(pci);
		return 0;
	}

	ret = pci_enable_device(pci);
	if (ret) {
		dev_err(&pci->dev, "can't enable pci device\n");
		goto err;
	}

	combo6 = __combo6_alloc(0, THIS_MODULE);
	if (combo6 == NULL) {
		dev_err(&pci->dev, "can't allocate combo6\n");
		ret = -ENOMEM;
		goto err_dis;
	}

	combo6->pci = pci;
	combo6->ops = &ops;

	combo6->is_v2 = 1;
	combo6->u.cv3.fwnum = -1;	// Actual fimware in flash booted

	set_bit(COMBO_BOOT_OK, combo6->flags);

	ret = pci_enable_msi(pci);
	if (ret)
		dev_err(&pci->dev, "can't enable msi\n");
	else
		set_bit(COMBO_MSI, combo6->flags);

	ret = pci_set_dma_mask(pci, DMA_BIT_MASK(64));
	if (ret)
		goto err_free;

	if(use64b) {
		ret = pci_set_consistent_dma_mask(pci, DMA_BIT_MASK(64));
		if (ret) {
			dev_err(&pci->dev, "Unable to set DMA consistent mask: %d\n", ret);
			goto err_free;
		}
	}

	pci_set_master(pci);

	ret = -ENOMEM;
	combo6->mem_phys = 0;
	combo6->mem_len = 0;
	combo6->mem_virt = 0;

	combo6->lmem_phys = pci_resource_start(pci, 0);
	combo6->lmem_len = pci_resource_len(pci, 0);
	if (!request_mem_region(combo6->lmem_phys, combo6->lmem_len,
			"ComboV3 mem")) {
		dev_err(&pci->dev, "unable to grab memory region "
			"0x%llx-0x%llx\n", (u64)combo6->lmem_phys,
			(u64)(combo6->lmem_phys + combo6->lmem_len - 1));
		goto err_free;
	}
	combo6->lmem_virt = ioremap_nocache(combo6->lmem_phys, combo6->lmem_len);
	if (combo6->lmem_virt == NULL) {
		dev_err(&pci->dev, "unable to remap memory region "
			"0x%llx-0x%llx\n", (u64)combo6->lmem_phys,
			(u64)(combo6->lmem_phys + combo6->lmem_len - 1));
		release_mem_region(combo6->lmem_phys, combo6->lmem_len);
		goto err_free;
	}

	ret = request_irq(pci->irq, combov3_interrupt, IRQF_SHARED, "ComboV3",
			combo6);
	if (ret) {
		dev_err(&pci->dev, "unable to allocate interrupt %d\n",
				pci->irq);
		goto err_free;
	}
	combo6->irq = pci->irq;

	pci_set_drvdata(pci, combo6);

	combov3_tuneup(pci);

	combov3_read_id(combo6, id);

	if(combo6->u.cv3.netcope_ver == 0)
		goto skip_flash;

	ret = -ENOMEM;

	map_flash0 = vzalloc(sizeof(struct map_info));
	if(!map_flash0)
		goto err_free;

	map_flash0->bankwidth = 2;
	map_flash0->name = "fb40_flash0";
	map_flash0->size = 1024*1024*64;
	map_flash0->read = combov3_flash_read;
	map_flash0->write = combov3_flash_write;
	map_flash0->copy_from = combov3_flash_copy_from;
	map_flash0->map_priv_1 = (long) combo6;

	combo6->u.cv3.map = map_flash0;
	combo6->u.cv3.mtd = do_map_probe("cfi_probe", map_flash0);

	if(!combo6->u.cv3.mtd)
	{
		dev_err(&pci->dev, "Map probe failed for flash, no flash\n");
	}

	combo6->u.cv3.part = vzalloc(sizeof(struct mtd_partition)*3);
	if(!combo6->u.cv3.part)
		goto err_destroymap;

	combo6->u.cv3.part[0].name   = "id";
	combo6->u.cv3.part[0].offset = 0;
	combo6->u.cv3.part[0].size   = 0x20000;

	combo6->u.cv3.part[1].name   = "bitstream";
	combo6->u.cv3.part[1].offset = 0x20000;
	combo6->u.cv3.part[1].size   = 0x2000000 - 0x20000;

	combo6->u.cv3.part[2].name   = "recovery";
	combo6->u.cv3.part[2].offset = 0x2000000;
	combo6->u.cv3.part[2].size   = 0x2000000;

	if(createmtd)
		add_mtd_partitions(combo6->u.cv3.mtd, combo6->u.cv3.part, 3);

skip_flash:
	ret = combo6_add(combo6);
	if (ret) {
		dev_err(&pci->dev, "too many cards in system\n");
		goto err_freepart;
	}
	return 0;


err_freepart:
	vfree(combo6->u.cv3.part);
err_destroymap:
	if(combo6->u.cv3.mtd)
		map_destroy(combo6->u.cv3.mtd);
	vfree(map_flash0);
err_free:
	combo6_free(combo6);
err_dis:
	pci_disable_device(pci);
err:
	return ret;
}

void __devexit combov3_remove(struct pci_dev *pci)
{
	struct combo6 *combo6 = (struct combo6 *) pci_get_drvdata(pci);
	if(combo6 == NULL)
		return;

	if(createmtd)
		del_mtd_partitions(combo6->u.cv3.mtd);
	if(combo6->u.cv3.mtd) {
		map_destroy(combo6->u.cv3.mtd);
		combo6->u.cv3.mtd = 0;
	}
	if(combo6->u.cv3.map) {
		vfree(combo6->u.cv3.map);
		combo6->u.cv3.map = 0;
	}
	if(combo6->u.cv3.part) {
		vfree(combo6->u.cv3.part);
		combo6->u.cv3.part = 0;
	}

	set_bit(COMBO_GOING_DOWN, combo6->flags);
	combo6_remove(combo6);
	combo6_ioctl_bus_detach(combo6);

	combo6_free(combo6);
	pci_set_drvdata(pci, NULL);
}

static void combov3_reload_firmware(struct combo6 * combo, int fwnum)
{
	struct pci_dev * pci = combo->pci;
	struct pci_bus *bus = pci->bus;
	unsigned short devfn = pci->devfn;
	int cap;
	u16 lctl;
	int part;

	dev_info(&pci->dev, "reloading firmware\n");
	cap = pci_find_capability(bus->self, PCI_CAP_ID_EXP);
	if (!cap) {
		dev_err(&pci->dev, "can't find PCI express capability on the "
				"parent bus\n");
		return;
	}
	switch(fwnum)
	{
		case 100:
			part = 1;
			break;
		case 101:
			part = 0;
			break;
		default:
			return;
	}

	combo6_fpga_writel(combo, COMBOV3_BOOT_BASE, part);
	combo6_fpga_writel(combo, COMBOV3_BOOT_BASE+4, 0xE0000000);

	pci_remove_bus_device(pci);
	/* turn off PCI-E downstream port */
	pci_read_config_word(bus->self, cap + PCI_EXP_LNKCTL, &lctl);
	pci_write_config_word(bus->self, cap + PCI_EXP_LNKCTL,
			lctl | PCI_EXP_LNKCTL_LD);
	/*
	 * 500ms for booting is experimentally obtained value (min. is approx.
	 * 400ms). Including fallback design boot. So wait COMBOV2_BOOTDELAY
	 * before the card starts the boot process (since PCR_FWCONTROL write)
	 * plus 500ms for booting itself.
	 */
	msleep(2000);
	/* turn on PCI-E downstream port */
	pci_write_config_word(bus->self, cap + PCI_EXP_LNKCTL, lctl);
	/*
	 * We should wait here for some time until the link wakes up (to train
	 * up or so). It is possible that it's not needed at all, but we stay
	 * on the safe side.
	 */
	msleep(600);
	pci_scan_slot(bus, devfn);
	pci = pci_get_slot(bus, devfn);
	if (pci == NULL) {
		dev_err(&bus->self->dev, "unable to find new PCI device!\n");
		return;
	}
	bus = pci->bus;
	pci_bus_assign_resources(bus);
	pci_bus_add_devices(bus);
	dev_info(&pci->dev, "firmware switch done\n");
	pci_dev_put(pci);
}


static int combov3_ioctl_boot(struct combo6 *combo6,
		struct combo6_boot __user *_boot)
{
	struct combo6_boot boot;
	struct erase_info ei;
	int res = 0;
	int part = 1;
	size_t ret_size;
	void *data;
	u32 size;

	if(!combo6->u.cv3.mtd)
		return -ENODEV;

	if (copy_from_user(&boot, _boot, sizeof(boot)))
		return -EFAULT;

	if(boot.d[0].size == 0)
		return -EINVAL;

	set_bit(COMBO_BOOTING, combo6->flags);

	size = boot.d[0].size;
	data = vmalloc(size);
	if (data == NULL)
		return -ENOMEM;
	if (copy_from_user(data, (void __user __force *)boot.d[0].data, size)) {
		vfree(data);
		return -EFAULT;
	}

	switch(boot.d[0].unit)
	{
		case 100:
			part = 2;
			break;
		case 101:
			part = 1;
			break;
		default:
			return -EINVAL;
	}
	if(size > combo6->u.cv3.part[part].size)
		return -ENOMEM;

	res = 0;

	ei.callback = 0;
	ei.mtd = combo6->u.cv3.mtd;
	ei.addr = combo6->u.cv3.part[part].offset;
	ei.len = (size + combo6->u.cv3.mtd->erasesize - 1) & ~(combo6->u.cv3.mtd->erasesize - 1);

	res = combo6->u.cv3.mtd->erase(combo6->u.cv3.mtd, &ei);
	if(res != 0)
		goto write_err;
	res = combo6->u.cv3.mtd->write(combo6->u.cv3.mtd, combo6->u.cv3.part[part].offset, size, &ret_size, data);
	if(res != 0)
		goto write_err;

	vfree(data);

	combo6->u.cv3.fwnum = boot.d[0].unit;
	clear_bit(COMBO_BOOTING, combo6->flags);
	return res;

write_err:
	vfree(data);
	dev_err(&combo6->pci->dev, "Write to Flash FAILED\n");
	clear_bit(COMBO_BOOT_OK, combo6->flags);
	clear_bit(COMBO_BOOTING, combo6->flags);
	return res;
}

static int combov3_ioctl_fwsel_read(struct combo6 *combo6, int __user *_fwnum)
{
	return put_user(combo6->u.cv3.fwnum, _fwnum);
}

static int combov3_ioctl_fwsel_write(struct combo6 *combo6, int __user *_fwnum)
{
	int fwnum;
	if (get_user(fwnum, _fwnum))
		return -EFAULT;
	switch (fwnum) {
	case 100 ... 101:
		break;
	default:
		return -EINVAL;
	}
	combo6->u.cv3.fwnum = fwnum;
	return 0;
}

static int combov3_ioctl(struct combo6 *combo6,
			 struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	void __user *argp = (void __user *)arg;

	switch (cmd) {
	case COMBO6_READ:
		return -EINVAL;
	case COMBO6_WRITE:
		return -EINVAL;
	case COMBO6_IOC_BOOT:
		return combov3_ioctl_boot(combo6, argp);
	case COMBO6_IOC_BUS_ATTACH:
		return combo6_ioctl_bus_attach(combo6);
	case COMBO6_IOC_BUS_DETACH:
		return combo6_ioctl_bus_detach(combo6);
	case COMBO6_IOC_INFO:
		return combo6_ioctl_info(combo6, argp);
	case COMBO6_IOC_DINFO:
		return -EFAULT;
	case COMBO6_IOC_FWSEL_READ:
		return combov3_ioctl_fwsel_read(combo6, argp);
	case COMBO6_IOC_FWSEL_WRITE:
		if (file->f_flags & O_APPEND)
			return -EBUSY;
		return combov3_ioctl_fwsel_write(combo6, argp);
	case COMBO6_IOC_I2C_OP:
		return -EFAULT;
	default:
		return -ENOTTY;
	}
}

static void combov3_read_id(struct combo6 *combo6, const struct pci_device_id *id)
{
	int i, tmp;

	tmp = combo6_fpga_readl(combo6, COMBOV3_ID_NEG);
	combo6_fpga_writel(combo6, COMBOV3_ID_NEG, tmp);
	if (combo6_fpga_readl(combo6, COMBOV3_ID_NEG) == ~tmp) {
		combo6->id_sw = combo6_fpga_readl(combo6, COMBOV3_ID_SW);
		combo6->id_hw = combo6_fpga_readl(combo6, COMBOV3_ID_HW);
		for (i = 0; i < 32; i += 4) {
			tmp = combo6_fpga_readl(combo6, COMBOV3_ID_TEXT + i);
			ID32_TO_STR(combo6->id_txt + i, tmp);
		}
		combo6->id_txt[32] = '\0';
		combo6->addon_interfaces = combo6_fpga_readl(combo6,
				COMBOV3_ID_CHANNELS) & 0xffff;
		tmp = combo6_fpga_readl(combo6, COMBOV3_ID_ID_VER);
		combo6->u.cv3.id_ver = tmp & 0xffff;
		combo6->u.cv3.netcope_ver = tmp >> 16;
	} else {
		dev_err(&combo6->pci->dev, "Device don't have ID Unit\n");
		combo6->u.cv3.netcope_ver = 0;
	}

	combo6->u.cv3.spgrade = 2;

	switch (id->device)
	{
	case 0xfb40:
		combo6->u.cv3.ctype = 0x01;
		combo6->u.cv3.cstype = 0x01;
		break;
	case 0xfbc1:
		combo6->u.cv3.ctype = 0x02;
		combo6->u.cv3.cstype = 0x02;
		break;
	default:
		combo6->u.cv3.ctype = 0x00;
		combo6->u.cv3.cstype = 0x00;
		break;
	}
}

static irqreturn_t combov3_interrupt(int irq, void *dev_id)
{
	struct combo6 *combo6 = dev_id;
	u32 source = 0;
	int ret;

	if (test_bit(COMBO_GOING_DOWN, combo6->flags))
		return IRQ_NONE;

	source = combo6_fpga_readl(combo6, COMBOV3_ID_IRQSTAT);

	if (source & COMBOV3_IRQS_SYSMON) {
		if (printk_ratelimit())
			dev_info(&combo6->pci->dev, "SYSMON alert\n");
		combo_queue_message(combo6->index, COMBOCTL_MSG_SYSMON);
		ret = IRQ_HANDLED;
	}

	if (source & COMBOV3_IRQS_LINK_CHG) {
		if (printk_ratelimit())
			dev_info(&combo6->pci->dev, "link changed\n");
		combo_queue_message(combo6->index, COMBOCTL_MSG_LINK);
		ret = IRQ_HANDLED;
	}

	source &= COMBOV3_IRQS_RX | COMBOV3_IRQS_TX;
	ret = IRQ_HANDLED;

	if (!source)
		return ret;

	spin_lock(&combo6->reg_lock);
	if (combo6->driver && combo6->driver->interrupt)
		combo6->driver->interrupt(combo6, source);
	spin_unlock(&combo6->reg_lock);

	return IRQ_HANDLED;
}

static int __init combov3_init(void)
{
	int err;

	if ((err = pci_register_driver(&driver)) < 0)
		return err;

	return 0;
}

static void __exit combov3_exit(void)
{
	pci_unregister_driver(&driver);
}

EXPORT_SYMBOL(combov3_probe);
EXPORT_SYMBOL(combov3_remove);

module_init(combov3_init)
module_exit(combov3_exit)

module_param(use64b, bool, S_IRUGO);
MODULE_PARM_DESC(use64b, "use 64b space for buffers [no]");
module_param(synchack, bool, S_IRUGO);
MODULE_PARM_DESC(synchack, "for designs with bug in async unit [yes]");
module_param(createmtd, bool, S_IRUGO);
MODULE_PARM_DESC(createmtd, "for Flash debug purposes [no]");

