/*
 *  core_proc.c: ipv6 hw router linux driver, proc interface
 *  Copyright (c) 2004,2005,2006,2007 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <generated/utsrelease.h>
#include <linux/module.h>

#include <linux/init.h>
#include <linux/mutex.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <linux/vmalloc.h>
#include <linux/fs.h>
#include <linux/mutex.h>
#include <linux/proc_fs.h>
#include <linux/uaccess.h>
#include <linux/io.h>

#include "combo6.h"
#include "combo6k.h"

static DEFINE_MUTEX(info_mutex);

struct combo6_info_private_data {
	struct combo6_info_buffer *rbuffer;
	struct combo6_info_entry *entry;
	void *file_private_data;
};

static int combo6_info_version_init(void);
static int combo6_info_version_done(void);
static void combo6_info_read_id(struct combo6_info_entry *entry,
		struct combo6_info_buffer *buffer);

/**
 * combo6_iprintf - printf on the procfs buffer
 * @buffer: the procfs buffer
 * @fmt: the printf format
 *
 * Outputs the string on the procfs buffer just like printf().
 *
 * Returns the size of output string.
 */
int combo6_iprintf(struct combo6_info_buffer *buffer, char *fmt, ...)
{
	va_list args;
	int res;
	char sbuffer[512];

	if (buffer->stop || buffer->error)
		return 0;
	va_start(args, fmt);
	res = vscnprintf(sbuffer, sizeof(sbuffer), fmt, args);
	va_end(args);
	if (buffer->size + res >= buffer->len) {
		buffer->stop = 1;
		return 0;
	}
	strcpy(buffer->curr, sbuffer);
	buffer->curr += res;
	buffer->size += res;
	return res;
}
EXPORT_SYMBOL(combo6_iprintf);

static struct proc_dir_entry *combo6_proc_root;

static void combo6_remove_proc_entry(struct proc_dir_entry *parent,
				     struct proc_dir_entry *de)
{
	if (de)
		remove_proc_entry(de->name, parent);
}

static ssize_t combo6_info_entry_read(struct file *file, char __user *buffer,
		size_t count, loff_t *offset)
{
	struct combo6_info_private_data *data;
	struct combo6_info_buffer *buf;
	size_t size = 0;
	loff_t pos;

	data = file->private_data;
	pos = *offset;
	if (pos < 0 || (long)pos != pos || (ssize_t)count < 0)
		return -EIO;
	if ((unsigned long)pos + (unsigned long)count < (unsigned long)pos)
		return -EIO;
	buf = data->rbuffer;
	if (buf == NULL)
		return -EIO;
	if (pos >= buf->size)
		return 0;
	size = buf->size - pos;
	size = min(count, size);
	if (copy_to_user(buffer, buf->buffer + pos, size))
		return -EFAULT;
	if ((ssize_t)size > 0)
		*offset = pos + size;
	return size;
}

static int combo6_info_entry_open(struct inode *inode, struct file *file)
{
	struct combo6_info_entry *entry;
	struct combo6_info_private_data *data;
	struct combo6_info_buffer *buffer;
	struct proc_dir_entry *p;
	int mode, err;

	mutex_lock(&info_mutex);
	p = PDE(inode);
	entry = p == NULL ? NULL : (struct combo6_info_entry *)p->data;
	if (entry == NULL) {
		mutex_unlock(&info_mutex);
		return -ENODEV;
	}
	if (!try_module_get(entry->module)) {
		err = -EFAULT;
		goto __error1;
	}
	mode = file->f_flags & O_ACCMODE;
	if (mode == O_RDONLY || mode == O_RDWR) {
		if (!entry->text.read_size) {
			err = -ENODEV;
			goto __error;
		}
	}
	if (mode == O_WRONLY || mode == O_RDWR) {
		if (printk_ratelimit())
			printk(KERN_INFO "%s: writing not implemented\n",
					__func__);
		err = -ENODEV;
		goto __error;
	}
	data = kmalloc(sizeof(struct combo6_info_private_data), GFP_KERNEL);
	if (data == NULL) {
		err = -ENOMEM;
		goto __error;
	}
	memset(data, 0, sizeof(struct combo6_info_private_data));
	data->entry = entry;
	if (mode == O_RDONLY || mode == O_RDWR) {
		buffer = kmalloc(sizeof(struct combo6_info_buffer),
				GFP_KERNEL);
		if (buffer == NULL) {
			kfree(data);
			err = -ENOMEM;
			goto __error;
		}
		memset(buffer, 0, sizeof(struct combo6_info_buffer));
		buffer->len = (entry->text.read_size +
			      (PAGE_SIZE - 1)) & ~(PAGE_SIZE - 1);
		buffer->buffer = vmalloc(buffer->len);
		if (buffer->buffer == NULL) {
			kfree(buffer);
			kfree(data);
			err = -ENOMEM;
			goto __error;
		}
		buffer->curr = buffer->buffer;
		data->rbuffer = buffer;
	}
	file->private_data = data;
	mutex_unlock(&info_mutex);
	if (mode == O_RDONLY || mode == O_RDWR) {
		if (entry->text.read) {
			mutex_lock(&entry->access);
			entry->text.read(entry, data->rbuffer);
			mutex_unlock(&entry->access);
		}
	}
	return 0;

__error:
	module_put(entry->module);
__error1:
	mutex_unlock(&info_mutex);
	return err;
}

static int combo6_info_entry_release(struct inode *inode, struct file *file)
{
	struct combo6_info_entry *entry;
	struct combo6_info_private_data *data;
	int mode;

	mode = file->f_flags & O_ACCMODE;
	data = file->private_data;
	entry = data->entry;
	if (mode == O_RDONLY || mode == O_RDWR) {
		vfree(data->rbuffer->buffer);
		kfree(data->rbuffer);
	}
	module_put(entry->module);
	kfree(data);
	return 0;
}

static struct file_operations combo6_info_entry_operations = {
	.owner =	THIS_MODULE,
	.read =		combo6_info_entry_read,
	.open =		combo6_info_entry_open,
	.release =	combo6_info_entry_release,
};

/**
 * combo6_create_proc_entry - create a procfs entry
 * @name: the name of the proc file
 * @mode: the file permission bits, S_Ixxx
 * @parent: the parent proc-directory entry
 *
 * Creates a new proc file entry with the given name and permission
 * on the given directory.
 *
 * Returns the pointer of new instance or NULL on failure.
 */
static struct proc_dir_entry *combo6_create_proc_entry(const char *name,
		mode_t mode, struct proc_dir_entry *parent)
{
	struct proc_dir_entry *p;
	p = create_proc_entry(name, mode, parent);
#ifdef CONFIG_PROC_ENTRY_HAS_OWNER
	if (p)
		p->owner = THIS_MODULE;
#endif
	return p;
}

int __init combo6_info_init(void)
{
	struct proc_dir_entry *p;

	p = combo6_create_proc_entry("driver/combo6",
			S_IFDIR | S_IRUGO | S_IXUGO, NULL);
	if (p == NULL)
		return -ENOMEM;
	combo6_proc_root = p;
	combo6_info_version_init();
	return 0;
}

int combo6_info_done(void)
{
	combo6_info_version_done();
	combo6_remove_proc_entry(combo6_proc_root->parent, combo6_proc_root);
	return 0;
}

/**
 * combo6_info_card_create - create a card proc file
 * @card: info regarding this card
 *
 * Called from core.c
 */
int combo6_info_card_create(struct combo6 *card)
{
	char str[8];
	struct combo6_info_entry *entry;

	sprintf(str, "card%i", card->index);
	entry = combo6_info_create_module_entry(card->module, str, NULL);
	if (entry == NULL)
		return -ENOMEM;
	entry->mode = S_IFDIR | S_IRUGO | S_IXUGO;
	if (combo6_info_register(entry) < 0) {
		combo6_info_free_entry(entry);
		return -ENOMEM;
	}
	card->proc_root = entry;
	entry = combo6_info_create_module_entry(card->module, "id",
			card->proc_root);
	if (entry) {
		combo6_info_set_text_ops(entry, card, 2048,
				combo6_info_read_id);
		if (combo6_info_register(entry) < 0) {
			combo6_info_free_entry(entry);
			entry = NULL;
		}
	}
	card->proc_id = entry;
	return 0;
}
EXPORT_SYMBOL(combo6_info_free_entry);

/**
 * combo6_info_card_free - de-register the card proc file
 * @card: which card info deregister
 *
 * called from init.c
 */
int combo6_info_card_free(struct combo6 *card)
{
	if (card->proc_id) {
		combo6_info_unregister(card->proc_id);
		card->proc_id = NULL;
	}
	if (card->proc_root) {
		combo6_info_unregister(card->proc_root);
		card->proc_root = NULL;
	}
	return 0;
}


/**
 * combo6_info_create_entry - create an info entry
 * @name: the proc file name
 *
 * Creates an info entry with the given file name and initializes as
 * the default state.
 *
 * Usually called from other functions such as
 * combo6_info_create_card_entry().
 *
 * Returns the pointer of the new instance, or NULL on failure.
 */
static struct combo6_info_entry *combo6_info_create_entry(const char *name)
{
	struct combo6_info_entry *entry;
	entry = kmalloc(sizeof(struct combo6_info_entry), GFP_KERNEL);
	if (entry == NULL)
		return NULL;
	memset(entry, 0, sizeof(struct combo6_info_entry));
	entry->name = kmalloc(strlen(name) + 1, GFP_KERNEL);
	if (entry->name == NULL) {
		kfree(entry);
		return NULL;
	}
	strcpy((char *)entry->name, name);
	entry->mode = S_IFREG | S_IRUGO;
	mutex_init(&entry->access);
	return entry;
}

/**
 * combo6_info_create_module_entry - create an info entry for the given module
 * @module: the module pointer
 * @name: the file name
 * @parent: the parent directory
 *
 * Creates a new info entry and assigns it to the given module.
 *
 * Returns the pointer of the new instance, or NULL on failure.
 */
struct combo6_info_entry *combo6_info_create_module_entry(struct module *module,
		const char *name, struct combo6_info_entry *parent)
{
	struct combo6_info_entry *entry = combo6_info_create_entry(name);
	if (entry) {
		entry->module = module;
		entry->parent = parent;
	}
	return entry;
}
EXPORT_SYMBOL(combo6_info_create_module_entry);

/**
 * combo6_info_create_card_entry - create an info entry for the given card
 * @card: the card instance
 * @name: the file name
 * @parent: the parent directory
 *
 * Creates a new info entry and assigns it to the given card.
 *
 * Returns the pointer of the new instance, or NULL on failure.
 */
struct combo6_info_entry *combo6_info_create_card_entry(struct combo6 *card,
		const char *name, struct combo6_info_entry *parent)
{
	struct combo6_info_entry *entry = combo6_info_create_entry(name);
	if (entry) {
		entry->module = card->module;
		entry->card = card;
		entry->parent = parent;
	}
	return entry;
}

/**
 * combo6_info_free_entry - release the info entry
 * @entry: the info entry
 *
 * Releases the info entry.  Don't call this after registered.
 */
void combo6_info_free_entry(struct combo6_info_entry *entry)
{
	if (entry == NULL)
		return;
	if (entry->name)
		kfree((char *)entry->name);
	if (entry->private_free)
		entry->private_free(entry);
	kfree(entry);
}

/**
 * combo6_info_register - register the info entry
 * @entry: the info entry
 *
 * Registers the proc info entry.
 *
 * Returns zero if successful, or a negative error code on failure.
 */
int combo6_info_register(struct combo6_info_entry *entry)
{
	struct proc_dir_entry *root, *p = NULL;

	root = entry->parent == NULL ? combo6_proc_root : entry->parent->p;
	mutex_lock(&info_mutex);
	p = combo6_create_proc_entry(entry->name, entry->mode, root);
	if (!p) {
		mutex_unlock(&info_mutex);
		return -ENOMEM;
	}
#ifdef CONFIG_PROC_ENTRY_HAS_OWNER
	p->owner = entry->module;
#endif
	if (!S_ISDIR(entry->mode))
		p->proc_fops = &combo6_info_entry_operations;
	p->size = entry->size;
	p->data = entry;
	entry->p = p;
	mutex_unlock(&info_mutex);
	return 0;
}
EXPORT_SYMBOL(combo6_info_register);

/**
 * combo6_info_unregister - de-register the info entry
 * @entry: the info entry
 *
 * De-registers the info entry and releases the instance.
 *
 * Returns zero if successful, or a negative error code on failure.
 */
int combo6_info_unregister(struct combo6_info_entry *entry)
{
	struct proc_dir_entry *root;

	root = entry->parent == NULL ? combo6_proc_root : entry->parent->p;
	mutex_lock(&info_mutex);
	combo6_remove_proc_entry(root, entry->p);
	mutex_unlock(&info_mutex);
	combo6_info_free_entry(entry);
	return 0;
}
EXPORT_SYMBOL(combo6_info_unregister);

static struct combo6_info_entry *combo6_info_version_entry;

static void combo6_info_version_read(struct combo6_info_entry *entry,
		struct combo6_info_buffer *buffer)
{
	static char *kernel_version = UTS_RELEASE;

	combo6_iprintf(buffer,
		    "Combo6 Driver Version " CONFIG_COMBO_VERSION ".\n"
		    "Compiled on " __DATE__ " " __TIME__ " for kernel %s"
#ifdef CONFIG_SMP
		    " (SMP)"
#endif
#ifdef MODVERSIONS
		    " with versioned symbols"
#endif
		    ".\n", kernel_version);
}

static int __init combo6_info_version_init(void)
{
	struct combo6_info_entry *entry;

	entry = combo6_info_create_module_entry(THIS_MODULE, "version", NULL);
	if (entry == NULL)
		return -ENOMEM;
	entry->text.read_size = 256;
	entry->text.read = combo6_info_version_read;
	if (combo6_info_register(entry) < 0) {
		combo6_info_free_entry(entry);
		return -ENOMEM;
	}
	combo6_info_version_entry = entry;
	return 0;
}

static int combo6_info_version_done(void)
{
	if (combo6_info_version_entry)
		combo6_info_unregister(combo6_info_version_entry);
	return 0;
}

static void combo6_info_read_id(struct combo6_info_entry *entry,
		struct combo6_info_buffer *buffer)
{
	struct combo6 *combo6 = entry->private_data;
	struct combo_driver *driver;
	const char *id_txt;
	char buf[8], buf1[8];
	unsigned int a;
	u32 id_sw, id_hw;

	if (test_bit(COMBO_BOOT_OK, combo6->flags)) {
		id_sw = combo6->id_sw;
		id_hw = combo6->id_hw;
		id_txt = combo6->id_txt;
	} else {
		id_sw = id_hw = 0;
		id_txt = "unknown";
	}

	combo6_iprintf(buffer, "Board    : %s\n",
			combo6_get_info(combo6, buf, COMBO6_INFO_TYPE, 0));
	combo6_iprintf(buffer, "Subtype  : %s\n",
			combo6_get_info(combo6, buf, COMBO6_INFO_SUBTYPE, 0));
	combo6_iprintf(buffer, "S/N      : %s\n",
			combo6_get_info(combo6, buf, COMBO6_INFO_SERNO, 0));
	combo6_iprintf(buffer, "Speedgr. : %s\n",
			combo6_get_info(combo6, buf, COMBO6_INFO_SPGRADE, 0));
	for (a = 0; a < COMBO6_ADDON_CARDS_MAX; a++) {
		combo6_iprintf(buffer, "Addon%u   : %s\n", a,
				combo6_get_info(combo6, buf,
					COMBO6_INFO_ADDON_CARD, a));
		combo6_iprintf(buffer, "Chip%u    : %s\n", a,
				combo6_get_info(combo6, buf,
					COMBO6_INFO_ADDON_CHIP, a));
		combo6_iprintf(buffer, "S/N%u     : %s\n", a,
				combo6_get_info(combo6, buf,
					COMBO6_INFO_SERNO, a + 1));
	}
	combo6_iprintf(buffer, "Channels : %s/%s (RX/TX)\n",
			combo6_get_info(combo6, buf, COMBO6_INFO_PORTS, 0),
			combo6_get_info(combo6, buf1, COMBO6_INFO_PORTS, 1));
	combo6_iprintf(buffer, "Firmware : %s\n",
			test_bit(COMBO_BOOT_OK, combo6->flags) ? "ok" : "none");
	combo6_iprintf(buffer, "SW       : 0x%08x\n", id_sw);
	combo6_iprintf(buffer, "HW       : 0x%08x\n", id_hw);
	combo6_iprintf(buffer, "Text     : %s\n", id_txt);
	if (combo6->ops->proc_info)
		combo6->ops->proc_info(combo6, buffer);

	mutex_lock(&combo6->drv_mutex);
	driver = combo6->driver;
	if (driver) {
		const struct combo_device_id *id;

		combo6_iprintf(buffer, "\n");
		combo6_iprintf(buffer, "Driver [%s] %s: active\n",
			driver->dhw == DHW_COMBO6X ? "combo6x" :
			driver->dhw == DHW_COMBOV2 ? "combov2" : "unknown",
			driver->drv.name);
		for (id = driver->id_table; id->id_hsw; id++)
			combo6_iprintf(buffer, "  (0x%x-0x%x) {%s}\n",
				id->id_lsw, id->id_hsw, id->id_text);
	}
	mutex_unlock(&combo6->drv_mutex);
}
