/*
 *  core.c: ipv6 hw router linux driver, main file
 *  Copyright (c) 2003,2004,2005,2006,2007 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/kdev_t.h>
#include <linux/kref.h>
#include <linux/time.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/mutex.h>
#include <linux/types.h>
#include <linux/uaccess.h>

#include "core.h"
#include "combo6.h"
#include "combo6k.h"

/*
 *  PCI initialization and detection part
 */

MODULE_AUTHOR("CESNET; Jaroslav Kysela <perex@perex.cz>");
MODULE_DESCRIPTION("Combo6 IPV6 HW Router Linux Driver");
MODULE_LICENSE("GPL");

struct combo6 *combo6_cards[COMBO6_CARDS];
static int dev, combo6_major;
static struct class *combo6_class;

static void __combo6_free(struct device *dev)
{
	struct combo6 *combo = container_of(dev, struct combo6, dev);

	kfree(combo);
}

/**
 * __combo6_alloc - allocate a new combo6 structure
 *
 * It shall be freed by combo6_free in any case.
 */
struct combo6 *__combo6_alloc(size_t priv_size, struct module *mod)
{
	struct combo6 *combo6;

	combo6 = kzalloc(sizeof(*combo6) + priv_size, GFP_KERNEL);
	if (combo6 == NULL)
		return NULL;

	combo6->module = mod;
	combo6->irq = -1;
	combo6->index = ~0;
	device_initialize(&combo6->dev);
	combo6->dev.release = __combo6_free;
	combo6->dev.bus = &combo_bus_type;
	spin_lock_init(&combo6->reg_lock);
	mutex_init(&combo6->drv_mutex);
	atomic_set(&combo6->openers, 0);
	atomic_set(&combo6->ref_count, 0);
	atomic_set(&combo6->detaching, 0);

	return combo6;
}
EXPORT_SYMBOL(__combo6_alloc);

static inline void combo6_get(struct combo6 *combo6)
{
	get_device(&combo6->dev);
}

static inline void combo6_put(struct combo6 *combo6)
{
	put_device(&combo6->dev);
}

/**
 * combo6_free - free allocated resources and combo6 structure itself
 * @combo6: structure to cleanup
 */
void combo6_free(struct combo6 *combo6)
{
	if (combo6 == NULL)
		return;
	if (combo6->index < COMBO6_CARDS)
		combo6_cards[combo6->index] = NULL;
	if (combo6->mem_virt) {
		iounmap(combo6->mem_virt);
		release_mem_region(combo6->mem_phys, combo6->mem_len);
	}
	if (combo6->lmem_virt) {
		iounmap(combo6->lmem_virt);
		release_mem_region(combo6->lmem_phys, combo6->lmem_len);
	}
	if (combo6->cmem_virt) {
		iounmap(combo6->cmem_virt);
		release_mem_region(combo6->cmem_phys, combo6->cmem_len);
	}
	if (combo6->irq >= 0)
		free_irq(combo6->irq, combo6);

	if (test_bit(COMBO_MSI, combo6->flags))
		pci_disable_msi(combo6->pci);

	combo6_put(combo6);
}
EXPORT_SYMBOL(combo6_free);

/**
 * combo6_add - add allocated combo6 structure
 * @combo6: structure to add
 *
 * Returns 0 on success, negative errno on error.
 */
int combo6_add(struct combo6 *combo6)
{
	int ret;

	if (dev >= COMBO6_CARDS)
		return -ENOMEM;
	while (dev < COMBO6_CARDS && combo6_cards[dev] != NULL)
		dev++;
	if (dev >= COMBO6_CARDS)
		return -ENOMEM;

	combo6->index = dev;
	combo6_cards[dev++] = combo6;

	combo6->dev.parent = &combo6->pci->dev;
	dev_set_name(&combo6->dev, "%d", combo6->index);

	ret = device_add(&combo6->dev);
	if (ret)
		combo6_cards[dev--] = NULL;

	combo6_info_card_create(combo6);
	device_create(combo6_class, &combo6->dev,
			MKDEV(combo6_major, combo6->index),
			NULL, "combosix%d", combo6->index);

	return ret;
}
EXPORT_SYMBOL(combo6_add);

/**
 * combo6_remove - remove combo6 structure
 * @combo6: structure to remove
 *
 * This has to be called after successful combo6_remove
 */
void combo6_remove(struct combo6 *combo6)
{
	device_del(&combo6->dev);
	combo6_cards[combo6->index] = NULL;
	dev = combo6->index;
	device_destroy(combo6_class, MKDEV(combo6_major, combo6->index));
	combo6_info_card_free(combo6);
}
EXPORT_SYMBOL(combo6_remove);

/*
 *  /dev/combo6/X devices
 */

static int combo6_open(struct inode *inode, struct file *file)
{
	unsigned int minor = ciminor(inode);
	struct combo6 *combo6;
	int ret;

	if (minor >= COMBO6_CARDS)
		return -ENODEV;
	/* XXX race with combo6_remove/combo6_free */
	combo6 = combo6_cards[minor];
	if (combo6 == NULL)
		return -ENODEV;
	combo6_get(combo6);
	if (!try_module_get(combo6->module)) {
		ret = -ENODEV;
		goto err_cput;
	}
	if (atomic_inc_return(&combo6->openers) > 1 &&
			!(file->f_flags & O_APPEND)) {
		ret = -EBUSY;
		goto err_dec;
	}
	file->private_data = combo6;

	return 0;
err_dec:
	atomic_dec(&combo6->openers);
	module_put(combo6->module);
err_cput:
	combo6_put(combo6);
	return ret;
}

static int combo6_release(struct inode *inode, struct file *file)
{
	static DEFINE_MUTEX(open_lock);
	struct combo6 *combo6 = file->private_data;

	mutex_lock(&open_lock);
	if (combo6->ops->go_release)
		combo6->ops->go_release(combo6);
	module_put(combo6->module);
	atomic_dec(&combo6->openers);
	combo6_put(combo6);
	mutex_unlock(&open_lock);

	return 0;
}

/**
 * combo6_ioctl_info - store combo6_info structure to userspace buffer
 * @combo6: structure to get info from
 * @_info: structure to fill up
 *
 * Returns 0 on success, otherwise negative errno.
 */
int combo6_ioctl_info(struct combo6 *combo6, struct combo6_info __user *_info)
{
	struct combo6_info info = {
		.ci_win_size = combo6->lmem_len,
	};
	unsigned int a;

	strcpy(info.ci_board_name,
		combo6_get_info(combo6, NULL, COMBO6_INFO_TYPE, 0));
	strcpy(info.ci_board_subtype,
		combo6_get_info(combo6, NULL, COMBO6_INFO_SUBTYPE, 0));
	for (a = 0; a < COMBO6_ADDON_CARDS_MAX; a++) {
		strcpy(info.ci_addon[a].card,
			combo6_get_info(combo6, NULL,
				COMBO6_INFO_ADDON_CARD, a));
		strcpy(info.ci_addon[a].chip,
			combo6_get_info(combo6, NULL,
				COMBO6_INFO_ADDON_CHIP, a));
	}
	if (test_bit(COMBO_BOOT_OK, combo6->flags)) {
		info.ci_id_sw = combo6->id_sw;
		info.ci_id_hw = combo6->id_hw;
		strcpy(info.ci_id_txt, combo6->id_txt);
		info.ci_fw_ok |= COMBO6_OK_FPGA0;
	}
	if (combo6->is_6x) {
		if (combo6->u.two.pcippc_ok)
			info.ci_fw_ok |= COMBO6_OK_PCRPPC;
		memcpy(&info.ci_pcippc, &combo6->u.two.pcippc,
				sizeof(struct combo6_eppc_id));
	}
	if (copy_to_user(_info, &info, sizeof(info)))
		return -EFAULT;
	return 0;
}
EXPORT_SYMBOL(combo6_ioctl_info);

static long combo6_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	static DEFINE_MUTEX(ioctl_lock);
	struct combo6 *combo6 = file->private_data;
	int ret = -ENXIO;

	/*
	 * note, if FPGA is not booted, only limited set of commands can be
	 * used
	 */
	switch (cmd) {
	case COMBO6_READ:
	case COMBO6_WRITE:
	case COMBO6_IOC_BOOT:
	case COMBO6_IOC_INFO:
	case COMBO6_IOC_FWSEL_READ:
	case COMBO6_IOC_FWSEL_WRITE:
		break;
	default:
		if (!test_bit(COMBO_BOOT_OK, combo6->flags))
			return -EBADFD;
	}

	if (combo6->ops->ioctl) {
		mutex_lock(&ioctl_lock);
		ret = combo6->ops->ioctl(combo6, NULL, file, cmd, arg);
		mutex_unlock(&ioctl_lock);
	}

	return ret;
}

static int combo6_fpga_mmap(struct combo6 *combo6, struct vm_area_struct *area)
{
	unsigned long size = area->vm_end - area->vm_start;
	unsigned long offset = area->vm_pgoff << PAGE_SHIFT;
	unsigned long mmap_bytes;

	if (!test_bit(COMBO_BOOT_OK, combo6->flags))
		return -EBADFD;
	if (!(area->vm_flags & (VM_WRITE|VM_READ)))
		return -EINVAL;
	mmap_bytes = PAGE_ALIGN(combo6->lmem_len);
	if (size > mmap_bytes)
		return -EINVAL;
	if (offset > mmap_bytes - size)
		return -EINVAL;
#ifdef pgprot_noncached
	area->vm_page_prot = pgprot_noncached(area->vm_page_prot);
#endif
	area->vm_private_data = combo6;
	area->vm_flags |= VM_IO;
	if (io_remap_pfn_range(area, area->vm_start,
			       (combo6->lmem_phys + offset) >> PAGE_SHIFT,
			       size, area->vm_page_prot))
		return -EAGAIN;
	return 0;
}

static int combo6_mmap(struct file *file, struct vm_area_struct *area)
{
	struct combo6 *combo6 = file->private_data;

	if (combo6->lmem_len == 0)
		return -ENXIO;
	return combo6_fpga_mmap(combo6, area);
}

static struct file_operations combo6_fops = {
	.owner		= THIS_MODULE,
	.open		= combo6_open,
	.release	= combo6_release,
	.unlocked_ioctl	= combo6_ioctl,
	.mmap		= combo6_mmap
};


/*
 * real initialization
 */

static int __init combo6_card_init(void)
{
	int ret;

	ret = combo_bus_register();
	if (ret)
		goto err;

	ret = comboctl_init();
	if (ret)
		goto err_bus;

	combo6_info_init();
	combo6_major = register_chrdev(0, "combo", &combo6_fops);
	if (combo6_major < 0) {
		printk(KERN_ERR "combo6: unable to register chardev\n");
		ret = combo6_major;
		goto err_info;
	}

	combo6_class = class_create(THIS_MODULE, "combo");
	if (IS_ERR(combo6_class)) {
		ret = PTR_ERR(combo6_class);
		goto err_chr;
	}

	return 0;
err_chr:
	unregister_chrdev(combo6_major, "combo");
err_info:
	combo6_info_done();
	comboctl_exit();
err_bus:
	combo_bus_unregister();
err:
	return ret;
}

static void __exit combo6_card_exit(void)
{
	unregister_chrdev(combo6_major, "combo");
	combo6_info_done();
	class_destroy(combo6_class);
	comboctl_exit();
	combo_bus_unregister();
}

module_init(combo6_card_init)
module_exit(combo6_card_exit)
