/*
 * Copyright (c) 2003-2010 CESNET
 * Author(s): Jiri Slaby <jirislaby@gmail.com>
 * 	      Jaroslav Kysela <perex@perex.cz>
 *
 * Licensed under the GPLv2
 */


#include <linux/device.h>
#include <linux/module.h>
#include <linux/pci.h> /* dev_* device, remove later */

#include "combo6k.h"
#include "core.h"

static const struct combo_device_id *combo_match_id(const struct combo6 *combo,
		const struct combo_driver *cdrv)
{
	const struct combo_device_id *id;

	for (id = cdrv->id_table; id->id_hsw; id++)
		if (id->id_lsw <= combo->id_sw && combo->id_sw <= id->id_hsw &&
		    /* text matching */
		    (id->id_text[0] == '\0' ||
		     !strcmp(id->id_text, combo->id_txt)))
			return id;

	return NULL;
}

static int combo_bus_match(struct device *dev, struct device_driver *drv)
{
	struct combo_driver *cdrv = container_of(drv, struct combo_driver, drv);
	struct combo6 *combo = container_of(dev, struct combo6, dev);

	if ((combo->is_6x && cdrv->dhw != DHW_COMBO6X) ||
	    (combo->is_v2 && cdrv->dhw != DHW_COMBOV2))
		return 0;

	return combo_match_id(combo, cdrv) != NULL;
}

static int combo_device_probe(struct device *dev)
{
	struct combo_driver *cdrv =
		container_of(dev->driver, struct combo_driver, drv);
	struct combo6 *combo6 = container_of(dev, struct combo6, dev);
	const struct combo_device_id *id;
	int interfaces, err;

	if (combo6->driver)
		return -ENODEV;

	if (!test_bit(COMBO_BOOT_OK, combo6->flags) ||
			test_bit(COMBO_BOOTING, combo6->flags))
		return -EBADFD;

	if (!combo6->is_v2 && !try_module_get(cdrv->drv.owner))
		return -EBUSY;

	id = combo_match_id(combo6, cdrv);
	if (!id)
		return -ENODEV;

	if (combo6->ops->go_attach)
		combo6->ops->go_attach(combo6);
	interfaces = combo6->addon_interfaces;
	if (id->intfs > 0 && id->intfs < interfaces)
		interfaces = id->intfs;
	err = cdrv->attach(combo6, id, interfaces);
	if (err < 0) {
		if (combo6->ops->detached)
			combo6->ops->detached(combo6);
		if (!combo6->is_v2)
			module_put(cdrv->drv.owner);
		return err;
	}
	dev_info(&combo6->pci->dev, "driver 0x%x (%s) successfully "
			"attached\n", combo6->id_sw, combo6->id_txt);
	/* enable interrupts and so on */
	if (combo6->ops->attached)
		combo6->ops->attached(combo6);

	/* protect against proc_info */
	mutex_lock(&combo6->drv_mutex);
	combo6->driver = cdrv;
	mutex_unlock(&combo6->drv_mutex);

	return 0;
}

static int combo_device_remove(struct device *dev)
{
	struct combo6 *combo6 = container_of(dev, struct combo6, dev);
	struct combo_driver *cdrv = combo6->driver;
	int err;

	if (cdrv) {
		atomic_inc(&combo6->detaching);
		if (atomic_read(&combo6->ref_count) > 0) {
			err = -EBUSY;
			goto err_dec;
		}
		/* disable interrupts and so on */
		if (combo6->ops->go_detach)
			combo6->ops->go_detach(combo6);
		err = cdrv->detach(combo6);
		if (err < 0)
			goto err_dec;

		if (combo6->ops->detached)
			combo6->ops->detached(combo6);
		dev_info(&combo6->pci->dev, "driver 0x%x (%s) "
				"successfully detached\n",
				combo6->id_sw, combo6->id_txt);
		mutex_lock(&combo6->drv_mutex);
		combo6->driver = NULL;
		mutex_unlock(&combo6->drv_mutex);
		if (!combo6->is_v2)
			module_put(cdrv->drv.owner);
		atomic_dec(&combo6->detaching);
	}

	return 0;
err_dec:
	atomic_dec(&combo6->detaching);
	return err;
}

static int combo_uevent(struct device *dev, struct kobj_uevent_env *env)
{
	return 0;
}

struct bus_type combo_bus_type = {
	.name	= "combo",
	.match	= combo_bus_match,
	.probe	= combo_device_probe,
	.remove	= combo_device_remove,
	.uevent	= combo_uevent,
};

int combo_bus_register(void)
{
	return bus_register(&combo_bus_type);
}

void combo_bus_unregister(void)
{
        bus_unregister(&combo_bus_type);
}

/**
 * combo6_ioctl_bus_attach - detach device from its driver
 * @combo6: structure to attach
 *
 * Returns 0 on success, otherwise negative errno.
 */
int combo6_ioctl_bus_attach(struct combo6 *combo6)
{
	return device_attach(&combo6->dev);
}
EXPORT_SYMBOL(combo6_ioctl_bus_attach);

/**
 * combo6_ioctl_bus_detach - detach device from its driver
 * @combo6: structure to detach
 *
 * Returns 0 on success, otherwise negative errno.
 */
int combo6_ioctl_bus_detach(struct combo6 *combo6)
{
	device_release_driver(&combo6->dev);

	return 0;
}
EXPORT_SYMBOL(combo6_ioctl_bus_detach);

/**
 * combo6_device_register - register a new combo device driver
 * @drv: driver to register
 * @combo6: combo to bind with @drv
 *
 * Returns 0 on success, otherwise negative errno.
 */
int combo_register_driver(struct combo_driver *drv)
{
	drv->drv.bus = &combo_bus_type;

	return driver_register(&drv->drv);
}
EXPORT_SYMBOL_GPL(combo_register_driver);

/**
 * combo_unregister_driver - unregister a combo device driver
 * @drv: driver to unregister
 * @combo6: combo to unbind from @drv
 *
 * Returns 0 on success, otherwise negative errno.
 */
void combo_unregister_driver(struct combo_driver *drv)
{
	driver_unregister(&drv->drv);
}
EXPORT_SYMBOL_GPL(combo_unregister_driver);
