#include "../../../kernel/drivers/third/cv2eth.h"

#define TX_UP_SIZE 8  // 8B = 2 * 32bits

struct cv2eth_desc {
	u32 length;
	u32 flags;
	u64 addr;
};

#define DESC_SIZE  (sizeof(struct cv2eth_desc))

struct mem_block {
	unsigned int size;
	void* virt;
	dma_addr_t phys;
};

struct tx_info {
	struct mem_block up;
	struct mem_block ring;
	unsigned int pkt_size;
	unsigned int ifc_index;
};

inline void cv2eth_write_phys_addr(struct combo6* combo6, u32 offset, dma_addr_t phys) {
	u32* val;
	u32 high;
	val = (u32*)&phys;
	combo6_fpga_writel(combo6, offset, *val);
	high = (sizeof(dma_addr_t) == 4) ? 0x0 : *(val + 1);
	combo6_fpga_writel(combo6, offset + 4, high);
}

static inline void cv2eth_fill_desc(struct cv2eth_desc *d, u64 addr, u32 len, u32 f) {
	d->flags = cpu_to_le32(f);
	d->length = cpu_to_le32(len);
	d->addr = cpu_to_le64(addr);
}

static void pac_init_tx(struct combo6 *combo6, struct tx_info *tx)
{
	struct cv2eth_desc* p;
	int i = 0;
	unsigned int set_intf = 0;
	unsigned int ifc = tx->ifc_index;


	combo6->private_data = tx;
	tx->up.size = TX_UP_SIZE;
	tx->up.virt = pci_alloc_consistent(combo6->pci, tx->up.size,
		               &tx->up.phys);

	/* set TX global upload addr */
	cv2eth_write_phys_addr(combo6,
				PAC_HW_REG(0, PAC_DIR_TX, PAC_HW_REG_TXGL), tx->up.phys);

	tx->ring.size = PAGE_SIZE;
	tx->ring.virt = pci_alloc_consistent(combo6->pci, tx->ring.size,
		               &tx->ring.phys);

	p = (struct cv2eth_desc*) tx->ring.virt;
	for ( i = 0; i < (PAGE_SIZE/DESC_SIZE) - 1; i++, p++) {
   		set_intf = ((i & 0x1F) == 0) ? CV2ETH_HW_INTF : 0;
		cv2eth_fill_desc(p, tx->ring.phys, tx->pkt_size, CV2ETH_HW_LFF | set_intf);

	}
	cv2eth_fill_desc(p, tx->ring.phys, tx->pkt_size, CV2ETH_HW_NDF);

	cv2eth_write_phys_addr(combo6, PAC_HW_DESC_CHANNEL(ifc, PAC_DIR_TX), 0);
	PDEBUG("TX%d clearing head %x\n", ifc, PAC_HW_DESC_CHANNEL(ifc, PAC_DIR_TX));

	cv2eth_write_phys_addr(combo6, PAC_HW_DESC_INIT(ifc, PAC_DIR_TX), tx->ring.phys);
	PDEBUG("TX%d initial desc address %lu\n", ifc, (unsigned long)tx->ring.phys);

	combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_TAIL), 0);
	PDEBUG("TX%d initial tail %u\n", ifc, 0);

	combo6_fpga_writel(combo6, CV2ETH_HW_OBUF_EN(ifc), 0x1); /* enable obuf */

	combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_TOUT),
			PAC_HW_TX_TIMEOUT); /* set TX timeout */
	PDEBUG("TX%d initial timeout %u offset 0x%X\n", ifc, PAC_HW_TX_TIMEOUT,
			PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_TOUT));
}

static void pac_start_tx(struct combo6 *combo6)
{
	struct tx_info *tx = combo6->private_data;
	unsigned int ifc = tx->ifc_index;
	unsigned int stat = 0;
	unsigned int counter = 0;

	combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_CTRL),
			PAC_HW_CTRL_RUN); /* start Tx */

	while (1) {
		stat = combo6_fpga_readl(combo6, PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_STATUS));

		if (stat == PAC_HW_STAT_RUN)
			break;
		if (counter++ > 100) {
			dev_warn(&combo6->pci->dev, "card didn't become ready "
					"in 100 us (no running after "
					"start), stat tx=%.8x!\n", stat);
			break;
		}
		udelay(1);
	}
}

static void pac_stop_tx(struct combo6 *combo6)
{
	struct tx_info *tx = combo6->private_data;
	unsigned int ifc = tx->ifc_index;
	unsigned int stat = 0;
	unsigned int counter = 0;

	combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_CTRL),
			PAC_HW_CTRL_STP); /* stop TX */

	while (1) {
		stat = combo6_fpga_readl(combo6, PAC_HW_REG(ifc, PAC_DIR_TX,
					PAC_HW_REG_STATUS));

		if (stat == PAC_HW_STAT_STP)
			break;
		if (counter++ > 100) {
			dev_warn(&combo6->pci->dev, "card didn't stop "
					"in 100 us (no stopped after "
					"stop), stat tx=%.8x!\n", stat);
			break;
		}
		udelay(1);
	}
}

static void pac_tx_irq(struct combo6 *combo6)
{
	struct tx_info *tx = combo6->private_data;
	unsigned int ifc = tx->ifc_index;
	volatile u32* tx_counters;
	unsigned int i = 0;
	u32 new_pkts;

	tx_counters = (u32*)tx->up.virt;          /* get the tx counters */
	for (i = 0; i < 2; i++) {
		new_pkts = le32_to_cpu(tx_counters[i]);
		combo6_fpga_writel(combo6,
				PAC_HW_REG(i, PAC_DIR_TX, PAC_HW_REG_TOUT),
				PAC_HW_TX_TIMEOUT); /* set TX timeout */
		if (printk_ratelimit())
			PDEBUG("TX%d new pkts %d\n", i, new_pkts);
	    combo6_fpga_writel(combo6, PAC_HW_REG(ifc, PAC_DIR_TX, PAC_HW_REG_TAIL),
				                    new_pkts + 16); /* set TX tail */

	}
}

