#include "kocompat.h"

#include <linux/version.h>

#include <linux/kobject.h> /* needed for old kernels before cdev.h */
#include <linux/fs.h> /* needed for old kernels before cdev.h */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 23)
#include <szedata.h>
#include <szedatak.h>

struct szedata2_app;

static int szedata2_ring_mmap(struct szedata2_app *, struct vm_area_struct *);
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 18)
static int szedata2_mmap(struct file *, struct vm_area_struct *);
#endif

#include "../../../kernel/drivers/szedata2/char.c"

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 23)
static struct page *szedata2_ring_mmap_nopage(struct vm_area_struct *vma,
		unsigned long address, int *type)
{
	struct szedata2_app *app = vma->vm_private_data;
	struct szedata2 *sd = app->sd;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
	unsigned int space = szedata2_find_space(app, offset);
	struct page *page;
	void *virt;

	offset += address - vma->vm_start;
	offset -= app->status_page->offsets[space];
	if (offset >= app->status_page->sizes[space])
		return NOPAGE_SIGBUS;

	szedata2_get_virt(sd, space, 0, &virt, offset, PAGE_SIZE);
	page = virt_to_page(virt);
	get_page(page);

	if (type)
		*type = VM_FAULT_MINOR;

	return page;
}

static int szedata2_ring_mmap(struct szedata2_app *app,
		struct vm_area_struct *vma)
{
	static struct vm_operations_struct vm_ops = {
		.nopage = szedata2_ring_mmap_nopage
	};
	unsigned long size = vma->vm_end - vma->vm_start;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
	unsigned int space = szedata2_find_space(app, offset);

	if (space == 0 && !app->sd->ring[0].blk_count)
		return -ENODEV;

	if (space != SZE2_MMIO_TX &&
			(vma->vm_flags & (VM_WRITE | VM_READ)) != VM_READ)
		return -EINVAL;

	if (size > app->status_page->sizes[space])
		return -EINVAL;
	if (offset - app->status_page->offsets[space] >
			app->status_page->sizes[space] - size)
		return -EINVAL;

	vma->vm_ops = &vm_ops;
	vma->vm_private_data = app;
	vma->vm_flags |= VM_RESERVED;

	return 0;
}
#endif /* < 2.6.23 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 18)
static struct page *szedata2_status_mmap_nopage(struct vm_area_struct *vma,
		unsigned long address, int *type)
{
	struct szedata2_app *app = vma->vm_private_data;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
	struct page *page;

	offset += address - vma->vm_start;
	if (offset != 0)
		return NOPAGE_SIGBUS;
	page = vmalloc_to_page(app->status_page);
	get_page(page);

	if (type)
		*type = VM_FAULT_MINOR;

	return page;
}

static int szedata2_status_mmap(struct szedata2_app *app,
		struct vm_area_struct *vma)
{
	static struct vm_operations_struct vm_ops = {
		.nopage = szedata2_status_mmap_nopage
	};
	unsigned long size = vma->vm_end - vma->vm_start;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
	unsigned long mmap_bytes = PAGE_ALIGN(sizeof(*app->status_page));

	if (size > mmap_bytes)
		return -EINVAL;
	if (offset > mmap_bytes - size)
		return -EINVAL;

	vma->vm_ops = &vm_ops;
	vma->vm_private_data = app;
	vma->vm_flags |= VM_RESERVED;

	return 0;
}

static struct page *szedata2_write_size_mmap_nopage(struct vm_area_struct *vma,
		unsigned long address, int *type)
{
	struct szedata2_app *app = vma->vm_private_data;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
	struct page *page;

	offset += address - vma->vm_start;
	if (offset != PAGE_SIZE)
		return NOPAGE_SIGBUS;
	page = vmalloc_to_page(app->write_size);
	get_page(page);

	if (type)
		*type = VM_FAULT_MINOR;

	return page;
}

static int szedata2_write_size_mmap(struct szedata2_app *app,
		struct vm_area_struct *vma)
{
	static struct vm_operations_struct vm_ops = {
		.nopage = szedata2_write_size_mmap_nopage
	};
	unsigned long size = vma->vm_end - vma->vm_start;
	unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
	unsigned long mmap_bytes = PAGE_SIZE;

	if (size > mmap_bytes)
		return -EINVAL;
	if (offset - PAGE_SIZE > mmap_bytes - size)
		return -EINVAL;

	vma->vm_ops = &vm_ops;
	vma->vm_private_data = app;
	vma->vm_flags |= VM_RESERVED;

	return 0;
}

static int szedata2_mmap(struct file *filp, struct vm_area_struct *vma)
{
	struct szedata2_app *app = filp->private_data;

	if (vma->vm_pgoff >=
			((PAGE_ALIGN(sizeof(*app->status_page)) + PAGE_SIZE) >> PAGE_SHIFT))
		return szedata2_ring_mmap(app, vma);

	if (vma->vm_pgoff >=
			(PAGE_ALIGN(sizeof(*app->status_page)) >> PAGE_SHIFT))
		return szedata2_write_size_mmap(app, vma);

	if ((vma->vm_flags & (VM_WRITE|VM_READ)) != VM_READ)
		return -EINVAL;

	return szedata2_status_mmap(app, vma);
}
#endif /* < 2.6.18 */
