-- tree_stage.vhd: One stage (level) of binary search tree 
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree_stage.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
use WORK.tree_func.all;
use work.bmem_func.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture arch of tree_stage is
  constant ITEMS                : integer := 2**TREE_LEVEL;
  constant ITEM_WIDTH           : integer := 1+KEY_WIDTH+1+log2(KEY_WIDTH);
  
  signal reg_in_addr            : std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
  signal reg_in_key             : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_in_vld             : std_logic := '0';
  signal reg_in_foundable       : std_logic := '0';
  signal reg_in_chen            : std_logic := '0';
  
  signal mem_addr               : std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
  signal mem_key                : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal mem_vld                : std_logic := '0';
  signal mem_item               : std_logic_vector(ITEM_WIDTH-1 downto 0);
  signal mem_chen               : std_logic;
  signal mem_foundable          : std_logic;
  
  signal reg_mem_addr           : std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
  signal reg_mem_key            : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_mem_vld            : std_logic := '0';
  signal reg_mem_chen           : std_logic := '0';
  signal reg_mem_foundable      : std_logic := '0';
  signal reg_mem_item           : std_logic_vector(ITEM_WIDTH-1 downto 0);
  
  signal cmp_item_vld       : std_logic;
  signal cmp_item_key       : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal cmp_item_isend     : std_logic;
  signal cmp_item_length    : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal cmp                : std_logic;
  signal cmp_res            : std_logic;
  
  signal reg_out_addr       : std_logic_vector(TREE_LEVEL downto 0);
  signal reg_out_key        : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_out_vld        : std_logic := '0';
  signal reg_out_foundable  : std_logic := '0';
  signal reg_out_chen       : std_logic := '0';
  
  signal cfg_mem_re         : std_logic;
  signal cfg_mem_we         : std_logic;
  signal cfg_do_reg         : std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
  signal cfg_addr_reg       : std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
   
  type regarray_t is array (0 to ITEMS-1) of std_logic_vector(ITEM_WIDTH-1 downto 0); 
  signal regmem             : regarray_t; 
  signal regmem_cs          : std_logic_vector(ITEMS-1 downto 0);
  signal regmem_we          : std_logic_vector(ITEMS-1 downto 0);    
begin    
-- memory stage ----------------------------------------------------------------
  -- computing of write and read enable for memory
  cfg_mem_re <= CFG_EN and not CFG_WE;
  cfg_mem_we <= CFG_EN and CFG_WE;
    
  -- use BRAMs
  bram_mem_gen : if MEM_TYPE=mem_bmem generate
    -- memory
    mem : entity work.DP_BMEM
      generic map(
        DATA_WIDTH     => ITEM_WIDTH,
        ITEMS          => ITEMS,
        BRAM_TYPE      => GET_BEST_BRAM_TYPE(ITEMS,ITEM_WIDTH),
        OUTPUT_REG     => false
      ) port map (
        RSTA           => RESET,
        CLKA           => CLK,
        PIPE_ENA       => '1',
        REA            => reg_in_vld,
        WEA            => '0',
        ADDRA          => reg_in_addr,
        DIA            => (CFG_DI'range => '0'),
        DOA_DV         => mem_vld,
        DOA            => mem_item,
        RSTB           => RESET,
        CLKB           => CLK,
        PIPE_ENB       => '1',
        REB            => cfg_mem_re,
        WEB            => cfg_mem_we,
        ADDRB          => CFG_ADDR,
        DIB            => CFG_DI,
        DOB_DV         => CFG_DRDY,
        DOB            => CFG_DO
      );
    -- synchronization of addr and key to BRAM read  
    bmem_sync_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        mem_addr      <= reg_in_addr;
        mem_key       <= reg_in_key;
        mem_foundable <= reg_in_foundable;
        mem_chen      <= reg_in_chen;  
      end if;
    end process;
  end generate;
  
  -- use DISTMEMs
  distmem_mem_gen : if MEM_TYPE=mem_distmem generate
    -- memory
    mem : entity work.DP_DISTMEM
      generic map (
        DATA_WIDTH     => ITEM_WIDTH,
        ITEMS          => ITEMS,
        DISTMEM_TYPE   => GET_DISTMEM_TYPE(ITEMS),
        DEBUG          => false
      ) port map (
        RESET          => RESET,
        DI             => CFG_DI,
        WE             => cfg_mem_we,
        WCLK           => CLK,
        ADDRA          => CFG_ADDR,
        DOA            => cfg_do_reg,
        ADDRB          => reg_in_addr,
        DOB            => mem_item
      );
    -- synchronization of distmem inputs and outputs
    do_reg : process(CLK)
    begin
      if CLK'event and CLK='1' then
        CFG_DO <= cfg_do_reg;
      end if;
    end process;
    drdy_reg : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          CFG_DRDY <= '0';
        else
          CFG_DRDY <= cfg_mem_re;
        end if;
      end if;
    end process;
    mem_vld  <= reg_in_vld;
    mem_foundable <= reg_in_foundable;
    mem_chen <= reg_in_chen;
    mem_addr <= reg_in_addr;
    mem_key  <= reg_in_key;  
  end generate;
  
  -- use array of registers
  regarray_mem_gen : if MEM_TYPE=mem_regarray generate
    -- memory
    mem_gen : for i in 0 to ITEMS-1 generate
      mem_item : process (CLK)
      begin
        if CLK'event and CLK='1' then
          if RESET='1' then
            regmem(i) <= (others => '0');
          elsif regmem_we(i)='1' then
            regmem(i) <= CFG_DI;
          end if;
        end if;
      end process;
    end generate;
    -- input and output regarray interface
    mem_item_mx_gen : if ITEMS>1 generate
      mem_item <= regmem(conv_integer(reg_in_addr));
    end generate;
    mem_item_nomx_gen : if ITEMS=1 generate
      mem_item <= regmem(0);
    end generate;
    mem_vld  <= reg_in_vld;
    mem_foundable <= reg_in_foundable;
    mem_chen <= reg_in_chen;
    mem_addr <= reg_in_addr;
    mem_key  <= reg_in_key;
    -- configuration regarray interface
    addr_reg : process(CLK)
    begin
      if CLK'event and CLK='1' then
        cfg_addr_reg <= CFG_ADDR;
      end if;
    end process;
    drdy_reg : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          CFG_DRDY <= '0';
        else
          CFG_DRDY <= cfg_mem_re;
        end if;
      end if;
    end process;
    cfg_do_mx_gen : if ITEMS>1 generate
      CFG_DO <= regmem(conv_integer(cfg_addr_reg));
    end generate;
    cfg_do_nomx_gen : if ITEMS=1 generate
      CFG_DO <= regmem(0);
    end generate;
    -- write enables for regarray
    regmem_chipselect_gen : if ITEMS>1 generate
      regmem_chip_select : process(CFG_ADDR)
      begin
        regmem_cs <= (others => '0');
        for i in 0 to (ITEMS-1) loop
          if (conv_std_logic_vector(i,log2(ITEMS))=CFG_ADDR) then
            regmem_cs(i) <= '1';
          end if;
        end loop;
      end process;
    end generate;
    regmem_nochipselect_gen : if ITEMS=1 generate
      regmem_cs <= "1";
    end generate; 
    regmem_we <= regmem_cs and (ITEMS-1 downto 0 => cfg_mem_we);
  end generate;
  
  

-- comparator stage ------------------------------------------------------------  
  -- memory item decoder
  cmp_item_vld    <= reg_mem_item(0);       
  cmp_item_isend  <= reg_mem_item(1);  
  cmp_item_length <= reg_mem_item(log2(KEY_WIDTH)+1 downto 2);
  cmp_item_key    <= reg_mem_item(ITEM_WIDTH-1 downto ITEM_WIDTH-KEY_WIDTH);                
  
  -- comparator of keys (NOTE: different behaviour on key equality for start and end of range)  
  cmp <= '1' when (reg_mem_key&'0')>=(cmp_item_key&cmp_item_isend) else '0';
  
  -- comparision result validation
  cmp_res <= cmp and cmp_item_vld;
  
  -- results computing
  not_level0_gen : if TREE_LEVEL>0 generate
    reg_out_addr(TREE_LEVEL downto 1) <= reg_mem_addr;
  end generate;
  reg_out_addr(0) <= cmp_res;
  reg_out_key     <= reg_mem_key;
  reg_out_vld     <= reg_mem_vld;
  reg_out_foundable <= reg_mem_foundable;
  reg_out_chen    <= reg_mem_chen;

  
  
-- registers -------------------------------------------------------------------
  -- input interface registers
  input_reg_gen : if INPUT_REG generate
    input_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        reg_in_addr <= IN_ADDR;
        reg_in_key  <= IN_KEY;
        reg_in_vld  <= IN_VLD;
        reg_in_foundable <= IN_FOUNDABLE;
        reg_in_chen <= IN_CHANGE_EN;    
      end if;
    end process;
  end generate;  
  input_noreg_gen : if not INPUT_REG generate
    reg_in_addr <= IN_ADDR;
    reg_in_key  <= IN_KEY;
    reg_in_vld  <= IN_VLD;
    reg_in_foundable <= IN_FOUNDABLE;
    reg_in_chen <= IN_CHANGE_EN;  
  end generate;

  -- middle registers
  middle_reg_gen : if MIDDLE_REG generate
    middle_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        reg_mem_addr <= mem_addr;
        reg_mem_key  <= mem_key;
        reg_mem_vld  <= mem_vld;
        reg_mem_item <= mem_item;
        reg_mem_foundable <= mem_foundable;
        reg_mem_chen <= mem_chen;
      end if;
    end process;
  end generate;
  middle_noreg_gen : if not MIDDLE_REG generate
    reg_mem_addr <= mem_addr;
    reg_mem_key  <= mem_key;
    reg_mem_vld  <= mem_vld;
    reg_mem_item <= mem_item;
    reg_mem_foundable <= mem_foundable;
    reg_mem_chen <= mem_chen;  
  end generate;
  
  -- output interface registers
  output_reg_gen : if OUTPUT_REG generate
    output_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        OUT_ADDR      <= reg_out_addr;
        OUT_KEY       <= reg_out_key;
        OUT_VLD       <= reg_out_vld;
        OUT_FOUNDABLE <= reg_out_foundable;
        OUT_CHANGE_EN <= reg_out_chen;  
      end if;
    end process;
  end generate;
  output_noreg_gen : if not OUTPUT_REG generate
    OUT_ADDR      <= reg_out_addr;
    OUT_KEY       <= reg_out_key;
    OUT_VLD       <= reg_out_vld;
    OUT_FOUNDABLE <= reg_out_foundable;
    OUT_CHANGE_EN <= reg_out_chen;   
  end generate;
  
end architecture;
