-- tag_manager.vhd: Tag manager component - architecture
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                        Architecture declaration
-- ----------------------------------------------------------------------------

architecture tag_manager_arch of tag_manager is

   -- Signals declaration -----------------------------------------------------

   -- tags utilization management signals
   signal utilized_tags          : std_logic_vector(31 downto 0);
   signal reg_utilized_tags      : std_logic_vector(31 downto 0);
   signal tag_offer_sig          : std_logic_vector(4 downto 0);
   signal tag_offer_vld_sig      : std_logic;
   signal offer_out_regs_we      : std_logic;
   signal reg_tag_offer          : std_logic_vector(4 downto 0);
   signal reg_tag_offer_vld      : std_logic;

   -- translation memory signals
   signal translation_write_data : std_logic_vector(15 downto 0);
   signal translation_we          : std_logic;
   signal translation_read_data  : std_logic_vector(15 downto 0);


-- ----------------------------------------------------------------------------
--                             Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --          Management of available and utilized tag translations
   -- -------------------------------------------------------------------------

   -- utilized tags management logic ------------------------------------------
   --    '0' on bit i means, that PCIe tag i is not currently utilized
   --    '1' on bit i means, that PCIe tag i is mapped on the tuple (DMA tag,
   --        DMA ID) stored on address i in the translation memory
   utilized_tags_mgmt_p : process(reg_utilized_tags, tag_offer_sig,
                                  tag_offer_vld_sig, offer_out_regs_we, TAG,
                                  TAG_RELEASE)
   begin
      -- default value
      utilized_tags <= reg_utilized_tags;

      -- add new tag mapping
      if (tag_offer_vld_sig = '1' AND offer_out_regs_we = '1') then
         utilized_tags(conv_integer(tag_offer_sig)) <= '1';
      end if;

      -- remove old tag mapping
      if (TAG_RELEASE = '1') then
         utilized_tags(conv_integer(TAG)) <= '0';
      end if;
   end process utilized_tags_mgmt_p;

   -- utilized tags register --------------------------------------------------
   reg_utilized_tags_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            reg_utilized_tags <= (others => '0');
         else
            reg_utilized_tags <= utilized_tags;
         end if;
      end if;
   end process reg_utilized_tags_p;

   -- logic to determine the lowest available tag -----------------------------
   lowest_available_tag_p : process(reg_utilized_tags)
   begin
      -- default values
      tag_offer_sig     <= (others => '0');
      tag_offer_vld_sig <= '0';

      for i in 0 to 31 loop
         if (reg_utilized_tags(i) = '0') then
            tag_offer_sig     <= conv_std_logic_vector(i,5);
            tag_offer_vld_sig <= '1';
            exit;
         end if;
      end loop;
   end process lowest_available_tag_p;

   -- write enable logic for offer output registers ---------------------------
   offer_out_regs_we <= -- acceptation of valid offer
                        '1' when (translation_we = '1') else
                        -- new offer is available
                        '1' when (reg_tag_offer_vld = '0' AND
                                  tag_offer_vld_sig = '1') else
                        '0';

   -- tag_offer output register -----------------------------------------------
   reg_tag_offer_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (offer_out_regs_we = '1') then
            reg_tag_offer <= tag_offer_sig;
         end if;
      end if;
   end process reg_tag_offer_p;

   -- tag_offer_vld output register -------------------------------------------
   reg_tag_offer_vld_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            reg_tag_offer_vld <= '0';
         elsif (offer_out_regs_we = '1') then
            reg_tag_offer_vld <= tag_offer_vld_sig;
         end if;
      end if;
   end process reg_tag_offer_vld_p;


   -- -------------------------------------------------------------------------
   --                         Translation memory 
   -- -------------------------------------------------------------------------

   -- composition of input signals to the translation memory
   translation_write_data <= DMA_UP_HDR_TAG & DMA_UP_HDR_ID;
   translation_we         <= reg_tag_offer_vld AND TAG_OFFER_ACCEPT;

   -- instantiation of memory for translations
   translation_mem_i : entity work.DP_DISTMEM
   generic map (
      DATA_WIDTH => 16,
      ITEMS      => 32
   )
   port map (
      -- Common interface
      WCLK       => CLK,
      RESET      => RESET,

      -- R/W Port
      ADDRA      => reg_tag_offer,
      DI         => translation_write_data,
      WE         => translation_we,
      DOA        => open,

      -- Read Port
      ADDRB      => TAG,
      DOB        => translation_read_data
   );


   -- -------------------------------------------------------------------------
   --                  Assigning values to output ports
   -- -------------------------------------------------------------------------

   -- ports of interface to dma_up_int component
   TAG_OFFER        <= reg_tag_offer;
   TAG_OFFER_VLD    <= reg_tag_offer_vld;

   -- ports of interface to dma_down_int component
   DMA_DOWN_HDR_TAG <= translation_read_data(15 downto 8); 
   DMA_DOWN_HDR_ID  <= translation_read_data(7 downto 0); 

end architecture tag_manager_arch;
