-- dma_ctrl_update.vhd: Update process for DMA Controller
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- TODO: fix reset timeout, remove tri-state logic

library IEEE;
use IEEE.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;

library work;
use work.math_pack.all;
use work.dma_pkg.all;

entity dma_ctrl_update is
   generic (
      --! Count of channels
      CHANNELS          : integer;
      --! Unique DMA Bus Unit ID
      UNIT_ID           : std_logic_vector(7 downto 0);
      --! DMA Bus data width
      DMA_DATA_WIDTH    : integer := 512;
      --! Data width of one value
      VALUE_WIDTH       : integer := 32;
      --! Generate FLUSH before update
      FLUSH             : boolean := false
   );
   port (
      CLK               : in  std_logic;
      RESET             : in  std_logic;

      --! Enable for each channel
      ENABLE            : in  std_logic_vector(CHANNELS-1 downto 0);

      --! Address in RAM for value update of first index. Must be valid when any index is enabled.
      UPDATE_ADDR       : in  std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);

      --! Interface for DMA opertion. When UPDATE is 1, the other signals must be valid
      --! Update signal
      UPDATE            : in  std_logic;
      --! Channel of value to be updated
      UPDATE_CHANNEL    : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      --! New value
      UPDATE_VALUE      : in  std_logic_vector(VALUE_WIDTH-1 downto 0);
      --! Generate interrupt after value update
      UPDATE_INTERRUPT  : in  std_logic;

      --! Interface for getting Timeout register value
      TIMEOUT_CHANNEL   : out std_logic_vector(log2(CHANNELS)-1 downto 0);
      TIMEOUT           : in  std_logic_vector(VALUE_WIDTH-1 downto 0);

      --! Generated interrupt
      INTERRUPT_CHANNEL : out std_logic_vector(log2(CHANNELS)-1 downto 0);
      INTERRUPT         : out std_logic;

      --! Interface for generating DMA requests
      DMA_UP_DATA       : out std_logic_vector(DMA_DATA_WIDTH-1 downto 0);
      DMA_UP_HDR        : out std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
      DMA_UP_SOP        : out std_logic;
      DMA_UP_EOP        : out std_logic;
      DMA_UP_SRC_RDY    : out std_logic;
      DMA_UP_DST_RDY    : in  std_logic;

      --! Interface for receiving completed requests
      DMA_DOWN_DATA     : in  std_logic_vector(DMA_DATA_WIDTH-1 downto 0);
      DMA_DOWN_HDR      : in  std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
      DMA_DOWN_SOP      : in  std_logic;
      DMA_DOWN_EOP      : in  std_logic;
      DMA_DOWN_SRC_RDY  : in  std_logic;
      DMA_DOWN_DST_RDY  : out std_logic
    );
end entity;


architecture behavioral of dma_ctrl_update is

   type t_state is (S_RESET, S_INIT, S_FLUSH, S_FLUSH_WAIT, S_UPDATE, S_SYNC, S_SYNC_WAIT, S_INTERRUPT, S_NEXT);
   signal present_state, next_state : t_state;

   type   t_ptr                 is array(CHANNELS-1 downto 0) of std_logic_vector(VALUE_WIDTH-1 downto 0);

   --! UPDATE signals
   signal iUPDATE_CHANNEL     : integer := 0;

   signal reg_value           : t_ptr;
   signal reg_interrupt       : std_logic_vector(CHANNELS-1 downto 0);
   signal reg_enableUpdate    : std_logic_vector(CHANNELS-1 downto 0);
   signal reg_needUpdate      : std_logic_vector(CHANNELS-1 downto 0);

   --! Timeout signals
   signal reg_timeout_w       : std_logic_vector(VALUE_WIDTH-1 downto 0);
   signal reg_timeout_we      : std_logic;
   signal reg_timeout         : std_logic_vector(VALUE_WIDTH-1 downto 0);

   signal reg_timeout_channel : std_logic_vector(log2(CHANNELS)-1 downto 0) := (others => '0');
   signal iReg_timeout_channel: integer := 0;

   --! FSM signals
   signal reg_channel         : std_logic_vector(log2(CHANNELS)-1 downto 0) := (others => '0');
   signal iReg_channel        : integer := 0;

   signal reg_flushValue      : std_logic_vector(VALUE_WIDTH-1 downto 0);

   signal hdr_read            : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal hdr_flush           : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
   signal hdr_update          : std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);

   signal reg_update_addr     : std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);

begin

   -- Update process is divided to 3 parts:
   -- 1) Input process     - New value for update is asserted by UPDATE signal - Enable update is set to active
   -- 2) Timeout process   - When enable update is active and timeout reached, the value needs to be updated
   -- 3) FSM process       - Stored value that needs update is sent to RAM

   DMA_DOWN_DST_RDY                 <= '1';

   DMA_UP_EOP                       <= '1';
   DMA_UP_SOP                       <= '1';

   DMA_UP_DATA                      <= std_logic_vector(to_unsigned(0, DMA_DATA_WIDTH-VALUE_WIDTH))
                                       & reg_flushValue(VALUE_WIDTH-1 downto 0);

   hdr_read(DMA_REQUEST_GLOBAL)     <= reg_update_addr;
   hdr_read(DMA_REQUEST_LENGTH)     <= "00000000001";
   hdr_read(DMA_REQUEST_TYPE)       <= DMA_TYPE_READ;
   hdr_read(DMA_REQUEST_TAG)        <= (others => '0');
   hdr_read(DMA_REQUEST_UNITID)     <= UNIT_ID;

   hdr_flush(DMA_REQUEST_GLOBAL)    <= reg_update_addr;
   hdr_flush(DMA_REQUEST_LENGTH)    <= "00000000001";
   hdr_flush(DMA_REQUEST_TYPE)      <= DMA_TYPE_FLUSH;
   hdr_flush(DMA_REQUEST_TAG)       <= (others => '0');
   hdr_flush(DMA_REQUEST_UNITID)    <= UNIT_ID;

   hdr_update(DMA_REQUEST_GLOBAL)   <= reg_update_addr;
   hdr_update(DMA_REQUEST_LENGTH)   <= "00000000001";
   hdr_update(DMA_REQUEST_TYPE)     <= DMA_TYPE_WRITE;
   hdr_update(DMA_REQUEST_TAG)      <= (others => '0');
   hdr_update(DMA_REQUEST_UNITID)   <= UNIT_ID;

   INTERRUPT_CHANNEL                <= reg_channel;
   TIMEOUT_CHANNEL                  <= reg_timeout_channel;
   iUPDATE_CHANNEL                  <= conv_integer(UPDATE_CHANNEL);
   iReg_timeout_channel             <= conv_integer(reg_timeout_channel);
   iReg_channel                     <= conv_integer(reg_channel);

   reg_timeout_w                    <= reg_timeout - CHANNELS when reg_timeout >= CHANNELS else TIMEOUT;
   reg_timeout_we                   <= '1' when reg_timeout >= CHANNELS or reg_enableUpdate(iReg_timeout_channel) = '1'  else '0';

   processes_p: process(CLK, RESET)
   begin
      if(CLK'event AND CLK = '1') then
         reg_timeout_channel                 <= reg_timeout_channel + 1;
         reg_update_addr                     <= UPDATE_ADDR + (reg_channel & "00");

         -- Input process
         if(UPDATE = '1') then
            reg_enableUpdate(iUPDATE_CHANNEL)<= '1';
            reg_value(iUPDATE_CHANNEL)       <= UPDATE_VALUE;
            reg_interrupt(iUPDATE_CHANNEL)   <= UPDATE_INTERRUPT;
         end if;

         -- Timeout process
         if(reg_timeout < CHANNELS and reg_enableUpdate(iReg_timeout_channel) = '1') then
            reg_needUpdate(iReg_timeout_channel)   <= '1';
            reg_enableUpdate(iReg_timeout_channel) <= '0';
         end if;

         if(present_state = S_NEXT) then
            reg_channel                      <= reg_channel + 1;
         end if;

         -- FSM process
         if(present_state = S_FLUSH) then
            reg_flushValue                   <= reg_value(iReg_channel);
            reg_needUpdate(iReg_channel)     <= '0';
         elsif(present_state = S_RESET) then
            reg_needUpdate(iReg_channel)     <= '0';
            reg_enableUpdate(iReg_channel)   <= '0';
         end if;
      end if;
   end process;

   -- --------------- Sync logic -------------------------------------------
   sync_logic: process(CLK, RESET, next_state)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            present_state <= S_INIT;
         else
            present_state <= next_state;
         end if;
      end if;
   end process;

   -- ------------------ Next state logic -------------------------------------
   next_state_logic: process(present_state, ENABLE, reg_needUpdate, iReg_channel, DMA_UP_DST_RDY, DMA_DOWN_SRC_RDY, reg_interrupt)
   begin
      next_state <= present_state;

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT =>
            if(ENABLE(iReg_channel) = '0') then
               next_state <= S_RESET;
            elsif(reg_needUpdate(iReg_channel) = '1') then
               next_state <= S_FLUSH;
            else
               next_state <= S_NEXT;
            end if;
         -- ---------------------------------------------
         when S_FLUSH =>
            if(FLUSH = false) then
               next_state <= S_UPDATE;
            elsif(DMA_UP_DST_RDY = '1') then
               next_state <= S_FLUSH_WAIT;
            end if;
         -- ---------------------------------------------
         when S_FLUSH_WAIT =>
            if(DMA_DOWN_SRC_RDY = '1') then
               next_state <= S_UPDATE;
            end if;
         -- ---------------------------------------------
         when S_UPDATE =>
            if(DMA_UP_DST_RDY = '1') then
               if(reg_interrupt(iReg_channel) = '1') then
                  next_state <= S_SYNC;
               else
                  next_state <= S_NEXT;
               end if;
            end if;
         -- ---------------------------------------------
         when S_SYNC =>
            if(DMA_UP_DST_RDY = '1') then
               next_state <= S_SYNC_WAIT;
            end if;
         -- ---------------------------------------------
         when S_SYNC_WAIT =>
            if(DMA_DOWN_SRC_RDY = '1') then
               next_state <= S_INTERRUPT;
            end if;
         -- ---------------------------------------------
         when S_INTERRUPT =>
            next_state <= S_NEXT;
         -- ---------------------------------------------
         when S_RESET =>
            next_state <= S_NEXT;
         -- ---------------------------------------------
         when S_NEXT  =>
            next_state <= S_INIT;
         -- ---------------------------------------------
      end case;
   end process;

   -- ------------------ Output logic -----------------------------------------
   output_logic: process(present_state, hdr_update, hdr_flush, DMA_UP_DST_RDY, iReg_channel, hdr_read)
   begin
      DMA_UP_SRC_RDY             <= '0';
      DMA_UP_HDR                 <=  hdr_update;
      INTERRUPT                  <= '0';

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT =>
         -- ---------------------------------------------
         when S_FLUSH =>
            if(FLUSH = true) then
               DMA_UP_HDR        <= hdr_flush;
               DMA_UP_SRC_RDY    <= '1';
            end if;
         -- ---------------------------------------------
         when S_FLUSH_WAIT =>
         -- ---------------------------------------------
         when S_UPDATE =>
            DMA_UP_SRC_RDY       <= '1';
         -- ---------------------------------------------
         when S_SYNC =>
            DMA_UP_HDR           <= hdr_read;
            DMA_UP_SRC_RDY       <= '1';
         -- ---------------------------------------------
         when S_INTERRUPT =>
            INTERRUPT            <= '1';
         -- ---------------------------------------------
         when S_RESET =>
         -- ---------------------------------------------
         when S_NEXT =>
         -- ---------------------------------------------
         when others =>
      end case;
   end process;

   reg_timeout_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 32,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> reg_timeout_w,
      WE      			=> reg_timeout_we,
      ADDR   			=> reg_timeout_channel,
      DO     			=> reg_timeout
   );

end architecture;
