-- dma_ctrl_rx.vhd: Generic RX DMA Controller - Top level
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library IEEE;
use IEEE.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;
use work.math_pack.all;

use work.dma_pkg.all;

entity dma_ctrl_tx is
   generic (
      --! Channel count, needs to be a power of 2
      CHANNELS          : integer := 8;
      --! Size of HW Buffer in bytes
      BUFFER_SIZE       : integer := 8192;
      --! Buffer count. It depends on data width and align
      BUFFERS           : integer := 8;
      --! Data align
      DATA_ALIGN        : integer := 8;

      TAG_WIDTH         : integer := 4
   );
   port (
      --! Common signals
      CLK               : in  std_logic;
      RESET             : in  std_logic;

      --! Reflects state of an channel
      ENABLE            : out std_logic_vector(CHANNELS-1 downto 0);
      --! Signal for generating an interrupt
      INTERRUPT         : out std_logic;

      --! RAM address for pointer update
      CONFIG_ADDR       : in  std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);

      --! Interface from DMA Buffer - when new data arrives.
      PACKET_SENT       : in  std_logic;
      PACKET_CHANNEL    : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      PACKET_LENGTH     : in  std_logic_vector(log2(BUFFER_SIZE) downto 0);

      --! Interface to DMA Buffer - when data needs to be transfered.
      BUFFER_WRITE      : out std_logic_vector(log2(BUFFERS) downto 0);
      BUFFER_CHANNEL    : out std_logic_vector(log2(CHANNELS)-1 downto 0);
      BUFFER_DATA       : out std_logic_vector(511 downto 0);
      BUFFER_ADDRESS    : out std_logic_vector(log2(BUFFER_SIZE/DATA_ALIGN)-1 downto 0);
      BUFFER_DONE       : out std_logic;
      BUFFER_DONE_SIZE  : out std_logic_vector(log2(BUFFER_SIZE/DATA_ALIGN) downto 0);
      BUFFER_DONE_CHANNEL : out std_logic_vector(log2(CHANNELS)-1 downto 0);

      --! Interface from Descriptor manager
      DESC_DATA         : in  std_logic_vector(DMA_DESC_WIDTH-1 downto 0);
      DESC_RDY          : in  std_logic;
      DESC_UPDATE       : out std_logic;
      DESC_UPDATE_LEN   : out std_logic_vector(PAGE_WIDTH downto 0);
      DESC_CHANNEL      : out std_logic_vector(log2(CHANNELS)-1 downto 0);

      --! Interface for generating DMA requests
      DMA_UP_DATA       : out std_logic_vector(511 downto 0);
      DMA_UP_HDR        : out std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
      DMA_UP_SOP        : out std_logic;
      DMA_UP_EOP        : out std_logic;
      DMA_UP_SRC_RDY    : out std_logic;
      DMA_UP_DST_RDY    : in  std_logic;

      --! Interface for receiving completed requests
      DMA_DOWN_DATA     : in  std_logic_vector(511 downto 0);
      DMA_DOWN_HDR      : in  std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
      DMA_DOWN_SOP      : in  std_logic;
      DMA_DOWN_EOP      : in  std_logic;
      DMA_DOWN_SRC_RDY  : in  std_logic;
      DMA_DOWN_DST_RDY  : out std_logic;

      --! SW Access for configuration
      MI_ADDR           : in  std_logic_vector(31 downto 0);
      MI_ARDY           : out std_logic;
      MI_BE             : in  std_logic_vector(3 downto 0);
      MI_DRD            : out std_logic_vector(31 downto 0);
      MI_DRDY           : out std_logic;
      MI_DWR            : in  std_logic_vector(31 downto 0);
      MI_RD             : in  std_logic;
      MI_WR             : in  std_logic
   );
end;

architecture behavioral of dma_ctrl_tx is

   constant zeros                   : std_logic_vector(63 downto 0) := (others => '0');
   constant SWPTR_WIDTH             : integer := 32;

   --! Pending FIFO constants

   constant FIFO_COMPLETE           :integer := 0;
   subtype FIFO_CHANNEL       is natural range PAGE_WIDTH-log2(DATA_ALIGN)+log2(CHANNELS) downto PAGE_WIDTH-log2(DATA_ALIGN);
   subtype FIFO_LENGTH        is natural range PAGE_WIDTH-log2(DATA_ALIGN) downto  1;

   --! New data process signals
   signal newdata_control           : std_logic_vector(1 downto 0);
   signal newdata_status            : std_logic_vector(7 downto 0);
   signal newdata_channel           : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal newdata_swendptr          : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal newdata_swbufmask         : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal newdata_interrupt         : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal newdata_interrupt_en      : std_logic;
   signal newdata_maxreq            : std_logic_vector(PAGE_WIDTH downto 0);

   signal ctrl_enable               : std_logic_vector(CHANNELS-1 downto 0);

   --! DMA Request process signals
   signal dma_length                : std_logic_vector(PAGE_WIDTH - log2(DATA_ALIGN) downto 0);
   signal dma_req                   : std_logic;
   signal dma_rdy                   : std_logic;
   signal dma_done                  : std_logic;
   signal dma_pointer               : std_logic_vector(31 downto 0);

   signal update_dma_donecomplete   : std_logic;
   signal update_dma_interrupt      : std_logic;
   signal update_dma_pointer        : std_logic_vector(31 downto 0);
   signal update_dma_channel        : std_logic_vector(log2(CHANNELS)-1 downto 0);

   signal reg_update_dma_donecomplete: std_logic;
   signal reg_update_dma_pointer    : std_logic_vector(31 downto 0);
   signal reg_update_dma_channel    : std_logic_vector(log2(CHANNELS)-1 downto 0);


   signal update_channel            : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal update_timeout            : std_logic_vector(31 downto 0);
   signal reg_interrupt_en          : std_logic_vector(CHANNELS-1 downto 0);
   signal interrupt_rst             : std_logic;
   signal interrupt_dataflow        : std_logic;
   signal interrupt_dataflow_rst    : std_logic;

   signal reldata_channel           : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal reldata_swendptr          : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reldata_swendptr_we       : std_logic;

   --! Software - MI signals
   signal reg_control               : std_logic_vector(1 downto 0);
   signal reg_status                : std_logic_vector(7 downto 0);
   signal reg_swstrptr              : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_swendptr              : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_swbufmask             : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_interrupt             : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_timeout               : std_logic_vector(31 downto 0);
   signal reg_maxreq                : std_logic_vector(PAGE_WIDTH downto 0);
   signal sw_regsel                 : std_logic_vector(15 downto 0);
   signal sw_channel                : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal sw_interrupt_en           : std_logic;

   --! DMA Bus Signals
   signal dma_up0_data              : std_logic_vector(511 downto 0);
   signal dma_up0_hdr               : std_logic_vector(95 downto 0);
   signal dma_up0_sop               : std_logic;
   signal dma_up0_eop               : std_logic;
   signal dma_up0_src_rdy           : std_logic;
   signal dma_up0_dst_rdy           : std_logic;

   signal dma_up1_data              : std_logic_vector(511 downto 0);
   signal dma_up1_hdr               : std_logic_vector(95 downto 0);
   signal dma_up1_sop               : std_logic;
   signal dma_up1_eop               : std_logic;
   signal dma_up1_src_rdy           : std_logic;
   signal dma_up1_dst_rdy           : std_logic;

   signal dma_down0_data            : std_logic_vector(511 downto 0);
   signal dma_down0_hdr             : std_logic_vector(31 downto 0);
   signal dma_down0_sop             : std_logic;
   signal dma_down0_eop             : std_logic;
   signal dma_down0_src_rdy         : std_logic;
   signal dma_down0_dst_rdy         : std_logic;
   signal dma_down1_data            : std_logic_vector(511 downto 0);
   signal dma_down1_hdr             : std_logic_vector(31 downto 0);
   signal dma_down1_sop             : std_logic;
   signal dma_down1_eop             : std_logic;
   signal dma_down1_src_rdy         : std_logic;
   signal dma_down1_dst_rdy         : std_logic;

   signal dma_interrupt             : std_logic;
   signal update_interrupt          : std_logic;

   signal dma_hwaddress             : std_logic_vector(log2(BUFFER_SIZE/DATA_ALIGN)-1 downto 0);
   signal dma_hwaddress_nd          : std_logic_vector(log2(BUFFER_SIZE)-1 downto 0);

begin

   ENABLE                  <= ctrl_enable;

   DESC_CHANNEL            <= newdata_channel;
   DESC_UPDATE             <= dma_req;
   DESC_UPDATE_LEN         <= dma_length & zeros(log2(DATA_ALIGN)-1 downto 0);

   INTERRUPT               <= update_interrupt and reg_interrupt_en(conv_integer(update_channel));

   BUFFER_DONE_CHANNEL     <= update_dma_channel;

   dma_hwaddress           <= dma_hwaddress_nd(log2(BUFFER_SIZE)-1 downto log2(DATA_ALIGN));

   newdata_i : entity work.dma_ctrl_tx_newdata
   generic map(
      CHANNELS          => CHANNELS,
      BUFFER_SIZE       => BUFFER_SIZE,
      DATA_ALIGN        => DATA_ALIGN
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      PACKET_SENT       => PACKET_SENT,
      PACKET_CHANNEL    => PACKET_CHANNEL,
      PACKET_LENGTH     => PACKET_LENGTH,

      DMA_REQ           => dma_req,
      DMA_RDY           => dma_rdy,
      DMA_LENGTH        => dma_length,
      DMA_HWADDRESS     => dma_hwaddress_nd,
      DMA_INTERRUPT     => dma_interrupt,
      DMA_POINTER       => dma_pointer,

      ENABLE            => ctrl_enable,

      SW_CHANNEL        => sw_channel,
      SW_CONTROL_WR     => sw_regsel(0),
      SW_PTR_WR         => sw_regsel(2),

      SW_CONTROL        => newdata_control,
      SW_STATUS         => newdata_status,
      CHANNEL           => newdata_channel,
      DESC_RDY          => DESC_RDY,
      DESC_DATA         => DESC_DATA,

      SWENDPTR          => newdata_swendptr,
      MAXREQUEST        => newdata_maxreq,
      SWBUFFER_MASK     => newdata_swbufmask,
      INTERRUPT         => newdata_interrupt,
      INTERRUPT_EN      => newdata_interrupt_en,
      INTERRUPT_RST     => interrupt_rst
   );

   request_i : entity work.dma_ctrl_tx_request
   generic map(
      CHANNELS          => CHANNELS,
      BUFFERS           => BUFFERS,
      BUFFER_SIZE       => BUFFER_SIZE,
      DATA_ALIGN        => DATA_ALIGN,
      TAG_WIDTH         => TAG_WIDTH
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_REQ           => dma_req,
      DMA_RDY           => dma_rdy,
      DMA_LENGTH        => dma_length,
      DMA_HWADDRESS     => dma_hwaddress,
      DMA_SWSTRPTR      => dma_pointer,
      DMA_INTERRUPT     => dma_interrupt,
      DMA_CHANNEL       => newdata_channel,
      DMA_DESC          => desc_data,

      BUFFER_WRITE      => BUFFER_WRITE,
      BUFFER_ADDRESS    => BUFFER_ADDRESS,
      BUFFER_CHANNEL    => BUFFER_CHANNEL,
      BUFFER_DATA       => BUFFER_DATA,
      BUFFER_DONE       => BUFFER_DONE,
      BUFFER_DONE_SIZE  => BUFFER_DONE_SIZE,
      BUFFER_DONE_CHANNEL => update_dma_channel,

      SWSTRPTR          => update_dma_pointer,
      SWSTRPTR_UPDATE   => update_dma_donecomplete,
      SWSTRPTR_INTERRUPT=> update_dma_interrupt,

      DMA_TX_DATA       => dma_up0_data,
      DMA_TX_HDR        => dma_up0_hdr,
      DMA_TX_SOP        => dma_up0_sop,
      DMA_TX_EOP        => dma_up0_eop,
      DMA_TX_SRC_RDY    => dma_up0_src_rdy,
      DMA_TX_DST_RDY    => dma_up0_dst_rdy,

      DMA_RX_DATA       => dma_down0_data,
      DMA_RX_HDR        => dma_down0_hdr,
      DMA_RX_SOP        => dma_down0_sop,
      DMA_RX_EOP        => dma_down0_eop,
      DMA_RX_SRC_RDY    => dma_down0_src_rdy,
      DMA_RX_DST_RDY    => dma_down0_dst_rdy

   );

   update_i : entity work.dma_ctrl_update
   generic map(
      CHANNELS          => CHANNELS,
      UNIT_ID           => X"03"
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      ENABLE            => ctrl_enable,

      UPDATE_ADDR       => CONFIG_ADDR,

      UPDATE            => update_dma_donecomplete,
      UPDATE_CHANNEL    => update_dma_channel,
      UPDATE_VALUE      => update_dma_pointer,
      UPDATE_INTERRUPT  => update_dma_interrupt,

      TIMEOUT           => update_timeout,
      TIMEOUT_CHANNEL   => update_channel,
      INTERRUPT         => update_interrupt,

      DMA_UP_DATA       => dma_up1_data,
      DMA_UP_HDR        => dma_up1_hdr,
      DMA_UP_SOP        => dma_up1_sop,
      DMA_UP_EOP        => dma_up1_eop,
      DMA_UP_SRC_RDY    => dma_up1_src_rdy,
      DMA_UP_DST_RDY    => dma_up1_dst_rdy,

      DMA_DOWN_DATA     => dma_down1_data,
      DMA_DOWN_HDR      => dma_down1_hdr,
      DMA_DOWN_SOP      => dma_down1_sop,
      DMA_DOWN_EOP      => dma_down1_eop,
      DMA_DOWN_SRC_RDY  => dma_down1_src_rdy,
      DMA_DOWN_DST_RDY  => dma_down1_dst_rdy
   );

   switch_i: entity work.dma_switch
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 96
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_TX_DATA       => DMA_UP_DATA,
      DMA_TX_HDR        => DMA_UP_HDR,
      DMA_TX_SOP        => DMA_UP_SOP,
      DMA_TX_EOP        => DMA_UP_EOP,
      DMA_TX_SRC_RDY    => DMA_UP_SRC_RDY,
      DMA_TX_DST_RDY    => DMA_UP_DST_RDY,

      DMA_RX0_DATA      => dma_up0_data,
      DMA_RX0_HDR       => dma_up0_hdr,
      DMA_RX0_SOP       => dma_up0_sop,
      DMA_RX0_EOP       => dma_up0_eop,
      DMA_RX0_SRC_RDY   => dma_up0_src_rdy,
      DMA_RX0_DST_RDY   => dma_up0_dst_rdy,

      DMA_RX1_DATA      => dma_up1_data,
      DMA_RX1_HDR       => dma_up1_hdr,
      DMA_RX1_SOP       => dma_up1_sop,
      DMA_RX1_EOP       => dma_up1_eop,
      DMA_RX1_SRC_RDY   => dma_up1_src_rdy,
      DMA_RX1_DST_RDY   => dma_up1_dst_rdy
   );

   splitter_rx_i: entity work.dma_splitter
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 32,
      HDR_BIT           => 25
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_RX_DATA       => DMA_DOWN_DATA,
      DMA_RX_HDR        => DMA_DOWN_HDR,
      DMA_RX_SOP        => DMA_DOWN_SOP,
      DMA_RX_EOP        => DMA_DOWN_EOP,
      DMA_RX_SRC_RDY    => DMA_DOWN_SRC_RDY,
      DMA_RX_DST_RDY    => DMA_DOWN_DST_RDY,

      DMA_TX0_DATA      => dma_down0_data,
      DMA_TX0_HDR       => dma_down0_hdr,
      DMA_TX0_SOP       => dma_down0_sop,
      DMA_TX0_EOP       => dma_down0_eop,
      DMA_TX0_SRC_RDY   => dma_down0_src_rdy,
      DMA_TX0_DST_RDY   => dma_down0_dst_rdy,

      DMA_TX1_DATA      => dma_down1_data,
      DMA_TX1_HDR       => dma_down1_hdr,
      DMA_TX1_SOP       => dma_down1_sop,
      DMA_TX1_EOP       => dma_down1_eop,
      DMA_TX1_SRC_RDY   => dma_down1_src_rdy,
      DMA_TX1_DST_RDY   => dma_down1_dst_rdy
   );

   gen_interrupt: for i in 0 to CHANNELS-1 generate
      reg_swendptrp : process (CLK)
      begin
         if(CLK = '1' and CLK'event) then
            if(RESET = '1') then
               reg_interrupt_en(i) <= '0';
            elsif(sw_regsel(5) = '1' and sw_channel = i) then
               reg_interrupt_en(i) <= MI_DWR(0);
            elsif(update_interrupt = '1' and update_channel = i) then
               reg_interrupt_en(i) <= '0';
            end if;
         end if;
      end process;
   end generate;

   -- ------------   SW Registers   -----------------------------

   MI_DRDY                 <= MI_RD;
   MI_ARDY                 <= MI_WR or MI_RD;
   sw_channel              <= MI_ADDR(log2(CHANNELS) + 5 downto 6);

   addrdec_p: process(MI_ADDR, MI_WR)
   begin
      sw_regsel <= (others => '0');
      for i in 0 to 15 loop
         if(MI_WR = '1' and conv_std_logic_vector(i, 4) = MI_ADDR(5 downto 2)) then
            sw_regsel(i) <= '1';
         end if;
      end loop;
   end process;

   with MI_ADDR(5 downto 2) select
      MI_DRD   <= zeros(31 downto 2) & reg_control    when X"0",
                  zeros(31 downto 8) & reg_status     when X"1",
                                       reg_swstrptr   when X"2",
                                       reg_swendptr   when X"3",
                                       reg_swbufmask  when X"4",
                                       reg_interrupt  when X"5",
                                       reg_timeout    when X"6",
                  zeros(31 downto PAGE_WIDTH+1) & reg_maxreq when X"7",
                                       X"00000000"    when others;

   reg_updateptrp : process (CLK)
   begin
      if(CLK = '1' and CLK'event) then
         reg_update_dma_donecomplete   <= update_dma_donecomplete;
         reg_update_dma_pointer        <= update_dma_pointer;
         reg_update_dma_channel        <= update_dma_channel;
      end if;
   end process;

   reg_control_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 2,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,

      DI     			=> MI_DWR(1 downto 0),
      WE      			=> sw_regsel(0),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_control,
      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_control
   );

   reg_status_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 8,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,

      DI     			=> newdata_status,
      WE      			=> '1',
      ADDRA   			=> newdata_channel,
      DOA     			=> open,

      ADDRB   			=> sw_channel,
      DOB     			=> reg_status
   );

   reg_swendptr_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> SWPTR_WIDTH,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI(31 downto 1)=> MI_DWR(31 downto 1),
      DI(0)    		=> '0',
      WE      			=> sw_regsel(2),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_swendptr,

      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_swendptr
   );

   reg_swstrptr_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> SWPTR_WIDTH,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI             => reg_update_dma_pointer,
      WE      			=> reg_update_dma_donecomplete,
      ADDRA   			=> reg_update_dma_channel,
      DOA     			=> open,

      ADDRB   			=> sw_channel,
      DOB     			=> reg_swstrptr
   );

   reg_swbufmask_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> SWPTR_WIDTH,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> MI_DWR(31 downto 0),
      WE      			=> sw_regsel(4),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_swbufmask,

      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_swbufmask
   );

   reg_interrupt_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 32,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> MI_DWR,
      WE      			=> sw_regsel(5),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_interrupt,

      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_interrupt
   );

   reg_timeout_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 32,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> MI_DWR,
      WE      			=> sw_regsel(6),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_timeout,

      ADDRB   			=> update_channel,
      DOB     			=> update_timeout
   );

   reg_maxreq_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> PAGE_WIDTH+1,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> MI_DWR(PAGE_WIDTH downto 0),
      WE      			=> sw_regsel(7),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_maxreq,

      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_maxreq
   );

end architecture;
