-- hw_txbuf_fifo.vhd: FIFO for HW TX Buffer
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------

entity DMA_BUFFER_TX_FIFO is
   generic(
      DATA_WIDTH           : integer := 64;
      ITEMS                : integer := 64
   );
   port(
      --! Common interface
      CLK                  : in  std_logic;
      ENABLE               : in  std_logic;

      --! Data in
      IN_DATA              : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      --! FIFO Write signal. User must check, if buffer isn't full!
      IN_WRITE             : in  std_logic;
      --! Addres for Write
      IN_ADDR              : in  std_logic_vector(log2(ITEMS)-1 downto 0);

      --! FIFO Read request. User must check, if buffer isn't empty!
      OUT_READ             : in  std_logic;
      --! Data out, valid in next cycle after READ
      OUT_DATA             : out std_logic_vector(DATA_WIDTH-1 downto 0)
   );
end entity;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture behavioral of DMA_BUFFER_TX_FIFO is

   type t_mem     is array(0 to ITEMS-1) of std_logic_vector(DATA_WIDTH-1 downto 0);

   shared variable memory                 : t_mem;

   signal reg_addr_read                   : std_logic_vector(log2(ITEMS)-1 downto 0) := (others =>'0');

   signal write_address                   : std_logic_vector(log2(ITEMS)-1 downto 0);
   signal read_address                    : std_logic_vector(log2(ITEMS)-1 downto 0);

   signal reg_valid                       : std_logic_vector(ITEMS-1 downto 0) := (others => '0');
   signal empty                           : std_logic;

   attribute ram_style                    : string;                     -- for XST
   attribute block_ram                    : boolean;                    -- for precision
   attribute ram_style of memory          : variable is "block";        -- auto,block,distributed
   attribute block_ram of memory          : variable is true;           -- true,false

begin

   write_address     <= IN_ADDR;
   read_address      <= reg_addr_read;

   --! Write input data to memory
   p_write: process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (IN_WRITE = '1') then
            memory(conv_integer(write_address)) := IN_DATA;
         end if;
      end if;
   end process;

   --! Read from memory
   process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         OUT_DATA    <= memory(conv_integer(read_address));
      end if;
   end process;

   --! Update pointers
   process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (ENABLE = '0') then
            reg_addr_read <= (others => '0');
         elsif (OUT_READ = '1') then
            reg_addr_read <= read_address + 1;
         end if;
      end if;
   end process;

end architecture behavioral;
