-- asfifo_dma_bus.vhd : Architecture of asynchronous FIFO for DMA bus
--!
--! \file
--! \brief Architecture of asynchronous FIFO for DMA bus
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                        Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of asfifo_dma_bus is

   --! Constants declaration
   -- -------------------------------------------------------------------------

   constant INTERNAL_DATA_WIDTH : integer := DATA_WIDTH + HDR_WIDTH + 2;

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! write interface
   signal wr_data   : std_logic_vector(INTERNAL_DATA_WIDTH-1 downto 0);
   signal wr_enable : std_logic;
   signal full      : std_logic;

   --! read interface
   signal rd_data   : std_logic_vector(INTERNAL_DATA_WIDTH-1 downto 0);
   signal rd_enable : std_logic;
   signal empty     : std_logic;


-- ----------------------------------------------------------------------------
--                            Architecture body
-- ----------------------------------------------------------------------------

begin

   --! aggregation of input port to write data signal
   wr_data        <= WR_DMA_EOP & WR_DMA_SOP & WR_DMA_HDR & WR_DMA_DATA;

   --! wite interface logic
   wr_enable      <= WR_DMA_SRC_RDY AND NOT full;
   WR_DMA_DST_RDY <= NOT full;

   --! DMA asynchronous FIFO for downstream
   asfifo_dma_bus_i : entity work.ASFIFO_BRAM_7SERIES
   generic map (
      DATA_WIDTH          => INTERNAL_DATA_WIDTH,
      ALMOST_FULL_OFFSET  => X"0004",
      ALMOST_EMPTY_OFFSET => X"0006",
      SAFE_RESET          => SAFE_RESET
   )
   port map (
      --! Write interface
      CLK_WR              => WR_CLK,
      RST_WR              => WR_RESET,
      DI                  => wr_data,
      WR                  => wr_enable,
      AFULL               => open,
      FULL                => full,

      --! Read interface
      CLK_RD              => RD_CLK,
      RST_RD              => RD_RESET,
      DO                  => rd_data,
      RD                  => rd_enable,
      AEMPTY              => open,
      EMPTY               => empty
   );

   --! de-aggregation of read data signal to output ports
   RD_DMA_DATA <= rd_data(DATA_WIDTH-1 downto 0);
   RD_DMA_HDR  <= rd_data(INTERNAL_DATA_WIDTH-3 downto DATA_WIDTH);
   RD_DMA_SOP  <= rd_data(INTERNAL_DATA_WIDTH-2);
   RD_DMA_EOP  <= rd_data(INTERNAL_DATA_WIDTH-1);

   --! read interface logic
   rd_enable      <= RD_DMA_DST_RDY AND NOT empty;
   RD_DMA_SRC_RDY <= NOT empty;

end architecture full;
