--!
--! reset.vhd: Reset synchronizer.
--! Copyright (C) 2014 CESNET
--! Author(s): Jakub Cabal <jakubcabal@gmail.com>
--!            Lukas Kekely <kekely@cesnet.cz>
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!
--! $Id$
--!

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

--! pragma translate_off
library UNISIM;
use UNISIM.vcomponents.all;
--! pragma translate_on

entity ASYNC_RESET_MAIN is
   Generic (
      TWO_REG  : BOOLEAN := false;     --! For two reg = true, for three reg = false
      OUT_REG  : BOOLEAN := false;     --! Registering of output reset by single normal register in destination clock domain.
      REPLICAS : INTEGER := 1          --! Number of output register replicas (registers actually replicated only when OUT_REG is true).
   );    
   Port (
      --! A clock domain 
      CLK        : in  STD_LOGIC;   --! Clock
      ASYNC_RST  : in  STD_LOGIC;   --! Asynchronous reset
      OUT_RST    : out STD_LOGIC_VECTOR(REPLICAS-1 downto 0)    --! Output reset
   );
end ASYNC_RESET_MAIN;

   --! -------------------------------------------------------------------------
   --!                      Architecture declaration
   --! -------------------------------------------------------------------------

architecture FULL of ASYNC_RESET_MAIN is

   --! -------------------------------------------------------------------------
   --! Signals
   --! -------------------------------------------------------------------------
   
   signal rff1    : std_logic := '1';
   signal rff2    : std_logic := '1';
   signal rff_out : std_logic := '1';

   --! Attributes declarations
   attribute dont_touch              : string;
   attribute shreg_extract           : string;
   attribute async_reg               : string;
   
   --! Attributes for rff1_reg and rff2_reg
   attribute dont_touch of rff1    : signal is "true";
   attribute shreg_extract of rff1 : signal is "no";
   attribute async_reg of rff1     : signal is "true";

   attribute shreg_extract of rff2 : signal is "no";
   attribute async_reg of rff2     : signal is "true";
   --! -------------------------------------------------------------------------
 
begin

   --! -------------------------------------------------------------------------

   --! Two synchronization registers
   rff1_sync_reg : process(CLK, ASYNC_RST)
   begin
      if (ASYNC_RST = '1') then
         rff1 <= '1';
      elsif (rising_edge(CLK)) then
         rff1 <= '0';
      end if;
   end process;

   rff2_sync_reg : process(CLK, ASYNC_RST)
   begin
      if (ASYNC_RST = '1') then
         rff2 <= '1';
      elsif (rising_edge(CLK)) then
         rff2 <= rff1;
      end if;
   end process;

   --! -------------------------------------------------------------------------
   
   --! Generics two synchronization registers
   two_reg_sync : if TWO_REG generate

      rff_out <= rff2;
  
   end generate;  
  
   --! -------------------------------------------------------------------------
  
   --! Generics three synchronization registers
   three_reg_sync : if NOT TWO_REG generate
      
      --! Signals
      signal rff3                       : std_logic := '1';
      
      --! Attribute for rff3_reg
      attribute shreg_extract of rff3 : signal is "no";
      attribute async_reg of rff3     : signal is "true";
   
      begin
   
      rff3_reg : process(CLK, ASYNC_RST)
         begin
         if (ASYNC_RST = '1') then
            rff3 <= '1';
         elsif (rising_edge(CLK)) then
            rff3 <= rff2; 
         end if;
      end process;
 
      rff_out <= rff3;
 
   end generate;

   --! -------------------------------------------------------------------------

   --! Generics do not use output register
   no_out_reg_gen : if not OUT_REG generate

      OUT_RST <= (others => rff_out);

   end generate;

   --! -------------------------------------------------------------------------

   --! Generics output registers usage
   out_reg_gen : if OUT_REG generate

      replicas_gen : for i in 0 to REPLICAS-1 generate

         signal rff_reg_out                  : std_logic := '1';
         attribute dont_touch of rff_reg_out : signal is "true";

         begin

         out_reg : process(CLK)
            begin
            if (rising_edge(CLK)) then
               rff_reg_out <= rff_out;
            end if;
         end process;

         OUT_RST(i) <= rff_reg_out;

      end generate;

   end generate;

   --! -------------------------------------------------------------------------

end architecture FULL;
