-- mem_ent.vhd: Mem unit of CGMII Input buffer - entity declaration
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                           Entity declaration
-- ----------------------------------------------------------------------------

entity mem is
   generic(
      -- Number of MAC addresses that can be placed into CAM (at most 16)
      MAC_COUNT         : integer := 16;
      -- Remove FCS from the packet (false -> remove, true -> don't remove)
      INBANDFCS         : boolean := true;
      -- Synthesise design without resets, default values of registers are set
      -- by initialization of signals
      RESET_BY_INIT     : boolean := false
   );
   port(
      -- Common signals -------------------------------------------------------
      -- Clock signal
      CLK               : in std_logic;
      -- Global reset signal
      RESET             : in std_logic;
      -- Sofware reset 
      SW_RESET          : in std_logic;

      -- General data signals (mem <-> address decoder) -----------------------
      -- Data to be written (into CAM or registers)
      DATA_WR           : in std_logic_vector(31 downto 0);
      -- Read data (from CAM or registers)
      DATA_RD           : out std_logic_vector(31 downto 0);
      -- Port DATA_RD contains valid data
      DATA_VLD          : out std_logic;
      

      -- Registers write interface (address decoder -> mem) -------------------
      -- Write to IBUF enable register
      IBUF_EN_WE        : in std_logic;
      -- Write to error mask register
      ERROR_MASK_WE     : in std_logic;
      -- Write to IBUF status register
      STATUS_WE         : in std_logic;
      -- Write to minimum allowed frame length register
      MIN_FRAME_LEN_WE  : in std_logic;
      -- Write to frame MTU register
      MAX_FRAME_LEN_WE  : in std_logic;
      -- Write to MAC check mode register
      MAC_CHECK_MODE_WE : in std_logic;


      -- Registers read interface (mem -> address decoder) --------------------
      -- Read from IBUF enable register
      IBUF_EN_RD        : in std_logic;
      -- Read from error mask register
      ERROR_MASK_RD     : in std_logic;
      -- Read from IBUF status register
      STATUS_RD         : in std_logic;
      -- Read from minimum allowed frame length register
      MIN_FRAME_LEN_RD  : in std_logic;
      -- Read from frame MTU register
      MAX_FRAME_LEN_RD  : in std_logic;
      -- Read from MAC check mode register
      MAC_CHECK_MODE_RD : in std_logic;

      -- CAM read/write interface (mem <-> address decoder) -------------------
      -- Address within CAM + low/high part of MAC selector (the lowest bit)
      CAM_ADDR          : in std_logic_vector(log2(MAC_COUNT) downto 0);
      -- Write enable signal for CAM
      CAM_WE            : in std_logic;
      -- Read enable signal for CAM
      CAM_RE            : in std_logic;
      -- Busy signal for CAM
      CAM_BUSY          : out std_logic;
      
      -- CAM match interface (mem <-> check unit) -----------------------------
      -- MAC address to be searched
      CAM_DI            : in std_logic_vector(63 downto 0);
      -- MAC address search enable
      CAM_MATCH_EN      : in std_logic;
      -- MAC match reset
      CAM_MATCH_RST     : in std_logic;
      -- Addresses found in CAM
      CAM_MATCH_BUS     : out std_logic_vector(MAC_COUNT-1 downto 0);
      -- CAM_MATCH_BUS is valid, active in '1'
      CAM_MATCH_BUS_VLD : out std_logic;
      
      -- Check registers interface (mem -> check unit) ------------------------
      -- Value of minimum allowed frame length register
      MIN_FRAME_LEN     : out std_logic_vector(15 downto 0);
      -- Value of frame MTU register
      MAX_FRAME_LEN     : out std_logic_vector(15 downto 0);
      -- Value of MAC check mode register
      -- 0x0: promiscuous
      -- 0x1: accept only MACs from CAM
      -- 0x2: mode1 + broadcast
      -- 0x3: mode2 + multicast
      MAC_CHECK_MODE    : out std_logic_vector(1 downto 0);

      -- Buf registers interface (mem <-> buf) --------------------------------
      -- Value of IBUF enable register
      IBUF_EN           : out std_logic;
      -- Value of error mask register
      ERROR_MASK        : out std_logic_vector(4 downto 0);
      -- Value of IBUF status register
      STATUS            : in std_logic_vector(16 downto 0);
	
      -- CGMII decoder interface (cgmii_dec -> mem) ---------------------------
      -- Link state (0=no link, 1=link ok)
      LINK_STATUS       : in std_logic
   );
end entity mem;

