-- ibuf_cgmii.vhd: CGMII Input buffer
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--            Pavel Benacek <benacek@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------

architecture ibuf_cgmii_arch of ibuf_cgmii is
   -- Constant declaration ----------------------------------------------------
      -- Statistics synchronization FIFO configuration
   constant STAT_FIFO_DATA_WIDTH    : integer := 22;
   constant STAT_FIFO_DISTMEM_TYPE  : integer := 16;
   constant STAT_FIFO_ITEMS         : integer := 16;
   constant STAT_FIFO_BLOCK_SIZE    : integer := 0;

   -- Signals declaration -----------------------------------------------------  
   -- cgmii_dec module output signals
   signal cgmii_dec_tx_data           : std_logic_vector(511 downto 0);
   signal cgmii_dec_tx_sop            : std_logic;
   signal cgmii_dec_tx_sop_pos        : std_logic_vector(2 downto 0);
   signal cgmii_dec_tx_eop            : std_logic;
   signal cgmii_dec_tx_eop_pos        : std_logic_vector(5 downto 0);
   signal cgmii_dec_tx_err            : std_logic;

   signal cgmii_dec_link_up           : std_logic;
   signal cgmii_dec_incoming_packet   : std_logic;

   -- check module output signals  
   signal check_tx_data               : std_logic_vector(511 downto 0);
   signal check_tx_sop                : std_logic;
   signal check_tx_sop_pos            : std_logic_vector(2 downto 0);
   signal check_tx_eop                : std_logic;
   signal check_tx_eop_pos            : std_logic_vector(5 downto 0);

   signal check_mac_err               : std_logic;
   signal check_mintu_err             : std_logic;
   signal check_mtu_err               : std_logic;
   signal check_sau_err               : std_logic;
   signal check_crc_err               : std_logic;
   signal check_rem_err               : std_logic;
   signal check_frame_err             : std_logic;
   signal check_frame_len             : std_logic_vector(15 downto 0);
   signal check_frame_len_fix         : std_logic_vector(15 downto 0);
   signal check_mac_mcast             : std_logic;
   signal check_mac_bcast             : std_logic;
   
   signal stat_fifo_in                : std_logic_vector(21 downto 0);
   signal stat_fifo_wr                : std_logic;
   signal stat_fifo_out               : std_logic_vector(21 downto 0);
   signal stat_fifo_rd                : std_logic;
   signal stat_fifo_empty             : std_logic;
   signal stat_dv                     : std_logic;
   
   signal check_cam_di                : std_logic_vector(63 downto 0);
   signal check_cam_match_en          : std_logic;
   signal check_cam_match_rst         : std_logic;

   -- addr_dec module output signals
   signal addr_dec_data_wr            : std_logic_vector(31 downto 0);

   signal addr_dec_ibuf_en_we         : std_logic;
   signal addr_dec_error_mask_we      : std_logic;
   signal addr_dec_status_we          : std_logic;
   signal addr_dec_min_frame_len_we   : std_logic;
   signal addr_dec_max_frame_len_we   : std_logic;
   signal addr_dec_mac_check_mode_we  : std_logic;

   signal addr_dec_ibuf_en_rd         : std_logic;
   signal addr_dec_error_mask_rd      : std_logic;
   signal addr_dec_status_rd          : std_logic;
   signal addr_dec_min_frame_len_rd   : std_logic;
   signal addr_dec_max_frame_len_rd   : std_logic;
   signal addr_dec_mac_check_mode_rd  : std_logic;
      
   signal addr_dec_cam_addr           : std_logic_vector(log2(MAC_COUNT) downto 0);
   signal addr_dec_cam_we             : std_logic;
   signal addr_dec_cam_re             : std_logic;

   signal addr_dec_sw_reset           : std_logic;
   signal addr_dec_read_en            : std_logic;
   signal addr_dec_last_addr_en       : std_logic;

   -- mem module output signals   
   signal mem_data_rd                 : std_logic_vector(31 downto 0);
   signal mem_data_vld                : std_logic;

   signal mem_cam_busy                : std_logic;
   signal mem_cam_match_bus           : std_logic_vector(MAC_COUNT-1 downto 0);
   signal mem_cam_match_bus_vld       : std_logic;

   signal mem_min_frame_len           : std_logic_vector(15 downto 0);
   signal mem_max_frame_len           : std_logic_vector(15 downto 0);
   signal mem_mac_check_mode          : std_logic_vector(1 downto 0);
   
   signal mem_ibuf_en                 : std_logic;
   signal mem_error_mask              : std_logic_vector(4 downto 0);

   -- buf module output signals     
   signal buf_frame_received          : std_logic;
   signal buf_frame_discarded         : std_logic;
   signal buf_frame_discarded_buf_ovf : std_logic;
  
   signal buf_ctrl_stat_mac_err       : std_logic;
   signal buf_ctrl_stat_mintu_err     : std_logic;
   signal buf_ctrl_stat_mtu_err       : std_logic;
   signal buf_ctrl_stat_crc_err       : std_logic;
   signal buf_ctrl_stat_frame_err     : std_logic;
   signal buf_ctrl_stat_frame_len     : std_logic_vector(15 downto 0);  
   signal buf_ctrl_stat_dv            : std_logic;

   signal buf_status                  : std_logic_vector(16 downto 0);

   -- stat_unit module output signals    
   signal stat_unit_total              : std_logic_vector(63 downto 0);
   signal stat_unit_received           : std_logic_vector(63 downto 0);
   signal stat_unit_discarded          : std_logic_vector(63 downto 0);
   signal stat_unit_discarded_buf_ovf  : std_logic_vector(63 downto 0);
   signal stat_unit_octets_received_ok : std_logic_vector(63 downto 0);
   signal stat_unit_octets_received    : std_logic_vector(63 downto 0);
   signal stat_unit_crc_err            : std_logic_vector(63 downto 0);
   signal stat_unit_over_mtu           : std_logic_vector(63 downto 0);
   signal stat_unit_below_min          : std_logic_vector(63 downto 0);
   signal stat_unit_bcast_frames       : std_logic_vector(63 downto 0);
   signal stat_unit_mcast_frames       : std_logic_vector(63 downto 0);
   signal stat_unit_fragment_frames    : std_logic_vector(63 downto 0);
   signal stat_unit_jabber_frames      : std_logic_vector(63 downto 0);
   signal stat_unit_frames_64          : std_logic_vector(63 downto 0);
   signal stat_unit_frames_65_127      : std_logic_vector(63 downto 0);
   signal stat_unit_frames_128_255     : std_logic_vector(63 downto 0);
   signal stat_unit_frames_256_511     : std_logic_vector(63 downto 0);
   signal stat_unit_frames_512_1023    : std_logic_vector(63 downto 0);
   signal stat_unit_frames_1024_1518   : std_logic_vector(63 downto 0);
   signal stat_unit_dv                 : std_logic;

begin

   -- -------------------------------------------------------------------------
   --                           CGMII Decoder unit
   -- -------------------------------------------------------------------------
   
   cgmii_dec_i : entity work.cgmii_dec
   generic map(
      CNT_ERROR_LENGTH  => CNT_ERROR_LENGTH,
      RESET_BY_INIT     => RESET_BY_INIT
   )
   port map(
      -- Common signals
      CLK             => CGMII_RXCLK,
      RESET           => CGMII_RXRESET,

      -- Input CGMII interface
      RXD             => CGMII_RXD,
      RXC             => CGMII_RXC,

      -- Output FLU-like interface
      TX_DATA         => cgmii_dec_tx_data,
      TX_SOP          => cgmii_dec_tx_sop,
      TX_SOP_POS      => cgmii_dec_tx_sop_pos,
      TX_EOP          => cgmii_dec_tx_eop,
      TX_EOP_POS      => cgmii_dec_tx_eop_pos,
      TX_ERR          => cgmii_dec_tx_err,

      -- Link status interface
      LINK_UP         => cgmii_dec_link_up,
      INCOMING_PACKET => cgmii_dec_incoming_packet
   );

   -- Output ports assignment
   LINK_UP         <= cgmii_dec_link_up;
   INCOMING_PACKET <= cgmii_dec_incoming_packet;
   

   -- -------------------------------------------------------------------------
   --                              Check unit
   -- -------------------------------------------------------------------------

   check_i : entity work.check
   generic map(
      MAC_COUNT         => MAC_COUNT,
      INBANDFCS         => INBANDFCS,
      RESET_BY_INIT     => RESET_BY_INIT
   )
   port map(
      -- Common signals
      CLK               => CGMII_RXCLK,
      RESET             => CGMII_RXRESET,

      -- Datapath input
      RX_DATA           => cgmii_dec_tx_data,
      RX_SOP            => cgmii_dec_tx_sop,
      RX_SOP_POS        => cgmii_dec_tx_sop_pos,
      RX_EOP            => cgmii_dec_tx_eop,
      RX_EOP_POS        => cgmii_dec_tx_eop_pos,
      RX_ERR            => cgmii_dec_tx_err,

      -- Datapath output
      TX_DATA           => check_tx_data,
      TX_SOP            => check_tx_sop,
      TX_SOP_POS        => check_tx_sop_pos,
      TX_EOP            => check_tx_eop,
      TX_EOP_POS        => check_tx_eop_pos,

      -- Statistics (valid when TX_EOP = '1')
      MAC_ERR           => check_mac_err,
      MINTU_ERR         => check_mintu_err,
      MTU_ERR           => check_mtu_err,
      SAU_ERR           => check_sau_err,
      CRC_ERR           => check_crc_err,
      CRC_REM_ERR       => check_rem_err,
      FRAME_ERR         => check_frame_err,
      FRAME_LEN         => check_frame_len,
      MAC_MCAST         => check_mac_mcast,
      MAC_BCAST         => check_mac_bcast,
      
      -- CAM connection
      CAM_DI            => check_cam_di,
      CAM_MATCH_EN      => check_cam_match_en,
      CAM_MATCH_RST     => check_cam_match_rst,
      CAM_MATCH_BUS     => mem_cam_match_bus,
      CAM_MATCH_BUS_VLD => mem_cam_match_bus_vld,

      -- Registers connection
      MIN_FRAME_LEN     => mem_min_frame_len,
      MAX_FRAME_LEN     => mem_max_frame_len,
      MAC_CHECK_MODE    => mem_mac_check_mode,

      -- Sampling unit interface, signals active in '1'
      SAU_CLK           => SAU_CLK,
      SAU_RESET         => SAU_RESET,
      SAU_REQ           => SAU_REQ,
      SAU_ACCEPT        => SAU_ACCEPT,
      SAU_DV            => SAU_DV
   );
   

   -- -------------------------------------------------------------------------
   --                          Address decoder unit
   -- -------------------------------------------------------------------------

   addr_dec_i : entity work.addr_dec
   generic map(
      MAC_COUNT         => MAC_COUNT,
      RESET_BY_INIT     => RESET_BY_INIT
   )
   port map(
      -- Common signals
      INT_CLK                      => CGMII_RXCLK,
      INT_RESET                    => CGMII_RXRESET,
      MI_CLK                       => MI_CLK,
      MI_RESET                     => MI_RESET,

      -- MI32 input interface
      MI_DWR                       => MI_DWR,
      MI_ADDR                      => MI_ADDR,
      MI_RD                        => MI_RD,
      MI_WR                        => MI_WR,
      MI_BE                        => MI_BE,
      MI_DRD                       => MI_DRD,
      MI_ARDY                      => MI_ARDY,
      MI_DRDY                      => MI_DRDY,
      
      -- General data signals (mem <-> address decoder)
      DATA_WR                      => addr_dec_data_wr,
      DATA_RD                      => mem_data_rd,
      DATA_VLD                     => mem_data_vld,

      -- Registers write interface (address decoder -> mem)
      IBUF_EN_WE                   => addr_dec_ibuf_en_we,
      ERROR_MASK_WE                => addr_dec_error_mask_we,
      STATUS_WE                    => addr_dec_status_we,
      MIN_FRAME_LEN_WE             => addr_dec_min_frame_len_we,
      MAX_FRAME_LEN_WE             => addr_dec_max_frame_len_we,
      MAC_CHECK_MODE_WE            => addr_dec_mac_check_mode_we,

      -- Registers read interface (address decoder -> mem)
      IBUF_EN_RD                   => addr_dec_ibuf_en_rd,
      ERROR_MASK_RD                => addr_dec_error_mask_rd,
      STATUS_RD                    => addr_dec_status_rd,
      MIN_FRAME_LEN_RD             => addr_dec_min_frame_len_rd,
      MAX_FRAME_LEN_RD             => addr_dec_max_frame_len_rd,
      MAC_CHECK_MODE_RD            => addr_dec_mac_check_mode_rd,

      -- CAM read/write interface (mem <-> address decoder)
      CAM_ADDR                     => addr_dec_cam_addr,
      CAM_WE                       => addr_dec_cam_we,
      CAM_RE                       => addr_dec_cam_re,
      CAM_BUSY                     => mem_cam_busy,

      -- Control interface
      SW_RESET                     => addr_dec_sw_reset,
      READ_EN                      => addr_dec_read_en,
      LAST_ADDR_EN                 => addr_dec_last_addr_en,

      -- Statistics inputs
      TOTAL_FRAMES_CNT             => stat_unit_total,
      CORRECT_FRAMES_CNT           => stat_unit_received,
      DISCARDED_FRAMES_CNT         => stat_unit_discarded,
      DISCARDED_BUF_OVF_FRAMES_CNT => stat_unit_discarded_buf_ovf,
      OCTETS_RECEIVED_OK_CNT       => stat_unit_octets_received_ok,
      TRANSFERED_OCTETS_CNT        => stat_unit_octets_received, 
      CRC_ERROR_CNT                => stat_unit_crc_err,
      OVER_MTU_CNT                 => stat_unit_over_mtu,
      BELOW_MIN_CNT                => stat_unit_below_min,
      BCAST_FRAMES_CNT             => stat_unit_bcast_frames,
      MCAST_FRAMES_CNT             => stat_unit_mcast_frames,
      FRAGMENT_FRAMES_CNT          => stat_unit_fragment_frames,
      JABBER_FRAMES_CNT            => stat_unit_jabber_frames,
      FRAMES_64_CNT                => stat_unit_frames_64,
      FRAMES_65_127_CNT            => stat_unit_frames_65_127,
      FRAMES_128_255_CNT           => stat_unit_frames_128_255,
      FRAMES_256_511_CNT           => stat_unit_frames_256_511,
      FRAMES_512_1023_CNT          => stat_unit_frames_512_1023,
      FRAMES_1024_1518_CNT         => stat_unit_frames_1024_1518,

      CNT_VALID                    => stat_unit_dv
   );


   -- -------------------------------------------------------------------------
   --                              Memory unit
   -- -------------------------------------------------------------------------
   
   mem_i : entity work.mem
   generic map(
      -- Number of MAC addresses that can be placed into CAM (at most 16)
      MAC_COUNT         => MAC_COUNT,
      INBANDFCS         => INBANDFCS,
      RESET_BY_INIT     => RESET_BY_INIT
   )
   port map(
      -- Common signals
      CLK               => CGMII_RXCLK,
      RESET             => CGMII_RXRESET,
      SW_RESET          => addr_dec_sw_reset,

      -- General data signals (mem <-> address decoder)
      DATA_WR           => addr_dec_data_wr,
      DATA_RD           => mem_data_rd,
      DATA_VLD          => mem_data_vld,

      -- Registers write interface (address decoder -> mem)
      IBUF_EN_WE        => addr_dec_ibuf_en_we,
      ERROR_MASK_WE     => addr_dec_error_mask_we,
      STATUS_WE         => addr_dec_status_we,
      MIN_FRAME_LEN_WE  => addr_dec_min_frame_len_we,
      MAX_FRAME_LEN_WE  => addr_dec_max_frame_len_we,
      MAC_CHECK_MODE_WE => addr_dec_mac_check_mode_we,

      -- Registers read interface (mem -> address decoder)
      IBUF_EN_RD        => addr_dec_ibuf_en_rd,
      ERROR_MASK_RD     => addr_dec_error_mask_rd,
      STATUS_RD         => addr_dec_status_rd,
      MIN_FRAME_LEN_RD  => addr_dec_min_frame_len_rd,
      MAX_FRAME_LEN_RD  => addr_dec_max_frame_len_rd,
      MAC_CHECK_MODE_RD => addr_dec_mac_check_mode_rd,

      -- CAM read/write interface (mem <-> address decoder)
      CAM_ADDR          => addr_dec_cam_addr,
      CAM_WE            => addr_dec_cam_we,
      CAM_RE            => addr_dec_cam_re,
      CAM_BUSY          => mem_cam_busy,

      -- CAM match interface (mem <-> check unit
      CAM_DI            => check_cam_di,
      CAM_MATCH_EN      => check_cam_match_en,
      CAM_MATCH_RST     => check_cam_match_rst,
      CAM_MATCH_BUS     => mem_cam_match_bus,
      CAM_MATCH_BUS_VLD => mem_cam_match_bus_vld,

      -- Check registers interface (mem -> check unit)
      MIN_FRAME_LEN     => mem_min_frame_len,
      MAX_FRAME_LEN     => mem_max_frame_len,
      MAC_CHECK_MODE    => mem_mac_check_mode,

      -- Buf registers interface (mem <-> buf)
      IBUF_EN           => mem_ibuf_en,
      ERROR_MASK        => mem_error_mask,
      STATUS            => buf_status,

      -- CGMII decoder interface (cgmii_dec -> mem)
      LINK_STATUS       => cgmii_dec_link_up
   );
   
   
   -- -------------------------------------------------------------------------
   --                             Buffering unit
   -- -------------------------------------------------------------------------
   
   cgmii_ibuf_buf_i : entity work.cgmii_ibuf_buf
   generic map(
      DFIFO_SIZE              => DFIFO_SIZE,
      HFIFO_SIZE              => HFIFO_SIZE,
      HDR_WIDTH               => HDR_WIDTH,
      RESET_BY_INIT           => RESET_BY_INIT
   )
   port map(
      -- Common signals
      CGMII_CLK               => CGMII_RXCLK,
      CGMII_RESET             => CGMII_RXRESET,
      FLU_CLK                 => TX_CLK,
      FLU_RESET               => TX_RESET,

      -- Data input interface
      RX_DATA                 => check_tx_data,
      RX_SOP                  => check_tx_sop,
      RX_SOP_POS              => check_tx_sop_pos,
      RX_EOP                  => check_tx_eop,
      RX_EOP_POS              => check_tx_eop_pos,

      -- Data payload output
      TX_DATA                 => TX_DATA,
      TX_SOP_POS              => TX_SOP_POS,
      TX_EOP_POS              => TX_EOP_POS,
      TX_SOP                  => TX_SOP,
      TX_EOP                  => TX_EOP,
      TX_SRC_RDY              => TX_SRC_RDY,
      TX_DST_RDY              => TX_DST_RDY,

      -- Data header/footer output
      TX_HDATA                => TX_HDATA,
      TX_HSOP_N               => TX_HSOP_N,
      TX_HEOP_N               => TX_HEOP_N,
      TX_HREM                 => TX_HREM,
      TX_HSRC_RDY_N           => TX_HSRC_RDY_N,
      TX_HDST_RDY_N           => TX_HDST_RDY_N,

      -- Statistics input (valid when RX_EOP = '1')
      RX_MAC_ERR              => check_mac_err,
      RX_MINTU_ERR            => check_mintu_err,
      RX_MTU_ERR              => check_mtu_err,
      RX_SAU_ERR              => check_sau_err,
      RX_CRC_ERR              => check_crc_err,
      RX_FRAME_ERR            => check_frame_err,
      RX_FRAME_LEN            => check_frame_len,

      -- Frame processing signals
      FRAME_RECEIVED          => buf_frame_received,
      FRAME_DISCARDED         => buf_frame_discarded,
      FRAME_DISCARDED_BUF_OVF => buf_frame_discarded_buf_ovf,

      -- Statistics for PACODAG
      CTRL_STAT_FRAME_ERR     => buf_ctrl_stat_frame_err,
      CTRL_STAT_MAC_ERR       => buf_ctrl_stat_mac_err,
      CTRL_STAT_MINTU_ERR     => buf_ctrl_stat_mintu_err,
      CTRL_STAT_MTU_ERR       => buf_ctrl_stat_mtu_err,
      CTRL_STAT_CRC_ERR       => buf_ctrl_stat_crc_err,
      CTRL_STAT_FRAME_LEN     => buf_ctrl_stat_frame_len,

      -- PACODAG interface
      CTRL_CLK                => CTRL_CLK,
      CTRL_RESET              => CTRL_RESET,
      CTRL_DATA               => CTRL_DATA,
      CTRL_SOP_N              => CTRL_SOP_N,
      CTRL_EOP_N              => CTRL_EOP_N,
      CTRL_REM                => CTRL_REM,
      CTRL_SRC_RDY_N          => CTRL_SRC_RDY_N,
      CTRL_DST_RDY_N          => CTRL_DST_RDY_N,
      CTRL_SOP                => CTRL_SOP,
      CTRL_STAT_DV            => buf_ctrl_stat_dv,
      CTRL_RDY                => CTRL_RDY,

      -- Value of registers placed in Mem module
      IBUF_EN                 => mem_ibuf_en,
      ERROR_MASK              => mem_error_mask,

      -- Input value for register placed in Mem module
      STATUS                  => buf_status
   );
   
   -- Output ports assignment
   CTRL_STAT_FRAME_ERR  <= buf_ctrl_stat_frame_err;
   CTRL_STAT_MAC_ERR    <= buf_ctrl_stat_mac_err;
   CTRL_STAT_MINTU_ERR  <= buf_ctrl_stat_mintu_err;
   CTRL_STAT_MTU_ERR    <= buf_ctrl_stat_mtu_err;
   CTRL_STAT_CRC_ERR    <= buf_ctrl_stat_crc_err;
   CTRL_STAT_FRAME_LEN  <= buf_ctrl_stat_frame_len;

   CTRL_STAT_DV         <= buf_ctrl_stat_dv;
   
   
   -- -------------------------------------------------------------------------
   --                             Statistical unit
   -- -------------------------------------------------------------------------

   FRAME_CHECK_LEN_REP: if (INBANDFCS = false) generate
      check_frame_len_fix <= check_frame_len when (check_rem_err = '0')
                                   else check_frame_len - 4;
   end generate;

   FRAME_CHECK_LEN_NO_REP: if (INBANDFCS = true) generate
      check_frame_len_fix <= check_frame_len;
   end generate;

   -- Statistical FIFO input
   stat_fifo_in(0)   <= check_crc_err and mem_error_mask(1) and not(check_rem_err);
   stat_fifo_in(1)   <= check_mintu_err and mem_error_mask(2);
   stat_fifo_in(2)   <= check_mtu_err and mem_error_mask(3);
   stat_fifo_in(3)   <= check_mac_err and mem_error_mask(4);
   stat_fifo_in(19 downto 4)   <= check_frame_len_fix; 
   stat_fifo_in(20)  <= check_mac_bcast;
   stat_fifo_in(21)  <= check_mac_mcast;
   stat_fifo_wr      <= check_tx_eop;        

   -- FIFO for statistic informations from the CHECK module
   stat_sync_fifo_i:entity work.FIFO(behavioral)
   generic map(
      -- Set data width here
      DATA_WIDTH     => STAT_FIFO_DATA_WIDTH,

      -- Distributed RAM type, only 16, 32, 64 bits
      DISTMEM_TYPE   => STAT_FIFO_DISTMEM_TYPE,

      -- Set number of items in FIFO here
      ITEMS          => STAT_FIFO_ITEMS,

      -- for last block identification
      BLOCK_SIZE     => STAT_FIFO_BLOCK_SIZE
   )
   port map(
      RESET    => CGMII_RXRESET,
      CLK      => CGMII_RXCLK,

      -- Write interface
      DATA_IN     => stat_fifo_in,
      WRITE_REQ   => stat_fifo_wr,
      FULL        => open,
      LSTBLK      => open,

      -- Read interface
      DATA_OUT    => stat_fifo_out,
      READ_REQ    => stat_fifo_rd,
      EMPTY       => stat_fifo_empty
   );

   -- Fifo read signal generation
   stat_fifo_rd <= buf_frame_received or buf_frame_discarded;
   stat_dv  <= stat_fifo_rd and not(stat_fifo_empty);

   -- Statistical unit
   stat_unit_i:entity work.stat_unit
   generic map(
      --! IFG detector setup
      SOP_ALIGN_WIDTH   => 3,
      EOP_POS_WIDTH     => 6,
      ETH_CLK_SIZE      => 64,
      
      --! Counters&regs setup
      DELAY_EN                => false, 
      READ_DELAY_EN           => false, 
      MAC_EN                  => false, 
      CRC_EN                  => true, 
      MTU_EN                  => true, 
      SIZE_EN                 => true, 
      BCAST_MCAST_EN          => true, 
      FRAGMENT_JABBER_EN      => true, 
      PAYLOAD_HISTOGRAM_EN    => true,
      CNT_DSP                 => CNT_DSP,
      
      -- Remove FCS from the packet (false -> remove, true -> don't remove)
      INBANDFCS               => INBANDFCS
   )
   port map(
      -- -----------------------------------------------------
      --! \name Common input
      -- -----------------------------------------------------
      CLK   => CGMII_RXCLK,
      RESET	=> CGMII_RXRESET,
      
      -- -----------------------------------------------------
      --! \name Statistics input data
      -- -----------------------------------------------------
      -- Inputs for IFG computation
      SOP                     => cgmii_dec_tx_sop,
      SOP_ALIGN_SIZE          => cgmii_dec_tx_sop_pos,
      EOP                     => cgmii_dec_tx_eop,
      EOP_POS                 => cgmii_dec_tx_eop_pos,
      
      --! Statistical inputs(active when stat_dv is 1)
      STAT_FRAME_ERROR        => '0', -- This statistic is not used
      STAT_CRC_CHECK_FAILED   => stat_fifo_out(0),
      STAT_LEN_BELOW_MIN      => stat_fifo_out(1),
      STAT_LEN_OVER_MTU       => stat_fifo_out(2), 
      STAT_MAC_CHECK_FAILED   => stat_fifo_out(3),
      STAT_PAYLOAD_LEN        => stat_fifo_out(19 downto 4),
      STAT_MAC_BCAST          => stat_fifo_out(20),
      STAT_MAC_MCAST          => stat_fifo_out(21),
      STAT_DV                 => stat_dv,
      
      --! Frame processing flags
      STAT_FRAME_RECEIVED     => buf_frame_received,
      STAT_FRAME_DISCARDED    => buf_frame_discarded,
      STAT_BUFFER_OVF         => buf_frame_discarded_buf_ovf,

      -- -----------------------------------------------------
      --! \name Control interface
      -- -----------------------------------------------------
      START_EN                => mem_ibuf_en,
      SW_RESET                => addr_dec_sw_reset,
      RESET_AFTER_READ        => (others => '0'),  -- do not use this function
      READ_EN                 => addr_dec_read_en,
      LAST_ADDR_EN            => addr_dec_last_addr_en,

      -- -----------------------------------------------------
      --! \name Statistics output
      -- -----------------------------------------------------
      OUT_STAT_DV             => stat_unit_dv,
      OUT_FRAMES_RECEIVED     => stat_unit_received,
      OUT_FRAMES_DISCARDED    => stat_unit_discarded,
      OUT_TOTAL_PACKET_TRAF   => stat_unit_total,
      OUT_BUFFER_OVF          => stat_unit_discarded_buf_ovf,
      OUT_SIZE_SUM_OK         => stat_unit_octets_received_ok,
      OUT_SIZE_SUM            => stat_unit_octets_received,
      OUT_SIZE_SUM_COUNT      => open, 
      OUT_MAC_CHECK_FAILED    => open,
      OUT_CRC_ERR             => stat_unit_crc_err,
      OUT_OVER_MTU            => stat_unit_over_mtu,
      OUT_BELOW_MIN           => stat_unit_below_min,
      OUT_MAX_SIZE            => open, 
      OUT_MIN_SIZE            => open,
      OUT_MIN_DELAY           => open,
      OUT_MAX_DELAY           => open,
      OUT_LAST_READ_DELAY     => open,
      OUT_BCAST_FRAMES        => stat_unit_bcast_frames,
      OUT_MCAST_FRAMES        => stat_unit_mcast_frames,
      OUT_FRAGMENT_FRAMES     => stat_unit_fragment_frames,
      OUT_JABBER_FRAMES       => stat_unit_jabber_frames,
      OUT_FRAMES_64           => stat_unit_frames_64,
      OUT_FRAMES_65_127       => stat_unit_frames_65_127,
      OUT_FRAMES_128_255      => stat_unit_frames_128_255,
      OUT_FRAMES_256_511      => stat_unit_frames_256_511,
      OUT_FRAMES_512_1023     => stat_unit_frames_512_1023,
      OUT_FRAMES_1024_1518    => stat_unit_frames_1024_1518
   );
      
end architecture ibuf_cgmii_arch;
