-- flu_bloc_reord_arch.vhd: FLU_BLOCK_REORD full architecture.
-- Copyright (C) 2012 CESNET
-- Author(s): Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of FLU_BLOCK_REORD is
   --Input FLU pipeline signals ---------------------------
   signal flu_pipe_data    : std_logic_vector(511 downto 0);   --! Data pipeline register
   signal flu_pipe_sop     : std_logic := '0';                        --! SOP pipeline register
   signal flu_pipe_sop_pos : std_logic_vector(2 downto 0);     --! SOP_POS pipeline register
   signal flu_pipe_eop     : std_logic := '0';                        --! EOP pipeline register
   signal flu_pipe_eop_pos : std_logic_vector(5 downto 0);     --! EOP_POS pipeline register
   signal flu_pipe_src_rdy : std_logic := '0';

   --! Destination ready
   signal dst_rdy          : std_logic;

   --FLU frame gap counter --------------------------------
   --! FLU gap registers, 3 items fifo
   signal flu_gap_reg         : std_logic_vector(5 downto 0);
   signal reg_flu_gap_we      : std_logic;
   signal empty               : std_logic;
   --! output multiplexer signal
   signal flu_gap_out         : std_logic_vector(4 downto 0);
   signal reg_flu_gap_out     : std_logic_vector(4 downto 0);
   --! FLU gap temporary registers
   signal flu_gap_reg_tmp     : std_logic_vector(5 downto 0);
   signal flu_gap_reg_tmp2     : std_logic_vector(5 downto 0);

   --! FLU gap register enable
   signal flu_gap_reg_en         : std_logic;
   --! FLU gap temporary register enable
   signal flu_gap_reg_tmp_en     : std_logic;
   --! FLU gap acknowledge signal from fsm
   signal flu_gap_ack : std_logic;
   --! new flu gap ready signal
   signal flu_gap_new : std_logic;

   --! Repaired SOP_POS position
   signal flu_sop_pos_rep     : std_logic_vector(5 downto 0);
   --! Repaired SOP_POS position
   signal flu_eop_pos_rep     : std_logic_vector(5 downto 0);

   --Block multiplexer registers --------------------------
   --! FLU data delay register
   signal flu_data_in_reg0     : std_logic_vector(511 downto 0);
   --! SOP_VECT_IN delay register 
   signal sop_vect_in_reg0     : std_logic_vector(7 downto 0);
   --! EOP_VECT_IN delay register 
   signal eop_vect_in_reg0     : std_logic_vector(7 downto 0);
   --! EOP_POS in block delay register
   signal eop_pos_in_blk_reg0  : std_logic_vector(2 downto 0);

   --! FLU data delay register
   signal flu_data_in_reg3     : std_logic_vector(511 downto 0);
   --! SOP_VECT_IN delay register 
   signal sop_vect_in_reg3     : std_logic_vector(7 downto 0);
   --! EOP_VECT_IN delay register 
   signal eop_vect_in_reg3     : std_logic_vector(7 downto 0);
   --! EOP_POS in block delay register
   signal eop_pos_in_blk_reg3  : std_logic_vector(2 downto 0);
   --! FLU data delay register
   signal flu_data_in_reg4     : std_logic_vector(511 downto 0);
   --! EOP_POS in block delay register
   signal eop_pos_in_blk_reg4  : std_logic_vector(2 downto 0);

   --! FLU data delay register
   signal flu_data_in_reg     : std_logic_vector(511 downto 0);
   --! SOP_VECT_IN delay register 
   signal sop_vect_in_reg     : std_logic_vector(7 downto 0);
   --! EOP_VECT_IN delay register 
   signal eop_vect_in_reg     : std_logic_vector(7 downto 0);
   --! EOP_POS in block delay register
   signal eop_pos_in_blk_reg  : std_logic_vector(2 downto 0);

   --! FLU data delay register
   signal flu_data_in_reg2     : std_logic_vector(511 downto 0);
   --! SOP_VECT_IN delay register 
   signal sop_vect_in_reg2     : std_logic_vector(7 downto 0);
   --! EOP_VECT_IN delay register 
   signal eop_vect_in_reg2     : std_logic_vector(7 downto 0);
   --! EOP_POS in block delay register
   signal eop_pos_in_blk_reg2  : std_logic_vector(2 downto 0);

   --Detectors for SOP & EOP ------------------------------
   --! Actual selected SOP vector
   signal sop_vect_in      : std_logic_vector(7 downto 0);
   --! Actual selected EOP vector
   signal eop_vect_in      : std_logic_vector(7 downto 0);
   --! End of packet in block
   signal eop_pos_in_blk   : std_logic_vector(2 downto 0);

   --Block multiplexers -----------------------------------
   --Input signals --------------------
   --! Input data blocks
   signal flu_data_block_in   : std_logic_vector(1023 downto 0);
   signal reg_flu_data_block_in   : std_logic_vector(1023 downto 0);
   --! Input control SOP blocks
   signal flu_sop_block_in    : std_logic_vector(15 downto 0);
   --! Input control EOP blocks
   signal flu_eop_block_in    : std_logic_vector(15 downto 0);
   --! Input EOP_POS blocks   
   signal flu_eop_pos_blk_in  : std_logic_vector(5 downto 0);
   signal reg_flu_eop_pos_blk_in  : std_logic_vector(5 downto 0);

   --Selection signals ----------------
   --! Data blocks selection signals (8x4b)
   signal flu_data_block_sel : std_logic_vector(31 downto 0);
   signal reg_flu_data_block_sel : std_logic_vector(31 downto 0);
   --! Control blocks selection signals (8x4b)
   signal flu_control_sel    : std_logic_vector(31 downto 0);
   --! EOP_POS selection
   signal flu_eop_pos_blk_sel: std_logic_vector(0 downto 0);
   signal reg_flu_eop_pos_blk_sel: std_logic_vector(0 downto 0);

   --Output signals --------------------
   --! FLU output data blocks
   signal flu_data_block_out  : std_logic_vector(511 downto 0);
   --! FLU SOP control blocks
   signal flu_sop_block_out   : std_logic_vector(7 downto 0);
   --! FLU EOP control blocks
   signal flu_eop_block_out   : std_logic_vector(7 downto 0);

   --! FLU EOP_POS value
   signal flu_eop_pos_blk_out : std_logic_vector(2 downto 0);

   --Output {SOP,EOP}_V_OUT signals
   signal fsm_sop_v_out       : std_logic_vector(7 downto 0);
   signal fsm_eop_v_out       : std_logic_vector(7 downto 0);

   --! Output FLU pipeline register
   signal tx_data_reg         : std_logic_vector(511 downto 0);
   signal tx_sop_v_reg        : std_logic_vector(7 downto 0);
   signal tx_eop_v_reg        : std_logic_vector(7 downto 0);
   signal tx_sop_v_reg2        : std_logic_vector(7 downto 0);
   signal tx_eop_v_reg2        : std_logic_vector(7 downto 0);
   signal tx_eop_pos_reg      : std_logic_vector(2 downto 0);

   --! output FSM
   type fsm_t is (fsm_default, fsm_wait);
   signal current_state  : fsm_t;
   signal next_state     : fsm_t;

begin

   --------------------------------------------------------
   --1st process stage - FLU pipelining and FLU gap
   --------------------------------------------------------
   
   --! Component is still ready for data transmittion
   RX_DST_RDY <= dst_rdy;

   --! \brief Input FLU pipeline register
   --! \details This register is used for better timing results
   --! and it is also used for SOP and gap value synchronization.

   flu_pipep:process(CLK)
   begin
      if(CLK='1' and CLK'event)then
         if(RESET = '1')then
            flu_pipe_sop <= '0';
            flu_pipe_eop <= '0';
            flu_pipe_src_rdy <= '0';
         else
            if(dst_rdy = '1')then
               --Rewrite input data if and only if destination and source are ready
               flu_pipe_sop     <= RX_SOP;
               flu_pipe_eop     <= RX_EOP;
               flu_pipe_src_rdy <= RX_SRC_RDY;
            end if;
         end if;
      end if;
   end process;

   flu_pipe2p:process(CLK)
   begin
      if(CLK='1' and CLK'event)then
            if(dst_rdy = '1')then
               --Rewrite input data if and only if destination and source are ready
               flu_pipe_data    <= RX_DATA;
               flu_pipe_sop_pos <= RX_SOP_POS;
               flu_pipe_eop_pos <= RX_EOP_POS;
            end if;
      end if;
   end process;

   --! SOP_POS mulptiplyed by 8
   flu_sop_pos_rep <= flu_pipe_sop_pos & "000";
   flu_eop_pos_rep <= flu_pipe_eop_pos;

   --! Measuring gap between frames
   flu_gap_regp:process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (RESET = '1')  then
            -- after reset keep maximal FLU gap value
            flu_gap_reg <= (others => '1');
            flu_gap_reg_en <= '0';
           -- flu_gap_reg_tmp <= (others => '1');
            flu_gap_reg_tmp_en <= '0';
           -- flu_gap_reg_tmp2 <= (others => '1');
            flu_gap_new <= '0';
         else
            flu_gap_new <= '0';
	    if (dst_rdy = '1' and flu_pipe_src_rdy = '1') then
	       -- signals SOP and EOP are not always valid
	       if (flu_gap_reg_en = '1') then
		  -- EOP was detected in some previous flu word
		  if (flu_gap_reg_tmp_en = '1') then
		     -- gap is saved in flu_gap_reg_tmp register!
		     if (flu_pipe_sop = '0' and flu_pipe_eop = '0') then
			-- add length of flu word
			   flu_gap_reg_tmp2 <= conv_std_logic_vector(63,6);
			-- no more use for temporary register
			flu_gap_reg_tmp_en <= '0';
		     elsif (flu_pipe_eop = '0' and flu_pipe_sop = '1') then
			-- add length from beginning to the sop_pos of the flu word
			if (conv_integer(flu_gap_reg_tmp) + (conv_integer(flu_sop_pos_rep)) > 31) then
                           flu_gap_new <= '1';
			   flu_gap_reg <= conv_std_logic_vector(63,6);
			else
                           flu_gap_new <= '1';
			   flu_gap_reg <= flu_gap_reg_tmp + flu_sop_pos_rep;
			end if;
			-- no more use for temporary register
			flu_gap_reg_tmp_en <= '0';
			-- waiting for eop in next words
			flu_gap_reg_en <= '0';
		     else
			-- in this case sop < eop occured
			-- have to count length of the previous frame gap
			if (conv_integer(flu_gap_reg_tmp) + (conv_integer(flu_sop_pos_rep)) > 31) then
                           flu_gap_new <= '1';
			   flu_gap_reg <= conv_std_logic_vector(63,6);
			else
                           flu_gap_new <= '1';
			   flu_gap_reg <= flu_gap_reg_tmp + flu_sop_pos_rep;
			end if;
			-- and remember length of the new frame gap
			if (63 - conv_integer(flu_eop_pos_rep) > 31) then
			   flu_gap_reg_tmp <= conv_std_logic_vector(63, 6);
			else
			   flu_gap_reg_tmp <= 63 - (flu_eop_pos_rep);
			end if;
		     end if;                  
		  else
		     -- gap is saved in flu_gap_reg register!
		     if (flu_pipe_sop = '0' and flu_pipe_eop = '0') then
			-- add length of flu word
			   flu_gap_reg_tmp2 <= conv_std_logic_vector(63,6);
		     elsif (flu_pipe_eop = '0' and flu_pipe_sop = '1') then
			-- add length from beginning to the sop_pos of the flu word
			if (conv_integer(flu_gap_reg_tmp2) + (conv_integer(flu_sop_pos_rep)) > 31) then
                           flu_gap_new <= '1';
			   flu_gap_reg <= conv_std_logic_vector(63,6);
			else
                           flu_gap_new <= '1';
			   flu_gap_reg <= flu_gap_reg_tmp2 + flu_sop_pos_rep;
			end if;
			-- waiting for eop in next words
			flu_gap_reg_en <= '0';
		     else
			-- in this case sop < eop occured
			-- have to count length of the previous frame gap
			if (conv_integer(flu_gap_reg_tmp2) + (conv_integer(flu_sop_pos_rep)) > 31) then
                           flu_gap_new <= '1';
			   flu_gap_reg <= conv_std_logic_vector(63,6);
			else
                           flu_gap_new <= '1';
			   flu_gap_reg <= flu_gap_reg_tmp2 + flu_sop_pos_rep;
			end if;
			-- and remember length of the new frame gap
			if (63 - conv_integer(flu_eop_pos_rep) > 31) then
			   flu_gap_reg_tmp <= conv_std_logic_vector(63, 6);
			else
			   flu_gap_reg_tmp <= 63 - (flu_eop_pos_rep);
			end if;
			-- in the next clock cycle I will use length of the new frame gap
			flu_gap_reg_tmp_en <= '1';
		     end if;
		  end if;
	       else
		  -- EOP pos wasn't detected before
		  if (flu_pipe_eop = '1') then
		     -- waiting for EOP signal active
		     if (flu_pipe_sop = '1' and (flu_sop_pos_rep > flu_eop_pos_rep)) then
			-- frame gap starts and ends in the same flu word
			if ((conv_integer(flu_sop_pos_rep)) - conv_integer(flu_eop_pos_rep) - 1 > 31) then
                           flu_gap_new <= '1';
			   flu_gap_reg <= conv_std_logic_vector(63,6);
			else
                           flu_gap_new <= '1';
			   flu_gap_reg <= flu_sop_pos_rep - (flu_eop_pos_rep) - 1;
			end if;
		     else
			-- frame gap from eop_pos to the end of the flu word
			if (63 - conv_integer(flu_eop_pos_rep) > 31) then
			   flu_gap_reg_tmp2 <= conv_std_logic_vector(63,6);
			else
			   flu_gap_reg_tmp2 <= 63 - flu_eop_pos_rep;
			end if;
			-- continue counting in next words
			flu_gap_reg_en <= '1';
		     end if;
		  end if;
	       end if;
	    end if;
         end if;
      end if;
   end process;
   
   -- Trans FIFO instantiation
   flu_gap_fifo_i: entity work.fifo
   generic map (
      DATA_WIDTH    => 5,
      DISTMEM_TYPE  => 64,
      ITEMS         => 4
   )
   port map (
      CLK         => CLK,
      RESET       => RESET,
      DATA_IN     => flu_gap_reg(4 downto 0),
      DATA_OUT    => flu_gap_out,
      READ_REQ    => reg_flu_gap_we,
      WRITE_REQ   => flu_gap_new,
      FULL        => open,
      EMPTY       => empty,
      LSTBLK      => open
   );

   reg_flu_gap_outp: process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (RESET = '1') then
            reg_flu_gap_out <= (others => '1');
         elsif (reg_flu_gap_we = '1') then
            reg_flu_gap_out <= flu_gap_out;
         end if;
      end if;
   end process;

   --! Fifo FSM
   --! FSM process
   fsmp: process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            current_state <= fsm_default;
         else
            current_state <= next_state;
         end if;
      end if;
   end process;

   --! Fifo FSM output/next state logic
   next_state_logicp: process (current_state, flu_gap_ack, empty)
   begin
      --! Default values
      reg_flu_gap_we <= '0';

      case current_state is
         when fsm_default =>
            next_state <= current_state;
            if (flu_gap_ack = '1') then
               if (empty = '1') then
                  next_state <= fsm_wait;
               else
                  reg_flu_gap_we <= '1';
               end if;
            end if;
         when fsm_wait =>
            next_state <= current_state;
            if (empty = '0') then
                  next_state <= fsm_default;
                  reg_flu_gap_we <= '1';
            end if;
      end case;
   end process;


   --------------------------------------------------------
   --3rd stage - Block multiplexers
   --------------------------------------------------------
   --SOP & EOP block decoders ---------

   --! SOP one-hot decoder
   SOP_DETECT_GEN:for i in 0 to 7 generate
      sop_vect_in(i) <= '1' when(flu_pipe_sop = '1' and flu_pipe_sop_pos = conv_std_logic_vector(i,3) and flu_pipe_src_rdy = '1' and dst_rdy = '1')
                     else '0';
   end generate;

   --! EOP one-hot decoder
   EOP_VECT_GEN:for i in 0 to 7 generate
      eop_vect_in(i) <= '1' when(flu_pipe_eop = '1' and flu_pipe_eop_pos(5 downto 3) = conv_std_logic_vector(i,3) and flu_pipe_src_rdy = '1' and dst_rdy = '1')
                     else '0';
   end generate;

   --! EOP_POS in block  
   eop_pos_in_blk <= flu_pipe_eop_pos(2 downto 0); 

   --DATA,SOP_V,EOP_V registers -------
   --! \brief Delay registers for DATA and control informations.

   flu_pipe_data0p:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            sop_vect_in_reg0      <= "00000000";
            eop_vect_in_reg0      <= "00000000";
         else
            if(dst_rdy = '1')then
               sop_vect_in_reg0      <= sop_vect_in;
               eop_vect_in_reg0      <= eop_vect_in;
            end if;
         end if;
      end if;
   end process;

   flu_pipe_data3p:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
            if(dst_rdy = '1')then
               flu_data_in_reg0      <= flu_pipe_data;
               eop_pos_in_blk_reg0   <= eop_pos_in_blk;
            end if;
      end if;
   end process;

   --DATA,SOP_V,EOP_V registers -------
   --! \brief Delay registers for DATA and control informations.

   flu_pipe_data4p:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            sop_vect_in_reg3      <= "00000000";
            eop_vect_in_reg3      <= "00000000";
         else
            if(dst_rdy = '1')then
               sop_vect_in_reg3      <= sop_vect_in_reg0;
               eop_vect_in_reg3      <= eop_vect_in_reg0;
            end if;
         end if;
      end if;
   end process;

   flu_pipe_data5p:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
            if(dst_rdy = '1')then
               flu_data_in_reg3      <= flu_data_in_reg0;
               eop_pos_in_blk_reg3   <= eop_pos_in_blk_reg0;
            end if;
      end if;
   end process;

   --DATA,SOP_V,EOP_V registers -------
   --! \brief Delay registers for DATA and control informations.

   flu_pipe_datap:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            sop_vect_in_reg      <= "00000000";
            eop_vect_in_reg      <= "00000000";
         else
            if(dst_rdy = '1')then
               sop_vect_in_reg      <= sop_vect_in_reg3;
               eop_vect_in_reg      <= eop_vect_in_reg3;
            end if;
         end if;
      end if;
   end process;

   flu_pipe_data2p:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
            if(dst_rdy = '1')then
               flu_data_in_reg      <= flu_data_in_reg3;
               eop_pos_in_blk_reg   <= eop_pos_in_blk_reg3;
            end if;
      end if;
   end process;

   --DATA,SOP_V,EOP_V registers -------
   --! \brief Delay registers for DATA and control informations.

   flu_in_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            sop_vect_in_reg2      <= "00000000";
            eop_vect_in_reg2      <= "00000000";
         else
            if(dst_rdy = '1')then
               sop_vect_in_reg2      <= sop_vect_in_reg;
               eop_vect_in_reg2      <= eop_vect_in_reg;
            end if;
         end if;
      end if;
   end process;

   flu_in_reg2p:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
            if(dst_rdy = '1')then
               flu_data_in_reg2      <= flu_data_in_reg;
               eop_pos_in_blk_reg2   <= eop_pos_in_blk_reg;
            end if;
      end if;
   end process;

   --Block multiplexers -----------------------------------
   -- DATA ----------------------------
   --! Input data blocks
   flu_data_block_in(511  downto 0)   <= flu_data_in_reg;
   flu_data_block_in(1023 downto 512)  <= flu_data_in_reg2;

   reg_flu_data_block_inp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
            reg_flu_data_block_in      <= flu_data_block_in;
            reg_flu_eop_pos_blk_in     <= flu_eop_pos_blk_in;
      end if;
   end process;

   --! Data block multiplexer - lane 0
   DATA_LANE0_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_data_block_in,
      SEL         => reg_flu_data_block_sel(3 downto 0),
      DATA_OUT    => flu_data_block_out(63 downto 0)
   );

   --! Data block multiplexer - lane 1
   DATA_LANE1_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_data_block_in,
      SEL         => reg_flu_data_block_sel(7 downto 4),
      DATA_OUT    => flu_data_block_out(127 downto 64)
   );

   --! Data block multiplexer - lane 2
   DATA_LANE2_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_data_block_in,
      SEL         => reg_flu_data_block_sel(11 downto 8),
      DATA_OUT    => flu_data_block_out(191 downto 128)
   );

   --! Data block multiplexer - lane 3
   DATA_LANE3_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_data_block_in,
      SEL         => reg_flu_data_block_sel(15 downto 12),
      DATA_OUT    => flu_data_block_out(255 downto 192)
   );

   --! Data block multiplexer - lane 4
   DATA_LANE4_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_data_block_in,
      SEL         => reg_flu_data_block_sel(19 downto 16),
      DATA_OUT    => flu_data_block_out(319 downto 256)
   );

   --! Data block multiplexer - lane 5
   DATA_LANE5_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_data_block_in,
      SEL         => reg_flu_data_block_sel(23 downto 20),
      DATA_OUT    => flu_data_block_out(383 downto 320)
   );

   --! Data block multiplexer - lane 6
   DATA_LANE6_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_data_block_in,
      SEL         => reg_flu_data_block_sel(27 downto 24),
      DATA_OUT    => flu_data_block_out(447 downto 384)
   );

   --! Data block multiplexer - lane 7
   DATA_LANE7_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 64,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_data_block_in,
      SEL         => reg_flu_data_block_sel(31 downto 28),
      DATA_OUT    => flu_data_block_out(511 downto 448)
   );

   
   --SOP information ------------------
   --! Input SOP control blocks
   flu_sop_block_in(7 downto 0) <= sop_vect_in_reg;
   flu_sop_block_in(15 downto 8) <= sop_vect_in_reg2;

   --! Control SOP block multiplexer - lane 0
   SOP_LANE0_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_sop_block_in,
      SEL         => flu_control_sel(3 downto 0),
      DATA_OUT    => flu_sop_block_out(0 downto 0)
   );
   
   --! Control SOP block multiplexer - lane 1
   SOP_LANE1_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_sop_block_in,
      SEL         => flu_control_sel(7 downto 4),
      DATA_OUT    => flu_sop_block_out(1 downto 1)
   );

   --! Control SOP block multiplexer - lane 2
   SOP_LANE2_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_sop_block_in,
      SEL         => flu_control_sel(11 downto 8),
      DATA_OUT    => flu_sop_block_out(2 downto 2)
   );

   --! Control SOP block multiplexer - lane 3
   SOP_LANE3_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_sop_block_in,
      SEL         => flu_control_sel(15 downto 12),
      DATA_OUT    => flu_sop_block_out(3 downto 3)
   );
   --! Control SOP block multiplexer - lane 4
   SOP_LANE4_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_sop_block_in,
      SEL         => flu_control_sel(19 downto 16),
      DATA_OUT    => flu_sop_block_out(4 downto 4)
   );

   --! Control SOP block multiplexer - lane 5
   SOP_LANE5_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_sop_block_in,
      SEL         => flu_control_sel(23 downto 20),
      DATA_OUT    => flu_sop_block_out(5 downto 5)
   );
   --! Control SOP block multiplexer - lane 6
   SOP_LANE6_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_sop_block_in,
      SEL         => flu_control_sel(27 downto 24),
      DATA_OUT    => flu_sop_block_out(6 downto 6)
   );

   --! Control SOP block multiplexer - lane 7
   SOP_LANE7_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_sop_block_in,
      SEL         => flu_control_sel(31 downto 28),
      DATA_OUT    => flu_sop_block_out(7 downto 7)
   );

   --EOP information ------------------
   --! Input EOP control blocks
   flu_eop_block_in(7 downto 0) <= eop_vect_in_reg;
   flu_eop_block_in(15 downto 8) <= eop_vect_in_reg2;

   --! Control EOP block multiplexer - lane 0 
   EOP_LANE0_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_eop_block_in,
      SEL         => flu_control_sel(3 downto 0),
      DATA_OUT    => flu_eop_block_out(0 downto 0)
   );

   --! Control EOP block multiplexer - lane 1 
   EOP_LANE1_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_eop_block_in,
      SEL         => flu_control_sel(7 downto 4),
      DATA_OUT    => flu_eop_block_out(1 downto 1)
   );

   --! Control EOP block multiplexer - lane 2 
   EOP_LANE2_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_eop_block_in,
      SEL         => flu_control_sel(11 downto 8),
      DATA_OUT    => flu_eop_block_out(2 downto 2)
   );

   --! Control EOP block multiplexer - lane 3 
   EOP_LANE3_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_eop_block_in,
      SEL         => flu_control_sel(15 downto 12),
      DATA_OUT    => flu_eop_block_out(3 downto 3)
   );

   --! Control EOP block multiplexer - lane 4 
   EOP_LANE4_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_eop_block_in,
      SEL         => flu_control_sel(19 downto 16),
      DATA_OUT    => flu_eop_block_out(4 downto 4)
   );

   --! Control EOP block multiplexer - lane 5 
   EOP_LANE5_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_eop_block_in,
      SEL         => flu_control_sel(23 downto 20),
      DATA_OUT    => flu_eop_block_out(5 downto 5)
   );

   --! Control EOP block multiplexer - lane 6 
   EOP_LANE6_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_eop_block_in,
      SEL         => flu_control_sel(27 downto 24),
      DATA_OUT    => flu_eop_block_out(6 downto 6)
   );

   --! Control EOP block multiplexer - lane 7 
   EOP_LANE7_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 1,
      MUX_WIDTH   => 16   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => flu_eop_block_in,
      SEL         => flu_control_sel(31 downto 28),
      DATA_OUT    => flu_eop_block_out(7 downto 7)
   );

   
   --EOP_POS information --------------
   --! Input EOP_POS blocks
   flu_eop_pos_blk_in(2 downto 0) <= eop_pos_in_blk_reg;
   flu_eop_pos_blk_in(5 downto 3) <= eop_pos_in_blk_reg2;

   --! EOP_POS block multiplexer 
   EOP_POS_I:entity work.GEN_MUX
   generic map(
      DATA_WIDTH  => 3,
      MUX_WIDTH   => 2   -- multiplexer width (number of inputs)
   )
   port map(
      DATA_IN     => reg_flu_eop_pos_blk_in,
      SEL         => reg_flu_eop_pos_blk_sel,
      DATA_OUT    => flu_eop_pos_blk_out
   );

   --------------------------------------------------------
   --FSM
   --------------------------------------------------------

   CONTROL_I:entity work.FLU_BLOCK_REORD_CONTROL
   port map(
      -----------------------------------------------------
      --! \name Common interface
      -----------------------------------------------------
      --! Clock signal
      CLK         => CLK,
      --! Synchronous reset

      RESET       => RESET,

      -----------------------------------------------------
      --! \name Input Intefrace 
      -----------------------------------------------------
      --! Measured FLU inter-frame gap
      FLU_GAP     => reg_flu_gap_out,
      --! SOP block control information (one-hot) in selected reading window
      SOP_V_IN    => flu_sop_block_out,
      --! EOP block control information (one-hot) in selected reading window
      EOP_V_IN    => flu_eop_block_out,
      
      -----------------------------------------------------
      --! \name Output interface
      -----------------------------------------------------
      --! Data selection signal - lane 0
      DSEL_ADDR0 => flu_data_block_sel(3 downto 0), 
      --! Data selection signal - lane 1
      DSEL_ADDR1 => flu_data_block_sel(7 downto 4),
      --! Data selection signal - lane 2
      DSEL_ADDR2 => flu_data_block_sel(11 downto 8),
      --! Data selection signal - lane 3
      DSEL_ADDR3 => flu_data_block_sel(15 downto 12),
      --! Data selection signal - lane 4
      DSEL_ADDR4 => flu_data_block_sel(19 downto 16), 
      --! Data selection signal - lane 5
      DSEL_ADDR5 => flu_data_block_sel(23 downto 20),
      --! Data selection signal - lane 6
      DSEL_ADDR6 => flu_data_block_sel(27 downto 24),
      --! Data selection signal - lane 7
      DSEL_ADDR7 => flu_data_block_sel(31 downto 28),

      --! Control selection signal - lane 0
      CSEL_ADDR0 => flu_control_sel(3 downto 0),
      --! Control selection signal - lane 1
      CSEL_ADDR1 => flu_control_sel(7 downto 4),
      --! Control selection signal - lane 2
      CSEL_ADDR2 => flu_control_sel(11 downto 8),
      --! Control selection signal - lane 3
      CSEL_ADDR3 => flu_control_sel(15 downto 12),
      --! Control selection signal - lane 0
      CSEL_ADDR4 => flu_control_sel(19 downto 16),
      --! Control selection signal - lane 1
      CSEL_ADDR5 => flu_control_sel(23 downto 20),
      --! Control selection signal - lane 2
      CSEL_ADDR6 => flu_control_sel(27 downto 24),
      --! Control selection signal - lane 3
      CSEL_ADDR7 => flu_control_sel(31 downto 28),
   
      --! Delayed/actual EOP_POS in block selection
      EOP_POS_ADDR => flu_eop_pos_blk_sel(0),

      --! Output SOP vector (one-hot coding)
      SOP_V_OUT   => fsm_sop_v_out,
      --! Output EOP vector (one-hot coding)
      EOP_V_OUT   => fsm_eop_v_out,
      
      --! Destination ready
      DST_RDY     => dst_rdy,

      --! Flu gap acknowledge
      FLU_GAP_ACK => flu_gap_ack
   );

   --! FLU data block sel register
   reg_flu_data_block_selp: process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (RESET = '1') then
            reg_flu_data_block_sel <= "01110110010101000011001000010000";
            reg_flu_eop_pos_blk_sel <= "0";
         else
            reg_flu_data_block_sel <= flu_data_block_sel;
            reg_flu_eop_pos_blk_sel <= flu_eop_pos_blk_sel;
         end if; 
      end if;
   end process;

   --! Output FLU registers for better timing results
   tx_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            tx_sop_v_reg <= (others => '0');
            tx_eop_v_reg <= (others => '0');
            tx_sop_v_reg2 <= (others => '0');
            tx_eop_v_reg2 <= (others => '0');
         else
            tx_sop_v_reg <= fsm_sop_v_out;
            tx_eop_v_reg <= fsm_eop_v_out;
            tx_sop_v_reg2 <= tx_sop_v_reg;
            tx_eop_v_reg2 <= tx_eop_v_reg;
         end if;
      end if;
   end process;

   tx_reg2p:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
            tx_data_reg <= flu_data_block_out;
            tx_eop_pos_reg <= flu_eop_pos_blk_out;
         end if;
   end process;

   --------------------------------------------------------
   --Output signal map 
   --------------------------------------------------------
   --TX DATA
   TX_DATA    <= tx_data_reg;

   --TX_SOP_V
   TX_SOP_V    <= tx_sop_v_reg2;

   --TX_EOP_V
   TX_EOP_V    <= tx_eop_v_reg2;

   --TX_EOP_POS
   TX_EOP_POS  <= tx_eop_pos_reg; 

end architecture full;
