-- buf_ent.vhd: Buf unit of CGMII Input buffer - entity declaration
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                           Entity declaration
-- ----------------------------------------------------------------------------

entity cgmii_ibuf_buf is
   generic(
      -- Number of items in Data FIFO
      DFIFO_SIZE     : integer := 512;
      -- Number of items in Header FIFO
      HFIFO_SIZE     : integer := 256;
      -- Width of header buses in bits
      HDR_WIDTH      : integer := 128;
      -- Synthesise design without resets, default values of registers are set
      -- by initialization of signals
      RESET_BY_INIT  : boolean := false
   );
   port(
      -- Common signals -------------------------------------------------------
      -- Input clock signal
      CGMII_CLK     : in std_logic;
      -- Reset for CGMII_CLK domain
      CGMII_RESET   : in std_logic;     
      -- Output clock signal
      FLU_CLK         : in std_logic;
      -- Reset for FLU_CLK domain
      FLU_RESET       : in std_logic;
      
      
      -- ----------------------------------------------------------------------
      --                        Data inputs and outputs
      -- ----------------------------------------------------------------------

      -- Data input interface -------------------------------------------------
      -- Packet data
      RX_DATA        : in std_logic_vector(511 downto 0);
      -- Start of the packet
      RX_SOP         : in std_logic;
      -- Position of the start of the packet, valid if RX_SOP is set to '1'
      RX_SOP_POS     : in std_logic_vector(2 downto 0);
      -- End of the packet
      RX_EOP         : in std_logic;
      -- Position of the end of the packet, valid if RX_EOP is set to '1'
      RX_EOP_POS     : in std_logic_vector(5 downto 0);

      -- Data payload output --------------------------------------------------
      -- Payload data
      TX_DATA        : out std_logic_vector(511 downto 0);
      -- Position of the start of the payload, valid if TX_SOP is set to '1'
      TX_SOP_POS     : out std_logic_vector(2 downto 0);
      -- Position of the end of the payload, valid if TX_EOP is set to '1'
      TX_EOP_POS     : out std_logic_vector(5 downto 0);
      -- Start of the payload
      TX_SOP         : out std_logic;
      -- End of the payload
      TX_EOP         : out std_logic;
      -- Source is ready
      TX_SRC_RDY     : out std_logic;
      -- Destination is ready
      TX_DST_RDY     : in std_logic;

      -- Data header output ---------------------------------------------------
      -- Header data
      TX_HDATA       : out std_logic_vector(HDR_WIDTH-1 downto 0);
      -- Start of the header, active in '0'
      TX_HSOP_N      : out std_logic;
      -- End of the header, active in '0'
      TX_HEOP_N      : out std_logic;
      -- Position of the end of the header, valid if TX_HEOP_N is set to '0'
      TX_HREM        : out std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
      -- Source is ready, active in '0'
      TX_HSRC_RDY_N  : out std_logic;
      -- Destination is ready, active in '0'
      TX_HDST_RDY_N  : in std_logic;


      -- ----------------------------------------------------------------------
      --                   Statistics input and outputs
      -- ----------------------------------------------------------------------
      
      -- Statistics input (valid when RX_EOP = '1') ---------------------------
      -- MAC address is not accepted
      RX_MAC_ERR     : in std_logic;
      -- Frame does not have minimal length
      RX_MINTU_ERR   : in std_logic;
      -- Frame is longer than maximal length
      RX_MTU_ERR     : in std_logic;
      -- Discard the frame
      RX_SAU_ERR     : in std_logic;
      -- Frame has bad CRC
      RX_CRC_ERR     : in std_logic;
      -- Error inside the packet was detected
      RX_FRAME_ERR   : in std_logic;
      -- Frame length
      RX_FRAME_LEN   : in std_logic_vector(15 downto 0);
      
      -- Frame processing signals ---------------------------------------------
      -- Frame is received
      FRAME_RECEIVED          : out std_logic;
      -- Frame is discarded due to some error (including buffer overflow)
      FRAME_DISCARDED         : out std_logic;
      -- Frame is discarded due to buffer overflow
      FRAME_DISCARDED_BUF_OVF : out std_logic;


      -- ----------------------------------------------------------------------
      --                        PACODAG interface
      -- ----------------------------------------------------------------------
      
      -- Statistics for PACODAG -----------------------------------------------
      -- MAC address is not accepted
      CTRL_STAT_MAC_ERR    : out std_logic;
      -- Frame does not have minimal length
      CTRL_STAT_MINTU_ERR  : out std_logic;
      -- Frame is longer than maximal length
      CTRL_STAT_MTU_ERR    : out std_logic;
      -- Frame has bad CRC
      CTRL_STAT_CRC_ERR    : out std_logic;
      -- Incoming protocol error
      CTRL_STAT_FRAME_ERR  : out std_logic;
      -- Frame length
      CTRL_STAT_FRAME_LEN  : out std_logic_vector(15 downto 0);
      
      -- PACODAG interface ----------------------------------------------------
      -- Control CLK
      CTRL_CLK       : out std_logic;
      -- Control reset
      CTRL_RESET     : out std_logic;
      -- Control data
      CTRL_DATA      : in std_logic_vector(HDR_WIDTH-1 downto 0);
      -- Start of control data, active in '0'
      CTRL_SOP_N     : in std_logic;
      -- End of control data, active in '0'
      CTRL_EOP_N     : in std_logic;
      -- Position of the end of control data, valid if CTRL_EOP_N is set to '0'.
      CTRL_REM       : in std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
      -- Asserted when the input signals from control data generator are valid
      CTRL_SRC_RDY_N : in std_logic;
      -- Asserted when data from control data generator will be accepted
      CTRL_DST_RDY_N : out std_logic;
      -- New frame is being received; prepare to generate new control data
      CTRL_SOP       : out std_logic;
      -- Statistics for PACODAG are valid
      CTRL_STAT_DV   : out std_logic;
      -- Control data generator is ready to receive new request
      CTRL_RDY       : in std_logic;


      -- ----------------------------------------------------------------------
      --                        Registers interface
      -- ----------------------------------------------------------------------

      -- Value of registers placed in Mem module ------------------------------
      -- Value of IBUF enable register
      IBUF_EN        : in std_logic;
      -- Value of error mask register
      ERROR_MASK     : in std_logic_vector(4 downto 0);

      -- Input value for register placed in Mem module ------------------------
      -- Value for status register
      STATUS         : out std_logic_vector(16 downto 0)
   );
end entity cgmii_ibuf_buf;

