-- block_decode.vhd : 40/100GBASE-R PCS block decoder module
--
-- Copyright (C) 2011 CESNET
-- Author(s): Stepan Friedl <friedl@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the 
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: $

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity block_decode is
   generic (
      NUM_LANES        : natural := 4;
      STATE_SIMPLIFIED : boolean := true
   );
   port (
      RESET       : in std_logic;
      CLK         : in std_logic; -- RX clock, 156.25MHz
      -- Encoded data input
      D           : in std_logic_vector(NUM_LANES*66-1 downto 0); -- Encoded data
      --  XL(C)GMII output
      RXD         : out std_logic_vector(NUM_LANES*64-1 downto 0);  -- XL(C)GMII data
      RXC         : out std_logic_vector(NUM_LANES*8-1 downto 0);  -- XL(C)GMII command
      -- Control & status
      BYPASS      : in std_logic := '0'; -- Decoder bypass
      BLK_LOCK    : in std_logic := '1'; -- Block lock input
      LOOPBACK    : in std_logic := '0'; -- Loopback enable
      HI_BER      : in std_logic := '0'; -- Hi BER flag
      BLK_ERR_CNTR: out std_logic_vector(21 downto 0);  -- block error counter
      BLK_ERR_CLR : in std_logic := '0'; -- Clear block error counter
      DEC_ERROR   : out std_logic_vector(NUM_LANES-1 downto 0); -- Block decode error
      SEQ_ERROR   : out std_logic_vector(NUM_LANES-1 downto 0); -- Block sequencing error
      STATE_O     : out std_logic_vector(NUM_LANES*3-1 downto 0)
   );
end block_decode;

-- ----------------------------------------------------------------------------
--             Architecture declaration  --  ComboLXT Top Level              --
-- ----------------------------------------------------------------------------
architecture behavioral of block_decode is

signal reset_n       : std_logic;
signal errd_blks     : std_logic_vector(8*NUM_LANES-1 downto 0);
signal rxs           : std_logic_vector( 3*NUM_LANES-1 downto 0); -- TX encoder state
signal rxs_p         : std_logic_vector( 3*NUM_LANES-1 downto 0); -- previous lane encoder state
signal rxs_reg       : std_logic_vector( 2 downto 0); -- 

begin

reset_n <= not reset;

GEN_DECODER: for i in 0 to NUM_LANES-1 generate

   DECODER: entity work.Decode 
   generic map (
      STATE_SIMPLIFIED => STATE_SIMPLIFIED
   )
   port map (
      DeScr_RXD        => D(66*(i+1)-1 downto 66*i),
      blk_lock         => BLK_LOCK,
      bypass_66decoder => BYPASS,
      lpbk             => LOOPBACK,
      clear_errblk     => BLK_ERR_CLR, -- clk156 synchronous
      clk156           => CLK,
      hi_ber           => HI_BER,
      rstb156          => reset_n,
      rxlf             => open,
      errd_blks        => errd_blks(8*(i+1)-1 downto 8*i),
      rxcontrol        => RXC( 8*(i+1)-1 downto  8*i),
      rxdata           => RXD(64*(i+1)-1 downto 64*i),
      STATE_PREV       => rxs_p(3*(i+1)-1 downto 3*i), -- State of previous lane
      STATE            => rxs(3*(i+1)-1 downto i*3),   -- State of current decoder
      DEC_ERROR        => DEC_ERROR(i),
      SEQ_ERROR        => SEQ_ERROR(i),
      STATE_O          => STATE_O(3*(i+1)-1 downto 3*i)
   );
   
end generate;

ERR_COUNTER: process(CLK)
variable tmp : integer;
begin
   if CLK'event and CLK = '1' then
      tmp := conv_integer(errd_blks(8*1-1 downto 8*0));
      for i in 1 to NUM_LANES-1 loop
         tmp := tmp + conv_integer(errd_blks(8*(i+1)-1 downto 8*i));
      end loop;
      BLK_ERR_CNTR <= conv_std_logic_vector(tmp, 22);
   end if;
end process;

rxs_p(2 downto 0) <= rxs_reg;
GEN_STATE: for i in 0 to NUM_LANES-2 generate
   rxs_p(3*(i+2)-1 downto 3*(i+1)) <= rxs(3*(i+1)-1 downto 3*i);
end generate;

DEC_STATE_REGS: process(CLK)
begin
   if clk'event and clk = '1' then
      rxs_reg <= rxs(rxs'high downto rxs'high-2);
   end if;
end process;

end behavioral;
