-- 100ge_phy_arch.vhd : The complete PHY for the 100GBASE-R (PCS+PMA) for 
--                      Virtex7 with GTZ transceivers
--
-- Copyright (C) 2013 CESNET
-- Author(s): Stepan Friedl <friedl@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: $
--
-- NOTES:

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

architecture V7_GTZ of phy_100ge_caui4_main is

  -------------------------------------------------------------------
  --  ICON core component declaration
  -------------------------------------------------------------------
--   component icon
--     port
--     (
--       control0    : inout std_logic_vector(35 downto 0)
--     );
--   end component;
--
--   component icon2
--     port
--     (
--       control0    : inout std_logic_vector(35 downto 0);
--       control1    : inout std_logic_vector(35 downto 0)
--     );
--   end component;
--
--   component ila
--     port
--     (
--       control     : inout std_logic_vector(35 downto 0);
--       clk         : in    std_logic;
--       trig0       : in    std_logic_vector(63 downto 0);
--       trig1       : in    std_logic_vector( 7 downto 0)
--     );
--   end component;
--
--   component pcsila
--     port
--     (
--       control     : inout std_logic_vector(35 downto 0);
--       clk         : in    std_logic;
--       trig0       : in    std_logic_vector(71 downto 0);
--       trig1       : in    std_logic_vector(31 downto 0)
--     );
--   end component;
--
--   component pcsila_wide
--   PORT (
--     CONTROL : INOUT STD_LOGIC_VECTOR(35 DOWNTO 0);
--     CLK     : IN STD_LOGIC;
--     TRIG0   : IN STD_LOGIC_VECTOR(255 DOWNTO 0);
--     TRIG1   : IN STD_LOGIC_VECTOR(7  DOWNTO 0);
--     TRIG2   : IN STD_LOGIC_VECTOR(31 DOWNTO 0)
--    );
--
--    end component;
--
--   signal control0   : std_logic_vector(35 downto 0);
--   signal control1   : std_logic_vector(35 downto 0);
-- --   signal trig0_a    : std_logic_vector(63 downto 0);
-- --   signal trig1_a    : std_logic_vector( 7 downto 0);
--   signal trig0_a    : std_logic_vector(71 downto 0);
--   signal trig1_a    : std_logic_vector(31 downto 0);
--   signal trig0_b    : std_logic_vector(63 downto 0);
--   signal trig1_b    : std_logic_vector( 7 downto 0);
--   signal trig0w    : std_logic_vector(255 downto 0);
--   signal trig1w    : std_logic_vector(  7 downto 0);
--   signal trig2w    : std_logic_vector( 31 downto 0);
--
--   attribute noopt : boolean;
--   attribute noopt of ila   : component is TRUE;
--   attribute noopt of pcsila: component is TRUE;
--   attribute noopt of pcsila_wide: component is TRUE;
--   attribute noopt of icon  : component is TRUE;
--   attribute noopt of icon2 : component is TRUE;
--   attribute dont_touch : boolean;
--   attribute dont_touch of ila   : component is TRUE;
--   attribute dont_touch of icon  : component is TRUE;
--   attribute dont_touch of icon2 : component is TRUE;
  -------------------------------------------------------------------
  
component gtz_caui4_v3_0_top
generic 
(
    EXAMPLE_SIMULATION          : integer := 1;        -- Set to 1 for simulation
    SIM_GTZRESET_SPEEDUP        : string  := "FALSE";
    SIM_VERSION                 :  string := "1.0"
);
port 
(
    RESET_IN          : in std_logic;
    REFCLK_N_IN       : in std_logic;
    REFCLK_P_IN       : in std_logic;
    TXRESETDONE_OUT   : out std_logic;
    RXRESETDONE_OUT   : out std_logic;
    -- Parallel data
    TXCLK_OUT         : out std_logic;
    TXDATA_IN         : in std_logic_vector(66*20-1 downto 0);
    TXREADY_OUT       : out std_logic_vector(19 downto 0);
    RXCLK_OUT         : out std_logic;
    RXINTCLK_OUT      : out std_logic;
    RXDATA_OUT        : out std_logic_vector(20*66-1 downto 0);
    RXDATA_OK_IN      : in std_logic; -- _vector(19 downto 0);
    RXVALID_OUT       : out std_logic_vector(19 downto 0);
    RXCLK_STABLE      : out std_logic;
    RX_OK_OUT         : out std_logic_vector(3 downto 0);
    -- Status/control
    BLK_LOCK_OUT      : out std_logic_vector(19 downto 0);
    SIGNAL_DET        : in std_logic_vector(3 downto 0);
    POWERDOWN_IN      : in std_logic;
    FARLOOPBACKEN_IN  : in std_logic;
    NEARLOOPBACKEN_IN : in std_logic;
    TXPRBSEN_IN       : in std_logic;
    TXPRBSSEL_IN      : in std_logic_vector(2 downto 0);
    RXPRBSEN_IN       : in std_logic;
    RXPRBSSEL_IN      : in std_logic_vector(2 downto 0);
    RXPRBSPASS_OUT    : out std_logic_vector(3 downto 0);
    CTLE_RETUNE       : in std_logic;
    -- GTZ DRP port
    DRPCLK_IN         : in std_logic; 
    DRPSEL            : in std_logic;
    DRPEN             : in std_logic;
    DRPWE             : in std_logic;
    DRPDI             : in std_logic_vector(31 downto 0);
    DRPADDR           : in std_logic_vector(15 downto 0);
    DRPDO             : out std_logic_vector(31 downto 0);
    DRPRDY            : out std_logic;
    DRPSTATUS         : out std_logic_vector(2 downto 0);
    DRPWDTERR         : out std_logic;
    --
    TXPOLARITY_IN     : in std_logic_vector(3 downto 0);
    RXPOLARITY_IN     : in std_logic_vector(3 downto 0);
    TXN_OUT           : out std_logic_vector(3 downto 0);
    TXP_OUT           : out std_logic_vector(3 downto 0);
    RXN_IN            : in std_logic_vector(3 downto 0);
    RXP_IN            : in std_logic_vector(3 downto 0);
    dbg_state         : out std_logic_vector(31 downto 0)    
   ); 
end component;

signal force_pma_reset : std_logic;
signal pma_txreset_done : std_logic;
signal pma_txreset_done_r : std_logic := '0';
signal pma_txreset_done_s : std_logic := '0';
signal pma_rxreset_done : std_logic;
signal pma_rxreset_done_r : std_logic := '0';
signal pma_rxreset_done_s : std_logic := '0';
signal rxinit_done  : std_logic;
signal pcs_tx_reset_mi  : std_logic := '1';
signal pcs_tx_reset_mii : std_logic := '1';
signal pcs_tx_reset_in  : std_logic := '1';
signal pcs_tx_reset_int : std_logic := '1';
signal pcs_tx_reset_pc  : std_logic := '1';
signal pcs_tx_reset_pcs : std_logic := '1';
signal pcs_rx_reset_mi  : std_logic := '1';
signal pcs_rx_reset_mii : std_logic := '1';
signal pcs_rx_reset_in  : std_logic := '1';
signal pcs_rx_reset_int : std_logic := '1';
signal pcs_rx_reset_pc  : std_logic := '1';
signal pcs_rx_reset_pcs : std_logic := '1';
signal pma_txclk    : std_logic;
signal int_clk      : std_logic;
signal rxint_clk    : std_logic;
signal rxclk_stable : std_logic;
signal xlgmii_clk_stable : std_logic;
signal cgmii_clk   : std_logic;

signal pcs_txd    : std_logic_vector(20*66-1 downto 0);
signal pma_txd    : std_logic_vector(20*66-1 downto 0);
signal pcs_txd_en : std_logic_vector(20-1 downto 0);
signal pma_rx_ok  : std_logic_vector(3 downto 0);
signal pma_rxclk  : std_logic;
signal pcs_rxd    : std_logic_vector(20*66-1 downto 0);
signal pcs_rxd_vld: std_logic_vector(19 downto 0);
signal pcs_rx_ok  : std_logic; -- no errors in then receive  PCS 
signal pma_txd_ptrn : std_logic_vector(20*66-1 downto 0);
--
signal algn_locked   : std_logic;
signal bip_err_cntrs : std_logic_vector(20*16-1 downto 0);
signal lane_map      : std_logic_vector(20*5-1 downto 0);
signal lane_align    : std_logic_vector(20-1 downto 0);
signal bip_err_clr   : std_logic_vector(20-1 downto 0);
signal pma_near_lpbck: std_logic;
signal pma_far_lpbck : std_logic;
signal pma_reset     : std_logic;
signal pma_low_pwr   : std_logic;
signal pma_retune    : std_logic; -- Force CTLE tunning on
signal pma_ptrn_en   : std_logic; -- Force PMA to transmit a debug pattern
signal hi_ber        : std_logic;
signal dec_hi_ber    : std_logic;
signal blk_lock      : std_logic_vector(20-1 downto 0);
signal linkstatus    : std_logic;
signal ber_count     : std_logic_vector(21 downto 0);
signal clr_ber_cnt   : std_logic;
signal blk_err_cntr  : std_logic_vector(21 downto 0);
signal blk_err_clr   : std_logic;
signal scr_bypass    : std_logic_vector(1 downto 0);
signal scr_bypass_mgmt: std_logic_vector(1 downto 0);
signal pcs_reset     : std_logic;

signal am_cntr_o     : std_logic;
signal am_found_o    : std_logic_vector(3 downto 0);
signal bip_err_o     : std_logic_vector(3 downto 0);
signal fifo_dv_o     : std_logic_vector(3 downto 0);
signal fifo_rd_o     : std_logic_vector(3 downto 0);
signal fifo_empty_o  : std_logic_vector(3 downto 0);
signal fifo_full_o   : std_logic_vector(3 downto 0);
signal rxd_o         : std_logic_vector(66*4-1 downto 0);
signal rxd_ce        : std_logic;
signal dec_state     : std_logic_vector(4*3-1 downto 0);

attribute SHREG_EXTRACT : string;
attribute SHREG_EXTRACT of pma_txreset_done_s : signal is "no";
attribute SHREG_EXTRACT of pma_txreset_done_r : signal is "no";
attribute SHREG_EXTRACT of pma_rxreset_done_s : signal is "no";
attribute SHREG_EXTRACT of pma_rxreset_done_r : signal is "no";
attribute SHREG_EXTRACT of pcs_rx_reset_mi  : signal is "no";
attribute SHREG_EXTRACT of pcs_rx_reset_mii : signal is "no";
attribute SHREG_EXTRACT of pcs_rx_reset_pc  : signal is "no";
attribute SHREG_EXTRACT of pcs_rx_reset_pcs : signal is "no";
attribute SHREG_EXTRACT of pcs_rx_reset_in  : signal is "no";
attribute SHREG_EXTRACT of pcs_rx_reset_int : signal is "no";
attribute SHREG_EXTRACT of pcs_tx_reset_mi  : signal is "no";
attribute SHREG_EXTRACT of pcs_tx_reset_mii : signal is "no";
attribute SHREG_EXTRACT of pcs_tx_reset_in  : signal is "no";
attribute SHREG_EXTRACT of pcs_tx_reset_int : signal is "no";
attribute SHREG_EXTRACT of pcs_tx_reset_pc  : signal is "no";
attribute SHREG_EXTRACT of pcs_tx_reset_pcs : signal is "no";

type t_drp_fsm is (IDLE, READ, WRITE, DONE);
signal drp_state : t_drp_fsm := IDLE;

signal drpsel      : std_logic;
signal drpen       : std_logic;
signal drpwe       : std_logic;
signal drpdi       : std_logic_vector(31 downto 0);
signal drpaddr     : std_logic_vector(15 downto 0);
signal drpdo       : std_logic_vector(31 downto 0);
signal drprdy      : std_logic;
signal drpstatus   : std_logic_vector(2 downto 0);
signal drpwdterr   : std_logic;
--
signal drp_drd     : std_logic_vector(31 downto 0);
signal drp_done    : std_logic;

signal reg_mi_addr  : std_logic_vector(1 downto 0);
signal mi_drd_drp   : std_logic_vector(31 downto 0);
signal mi_drdy_drp  : std_logic;
signal mi_drd_mgmt  : std_logic_vector(31 downto 0);
signal mi_drdy_mgmt : std_logic;

signal drp_addr_mi  : std_logic_vector(15 downto 0);
signal drp_wr_mi    : std_logic;
signal drp_wr_mi_r1 : std_logic;
signal drp_wr_mi_r2 : std_logic;
signal drp_rd_mi    : std_logic;
signal drp_rd_mi_r1 : std_logic;
signal drp_rd_mi_r2 : std_logic;
signal drp_dwr_mi   : std_logic_vector(31 downto 0);
signal drp_status_r : std_logic_vector(2 downto 0);

signal seq_cntr     : std_logic_vector(7 downto 0) := (others => '0');
signal pma_dbg      : std_logic_vector(31 downto 0);

begin

MI_ARDY <= MI_RD or MI_WR;
MI_DRDY <= mi_drdy_drp when reg_mi_addr = "00" else mi_drdy_mgmt;
MI_DRD  <= mi_drd_drp  when reg_mi_addr = "00" else mi_drd_mgmt;

DRP_ACCESS: process(MI_CLK)
begin
   if MI_CLK'event and MI_CLK = '1' then
  
      reg_mi_addr <= MI_ADDR(17 downto 16);
      drp_wr_mi_r1 <= '0';
      drp_rd_mi_r1 <= '0';      
      
      if MI_ADDR(17 downto 16) = "00" then
         if MI_WR = '1' then
            case MI_ADDR(3 downto 2) is
               when "00" => -- 0x00000
                  drp_dwr_mi  <= MI_DWR;               
               when "01" => -- 0x00004
                  drp_addr_mi  <= MI_DWR(15 downto 0); 
                  drp_wr_mi_r1 <= MI_DWR(31);
                  drp_rd_mi_r1 <= not MI_DWR(31);
               when others => null;
            end case;
         end if;
      end if;
      -- Read
      case MI_ADDR(3 downto 2) is
         when "00" => 
            mi_drd_drp <= drp_drd;
         when "01" => 
            mi_drd_drp <= drp_done & drp_status_r & X"000" & drp_addr_mi;
         when "10" => -- 0x08 - read debug state  
            mi_drd_drp(pma_dbg'high downto 0) <= pma_dbg;
         when others => null;
      end case;               
      
      mi_drdy_drp  <= MI_RD;
      
      drp_wr_mi_r2 <= drp_wr_mi_r1;
      drp_rd_mi_r2 <= drp_rd_mi_r1;
      drp_wr_mi    <= drp_wr_mi_r1 or drp_wr_mi_r2;
      drp_rd_mi    <= drp_rd_mi_r1 or drp_rd_mi_r2;      
      
   end if;
end process;

DRP_FSM: process(DRPCLK)
begin
   if DRPCLK'event and DRPCLK = '1' then
      drpaddr   <= drp_addr_mi;
      drpdi     <= drp_dwr_mi;
      case drp_state is
         when IDLE => 
            drpsel <= '0';
            drpen  <= '0';
            if (drp_wr_mi = '1') then
               drpsel    <= '1';
               drpen     <= '1';
               drpwe     <= '1';
               drp_done  <= '0';
               drp_state <= WRITE;
            elsif (drp_rd_mi = '1') then
               drpsel    <= '1';
               drpen     <= '1';
               drpwe     <= '0';
               drp_done  <= '0';
               drp_state <= READ;
            end if;
            
         when WRITE =>
            drpen  <= '0';
            drpsel <= '1';
            if drprdy = '1' then
               drp_status_r <= drpstatus;
               drp_done  <= '1';
               drpsel    <= '0';
               drp_state <= DONE;
            end if;
            
         when READ =>
            drpen  <= '0';
            drpsel <= '1';
            if drprdy = '1' then
               drp_drd  <= drpdo;
               drpsel   <= '0';
               drp_status_r <= drpstatus;
               drp_state <= DONE;
            end if;
            
         when DONE =>
            drp_done  <= '1';
            drp_state <= IDLE;
            
      end case;
      if force_pma_reset = '1' then
         drp_state <= IDLE;
      end if;
   end if;
end process;

MGMT: entity work.mgmt
generic map (
   NUM_LANES     => 20,
   PMA_LANES     =>  4,
   SPEED100G     => '1',
   GBASE40_ABLE  => '0',
   GBASE100_ABLE => '1'
)
port map (
   RESET    => RESET,
   MI_CLK   => MI_CLK,
   MI_DWR   => MI_DWR,
   MI_ADDR  => MI_ADDR,
   MI_RD    => MI_RD,
   MI_WR    => MI_WR,
   MI_BE    => MI_BE,
   MI_DRD   => mi_drd_mgmt,
   MI_ARDY  => open,
   MI_DRDY  => mi_drdy_mgmt,
   PCSCLK   => cgmii_clk,
   PMACLK   => pma_txclk,
   -- PMA & PMD status/control
   PMA_LPBCK     => pma_near_lpbck,
   PMA_REM_LPBCK => pma_far_lpbck,
   PMA_RESET     => pma_reset,
   PMA_RETUNE    => pma_retune,
   PMA_PTRN_EN   => pma_ptrn_en,
   PMA_RX_OK     => pma_rx_ok,
   PMD_SIG_DET   => SIGNAL_DET, 
   PMA_LOPWR     => pma_low_pwr,
   -- PCS Lane align
   ALGN_LOCKED   => algn_locked,
   LANE_MAP      => lane_map,
   LANE_ALIGN    => lane_align,
   BIP_ERR_CNTRS => bip_err_cntrs,
   BIP_ERR_CLR   => bip_err_clr,
   -- PCS status
   HI_BER        => hi_ber,
   BLK_LOCK      => blk_lock,
   LINKSTATUS    => linkstatus,
   BER_COUNT     => ber_count,
   BER_COUNT_CLR => clr_ber_cnt,
   BLK_ERR_CNTR  => blk_err_cntr,
   BLK_ERR_CLR   => blk_err_clr,
   SCR_BYPASS    => scr_bypass_mgmt,
   PCS_RESET     => pcs_reset,
   PCS_LPBCK     => open
);

SIM: if (SIMULATION = 1) generate
   scr_bypass <= "11";
end generate;

NO_SIM: if (SIMULATION /= 1) generate
   scr_bypass <= scr_bypass_mgmt;
end generate;

-- TX PCS
TX_PCS: entity work.tx_path_100g
generic map (MAC_LANES => 8)
port map (
   RESET_MII => pcs_tx_reset_mii,
   RESET_INT => pcs_tx_reset_int,
   RESET_PCS => pcs_tx_reset_pcs,
   --
   CGMII_CLK => cgmii_clk,
   CGMII_TXD => CGMII_TXD,
   CGMII_TXC => CGMII_TXC,
   --
   CLKINT     => int_clk,
   ENC_BYPASS => '0',
   SCR_BYPASS => scr_bypass(1),
   -- HS interface
   PCS_CLK    => pma_txclk,
   PCS_TXD    => pcs_txd,
   PCS_TXD_EN => pcs_txd_en(0)
);

pma_txd <= pcs_txd;

-- For debug only: Generate Test pattern to PMA -------------------------------------------------
-- GEN_TXD: for i in 0 to 19 generate
--    pma_txd_ptrn((i+1)*66-1 downto i*66+2) <= conv_std_logic_vector(i,8) & X"0100" & seq_cntr &
--                                         conv_std_logic_vector(i,8) & X"0000" & seq_cntr ;
--    pma_txd_ptrn(i*66+1     downto i*66)   <= "01";
-- end generate;
-- 
-- BYTE_GEN: process(pma_txclk)
-- begin
--    if pma_txclk'event and pma_txclk = '1' then
--       if pcs_txd_en(0) = '1' then
--          seq_cntr <= seq_cntr + 1;
--       end if;
--    end if;
-- end process;
-- pma_txd <= pcs_txd when (pma_ptrn_en = '0') else pma_txd_ptrn;
--

RX_PATH: entity work.rx_path_100g
port map (
   RESET_MII     => pcs_rx_reset_mii,
   RESET_INT     => pcs_rx_reset_int,
   RESET_PCS     => pcs_rx_reset_pcs,
   --
   INT_CLK       => rxint_clk,
   CGMII_CLK     => cgmii_clk,
   CGMII_RXD     => CGMII_RXD,
   CGMII_RXC     => CGMII_RXC,
   -- HS interface
   RXCLK         => pma_rxclk,
   RXCLK_EN      => pcs_rxd_vld,
   RXD           => pcs_rxd,
   -- Status ports
   HI_BER        => hi_ber,
   DEC_HI_BER    => dec_hi_ber,
   BLK_LOCK      => blk_lock,
   LINKSTATUS    => linkstatus,
   BER_COUNT     => ber_count,
   CLR_BER_CNT   => clr_ber_cnt,
   BLK_ERR_CNTR  => blk_err_cntr,
   BLK_ERR_CLR   => blk_err_clr,
   --
   LANE_MAP      => lane_map,
   LANES_ALIGNED => lane_align,
   ALIGN_LOCKED  => algn_locked,
   BIP_ERR_CNTRS => bip_err_cntrs,
   BIP_ERR_CLR   => bip_err_clr,
   SCR_BYPASS    => scr_bypass(0)
   --
--    AM_CNTR_O     => am_cntr_o,
--    AM_FOUND_O    => am_found_o,
--    BIP_ERR_O     => bip_err_o,
--    FIFO_DV_O     => fifo_dv_o,
--    FIFO_RD_O     => fifo_rd_o,
--    FIFO_FULL_O   => fifo_full_o,
--    FIFO_EMPTY_O  => fifo_empty_o,
--    RXD_O         => rxd_o,
--    RXD_CE        => rxd_ce,
--    DEC_STATE     => dec_state
);

-- This is an auxilliary flag to indicate the PMA that the recieve data is OK
pcs_rx_ok <= (not dec_hi_ber) and (not hi_ber) and algn_locked;

PMA_I: gtz_caui4_v3_0_top
generic map
(
    EXAMPLE_SIMULATION          => SIMULATION,        -- Set to 1 for simulation
    SIM_GTZRESET_SPEEDUP        => "FALSE",
    SIM_VERSION                 => "1.0"
)
port map
(
    RESET_IN          => force_pma_reset,
    REFCLK_N_IN       => REFCLK_P,
    REFCLK_P_IN       => REFCLK_N,
    TXRESETDONE_OUT   => pma_txreset_done,
    RXRESETDONE_OUT   => pma_rxreset_done, 
    -- Parallel data
    TXCLK_OUT         => pma_txclk,    
    TXDATA_IN         => pma_txd,
    TXREADY_OUT       => pcs_txd_en,
    RXCLK_OUT         => pma_rxclk,
    RXINTCLK_OUT      => rxint_clk,
    RXDATA_OUT        => pcs_rxd,
    RXDATA_OK_IN      => pcs_rx_ok,  
    RXVALID_OUT       => pcs_rxd_vld,
    RXCLK_STABLE      => rxclk_stable,
    RX_OK_OUT         => pma_rx_ok,
    -- Status/control
    BLK_LOCK_OUT      => blk_lock,
    SIGNAL_DET        => SIGNAL_DET(3 downto 0),
    POWERDOWN_IN      => pma_low_pwr,
    FARLOOPBACKEN_IN  => pma_far_lpbck,
    NEARLOOPBACKEN_IN => pma_near_lpbck,
    -- LOOPBACK_IN       => "000",
    TXPRBSEN_IN       => '0',
    TXPRBSSEL_IN      => "000",  -- TXCLK domain
    RXPRBSEN_IN       => '0',
    RXPRBSSEL_IN      => "000",  -- RXCLK domain
    --RXPRBSERR_OUT     => open,
    RXPRBSPASS_OUT    => open,
    CTLE_RETUNE       => pma_retune,
    --                       
    DRPCLK_IN         => DRPCLK,
    DRPSEL            => drpsel,
    DRPEN             => drpen,
    DRPWE             => drpwe,
    DRPDI             => drpdi,
    DRPADDR           => drpaddr,
    DRPDO             => drpdo,
    DRPRDY            => drprdy ,
    DRPSTATUS         => drpstatus,
    DRPWDTERR         => drpwdterr,
    --
    TXPOLARITY_IN     => TXPOLARITY,     
    RXPOLARITY_IN     => RXPOLARITY,     
    TXN_OUT           => TXN(3 downto 0),
    TXP_OUT           => TXP(3 downto 0),
    RXN_IN            => RXN(3 downto 0),
    RXP_IN            => RXP(3 downto 0),
    dbg_state         => pma_dbg
); 

CLKGEN: entity work.gbase100_clkgen
port map
(
   CLK_IN     => pma_txclk,
   -- Clock out ports
   CGMII_CLK  => cgmii_clk,
   INT_CLK    => int_clk,
   PCS_CLK    => open, -- 80.xx MHz
   -- Status and control signals
   RESET      => RESET,
   LOCKED     => xlgmii_clk_stable
);

force_pma_reset <= RESET or pma_reset;

-- PCS RESET pulse synchronization - CGMII clock domain (common for TX and RX)
RXRESET_FF_CGMII : process(cgmii_clk, xlgmii_clk_stable)
begin
   if xlgmii_clk_stable = '0' then -- Gloal async reset
      pcs_tx_reset_mii <= '1';
      pcs_rx_reset_mii <= '1';
   elsif cgmii_clk'event and cgmii_clk = '1' then
      pma_txreset_done_r <= pma_txreset_done;
      pma_txreset_done_s <= pma_txreset_done_r;
      
      pma_rxreset_done_r <= pma_rxreset_done;
      pma_rxreset_done_s <= pma_rxreset_done_r;      

      pcs_tx_reset_mi  <= RESET or pcs_reset or (not pma_txreset_done_s);
      pcs_tx_reset_mii <= pcs_tx_reset_mi;
      
      pcs_rx_reset_mi  <= RESET or pcs_reset or (not pma_rxreset_done_s);
      pcs_rx_reset_mii <= pcs_rx_reset_mi;      
   end if;
end process;

-- PCS RESET pulse synchronization - TX INT clock domain
TXRESET_FF_INT : process(int_clk, xlgmii_clk_stable)
begin
   if xlgmii_clk_stable = '0' then -- Gloal async reset
      pcs_tx_reset_int <= '1';
   elsif int_clk'event and int_clk = '1' then
      pcs_tx_reset_in  <= RESET or pcs_reset or (not pma_txreset_done);
      pcs_tx_reset_int <= pcs_tx_reset_in;
   end if;
end process;

-- PCS RESET pulse synchronization -  TX clock domain
TXRESET_FF_PCS : process(pma_txclk, xlgmii_clk_stable)
begin
   if xlgmii_clk_stable = '0' then -- Gloal async reset
      pcs_tx_reset_pcs <= '1';
   elsif pma_txclk'event and pma_txclk = '1' then
      pcs_tx_reset_pc  <= RESET or pcs_reset or (not pma_txreset_done);
      pcs_tx_reset_pcs <= pcs_tx_reset_pc;
   end if;
end process;


-- PCS RESET pulse synchronization - RX INT clock domain
RXRESET_FF_INT : process(rxint_clk, rxclk_stable)
begin
   if rxclk_stable = '0' then -- Gloal async reset
      pcs_rx_reset_int <= '1';
   elsif rxint_clk'event and rxint_clk = '1' then
      pcs_rx_reset_in  <= RESET or pcs_reset or (not pma_rxreset_done);
      pcs_rx_reset_int <= pcs_rx_reset_in;
   end if;
end process;

-- PCS RESET pulse synchronization -  RX clock domain
RXRESET_FF_PCS : process(pma_rxclk, rxclk_stable)
begin
   if rxclk_stable = '0' then -- Gloal async reset
      pcs_rx_reset_pcs  <= '1';
   elsif pma_rxclk'event and pma_rxclk = '1' then
      pcs_rx_reset_pc  <= pcs_reset or (not pma_rxreset_done); -- or (not rxinit_done);
      pcs_rx_reset_pcs <= pcs_rx_reset_pc;
   end if;
end process;

RXCLK <= cgmii_clk;
TXCLK <= cgmii_clk;
CLK_STABLE <= xlgmii_clk_stable;

-------------------------------------------------------------------------------
-- Chipscope
-------------------------------------------------------------------------------

-- hi_ber,
-- blk_lock,
-- linkstatus,
-- ber_count,
-- clr_ber_cnt,
-- blk_err_cntr,
-- blk_err_clr,
-- algn_locked,
-- lane_map,
-- lane_align,
-- bip_err_cntrs,
-- bip_err_clr,

-- am_cntr_o
-- am_found_o
-- bip_err_o
-- fifo_dv_o
-- fifo_rd_o
-- fifo_empty_o
-- rxd_ce

--   trig0_a <= bip_err_cntrs(31 downto 16) & bip_err_cntrs(15 downto 0) & blk_err_cntr(15 downto 0) & ber_count(15 downto 0);
--   trig1_a <= fifo_full_o(0) & lane_align(0) & algn_locked & fifo_empty_o(0) &
--              fifo_rd_o(0) & fifo_dv_o(0) & am_found_o(0) & am_cntr_o;

--   trig0_a <= rxd_o(66*3+17 downto 66*3) & rxd_o(66*2+17 downto 66*2) & rxd_o(66*1+17 downto 66*1) & rxd_o(17 downto 0);
--   trig1_a <= ber_count(15 downto 0) & blk_err_cntr(15 downto 0);
-- --   trig1_a <= rxd_ce          & algn_locked     & am_cntr_o       & pcs_rx_reset &
-- --              lane_align(3)   & lane_align(2)   & lane_align(1)   & lane_align(0) &
-- --              fifo_full_o(3)  & fifo_full_o(2)  & fifo_full_o(1)  & fifo_full_o(0) &
-- --              fifo_empty_o(3) & fifo_empty_o(2) & fifo_empty_o(1) & fifo_empty_o(0) &
-- --              fifo_rd_o(3)    & fifo_rd_o(2)    & fifo_rd_o(1)    & fifo_rd_o(0) &
-- --              fifo_dv_o(3)    & fifo_dv_o(2)    & fifo_dv_o(1)    & fifo_dv_o(0) &
-- --              am_found_o(3)   & am_found_o(2)   & am_found_o(1)   & am_found_o(0) &
-- --              bip_err_o(3)    & bip_err_o(2)    & bip_err_o(1)    & bip_err_o(0);
--
--   trig0w <= rxd_o(263 downto 200) & rxd_o(197 downto 134) & rxd_o(131 downto 68) & rxd_o(65 downto 2); -- Data blocks
--   trig1w <= rxd_o(199 downto 198) & rxd_o(133 downto 132) & rxd_o(67 downto 66) & rxd_o(1 downto 0); -- Sync headers
--   trig2w <= blk_err_clr & rxd_ce & fifo_full_o(0) & fifo_empty_o(0) & dec_state & blk_err_cntr(15 downto 0);
--
--   I_ICON : icon -- 2
--     port map
--     (
--       control0  => control0 --,
--       --control1  => control1
--     );
--
--    i_ila_a : pcsila_wide
--     port map
--     (
--       control => control0,
--       clk     => pma_rxclk0,
--       trig0   => trig0w,
--       trig1   => trig1w,
--       trig2   => trig2w
--     );
--
--
-- --    i_ila_a : pcsila
-- --     port map
-- --     (
-- --       control => control0,
-- --       clk     => cgmii_clk,
-- --       trig0   => trig0_a,
-- --       trig1   => trig1_a
-- --     );
--
-- --   trig0_b <= mi_drd & mi_addr;
-- --   trig1_b <= mi_be & mi_ardy & mi_drdy & mi_wr & mi_rd;
--
-- --     i_ila_b : ila
-- --     port map
-- --     (
-- --       control => control1,
-- --       clk     => cgmii_clk,
-- --       trig0   => trig0_b,
-- --       trig1   => trig1_b(7 downto 0)
-- --     );

end V7_GTZ;




