-- top.vhd: HFE-M toplevel, connecting analyzer and extractor.
-- Copyright (C) 2011 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Entity declaration
-- ----------------------------------------------------------------------------

entity hfe_m_top is
   generic (
      --* FrameLink width
      DATA_WIDTH           : integer := 512;
      --* Use input pipe
      IN_PIPE_EN           : boolean := true;
      --* Use output register of input pipe
      IN_PIPE_OUTREG       : boolean := true;
      --* Use middle pipe
      MID_PIPE_EN          : boolean := true;
      --* Use output register of middle pipe
      MID_PIPE_OUTREG      : boolean := true;
      --* Use output pipe
      OUT_PIPE_EN          : boolean := true;
      --* Use output register of output pipe
      OUT_PIPE_OUTREG      : boolean := true;
      --* Width of OFFSET_* signals (which are in Bytes).
      --* 8 means that headers should fit in first 256 B of packet
      --* (realistic guess). 14 Eth + 4*4 VLAN/MPLS + min. 40 IPv6 = min. 70
      --* Don't set less than 8 unless you know what you're doing.
      OFFSET_WIDTH         : integer := 8;
      --* Skip this amount of FL words (SZE header typically)
      HEADER_SKIP          : integer := 1;
      --* Length of IPv6 Hdr Ext Len field.
      --* Should be 8, lower it to save logic.
      IPV6_EXT_LEN_BITS    : integer := 3;
      --* Maximum supported IPv6 Extension headers, max 8
      IPV6_MAX_EXT_HEADERS : integer := 2;
      --* Extractor latency.
      --* 0 or 1. Set to 1 for higher frequency and lower resources, 
      --* but results one cycle later
      EXTRACT_LATENCY      : integer := 1
   );
   port (
      CLK         : in std_logic;
      RESET       : in std_logic;

      --+ FrameLink input interface
      RX_DATA     : in std_logic_vector(DATA_WIDTH-1 downto 0);
      RX_DREM     : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      RX_SRC_RDY_N: in std_logic;
      RX_DST_RDY_N: out std_logic;
      RX_SOF_N    : in std_logic;
      RX_EOF_N    : in std_logic;
      RX_SOP_N    : in std_logic;
      RX_EOP_N    : in std_logic;

      --+ FrameLink output interface
      TX_DATA     : out std_logic_vector(DATA_WIDTH-1 downto 0);
      TX_DREM     : out std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      TX_SRC_RDY_N: out std_logic;
      TX_DST_RDY_N: in  std_logic;
      TX_SOF_N    : out std_logic;
      TX_EOF_N    : out std_logic;
      TX_SOP_N    : out std_logic;
      TX_EOP_N    : out std_logic;

      --------------------------------------------------------------------
      -- Analysis results
      --------------------------------------------------------------------
      -- Link layer ------------------------------------------------------
      --* Offset of the link layer (Ethernet) header start from SOF (in Bytes)
      LINK_OFFSET       : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Ethernet header present
      ACTIVE_ETHERNET   : out std_logic;
      --* Link layer analysis still running (above outputs are not valid)
      RUNNING_LINK      : out std_logic;

      -- VLAN and MPLS ---------------------------------------------------
      --* Offset of the first VLAN header start from SOF (in Bytes)
      VLAN0_OFFSET      : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the second VLAN header start from SOF (in Bytes)
      VLAN1_OFFSET      : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the first MPLS header start from SOF (in Bytes)
      MPLS0_OFFSET      : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the second MPLS header start from SOF (in Bytes)
      MPLS1_OFFSET      : out std_logic_vector(OFFSET_WIDTH-1 downto 0);

      --* First VLAN header present
      ACTIVE_VLAN0      : out std_logic;
      --* Second VLAN header present
      ACTIVE_VLAN1      : out std_logic;
      --* First MPLS header present
      ACTIVE_MPLS0      : out std_logic;
      --* Second MPLS header present
      ACTIVE_MPLS1      : out std_logic;

      --* VLAN & MPLS layer analysis still running (above outputs are not valid)
      RUNNING_VLAN_MPLS : out std_logic;

      -- Internet layer ------------------------------------------------
      --* Offset of the Internet layer (IP) header start from SOF (in Bytes)
      INTERNET_OFFSET   : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* IPv4 header present
      ACTIVE_IPV4       : out std_logic;
      --* IPv6 header present
      ACTIVE_IPV6       : out std_logic;
      --* Is set to 1 when packet is NOT fragmented or has
      --* fragment offset = 0
      FRAGMENT_FIRST : out std_logic;
      --* Is set to 1 when packet is NOT fragmented or is
      --* the last fragment of the fragmented packet
      FRAGMENT_LAST  : out std_logic;
      --* Internet layer analysis still running (above outputs are not valid)
      RUNNING_INTERNET  : out std_logic;
      --* IPv4 header present but has errors (may appear after RUNNING_INTERNET)
      ERROR_IPV4        : out std_logic;
      --* IPv6 header present but has errors (may appear after RUNNING_INTERNET)
      ERROR_IPV6        : out std_logic;

      -- Transport layer -----------------------------------------------
      -- Something returned only if IP Fragment Offset == 0
      --* Offset of the Transport layer header start from SOF (in Bytes)
      TRANSPORT_OFFSET  : out std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* TCP header present
      ACTIVE_TCP        : out std_logic;
      --* UDP header present
      ACTIVE_UDP        : out std_logic;
      --* ICMP header present
      ACTIVE_ICMP       : out std_logic;
      --* IGMP header present
      ACTIVE_IGMP       : out std_logic;
      --* MPLS in IP header present
      ACTIVE_MPLS_TUN   : out std_logic;
      --* SCTP header present
      ACTIVE_SCTP       : out std_logic;
      --* IPv6 in IP header present
      ACTIVE_IPV6_TUN   : out std_logic;
      --* IPv4 in IP header present
      ACTIVE_IPV4_TUN   : out std_logic;
      --* IPv6 No Next Header identifier present
      ACTIVE_IPV6_NO_NEXT:out std_logic;
      --* Transport layer analysis still running (above outputs are not valid)
      RUNNING_TRANSPORT : out std_logic;

      --------------------------------------------------------------------
      -- Extracted fields
      --------------------------------------------------------------------
      -- Internet layer -------------------------------------------------
      --+ Source IPv4 address
      SRCIPV4           : out std_logic_vector(31 downto 0);
      SRCIPV4_VLD       : out std_logic;
      --+ Destination IPv4 address
      DSTIPV4           : out std_logic_vector(31 downto 0);
      DSTIPV4_VLD       : out std_logic;
      --+ Source IPv6 address
      SRCIPV6           : out std_logic_vector(127 downto 0);
      SRCIPV6_VLD       : out std_logic;
      --+ Destination IPv6 address
      DSTIPV6           : out std_logic_vector(127 downto 0);
      DSTIPV6_VLD       : out std_logic;
      --+ Protocol
      PROTOCOL          : out std_logic_vector(7 downto 0);
      PROTOCOL_VLD      : out std_logic;

      -- Transport layer ------------------------------------------------
      -- Something returned only if IP Fragment Offset == 0
      --+ Source TCP port number
      SRCTCP            : out std_logic_vector(15 downto 0);
      SRCTCP_VLD        : out std_logic;
      --+ Destination TCP port number
      DSTTCP            : out std_logic_vector(15 downto 0);
      DSTTCP_VLD        : out std_logic;
      --+ Source UDP port number
      SRCUDP            : out std_logic_vector(15 downto 0);
      SRCUDP_VLD        : out std_logic;
      --+ Destination UDP port number
      DSTUDP            : out std_logic_vector(15 downto 0);
      DSTUDP_VLD        : out std_logic;

      --* Don't expect any new *_VLD to appear after this
      PARSING_DONE      : out std_logic
   );
end entity;

architecture full of hfe_m_top is

   -- HFE-M analysis results
   signal sig_link_offset      : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_active_ethernet  : std_logic;
   signal sig_running_link     : std_logic;
   signal sig_vlan0_offset     : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_vlan1_offset     : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_mpls0_offset     : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_mpls1_offset     : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_active_vlan0     : std_logic;
   signal sig_active_vlan1     : std_logic;
   signal sig_active_mpls0     : std_logic;
   signal sig_active_mpls1     : std_logic;
   signal sig_running_vlan_mpls: std_logic;
   signal sig_internet_offset  : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_active_ipv4      : std_logic;
   signal sig_active_ipv6      : std_logic;
   signal sig_fragment_first   : std_logic;
   signal sig_fragment_last    : std_logic;
   signal sig_running_internet : std_logic;
   signal sig_error_ipv4       : std_logic;
   signal sig_error_ipv6       : std_logic;
   signal sig_transport_offset : std_logic_vector(OFFSET_WIDTH-1 downto 0);
   signal sig_active_tcp       : std_logic;
   signal sig_active_udp       : std_logic;
   signal sig_active_icmp      : std_logic;
   signal sig_active_igmp      : std_logic;
   signal sig_active_mpls_tun  : std_logic;
   signal sig_active_sctp      : std_logic;
   signal sig_active_ipv6_tun  : std_logic;
   signal sig_active_ipv4_tun  : std_logic;
   signal sig_active_ipv6_no_next:std_logic;
   signal sig_protocol_in      : std_logic_vector(7 downto 0);
   signal sig_running_transport: std_logic;

   signal sig_tx_data     : std_logic_vector(DATA_WIDTH-1 downto 0);
   signal sig_tx_drem     : std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
   signal sig_tx_src_rdy_n: std_logic;
   signal sig_tx_sof_n    : std_logic;
   signal sig_tx_eof_n    : std_logic;
   signal sig_tx_sop_n    : std_logic;
   signal sig_tx_eop_n    : std_logic;

begin

   analyzer_i : entity work.hfe_m_full
   generic map (
      DATA_WIDTH           => DATA_WIDTH,
      IN_PIPE_EN           => IN_PIPE_EN,
      IN_PIPE_OUTREG       => IN_PIPE_OUTREG,
      MID_PIPE_EN          => MID_PIPE_EN,
      MID_PIPE_OUTREG      => MID_PIPE_OUTREG,
      OUT_PIPE_EN          => OUT_PIPE_EN,
      OUT_PIPE_OUTREG      => OUT_PIPE_OUTREG,
      OFFSET_WIDTH         => OFFSET_WIDTH,
      HEADER_SKIP          => HEADER_SKIP,
      IPV6_EXT_LEN_BITS    => IPV6_EXT_LEN_BITS,
      IPV6_MAX_EXT_HEADERS => IPV6_MAX_EXT_HEADERS
   )
   port map (
      CLK               => CLK,
      RESET             => RESET,

      -- Input interface
      RX_DATA           => RX_DATA,
      RX_DREM           => RX_DREM,
      RX_SRC_RDY_N      => RX_SRC_RDY_N,
      RX_DST_RDY_N      => RX_DST_RDY_N,
      RX_SOF_N          => RX_SOF_N,
      RX_EOF_N          => RX_EOF_N,
      RX_SOP_N          => RX_SOP_N,
      RX_EOP_N          => RX_EOP_N,

      -- Output interface
      TX_DATA           => sig_tx_data,
      TX_DREM           => sig_tx_drem,
      TX_SRC_RDY_N      => sig_tx_src_rdy_n,
      TX_DST_RDY_N      => TX_DST_RDY_N,
      TX_SOF_N          => sig_tx_sof_n,
      TX_EOF_N          => sig_tx_eof_n,
      TX_SOP_N          => sig_tx_sop_n,
      TX_EOP_N          => sig_tx_eop_n,

      LINK_OFFSET       => sig_link_offset,
      ACTIVE_ETHERNET   => sig_active_ethernet,
      RUNNING_LINK      => sig_running_link,
      -- VLAN and MPLS ---------------------------------------------------
      VLAN0_OFFSET      => sig_vlan0_offset,
      VLAN1_OFFSET      => sig_vlan1_offset,
      MPLS0_OFFSET      => sig_mpls0_offset,
      MPLS1_OFFSET      => sig_mpls1_offset,
      ACTIVE_VLAN0      => sig_active_vlan0,
      ACTIVE_VLAN1      => sig_active_vlan1,
      ACTIVE_MPLS0      => sig_active_mpls0,
      ACTIVE_MPLS1      => sig_active_mpls1,
      RUNNING_VLAN_MPLS => sig_running_vlan_mpls,
      -- Internet layer ------------------------------------------------
      INTERNET_OFFSET   => sig_internet_offset,
      ACTIVE_IPV4       => sig_active_ipv4,
      ACTIVE_IPV6       => sig_active_ipv6,
      FRAGMENT_FIRST    => sig_fragment_first,
      FRAGMENT_LAST     => sig_fragment_last,
      RUNNING_INTERNET  => sig_running_internet,
      ERROR_IPV4        => sig_error_ipv4,
      ERROR_IPV6        => sig_error_ipv6,
      -- Transport layer -----------------------------------------------
      TRANSPORT_OFFSET  => sig_transport_offset,
      ACTIVE_TCP        => sig_active_tcp,
      ACTIVE_UDP        => sig_active_udp,
      ACTIVE_ICMP       => sig_active_icmp,
      ACTIVE_IGMP       => sig_active_igmp,
      ACTIVE_MPLS_TUN   => sig_active_mpls_tun,
      ACTIVE_SCTP       => sig_active_sctp,
      ACTIVE_IPV6_TUN   => sig_active_ipv6_tun,
      ACTIVE_IPV4_TUN   => sig_active_ipv4_tun,
      ACTIVE_IPV6_NO_NEXT=>sig_active_ipv6_no_next,
      PROTOCOL          => sig_protocol_in,
      RUNNING_TRANSPORT => sig_running_transport
   );

   TX_DATA           <= sig_tx_data;
   TX_DREM           <= sig_tx_drem;
   TX_SRC_RDY_N      <= sig_tx_src_rdy_n;
   TX_SOF_N          <= sig_tx_sof_n;
   TX_EOF_N          <= sig_tx_eof_n;
   TX_SOP_N          <= sig_tx_sop_n;
   TX_EOP_N          <= sig_tx_eop_n;

   LINK_OFFSET        <= sig_link_offset;
   ACTIVE_ETHERNET    <= sig_active_ethernet;
   RUNNING_LINK       <= sig_running_link;
   VLAN0_OFFSET       <= sig_vlan0_offset;
   VLAN1_OFFSET       <= sig_vlan1_offset;
   MPLS0_OFFSET       <= sig_mpls0_offset;
   MPLS1_OFFSET       <= sig_mpls1_offset;
   ACTIVE_VLAN0       <= sig_active_vlan0;
   ACTIVE_VLAN1       <= sig_active_vlan1;
   ACTIVE_MPLS0       <= sig_active_mpls0;
   ACTIVE_MPLS1       <= sig_active_mpls1;
   RUNNING_VLAN_MPLS  <= sig_running_vlan_mpls;
   INTERNET_OFFSET    <= sig_internet_offset;
   ACTIVE_IPV4        <= sig_active_ipv4;
   ACTIVE_IPV6        <= sig_active_ipv6;
   RUNNING_INTERNET   <= sig_running_internet;
   ERROR_IPV4         <= sig_error_ipv4;
   ERROR_IPV6         <= sig_error_ipv6;
   TRANSPORT_OFFSET   <= sig_transport_offset;
   ACTIVE_TCP         <= sig_active_tcp;
   ACTIVE_UDP         <= sig_active_udp;
   ACTIVE_ICMP        <= sig_active_icmp;
   ACTIVE_IGMP        <= sig_active_igmp;
   ACTIVE_MPLS_TUN    <= sig_active_mpls_tun;
   ACTIVE_SCTP        <= sig_active_sctp;
   ACTIVE_IPV6_TUN    <= sig_active_ipv6_tun;
   ACTIVE_IPV4_TUN    <= sig_active_ipv4_tun;
   ACTIVE_IPV6_NO_NEXT<= sig_active_ipv6_no_next;
   RUNNING_TRANSPORT  <= sig_running_transport;

   extractor_i : entity work.hfe_m_extract_full
   generic map (
      DATA_WIDTH         => DATA_WIDTH,
      OFFSET_WIDTH       => OFFSET_WIDTH,
      LATENCY            => EXTRACT_LATENCY
   )
   port map (
      CLK         => CLK,
      RESET       => RESET,
      --+ FrameLink input interface (passive port)
      FL_DATA     => sig_tx_data,
      FL_DREM     => sig_tx_drem,
      FL_SRC_RDY_N=> sig_tx_src_rdy_n,
      FL_DST_RDY_N=> TX_DST_RDY_N,
      FL_SOF_N    => sig_tx_sof_n,
      FL_EOF_N    => sig_tx_eof_n,
      FL_SOP_N    => sig_tx_sop_n,
      FL_EOP_N    => sig_tx_eop_n,

      LINK_OFFSET       => sig_link_offset,
      ACTIVE_ETHERNET   => sig_active_ethernet,
      RUNNING_LINK      => sig_running_link,
      -- VLAN and MPLS ---------------------------------------------------
      VLAN0_OFFSET      => sig_vlan0_offset,
      VLAN1_OFFSET      => sig_vlan1_offset,
      MPLS0_OFFSET      => sig_mpls0_offset,
      MPLS1_OFFSET      => sig_mpls1_offset,
      ACTIVE_VLAN0      => sig_active_vlan0,
      ACTIVE_VLAN1      => sig_active_vlan1,
      ACTIVE_MPLS0      => sig_active_mpls0,
      ACTIVE_MPLS1      => sig_active_mpls1,
      RUNNING_VLAN_MPLS => sig_running_vlan_mpls,
      -- Internet layer ------------------------------------------------
      INTERNET_OFFSET   => sig_internet_offset,
      ACTIVE_IPV4       => sig_active_ipv4,
      ACTIVE_IPV6       => sig_active_ipv6,
      RUNNING_INTERNET  => sig_running_internet,
      ERROR_IPV4        => sig_error_ipv4,
      ERROR_IPV6        => sig_error_ipv6,
      -- Transport layer -----------------------------------------------
      TRANSPORT_OFFSET  => sig_transport_offset,
      ACTIVE_TCP        => sig_active_tcp,
      ACTIVE_UDP        => sig_active_udp,
      ACTIVE_ICMP       => sig_active_icmp,
      ACTIVE_IGMP       => sig_active_igmp,
      ACTIVE_MPLS_TUN   => sig_active_mpls_tun,
      ACTIVE_SCTP       => sig_active_sctp,
      ACTIVE_IPV6_TUN   => sig_active_ipv6_tun,
      ACTIVE_IPV4_TUN   => sig_active_ipv4_tun,
      ACTIVE_IPV6_NO_NEXT=>sig_active_ipv6_no_next,
      PROTOCOL_IN       => sig_protocol_in,
      FRAGMENT_FIRST_IN => sig_fragment_first,
      FRAGMENT_LAST_IN  => sig_fragment_last,
      RUNNING_TRANSPORT => sig_running_transport,

      -- Extracted fields
      -- Internet layer -------------------------------------------------
      SRCIPV4           => SRCIPV4,
      SRCIPV4_VLD       => SRCIPV4_VLD,
      DSTIPV4           => DSTIPV4,
      DSTIPV4_VLD       => DSTIPV4_VLD,
      SRCIPV6           => SRCIPV6,
      SRCIPV6_VLD       => SRCIPV6_VLD,
      DSTIPV6           => DSTIPV6,
      DSTIPV6_VLD       => DSTIPV6_VLD,
      PROTOCOL          => PROTOCOL,
      FRAGMENT_FIRST    => FRAGMENT_FIRST,
      FRAGMENT_LAST     => FRAGMENT_LAST,
      PROTOCOL_VLD      => PROTOCOL_VLD,
      -- Transport layer ------------------------------------------------
      SRCTCP            => SRCTCP,
      SRCTCP_VLD        => SRCTCP_VLD,
      DSTTCP            => DSTTCP,
      DSTTCP_VLD        => DSTTCP_VLD,
      SRCUDP            => SRCUDP,
      SRCUDP_VLD        => SRCUDP_VLD,
      DSTUDP            => DSTUDP,
      DSTUDP_VLD        => DSTUDP_VLD,

      PARSING_DONE      => PARSING_DONE
   );
end architecture full;
