-- dma_pipe.vhd: DMA Pipe
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use work.math_pack.all;

-- -----------------------------------------------------------------------------
--                              Entity declaration
-- -----------------------------------------------------------------------------
entity DMA_PIPE is
   generic (
      DATA_WIDTH        : integer := 512;
      HDR_WIDTH         : integer := 32;

      USE_OUTREG        : boolean := false;
      FAKE_PIPE         : boolean := false
   );
   port (
      CLK               : in std_logic;
      RESET             : in std_logic;

      RX_DATA       : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      RX_HDR        : in  std_logic_vector(HDR_WIDTH-1 downto 0);
      RX_SOP        : in  std_logic;
      RX_EOP        : in  std_logic;
      RX_SRC_RDY    : in  std_logic;
      RX_DST_RDY    : out std_logic;

      TX_DATA       : out std_logic_vector(DATA_WIDTH-1 downto 0);
      TX_HDR        : out std_logic_vector(HDR_WIDTH-1 downto 0);
      TX_SOP        : out std_logic;
      TX_EOP        : out std_logic;
      TX_SRC_RDY    : out std_logic;
      TX_DST_RDY    : in  std_logic
   );
end entity DMA_PIPE;

-- -----------------------------------------------------------------------------
--                           Architecture declaration
-- -----------------------------------------------------------------------------
architecture behavioral of DMA_PIPE is

   signal in_data    : std_logic_vector(DATA_WIDTH+HDR_WIDTH+1 downto 0);
   signal out_data   : std_logic_vector(DATA_WIDTH+HDR_WIDTH+1 downto 0);

begin

   pipe_i : entity work.PIPE
   generic map(
      DATA_WIDTH     => DATA_WIDTH + HDR_WIDTH + 2,
      USE_OUTREG     => USE_OUTREG,
      FAKE_PIPE      => FAKE_PIPE
   )
   port map(
      CLK            => CLK,
      RESET          => RESET,
      IN_DATA        => in_data,
      IN_SRC_RDY     => RX_SRC_RDY,
      IN_DST_RDY     => RX_DST_RDY,
      OUT_DATA       => out_data,
      OUT_SRC_RDY    => TX_SRC_RDY,
      OUT_DST_RDY    => TX_DST_RDY
   );

   in_data(DATA_WIDTH-1 downto 0)                     <= RX_DATA;
   in_data(DATA_WIDTH+HDR_WIDTH-1 downto DATA_WIDTH)  <= RX_HDR;
   in_data(DATA_WIDTH+HDR_WIDTH+0)                    <= RX_SOP;
   in_data(DATA_WIDTH+HDR_WIDTH+1)                    <= RX_EOP;

   TX_DATA  <= out_data(DATA_WIDTH-1 downto 0);
   TX_HDR   <= out_data(DATA_WIDTH+HDR_WIDTH-1 downto DATA_WIDTH);
   TX_SOP   <= out_data(DATA_WIDTH+HDR_WIDTH+0);
   TX_EOP   <= out_data(DATA_WIDTH+HDR_WIDTH+1);

end architecture;
