-- bootfpga.vhd : component for work with flash memory on Fiberblaze FB1CG card
--!
--! \file
--! \brief component for work with flash memory on Fiberblaze FB1CG card
--! \author Martin Spinler <spinler@cesnet.cz>
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz> 
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                             Entity declaration
-- ----------------------------------------------------------------------------

entity bootfpga is
port(
   --! \name Clock and reset
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   CLK     : in  std_logic;
   --! \brief Reset signal
   RESET   : in  std_logic;

   --! \name Input MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Data to be written
   MI_DWR  : in  std_logic_vector(31 downto 0);
   --! \brief Address bus
   MI_ADDR : in  std_logic_vector(31 downto 0);
   --! \brief Read request
   MI_RD   : in  std_logic;
   --! \brief Write request
   MI_WR   : in  std_logic;
   --! \brief Byte enable signals      
   MI_BE   : in  std_logic_vector(3 downto 0);
   --! \brief Read data
   MI_DRD  : out std_logic_vector(31 downto 0);
   --! \brief Address ready
   MI_ARDY : out std_logic;
   --! \brief Read data ready
   MI_DRDY : out std_logic;

   --! \name SPI interface to BootFPGA
   -- -------------------------------------------------------------------------
   --! \brief SPI clock
   BF_CLK  : out std_logic;
   --! \brief SPI not slave select
   BF_NSS  : out std_logic;
   --! \brief SPI data from Master to Slave
   BF_MOSI : out std_logic;
   --! \brief SPI data from Slave to Master
   BF_MISO : in  std_logic
);
end entity bootfpga;


-- ----------------------------------------------------------------------------
--                         Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of bootfpga is

   --! Signals declaration
   -- -------------------------------------------------------------------------

   signal boot_wr_data : std_logic_vector(63 downto 0);
   signal boot_wr_str  : std_logic;
   signal boot_rd_data : std_logic_vector(63 downto 0);

   signal mi_wr_str    :  std_logic := '0';
   signal timeout      :  std_logic_vector(25 downto 0) := (others => '0');
   signal timeout_en   :  std_logic := '0';

   -- -------------------------------------------------------------------------
   --                           Architecture body
   -- -------------------------------------------------------------------------

begin

   -- INFO: X"E" is a Reload FPGA Design Command, we send it delayed. (cca 135ms @ 250 MHz)
   -- INFO: Driver must detach device from the bus first. Otherwise is raised Kernel panic

   -- -------------------------------------------------------------------------
   --                          MI32 interface logic
   -- -------------------------------------------------------------------------

   --! ARDY/DRDY logic
   -- -------------------------------------------------------------------------

   MI_ARDY  <= MI_RD or MI_WR;
   MI_DRDY  <= MI_RD;
   boot_wr_str <= mi_wr_str or timeout(25);

   timeoutp: process(CLK)
   begin
      if CLK'event and CLK = '1' then
         if RESET = '1' then
            timeout_en  <= '0';
         elsif MI_WR = '1' and MI_ADDR(2) = '1' and MI_DWR(31 downto 28) = X"E" then
            timeout_en  <= '1';
         end if;

         if timeout_en = '1' and timeout(25) = '0' then
            timeout     <= timeout + 1;
         else
            timeout     <= (others =>'0');
         end if;
      end if;
   end process;

   --! write logic
   -- -------------------------------------------------------------------------

   mi_wr_regs_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         --! default value
         mi_wr_str <= '0';

         if (MI_WR = '1') then
            case MI_ADDR(2) is
               when '0'    => boot_wr_data(31 downto  0) <= MI_DWR;
               when '1'    => boot_wr_data(63 downto 32) <= MI_DWR;
                  if(MI_DWR(31 downto 28) /= X"E") then
                     mi_wr_str   <= '1';
                  end if;
               when others => null;
            end case;
         end if;
      end if;
   end process mi_wr_regs_p;

   --! read logic
   -- -------------------------------------------------------------------------

   mi_rd_p : process(MI_ADDR, boot_rd_data)
   begin
      case MI_ADDR(2) is
         when '0'    => MI_DRD <= boot_rd_data(31 downto  0);
         when '1'    => MI_DRD <= boot_rd_data(63 downto 32);
         when others => null;
      end case;
   end process mi_rd_p;


   -- -------------------------------------------------------------------------
   --                         SPI interface controller
   -- -------------------------------------------------------------------------

   bootfpga_ctrl_i : entity work.BootFPGA_ctrl
   port map (
      --! Clock and reset interface
      PCIe_clk        => CLK,
      reset           => RESET,

      -- SPI interface to BootFPGA
      bf_clk          => BF_CLK,
      bf_nss          => BF_NSS,
      bf_mosi         => BF_MOSI,
      bf_miso         => BF_MISO,

      -- Interface towards PCIe
      setup_wr        => boot_wr_data,
      setup_wr_strobe => boot_wr_str,
      setup_rd        => boot_rd_data
   );

end architecture full;
