-- axi2dma.vhd: AXI2DMA - Architecture declaration
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;


-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------

architecture axi2dma_arch of axi2dma is

   -- Signals declaration -----------------------------------------------------

   ----- interconnection of DMA UP interface and dma_up_int component -----
   -- in and out data signals of pipe between DMA UP interface and dma_up_int
   signal pipe_dma_up_int_data_in     : std_logic_vector(353 downto 0);
   signal pipe_dma_up_int_data_out    : std_logic_vector(353 downto 0);
   -- inputs to dma_up_int component
   signal dma_up_data_in              : std_logic_vector(255 downto 0);
   signal dma_up_hdr_in               : std_logic_vector(95 downto 0);
   signal dma_up_sop_in               : std_logic;
   signal dma_up_eop_in               : std_logic;
   signal dma_up_src_rdy_in           : std_logic;
   signal dma_up_dst_rdy_in           : std_logic;

   ----- interconnection of dma_up_int and rq_int components -----
   -- outputs from dma_up_int component
   signal dma_rq_data_out             : std_logic_vector(255 downto 0);
   signal dma_rq_hdr_dwords_out       : std_logic_vector(10 downto 0);
   signal dma_rq_hdr_type_out         : std_logic;
   signal dma_rq_hdr_addr_out         : std_logic_vector(63 downto 0);
   signal dma_rq_sop_out              : std_logic;
   signal dma_rq_eop_out              : std_logic;
   signal dma_rq_src_rdy_out          : std_logic;
   signal dma_rq_dst_rdy_out          : std_logic;
   signal dma_rq_pcie_tag_out         : std_logic_vector(4 downto 0);
   -- input and output data signals of pipe between dma_up_int and rq_int
   signal pipe_up_data_in             : std_logic_vector(338 downto 0);
   signal pipe_up_data_out            : std_logic_vector(338 downto 0);
   -- inputs to rq_int component
   signal dma_rq_data_in              : std_logic_vector(255 downto 0);
   signal dma_rq_hdr_dwords_in        : std_logic_vector(10 downto 0);
   signal dma_rq_hdr_type_in          : std_logic;
   signal dma_rq_hdr_addr_in          : std_logic_vector(63 downto 0);
   signal dma_rq_sop_in               : std_logic;
   signal dma_rq_eop_in               : std_logic;
   signal dma_rq_src_rdy_in           : std_logic;
   signal dma_rq_dst_rdy_in           : std_logic;
   signal dma_rq_pcie_tag_in          : std_logic_vector(4 downto 0);

   ----- interconnection of rq_int component and RQ interface -----
   -- outputs from rq_int component
   signal rq_data_out                 : std_logic_vector(255 downto 0);
   signal rq_user_out                 : std_logic_vector(59 downto 0);
   signal rq_last_out                 : std_logic;
   signal rq_keep_out                 : std_logic_vector(7 downto 0);
   signal rq_ready_out                : std_logic;
   signal rq_valid_out                : std_logic;
   -- input and output data signals of pipe between rq_int and RQ interface
   signal pipe_rq_int_data_in         : std_logic_vector(324 downto 0);
   signal pipe_rq_int_data_out        : std_logic_vector(324 downto 0);

   ----- interconnection of RC interface and rc_int component -----
   -- input and output data signals of pipe between rq_int and RQ interface
   signal pipe_rc_int_data_in         : std_logic_vector(265 downto 0);
   signal pipe_rc_int_data_out        : std_logic_vector(265 downto 0);
   -- inputs to rc_int component
   signal rc_data_in                  : std_logic_vector(255 downto 0);
   signal rc_user_sof_in              : std_logic;
   signal rc_last_in                  : std_logic;
   signal rc_keep_in                  : std_logic_vector(7 downto 0);
   signal rc_ready_in                 : std_logic;
   signal rc_valid_in                 : std_logic;

   ----- interconnection of rc_int and dma_down_int components -----
   -- outputs from rc_int component
   signal rc_dma_data_out             : std_logic_vector(255 downto 0);
   signal rc_dma_hdr_dwords_out       : std_logic_vector(10 downto 0);
   signal rc_dma_sop_out              : std_logic;
   signal rc_dma_eop_out              : std_logic;
   signal rc_dma_src_rdy_out          : std_logic;
   signal rc_dma_dst_rdy_out          : std_logic;
   signal rc_dma_pcie_tag_out         : std_logic_vector(4 downto 0);
   signal rc_dma_pcie_tag_release_out : std_logic;
   -- input and output data signals of pipe between rc_int and dma_down_int
   signal pipe_down_data_in           : std_logic_vector(274 downto 0);
   signal pipe_down_data_out          : std_logic_vector(274 downto 0);
   -- inputs to dma_down_int component
   signal rc_dma_data_in              : std_logic_vector(255 downto 0);
   signal rc_dma_hdr_dwords_in        : std_logic_vector(10 downto 0);
   signal rc_dma_sop_in               : std_logic;
   signal rc_dma_eop_in               : std_logic;
   signal rc_dma_src_rdy_in           : std_logic;
   signal rc_dma_dst_rdy_in           : std_logic;
   signal rc_dma_pcie_tag_in          : std_logic_vector(4 downto 0);
   signal rc_dma_pcie_tag_release_in  : std_logic;

   ----- interconnection of dma_down_int component and DMA DOWN interface -----
   -- outputs from dma_down_int component
   signal dma_down_data_out           : std_logic_vector(255 downto 0);
   signal dma_down_hdr_out            : std_logic_vector(31 downto 0);
   signal dma_down_sop_out            : std_logic;
   signal dma_down_eop_out            : std_logic;
   signal dma_down_src_rdy_out        : std_logic;
   signal dma_down_dst_rdy_out        : std_logic;
   -- in and out data signals of pipe between dma_down_int and DMA DOWN
   signal pipe_dma_down_int_data_in   : std_logic_vector(289 downto 0);
   signal pipe_dma_down_int_data_out  : std_logic_vector(289 downto 0);

   ----- connection of tag_manager component -----
   -- interconnection of dma_up_int and tag_manager components
   signal tag_offer                   : std_logic_vector(4 downto 0);
   signal tag_offer_vld               : std_logic;
   signal tag_offer_accept            : std_logic;
   signal dma_up_hdr_tag              : std_logic_vector(7 downto 0);
   signal dma_up_hdr_id               : std_logic_vector(7 downto 0);
   -- interconnection of dma_down_int and tag_manager components
   signal tag                         : std_logic_vector(4 downto 0);
   signal tag_release                 : std_logic;
   signal dma_down_hdr_tag            : std_logic_vector(7 downto 0);
   signal dma_down_hdr_id             : std_logic_vector(7 downto 0);

begin


   -- -------------------------------------------------------------------------
   --                      DMA UP interface connection
   -- -------------------------------------------------------------------------

   -- input data composition
   pipe_dma_up_int_data_in <=
      DMA_UP_DATA &
      DMA_UP_HDR  &
      DMA_UP_SOP  &
      DMA_UP_EOP;

   -- pipe instantiation
   pipe_dma_up_int_i : entity work.pipe
   generic map(
      DATA_WIDTH => 354,
      USE_OUTREG => DMA_PIPE_USE_OUTREG,
      FAKE_PIPE  => not DMA_PIPE
   )
   port map(
      -- Common signals
      CLK         => CLK,
      RESET       => RESET,

      -- Input interface
      IN_DATA     => pipe_dma_up_int_data_in,
      IN_SRC_RDY  => DMA_UP_SRC_RDY,
      IN_DST_RDY  => DMA_UP_DST_RDY,

      -- Output interface
      OUT_DATA    => pipe_dma_up_int_data_out,
      OUT_SRC_RDY => dma_up_src_rdy_in,
      OUT_DST_RDY => dma_up_dst_rdy_in
   );

   -- output data decomposition
   dma_up_data_in <= pipe_dma_up_int_data_out(353 downto 98);
   dma_up_hdr_in  <= pipe_dma_up_int_data_out(97 downto 2);
   dma_up_sop_in  <= pipe_dma_up_int_data_out(1);
   dma_up_eop_in  <= pipe_dma_up_int_data_out(0);


   -- -------------------------------------------------------------------------
   --                           DMA_UP_INT
   -- -------------------------------------------------------------------------

   dma_up_int_i: entity work.dma_up_int
   port map(
      -- Common signals
      CLK               => CLK,
      RESET             => RESET,

      -- DMA UP Interface (from user application to host system)
      DMA_UP_DATA       => dma_up_data_in,
      DMA_UP_HDR_DWORDS => dma_up_hdr_in(10 downto 0),
      DMA_UP_HDR_TYPE   => dma_up_hdr_in(15 downto 11),
      DMA_UP_HDR_TAG    => dma_up_hdr_in(23 downto 16),
      DMA_UP_HDR_ID     => dma_up_hdr_in(31 downto 24),
      DMA_UP_HDR_ADDR   => dma_up_hdr_in(95 downto 32),
      DMA_UP_SOP        => dma_up_sop_in,
      DMA_UP_EOP        => dma_up_eop_in,
      DMA_UP_SRC_RDY    => dma_up_src_rdy_in,
      DMA_UP_DST_RDY    => dma_up_dst_rdy_in,

      -- DMA-like interface to rq_int component
      DMA_DATA          => dma_rq_data_out,
      DMA_HDR_DWORDS    => dma_rq_hdr_dwords_out,
      DMA_HDR_TYPE      => dma_rq_hdr_type_out,
      DMA_HDR_ADDR      => dma_rq_hdr_addr_out,
      DMA_SOP           => dma_rq_sop_out,
      DMA_EOP           => dma_rq_eop_out,
      DMA_SRC_RDY       => dma_rq_src_rdy_out,
      DMA_DST_RDY       => dma_rq_dst_rdy_out,
      PCIE_TAG          => dma_rq_pcie_tag_out,
      
      -- Interface to tag_manager component
      TAG_OFFER         => tag_offer,
      TAG_OFFER_VLD     => tag_offer_vld,
      TAG_OFFER_ACCEPT  => tag_offer_accept,
      DMA_HDR_TAG       => dma_up_hdr_tag,
      DMA_HDR_ID        => dma_up_hdr_id
   );


   -- -------------------------------------------------------------------------
   --           Interconnection of DMA_UP_INT and RQ_INT components
   -- -------------------------------------------------------------------------

   -- input data composition
   pipe_up_data_in <=
      dma_rq_data_out       &
      dma_rq_hdr_dwords_out &
      dma_rq_hdr_type_out   &
      dma_rq_hdr_addr_out   &
      dma_rq_sop_out        &
      dma_rq_eop_out        &
      dma_rq_pcie_tag_out;

   -- pipe instantiation
   pipe_up_i : entity work.pipe
   generic map(
      DATA_WIDTH => 339,
      USE_OUTREG => INTERNAL_PIPE_USE_OUTREG,
      FAKE_PIPE  => not INTERNAL_PIPE
   )
   port map(
      -- Common signals
      CLK         => CLK,
      RESET       => RESET,

      -- Input interface
      IN_DATA     => pipe_up_data_in,
      IN_SRC_RDY  => dma_rq_src_rdy_out,
      IN_DST_RDY  => dma_rq_dst_rdy_out,

      -- Output interface
      OUT_DATA    => pipe_up_data_out,
      OUT_SRC_RDY => dma_rq_src_rdy_in,
      OUT_DST_RDY => dma_rq_dst_rdy_in
   );

   -- output data decomposition
   dma_rq_data_in       <= pipe_up_data_out(338 downto 83);
   dma_rq_hdr_dwords_in <= pipe_up_data_out(82 downto 72);
   dma_rq_hdr_type_in   <= pipe_up_data_out(71);
   dma_rq_hdr_addr_in   <= pipe_up_data_out(70 downto 7);
   dma_rq_sop_in        <= pipe_up_data_out(6);
   dma_rq_eop_in        <= pipe_up_data_out(5);
   dma_rq_pcie_tag_in   <= pipe_up_data_out(4 downto 0);


   -- -------------------------------------------------------------------------
   --                             RQ_INT
   -- -------------------------------------------------------------------------

   rq_int_i: entity work.rq_int
   port map(
      -- Common signals 
      CLK              => CLK,
      RESET            => RESET,

      -- Requester Request Interface (RQ) - reduced version
      RQ_DATA          => rq_data_out,
      RQ_USER          => rq_user_out,
      RQ_LAST          => rq_last_out,
      RQ_KEEP          => rq_keep_out,
      RQ_READY         => rq_ready_out,
      RQ_VALID         => rq_valid_out,

      -- DMA-like interface from dma_up_int component
      DMA_DATA         => dma_rq_data_in,
      DMA_HDR_DWORDS   => dma_rq_hdr_dwords_in,
      DMA_HDR_TYPE     => dma_rq_hdr_type_in,
      DMA_HDR_ADDR     => dma_rq_hdr_addr_in,
      DMA_SOP          => dma_rq_sop_in,
      DMA_EOP          => dma_rq_eop_in,
      DMA_SRC_RDY      => dma_rq_src_rdy_in,
      DMA_DST_RDY      => dma_rq_dst_rdy_in,
      PCIE_TAG         => dma_rq_pcie_tag_in
   );


   -- -------------------------------------------------------------------------
   --                         RQ interface connection
   -- -------------------------------------------------------------------------

   -- input data composition
   pipe_rq_int_data_in <=
      rq_data_out  &
      rq_user_out  &
      rq_last_out  &
      rq_keep_out;

   -- pipe instantiation
   pipe_rq_int_i : entity work.pipe
   generic map(
      DATA_WIDTH => 325,
      USE_OUTREG => AXI_PIPE_USE_OUTREG,
      FAKE_PIPE  => not AXI_PIPE
   )
   port map(
      -- Common signals
      CLK         => CLK,
      RESET       => RESET,

      -- Input interface
      IN_DATA     => pipe_rq_int_data_in,
      IN_SRC_RDY  => rq_valid_out,
      IN_DST_RDY  => rq_ready_out,

      -- Output interface
      OUT_DATA    => pipe_rq_int_data_out,
      OUT_SRC_RDY => RQ_VALID,
      OUT_DST_RDY => RQ_READY
   );

   -- output data decomposition
   RQ_DATA <= pipe_rq_int_data_out(324 downto 69);
   RQ_USER <= pipe_rq_int_data_out(68 downto 9);
   RQ_LAST <= pipe_rq_int_data_out(8);
   RQ_KEEP <= pipe_rq_int_data_out(7 downto 0);


   -- -------------------------------------------------------------------------
   --                         RC interface connection
   -- -------------------------------------------------------------------------

   -- input data composition
   pipe_rc_int_data_in <=
      RC_DATA     &
      RC_USER(32) &
      RC_LAST     &
      RC_KEEP;

   -- pipe instantiation
   pipe_rc_int_i : entity work.pipe
   generic map(
      DATA_WIDTH => 266,
      USE_OUTREG => AXI_PIPE_USE_OUTREG,
      FAKE_PIPE  => not AXI_PIPE
   )
   port map(
      -- Common signals
      CLK         => CLK,
      RESET       => RESET,

      -- Input interface
      IN_DATA     => pipe_rc_int_data_in,
      IN_SRC_RDY  => RC_VALID,
      IN_DST_RDY  => RC_READY,

      -- Output interface
      OUT_DATA    => pipe_rc_int_data_out,
      OUT_SRC_RDY => rc_valid_in,
      OUT_DST_RDY => rc_ready_in
   );

   -- output data decomposition
   rc_data_in     <= pipe_rc_int_data_out(265 downto 10);
   rc_user_sof_in <= pipe_rc_int_data_out(9);
   rc_last_in     <= pipe_rc_int_data_out(8);
   rc_keep_in     <= pipe_rc_int_data_out(7 downto 0);


   -- -------------------------------------------------------------------------
   --                             RC_INT
   -- -------------------------------------------------------------------------

   rc_int_i: entity work.rc_int
   port map(
      -- Common signals
      CLK              => CLK,
      RESET            => RESET,
      
      -- Requester Completion Interface (RC) - reduced version
      RC_DATA          => rc_data_in,
      RC_USER_SOF      => rc_user_sof_in,
      RC_LAST          => rc_last_in,
      RC_KEEP          => rc_keep_in,
      RC_VALID         => rc_valid_in,
      RC_READY         => rc_ready_in,

      -- DMA-like interface to dma_down_int component
      DMA_DATA         => rc_dma_data_out,
      DMA_HDR_DWORDS   => rc_dma_hdr_dwords_out,
      DMA_SOP          => rc_dma_sop_out,
      DMA_EOP          => rc_dma_eop_out,
      DMA_SRC_RDY      => rc_dma_src_rdy_out,
      DMA_DST_RDY      => rc_dma_dst_rdy_out,
      PCIE_TAG         => rc_dma_pcie_tag_out,
      PCIE_TAG_RELEASE => rc_dma_pcie_tag_release_out
   );


   -- -------------------------------------------------------------------------
   --          Interconnection of RC_INT and DMA_DOWN_INT components
   -- -------------------------------------------------------------------------

   -- input data composition
   pipe_down_data_in <=
      rc_dma_data_out              &
      rc_dma_hdr_dwords_out        &
      rc_dma_sop_out               &
      rc_dma_eop_out               &
      rc_dma_pcie_tag_out          &
      rc_dma_pcie_tag_release_out;

   -- pipe instantiation
   pipe_down_i : entity work.pipe
   generic map(
      DATA_WIDTH => 275,
      USE_OUTREG => INTERNAL_PIPE_USE_OUTREG,
      FAKE_PIPE  => not INTERNAL_PIPE
   )
   port map(
      -- Common signals
      CLK         => CLK,
      RESET       => RESET,

      -- Input interface
      IN_DATA     => pipe_down_data_in,
      IN_SRC_RDY  => rc_dma_src_rdy_out,
      IN_DST_RDY  => rc_dma_dst_rdy_out,

      -- Output interface
      OUT_DATA    => pipe_down_data_out,
      OUT_SRC_RDY => rc_dma_src_rdy_in,
      OUT_DST_RDY => rc_dma_dst_rdy_in
   );

   -- output data decomposition
   rc_dma_data_in             <= pipe_down_data_out(274 downto 19);
   rc_dma_hdr_dwords_in       <= pipe_down_data_out(18 downto 8);
   rc_dma_sop_in              <= pipe_down_data_out(7);
   rc_dma_eop_in              <= pipe_down_data_out(6);
   rc_dma_pcie_tag_in         <= pipe_down_data_out(5 downto 1);
   rc_dma_pcie_tag_release_in <= pipe_down_data_out(0);


   -- -------------------------------------------------------------------------
   --                          DMA_DOWN_INT
   -- -------------------------------------------------------------------------

   dma_down_int_i: entity work.dma_down_int
   port map(
      -- Common signals
      CLK              => CLK,
      RESET            => RESET,

      -- DMA DOWN INTERFace (from host system to user application)
      DMA_DOWN_DATA    => dma_down_data_out,
      DMA_DOWN_HDR     => dma_down_hdr_out,
      DMA_DOWN_SOP     => dma_down_sop_out,
      DMA_DOWN_EOP     => dma_down_eop_out,
      DMA_DOWN_SRC_RDY => dma_down_src_rdy_out,
      DMA_DOWN_DST_RDY => dma_down_dst_rdy_out,

      -- DMA-like interface from rc_int component
      DMA_DATA         => rc_dma_data_in,
      DMA_HDR_DWORDS   => rc_dma_hdr_dwords_in,
      DMA_SOP          => rc_dma_sop_in,
      DMA_EOP          => rc_dma_eop_in,
      DMA_SRC_RDY      => rc_dma_src_rdy_in,
      DMA_DST_RDY      => rc_dma_dst_rdy_in,
      PCIE_TAG         => rc_dma_pcie_tag_in,
      PCIE_TAG_RELEASE => rc_dma_pcie_tag_release_in,
      
      -- Interface to tag_manager component
      TAG              => tag,
      TAG_RELEASE      => tag_release,
      DMA_DOWN_HDR_TAG => dma_down_hdr_tag,
      DMA_DOWN_HDR_ID  => dma_down_hdr_id
   );


   -- -------------------------------------------------------------------------
   --                     DMA DOWN interface connection
   -- -------------------------------------------------------------------------

   -- input data composition
   pipe_dma_down_int_data_in <=
      dma_down_data_out &
      dma_down_hdr_out  &
      dma_down_sop_out  &
      dma_down_eop_out;

   -- pipe instantiation
   pipe_dma_down_int_i : entity work.pipe
   generic map(
      DATA_WIDTH => 290,
      USE_OUTREG => DMA_PIPE_USE_OUTREG,
      FAKE_PIPE  => not DMA_PIPE
   )
   port map(
      -- Common signals
      CLK         => CLK,
      RESET       => RESET,

      -- Input interface
      IN_DATA     => pipe_dma_down_int_data_in,
      IN_SRC_RDY  => dma_down_src_rdy_out,
      IN_DST_RDY  => dma_down_dst_rdy_out,

      -- Output interface
      OUT_DATA    => pipe_dma_down_int_data_out,
      OUT_SRC_RDY => DMA_DOWN_SRC_RDY,
      OUT_DST_RDY => DMA_DOWN_DST_RDY
   );

   -- output data decomposition
   DMA_DOWN_DATA <= pipe_dma_down_int_data_out(289 downto 34);
   DMA_DOWN_HDR  <= pipe_dma_down_int_data_out(33 downto 2);
   DMA_DOWN_SOP  <= pipe_dma_down_int_data_out(1);
   DMA_DOWN_EOP  <= pipe_dma_down_int_data_out(0);


   -- -------------------------------------------------------------------------
   --                          TAG_MANAGER
   -- -------------------------------------------------------------------------

   tag_manager_i: entity work.tag_manager
   port map(
      -- Common signals
      CLK              => CLK,
      RESET            => RESET,

      -- Interface to dma_up_int component
      TAG_OFFER        => tag_offer,
      TAG_OFFER_VLD    => tag_offer_vld,
      TAG_OFFER_ACCEPT => tag_offer_accept,
      DMA_UP_HDR_TAG   => dma_up_hdr_tag,
      DMA_UP_HDR_ID    => dma_up_hdr_id,

      -- Interface to dma_down_int component
      TAG              => tag,
      TAG_RELEASE      => tag_release,
      DMA_DOWN_HDR_TAG => dma_down_hdr_tag,
      DMA_DOWN_HDR_ID  => dma_down_hdr_id
   );

end architecture axi2dma_arch;
