-- asfifo_dma_bus_ent.vhd : Entity of asynchronous FIFO for DMA bus
--!
--! \file
--! \brief Entity of asynchronous FIFO for DMA bus
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;

entity asfifo_dma_bus is
generic (
   --! \brief Width of DMA data
   DATA_WIDTH     : integer := 512;
   --! \brief Width of DMA header
   HDR_WIDTH      : integer := 96;
   --! \brief There are some requirement for timing of RESET and RDEN to ensure
   SAFE_RESET     : boolean := false
);
port (
   --! \name Write interface
   -- -------------------------------------------------------------------------
   --! \brief Write clock
   WR_CLK         : in  std_logic;
   --! \brief Write reset
   WR_RESET       : in  std_logic;
   --! \brief DMA transaction data
   WR_DMA_DATA    : in  std_logic_vector(DATA_WIDTH-1 downto 0);
   --! \brief DMA transaction header
   --! \details Valid when WR_DMA_SOP is valid.
   WR_DMA_HDR     : in  std_logic_vector(HDR_WIDTH-1 downto 0);
   --! \brief Start of DMA transaction
   --! \details Valid when WR_DMA_SRC_RDY = WR_DMA_DST_RDY = '1'.
   WR_DMA_SOP     : in  std_logic;
   --! \brief End of DMA transaction
   --! \details Valid when WR_DMA_SRC_RDY = WR_DMA_DST_RDY = '1'.
   WR_DMA_EOP     : in  std_logic;
   --! \brief Source is ready to transmit DMA data
   WR_DMA_SRC_RDY : in  std_logic;
   --! \brief Destination is ready to receive DMA data
   WR_DMA_DST_RDY : out std_logic;

   --! \name Read interface
   -- -------------------------------------------------------------------------
   --! \brief Read clock
   RD_CLK         : in  std_logic;
   --! \brief Read reset
   RD_RESET       : in  std_logic;
   --! \brief DMA transaction data
   RD_DMA_DATA    : out std_logic_vector(DATA_WIDTH-1 downto 0);
   --! \brief DMA transaction header
   --! \details Valid when RD_DMA_SOP is valid.
   RD_DMA_HDR     : out std_logic_vector(HDR_WIDTH-1 downto 0);
   --! \brief Start of DMA transaction
   --! \details Valid when RD_DMA_SRC_RDY = RD_DMA_DST_RDY = '1'.
   RD_DMA_SOP     : out std_logic;
   --! \brief End of DMA transaction
   --! \details Valid when RD_DMA_SRC_RDY = RD_DMA_DST_RDY = '1'.
   RD_DMA_EOP     : out std_logic;
   --! \brief Source is ready to transmit DMA data
   RD_DMA_SRC_RDY : out std_logic;
   --! \brief Destination is ready to receive DMA data
   RD_DMA_DST_RDY : in  std_logic
);
end entity asfifo_dma_bus;
