-- tree_final_stage_fixed.vhd: The last stage (level) of binary search tree fixed for atomic configuration changes 
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree_final_stage_fixed.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture arch of tree_final_stage_fixed is
  signal tree_data  : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal tree_len   : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal tree_found : std_logic;
  signal tree_foundable : std_logic;
  signal tree_vld   : std_logic;
  signal tree_out_data  : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal tree_out_len   : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal tree_out_found : std_logic;
  signal tree_out_vld   : std_logic;
  
  signal fix_key_start_cmp    : std_logic;
  signal fix_key_end_cmp      : std_logic;
  signal reg_fix_key_start_cmp    : std_logic;
  signal reg_fix_key_end_cmp      : std_logic;
  signal reg_fix_active           : std_logic;
  signal reg_fix_data             : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal reg_fix_datalen          : std_logic_vector(log2(KEY_WIDTH)-1 downto 0); 
  signal reg_fix_len              : std_logic_vector(log2(KEY_WIDTH)-1 downto 0); 
  signal reg_fix_found            : std_logic;
  signal reg_fix_is_add           : std_logic;
  signal fix_key_cmp              : std_logic;
  
  signal fixed_data   : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal fixed_len    : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal fixed_found  : std_logic;
  signal fixed_vld    : std_logic;
  signal use_add_fix: std_logic;
  signal use_rem_fix: std_logic;
  signal use_fix    : std_logic;
begin
  -- Basic final stage of tree with synchronization
  basic_final_stage : entity work.tree_final_stage
    generic map (
      TREE_LEVEL     => TREE_STAGES,
      KEY_WIDTH      => KEY_WIDTH,
      DATA_WIDTH     => DATA_WIDTH,
      MEM_TYPE       => MEM_TYPE,
      INPUT_REG      => false,
      OUTPUT_REG     => false
    ) port map (
      CLK           => CLK,
      RESET         => RESET,
      IN_ADDR       => IN_ADDR,
      IN_VLD        => IN_VLD,
      OUT_DATA      => tree_out_data,
      OUT_LENGTH    => tree_out_len,
      OUT_FOUND     => tree_out_found,
      OUT_VLD       => tree_out_vld,
      CFG_DI        => CFG_DI,
      CFG_DO        => CFG_DO,
      CFG_DRDY      => CFG_DRDY,
      CFG_ADDR      => CFG_ADDR,
      CFG_WE        => CFG_WE,
      CFG_EN        => CFG_EN
    );
  noreg_tree_sync_gen : if MEM_TYPE=mem_bmem generate
    tree_data  <= tree_out_data;
    tree_len   <= tree_out_len;
    tree_found <= tree_out_found;
    tree_vld   <= tree_out_vld;
  end generate;
  reg_tree_sync_gen : if not(MEM_TYPE=mem_bmem) generate
    process(CLK)
    begin
      if CLK'event and CLK='1' then
        tree_data  <= tree_out_data;
        tree_len   <= tree_out_len;
        tree_found <= tree_out_found;
      end if;
    end process;
    process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          tree_vld   <= '0';
        else
          tree_vld   <= tree_out_vld;
        end if;
      end if;
    end process;
  end generate;
  
  -- Adding and removing fix comparators with synchronization
  fix_key_start_cmp <= '1' when IN_KEY >= FIX_KEY_START else '0';
  fix_key_end_cmp   <= '1' when IN_KEY <= FIX_KEY_END   else '0';
  FIX_ACTIVATE      <= IN_CHANGE_EN;
  process(CLK)
  begin
    if CLK'event and CLK='1' then
      reg_fix_key_start_cmp <= fix_key_start_cmp;
      reg_fix_key_end_cmp   <= fix_key_end_cmp;
      reg_fix_active        <= FIX_ACTIVE;
      reg_fix_data          <= FIX_DATA;
      reg_fix_datalen       <= FIX_DATALEN;
      reg_fix_len           <= FIX_LEN;
      reg_fix_found         <= FIX_FOUND;
      reg_fix_is_add        <= FIX_IS_ADD;
      tree_foundable        <= IN_FOUNDABLE;
    end if;
  end process;
  fix_key_cmp <= reg_fix_key_start_cmp and reg_fix_key_end_cmp and reg_fix_active;
  
  -- Result fixing
  use_add_fix <= '1' when fix_key_cmp='1' and reg_fix_len>tree_len and reg_fix_is_add='1' else '0';
  use_rem_fix <= '1' when fix_key_cmp='1' and reg_fix_len=tree_len and reg_fix_is_add='0' else '0';
  use_fix     <= use_add_fix or use_rem_fix;
  fixed_data  <= tree_data  when use_fix='0' else reg_fix_data;
  fixed_len   <= tree_len   when use_fix='0' else reg_fix_datalen;
  fixed_found <= tree_found when use_fix='0' else reg_fix_found;
  fixed_vld   <= tree_vld;
  
  -- Output interface registering
  out_reg_gen : if OUTPUT_REG generate
    process(CLK)
    begin
      if CLK'event and CLK='1' then
        OUT_DATA   <= fixed_data;
        OUT_LENGTH <= fixed_len;
        OUT_FOUND  <= fixed_found and tree_foundable;
      end if;
    end process;    
    process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          OUT_VLD <= '0';
        else
          OUT_VLD <= fixed_vld;
        end if;
      end if;
    end process;
  end generate;
  out_noreg_gen : if not OUTPUT_REG generate
    OUT_DATA   <= fixed_data;
    OUT_LENGTH <= fixed_len;
    OUT_FOUND  <= fixed_found and tree_foundable;
    OUT_VLD    <= fixed_vld;
  end generate;
end architecture;
