-- getbyte.vhd: HFE-M byte extractor
-- Copyright (C) 2011 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Entity declaration
-- ----------------------------------------------------------------------------

entity hfe_m_getbyte is
   generic (
      --* FrameLink width
      DATA_WIDTH  : integer;
      --* Width of WORD_COUNTER
      COUNTER_WIDTH:integer;
      --* Offset of extracted byte within header (Bytes from header start)
      BYTE_OFFSET : integer;
      --* Module latency, 0 or 1 cycles, set to 1 for higher frequency,
      --* but take care not to miss the byte you want in the FL stream!
      LATENCY     : integer := 0
   );
	port (
      CLK         : in std_logic;
      RESET       : in std_logic;

      --+ FrameLink interface (passive port)
      FL_DATA     : in std_logic_vector(DATA_WIDTH-1 downto 0);
      FL_DREM     : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      FL_SRC_RDY_N: in std_logic;
      FL_DST_RDY_N: in std_logic;
      FL_SOF_N    : in std_logic;
      FL_EOF_N    : in std_logic;
      FL_SOP_N    : in std_logic;
      FL_EOP_N    : in std_logic;

      --* Counter of FL words (incremented by 1 with every word)
      WORD_COUNTER: in std_logic_vector(COUNTER_WIDTH-1 downto 0);

      --* Offset of the header start from SOF (in Bytes)
      HEADER_OFFSET:in std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --* Enables the module (mainly the HEADER_OFFSET input)
      ACTIVE      : in std_logic;

      --* BYTE_OUT is valid
      VALID       : out std_logic;
      --* Extracted byte
      BYTE_OUT    : out std_logic_vector(7 downto 0)
   );
end entity;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------

architecture behavioral of hfe_m_getbyte is

   signal full_byte_offset : std_logic_vector
      (COUNTER_WIDTH+log2(DATA_WIDTH/8)-1 downto 0);
   signal word_offset      : std_logic_vector(COUNTER_WIDTH-1 downto 0);
   signal byte_inword_offset:std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);

   signal word_match       : std_logic;
   signal fl_active        : std_logic;

   signal data_mx1         : std_logic_vector(7 downto 0);
   signal reg_data         : std_logic_vector(7 downto 0);
   signal reg_found        : std_logic;
   signal data_mx2         : std_logic_vector(7 downto 0);

begin

   -- Offset of the desired byte from SOF
   full_byte_offset <= HEADER_OFFSET +
      conv_std_logic_vector(BYTE_OFFSET, COUNTER_WIDTH+log2(DATA_WIDTH/8));

   -- Offset of the desired word from SOF
   word_offset <= full_byte_offset
      (COUNTER_WIDTH+log2(DATA_WIDTH/8)-1 downto log2(DATA_WIDTH/8));

   -- Offset of the desired byte in the desired word
   byte_inword_offset <= full_byte_offset(log2(DATA_WIDTH/8)-1 downto 0);

   -- Compare counters
   word_match <= '1' when WORD_COUNTER = word_offset else '0';

   -- Check FL state
   fl_active <= '1' when FL_SRC_RDY_N = '0' and  FL_DST_RDY_N = '0' else '0';

   --* Behavioral MUX selecting byte from FrameLink
   data_mx1_p : process(FL_DATA, byte_inword_offset)
   begin
      data_mx1 <= FL_DATA(7 downto 0);
      for i in 0 to DATA_WIDTH/8-1 loop
         if conv_std_logic_vector(i, log2(DATA_WIDTH/8))
            = byte_inword_offset then
            data_mx1 <= FL_DATA((i+1)*8-1 downto i*8);
         end if;
      end loop;
   end process;

   --* Store extracted byte
   reg_data_p : process(CLK)
   begin
      if CLK'event and CLK = '1' then
         if word_match = '1' and fl_active = '1' then
            reg_data <= data_mx1;
         end if;
      end if;
   end process;

   --* Remember successful extraction
   reg_found_p : process(CLK)
   begin
      if CLK'event and CLK = '1' then
         if RESET = '1' then
            reg_found <= '0';
         else
            if fl_active = '1' then
               if (word_match = '1' and ACTIVE = '1') then
                  reg_found <= '1';
               end if;
-- !!! BUG HERE: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
-- !!!           active data in first word are ignored                             !!!
-- !!!           possible solution is to change this into one if with elsif branch !!!
-- !!!           change can possibly leads to some back compatibility issues       !!!
               if FL_SOF_N = '0' then
                  reg_found <= '0';
               end if;
            end if;
         end if;
      end if;
   end process;

   --* Behavioral MUX selecting direct or stored data
   data_mx2_p : process(data_mx1, reg_data, reg_found)
   begin
      if reg_found = '0' then
         data_mx2 <= data_mx1;
      else
         data_mx2 <= reg_data;
      end if;
   end process;

   nolatency_gen : if LATENCY = 0 generate
      BYTE_OUT <= data_mx2;
      VALID    <= reg_found or (fl_active and word_match);
   end generate;

   latency_gen : if LATENCY = 1 generate
      BYTE_OUT <= reg_data;
      VALID    <= reg_found;
   end generate;

end architecture behavioral;
