-- fl_hins.vhd: Architecture of FL header inserter into FL data strem
--              (header size can be at most one data word)
-- Copyright (c) 2014 Brno University of Technology 
-- Author: Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: header_insert.vhd 2542 2012-10-06 09:21:59Z xkekel00 $
--
-- TODO:
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                       Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of fl_hins is

   --! Types declaration
   -- --------------------------------------------------------------------------

   type fsm_states is (st_header, st_payload);

   --! Signals declaration
   -- --------------------------------------------------------------------------

   --! FSM state signals
   signal curr_state : fsm_states := st_header;
   signal next_state : fsm_states;

   --! Header interface buses extended to full data width
   signal rx_hdr_data_full : std_logic_vector(DATA_WIDTH-1 downto 0);
   signal rx_hdr_rem_full  : std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);


-- ----------------------------------------------------------------------------
--                            Architecture body
-- ----------------------------------------------------------------------------

begin

   --! FL header interface extension to full data width (if necessary)
   -- -------------------------------------------------------------------------

   --! Extension is necessary
   hdr_extension_gen : if (HDR_WIDTH < DATA_WIDTH) generate
   begin
      rx_hdr_data_full <= (DATA_WIDTH-1 downto HDR_WIDTH => '0') & RX_HDR_DATA;
      rx_hdr_rem_full  <= (log2(DATA_WIDTH/8)-1 downto log2(HDR_WIDTH/8) => '0') & RX_HDR_REM;
   end generate hdr_extension_gen;

   --! Extension is not necessary
   no_hdr_extension_gen : if (HDR_WIDTH = DATA_WIDTH) generate
   begin
      rx_hdr_data_full <= RX_HDR_DATA;
      rx_hdr_rem_full  <= RX_HDR_REM;
   end generate no_hdr_extension_gen;

   --! Current state logic
   -- -------------------------------------------------------------------------

   current_state_logic_p : process(CLK)
   begin
      if rising_edge(CLK) then
         if (RESET = '1') then
            curr_state <= st_header;
         else
            curr_state <= next_state;
         end if;
      end if;
   end process current_state_logic_p;

   --! Next state logic
   -- -------------------------------------------------------------------------

   next_state_logic_p : process(curr_state, RX_HDR_SOF_N, RX_HDR_SRC_RDY_N,
                                RX_PAYLOAD_SOF_N, RX_PAYLOAD_EOF_N,
                                RX_PAYLOAD_SRC_RDY_N, TX_DST_RDY_N)
   begin
      case (curr_state) is

         --! st_header
         when st_header =>
            if (RX_HDR_SOF_N = '0' AND RX_HDR_SRC_RDY_N = '0' AND
                RX_PAYLOAD_SOF_N = '0' AND RX_PAYLOAD_SRC_RDY_N = '0' AND
                TX_DST_RDY_N = '0') then
               next_state <= st_payload;
            else
               next_state <= st_header;
            end if;

         --! st_payload
         when st_payload =>
            if (RX_PAYLOAD_EOF_N = '0' AND RX_PAYLOAD_SRC_RDY_N = '0' AND
                TX_DST_RDY_N = '0') then
               next_state <= st_header;
            else
               next_state <= st_payload;
            end if;

         --! when others
         when others =>
            next_state <= st_header;

      end case;
   end process next_state_logic_p;

   --! Output logic
   -- -------------------------------------------------------------------------

   output_logic_p : process(curr_state,
                            rx_hdr_data_full, RX_HDR_SOF_N, RX_HDR_SOP_N,
                            RX_HDR_EOP_N, rx_hdr_rem_full, RX_HDR_SRC_RDY_N, 
                            RX_PAYLOAD_DATA, RX_PAYLOAD_SOF_N,
                            RX_PAYLOAD_SOP_N, RX_PAYLOAD_EOF_N,
                            RX_PAYLOAD_EOP_N, RX_PAYLOAD_REM,
                            RX_PAYLOAD_SRC_RDY_N,
                            TX_DST_RDY_N)
   begin

      --! Default values
      TX_DATA              <= (others => '0');
      TX_SOF_N             <= '1';
      TX_SOP_N             <= '1';
      TX_EOF_N             <= '1';
      TX_EOP_N             <= '1';
      TX_REM               <= (others => '0');
      TX_SRC_RDY_N         <= '1';
      RX_PAYLOAD_DST_RDY_N <= '1';
      RX_HDR_DST_RDY_N     <= '1';

      case (curr_state) is

         --! st_header
         when st_header =>

            if (RX_HDR_SOF_N = '0' AND RX_HDR_SRC_RDY_N = '0' AND
                RX_PAYLOAD_SOF_N = '0' AND RX_PAYLOAD_SRC_RDY_N = '0' AND
                TX_DST_RDY_N = '0') then

               --! Send header to the output interface
               TX_DATA          <= rx_hdr_data_full;
               TX_SOF_N         <= RX_HDR_SOF_N;
               TX_SOP_N         <= RX_HDR_SOP_N;
               TX_EOF_N         <= '1';
               TX_EOP_N         <= RX_HDR_EOP_N;
               TX_REM           <= rx_hdr_rem_full;
               TX_SRC_RDY_N     <= RX_HDR_SRC_RDY_N;
               RX_HDR_DST_RDY_N <= TX_DST_RDY_N;
            end if;

         --! st_payload
         when st_payload =>
            TX_DATA              <= RX_PAYLOAD_DATA;
            TX_SOF_N             <= '1';
            TX_SOP_N             <= RX_PAYLOAD_SOP_N;
            TX_EOF_N             <= RX_PAYLOAD_EOF_N;
            TX_EOP_N             <= RX_PAYLOAD_EOP_N;
            TX_REM               <= RX_PAYLOAD_REM;
            TX_SRC_RDY_N         <= RX_PAYLOAD_SRC_RDY_N;
            RX_PAYLOAD_DST_RDY_N <= TX_DST_RDY_N;

         --! when others
         when others =>
            null;

      end case;
   end process output_logic_p;

end architecture full;
