-- filter_core_config.vhd: Configuration parameters for advanced filter core
-- Copyright (C) 2013 Brno University of Technology
-- Author: Lukas Kekely <xkekel00@stud.fit.vutbr.cz> 
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: filter_core_config.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--

library IEEE;
use IEEE.std_logic_1164.all;
use work.math_pack.all;
use work.tree_func.all; 

package filter_core_v2_config is 
-- -----------------------------------------------------------------------------  
-- CONFIGURABLE PART 1/2
  
  -- Number of different supported rule types.
  --    values: 1-30  
  constant FILTER_RULE_TYPES                  : integer := 4;
  
-- ----------------------------------------------------------------------------- 
  
  type key_parts_t is
  record
    -- Length of source IP prefix used in key part of rule.
    --   values: 0-32 when IPv4 only (ipv4_only==true)
    --           0-128 otherwise (ipv4_only==false) 
    --   special: 0 means do not use source IP at all        
    srcip      : integer;
    -- Length of destination IP prefix used in key part of rule.
    --   values: 0-32 when IPv4 only (ipv4_only==true)
    --           0-128 otherwise (ipv4_only==false) 
    --   special: 0 means do not use destination IP at all  
    dstip      : integer;
    -- Usage of source port number in key part of rule.
    --   values: true = use
    --           false = do not use
    srcport    : boolean;
    -- Usage of destination port number in key part of rule.
    --   values: true = use
    --           false = do not use
    dstport    : boolean;
    -- Usage of protocol number in key part of rule.
    --   values: true = use
    --           false = do not use
    proto      : boolean;
    -- Supporting only IPv4 addresses (prefixes) in key part of rule.
    ipv4_only  : boolean;
  end record;
  
  type rule_type_config_t is
  record
    -- Width of rule key in bits.
    --   values: 0 or sum of lengths of used key parts
    --   special: 0 = disable this type of rules (matching is always false and table is always full)   
    key_width   : integer; 
    -- Which parts of 5-tuple will be used in rule key.     
    key         : key_parts_t;
    -- Number of items inside one hash table.
    items       : integer;
    -- Number of parallel hash tables.
    tables      : integer;
    -- Used type of BRAM memories.
    --   values: 1, 2, 4, 9, 18, 36
    bram_type   : integer;
    -- Insert register after CRC computing logic.
    use_crc_reg : boolean;
    -- Insert register inside key comparision logic.
    use_cmp_reg : boolean;
  end record;
  
  type config_t is array (0 to FILTER_RULE_TYPES-1) of rule_type_config_t;
  
-- -----------------------------------------------------------------------------  
-- CONFIGURABLE PART 2/2

-- Rule Types Configuration
--     must contain constant FILTER_CFG of type config_t with correct filter configuration
--     number of specified items in FILTER_CFG must be exactly FILTER_RULE_TYPES
--     rule types must be specified in ascending order by their priority
  constant FILTER_CFG : config_t := (
    -- Rule type -----------------------
    (
      key_width   => 24,
      key         => 
        (
          srcip     => 24,
          dstip     => 0,
          srcport   => false,
          dstport   => false,
          proto     => false,
          ipv4_only => true
        ),
      items       => 512,
      tables      => 3,
      bram_type   => 36,
      use_crc_reg => true,
      use_cmp_reg => true
    ),
    
    -- Rule type -----------------------
    (
      key_width   => 48,
      key         => 
        (
          srcip     => 48,
          dstip     => 0,
          srcport   => false,
          dstport   => false,
          proto     => false,
          ipv4_only => false
        ),
      items       => 512,
      tables      => 3,
      bram_type   => 36,
      use_crc_reg => true,
      use_cmp_reg => true
    ),
    
    -- Rule type -----------------------
    (
      key_width   => 64,
      key         => 
        (
          srcip     => 64,
          dstip     => 0,
          srcport   => false,
          dstport   => false,
          proto     => false,
          ipv4_only => false
        ),
      items       => 512,
      tables      => 3,
      bram_type   => 36,
      use_crc_reg => true,
      use_cmp_reg => true
    ),
    
    -- Rule type -----------------------
    (
      key_width   => 128,
      key         => 
        (
          srcip     => 128,
          dstip     => 0,
          srcport   => false,
          dstport   => false,
          proto     => false,
          ipv4_only => false
        ),
      items       => 512,
      tables      => 3,
      bram_type   => 36,
      use_crc_reg => true,
      use_cmp_reg => true
    )
  );

-- Init Vectors Configuration
--     must contain constant FILTER_IV_NUMBER equal or higher than maximum number of tables used in FILTER_CFG
--     must contain constant FILTER_IV_MAGIC with FILTER_IV_NUMBER different 32b magic numbers for hash functions 
  constant FILTER_IV_NUMBER            : integer := 3;
  constant FILTER_IV_MAGIC             : std_logic_vector(FILTER_IV_NUMBER*32-1 downto 0) := X"21620FF2DEADF00D00000000";

-- IP Prefix Tree Configuration
--     number of non-leaf tree stages (levels) (capacity is 2**(TS-1)-1 prefixes)
  constant FILTER_TREE_STAGES          : integer := 10;
--     types of memories used for stages (from root)
  constant FILTER_USE_STAGES           : STAGE_TYPE_ARRAY(0 to 31) := (STAGE_TYPE_REGARRAY, STAGE_TYPE_REGARRAY, STAGE_TYPE_REGARRAY, STAGE_TYPE_DIST3MEM, STAGE_TYPE_NOTHING, 
                                                                       STAGE_TYPE_NOTHING,  STAGE_TYPE_DISTMEM,  STAGE_TYPE_DISTMEM,  STAGE_TYPE_BMEM,     STAGE_TYPE_BMEM,
                                                                       STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,
                                                                       STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,
                                                                       STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,
                                                                       STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING,
                                                                       STAGE_TYPE_NOTHING,  STAGE_TYPE_NOTHING);
--     memory type used in leaf stage
  constant FILTER_USE_STAGE_LEAF       : MEMORY_TYPE := mem_bmem;
--     bitmap of register usage inside processing pipeline (width is TS*3+1)
  constant FILTER_USE_REGS             : std_logic_vector(0 to 32*3) := "001" & "001" & "001" & "001" & "001" &
                                                                        "001" & "001" & "001" & "001" & "001" &
                                                                        "1"   &
                                                                        "000" & "000" & "000" & "000" & "000" &
                                                                        "000" & "000" & "000" & "000" & "000" &
                                                                        "000" & "000" & "000" & "000" & "000" &
                                                                        "000" & "000" & "000" & "000" & "000" &
                                                                        "000" & "000";
  
-- ----------------------------------------------------------------------------- 
  
  constant FILTER_ADC_SIZE             : integer := log2(18+FILTER_IV_NUMBER+FILTER_RULE_TYPES);
end filter_core_v2_config;

package body filter_core_v2_config is
end filter_core_v2_config; 
