-- rx_path_100g.vhd : 100GBASE-R PCS - RX module top level
--
-- Copyright (C) 2012 CESNET
-- Author(s): Stepan Friedl <friedl@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: $

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity rx_path_100g is
   generic (
      LANE_ALIGN_SHIFTS : positive := 16
   );
   port (
      RESET_MII     : in std_logic;
      RESET_INT     : in std_logic;
      RESET_PCS     : in std_logic;
      -- 
      INT_CLK       : in std_logic; -- Internal (8 lane) clock
      CGMII_CLK     : in std_logic; -- CGMII clock, 195.xx MHz
      CGMII_RXD     : out std_logic_vector(511 downto 0); -- RX data
      CGMII_RXC     : out std_logic_vector(63 downto 0);  -- RX command
      -- 
      -- HS interface
      RXCLK         : in std_logic; -- RocketIO Clock for lane 0..19 (80.xxx MHz)
      RXCLK_EN      : in std_logic_vector(19 downto 0); -- RX clock enable
      RXD           : in std_logic_vector(20*66-1 downto 0); -- RX data from RocketIO - PCS lane 0..19
      -- Status ports
      HI_BER        : out std_logic;                     -- Block sync HI BER 
      DEC_HI_BER    : out std_logic;                     -- Decoder HI BER 
      BLK_LOCK      : in  std_logic_vector(19 downto 0); -- Block sync lock for each lane
      LINKSTATUS    : out std_logic;                     -- Block sync link status (=lock and !hi_ber)
      BER_COUNT     : out std_logic_vector(21 downto 0); -- Block sync number of errored block (all lanes)
      CLR_BER_CNT   : in std_logic := '0';               -- Clear BER counter in the block sync, RXCLKx synchronous
      BLK_ERR_CNTR  : out std_logic_vector(21 downto 0); 
      BLK_ERR_CLR   : in std_logic := '0';               -- Clear block err counter in the decoder
      -- 
      LANE_MAP      : out std_logic_vector((20*5)-1 downto 0); --
      LANES_ALIGNED : out std_logic_vector(20-1 downto 0); --
      ALIGN_LOCKED  : out std_logic;
      BIP_ERR_CNTRS : out std_logic_vector((20*16)-1 downto 0); -- BIP error counters
      BIP_ERR_CLR   : in  std_logic_vector(20-1 downto 0);
      SCR_BYPASS    : in  std_logic := '0'
      -- Debug ports
--       AM_CNTR_O     : out std_logic;
--       AM_FOUND_O    : out std_logic_vector(3 downto 0); -- 
--       BIP_ERR_O     : out std_logic_vector(3 downto 0); -- 
--       FIFO_DV_O     : out std_logic_vector(3 downto 0); -- 
--       FIFO_RD_O     : out std_logic_vector(3 downto 0); -- 
--       FIFO_FULL_O   : out std_logic_vector(3 downto 0); -- 
--       FIFO_EMPTY_O  : out std_logic_vector(3 downto 0); -- 
--       RXD_O         : out std_logic_vector(4*66-1 downto 0); -- 
--       RXD_CE        : out std_logic;
--       DEC_STATE     : out std_logic_vector(4*3-1 downto 0) -- Decoder state
      
   );
end rx_path_100g;

-- ----------------------------------------------------------------------------
--             Architecture declaration  --  ComboLXT Top Level              --
-- ----------------------------------------------------------------------------
architecture structural of rx_path_100g is

COMPONENT cgmii_ila
  PORT (
    clk : IN STD_LOGIC;
    probe0 : IN STD_LOGIC_VECTOR(511 DOWNTO 0);
    probe1 : IN STD_LOGIC_VECTOR(63 DOWNTO 0);
    probe2 : IN STD_LOGIC_VECTOR(31 DOWNTO 0)
  );
END COMPONENT;

signal probe0 : std_logic_vector(511 downto 0);
signal probe1 : std_logic_vector(63 downto 0);
signal probe2 : std_logic_vector(31 downto 0);
signal ila_clk : std_logic;

signal rxd_wswap       : std_logic_vector(65 downto 0);
signal rxd_bitswap     : std_logic_vector(65 downto 0);
signal rxd_wswap_bswap : std_logic_vector(65 downto 0);

signal match0 : std_logic_vector(19 downto 0);
signal match1 : std_logic_vector(19 downto 0);
signal match2 : std_logic_vector(19 downto 0);
signal match3 : std_logic_vector(19 downto 0);
signal found0 : std_logic;
signal found1 : std_logic;
signal found2 : std_logic;
signal found3 : std_logic;
signal data_from_scr_dbg  : std_logic_vector(64*20-1 downto 0);
signal data_to_scr_dbg  : std_logic_vector(64*20-1 downto 0);

constant PMA_LANES : natural := 10;
constant PCS_LANES : natural := 20;
constant MAC_LANES : natural :=  8;
  
signal zeros          : std_logic_vector(127 downto 0);
signal pwr            : std_logic;
signal data_to_scr    : std_logic_vector(64*MAC_LANES-1 downto 0);
signal data_from_scr  : std_logic_vector(64*MAC_LANES-1 downto 0);
signal data_to_fifo   : std_logic_vector(66*MAC_LANES-1 downto 0);
signal data_from_fifo : std_logic_vector(66*MAC_LANES-1 downto 0);
signal rxd_aligned    : std_logic_vector(66*PCS_LANES-1 downto 0);
signal rxd_merged     : std_logic_vector(66*MAC_LANES-1 downto 0);
signal rxd_aligned_ce : std_logic;
signal rxd_merged_ce  : std_logic;
signal rx_sync_ce     : std_logic_vector(PCS_LANES-1 downto 0);
signal hi_ber_i       : std_logic;
signal reasync_hi_ber : std_logic;
signal hi_ber_xclk    : std_logic;
signal blk_lock_i     : std_logic_vector(PCS_LANES-1 downto 0);
signal reasync_blk_lock : std_logic_vector(PCS_LANES-1 downto 0);
signal blk_lock_xclk  : std_logic_vector(PCS_LANES-1 downto 0);
signal blk_lock_all   : std_logic;
signal align_locked_i     : std_logic;
signal reasync_align_lock : std_logic;
signal align_locked_xclk  : std_logic;
signal reagsync_bip_err_clr : std_logic_vector(19 downto 0);
signal bip_err_clr_rxclk    : std_logic_vector(19 downto 0);
signal regasync_clr_ber_cnt : std_logic;  
signal clear_ber_count      : std_logic;
signal linkstatus_i   : std_logic;
signal data_to_ber_mon: std_logic_vector(2*PCS_LANES-1 downto 0);
signal cgmii_rxd_i     : std_logic_vector(CGMII_RXD'range);
signal cgmii_rxc_i     : std_logic_vector(CGMII_RXC'range);


signal rxd_sync_out  : std_logic_vector(66*PCS_LANES-1 downto 0);
signal rxd_blklocked : std_logic_vector(66*PCS_LANES-1 downto 0);

signal align_clk     : std_logic_vector(PCS_LANES-1 downto 0);     
signal align_en      : std_logic_vector(PCS_LANES-1 downto 0);
signal dec_err_flg   : std_logic_vector(MAC_LANES*2-1 downto 0);
signal dec_error     : std_logic_vector(MAC_LANES-1 downto 0);
signal seq_error     : std_logic_vector(MAC_LANES-1 downto 0);

signal am_found_dbg  :  std_logic_vector(PCS_LANES-1 downto 0);

attribute SHREG_EXTRACT : string;
attribute SHREG_EXTRACT of blk_lock_xclk : signal is "no"; 
attribute SHREG_EXTRACT of hi_ber_xclk  : signal is "no"; 
attribute SHREG_EXTRACT of bip_err_clr_rxclk : signal is "no"; 

function AND_REDUCE (D : in std_logic_vector) return std_logic is
variable tmp : std_logic;
begin
   tmp := D(D'right);
   for i in D'right+1 to D'right loop
      tmp := tmp and D(i);
   end loop;
   return tmp;
end function AND_REDUCE;

begin

zeros <= (others => '0');
pwr   <= '1';

CONTROL_RECLOCK: process(RXCLK)
begin

   if RXCLK'event and RXCLK = '1' then
      reagsync_bip_err_clr <= BIP_ERR_CLR;
      bip_err_clr_rxclk    <= reagsync_bip_err_clr;
      
      regasync_clr_ber_cnt <= CLR_BER_CNT;
      clear_ber_count      <= regasync_clr_ber_cnt;
   end if;

end process;

align_clk <= RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & 
             RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK & RXCLK;
             
align_en  <= RXCLK_EN;

-- ALIGN: entity work.lane_align 
-- generic map (
--    LANES => PCS_LANES
-- )
-- port map (
--    RESET         => RESET_PCS,
--    CLK           => align_clk,
--    BLK_LOCK      => BLK_LOCK,   
--    EN            => RXCLK_EN,
--    D             => RXD,
--    Q_CLK         => RXCLK,
--    Q             => rxd_aligned,
--    QV            => rxd_aligned_ce,
--    --
--    LOCKED        => align_locked_i,
--    LANE_MAP      => LANE_MAP,
--    LANES_ALIGNED => LANES_ALIGNED,
--    BIP_ERR_CNTRS => BIP_ERR_CNTRS,
--    CLR_ERR_CNTRS => bip_err_clr_rxclk,
--    --
--    AM_CNTR_O     => open, -- AM_CNTR_O,
--    AM_FOUND_O    => am_found_dbg, -- AM_FOUND_O,
--    BIP_ERR_O     => open, -- BIP_ERR_O,
--    FIFO_DV_O     => open, -- FIFO_DV_O,
--    FIFO_RD_O     => open, -- FIFO_RD_O,
--    FIFO_EMPTY_O  => open,
--    FIFO_FULL_O   => open
-- );

-- The newer version of the LANE_ALIGN (with shift registers) ---------------------
ALIGN: entity work.lane_align(behavioral)
generic map (
   LANES  => PCS_LANES,        -- number of parallel lines (4 or 20)
   SHIFTS => LANE_ALIGN_SHIFTS -- shift register width = lanes skew size
)
port map (
   RESET    => RESET_PCS,         --  
   CLK      => RXCLK,             
   EN       => RXCLK_EN,          -- Input enable for each lane
   D        => RXD,               -- Input data for each lane
   Q        => rxd_aligned,       -- Output data for each lane
   QV       => rxd_aligned_ce,    -- Otput data valid
   BLK_LOCK => BLK_LOCK,          -- Block lock for each lane
   -- Status ports
   LOCKED        => align_locked_i,    -- valid mark found on each lane
   LANE_MAP      => LANE_MAP,          -- output mapping for each lane
   LANES_ALIGNED => LANES_ALIGNED,     -- alignment confirmed for eac line
   BIP_ERR_CNTRS => BIP_ERR_CNTRS,     -- BIP error counters out
   CLR_ERR_CNTRS => bip_err_clr_rxclk,  -- BIP error counters reset
   -- Debug
--    shift_cntrs_reset_DBG   => AM_FOUND_O,
--    lanes_align_reset_DBG   => BIP_ERR_O,
--    lane_align_err_DBG      => FIFO_DV_O,
--    am_invld_cntrs_end_DBG  => FIFO_RD_O,
   --
--    am_match_DBG(11 downto 0)  => DEC_STATE,
--    am_match_DBG(15 downto 12) => FIFO_RD_O,
--    am_lock_DBG                => FIFO_DV_O,
--    am_cntrs_end_DBG           => FIFO_FULL_O,
      am_found_DBG               => am_found_dbg
--    bip_match_DBG              => FIFO_EMPTY_O,
--    bip_err_DBG                => BIP_ERR_O
);

GEN_BM_DATA: for i in 0 to PCS_LANES-1 generate
   data_to_ber_mon((i+1)*2-1 downto i*2) <= rxd_aligned(66*i+1 downto i*66); -- sync headers only
end generate;

BER_MONITOR: entity work.ber_mon 
generic map (
   NUM_LANES       => PCS_LANES
)
port map (
   RESET         => RESET_PCS,
   CLK           => RXCLK,
   CE            => rxd_aligned_ce,
   SH            => data_to_ber_mon,
   -- 
   BER_CNT       => open,
   BER_COUNT_CLR => clear_ber_count,
   BER_COUNT     => BER_COUNT,
   HI_BER        => hi_ber_i
);

-- Bus width conversion: 20 PCS lanes to 8 MAC lanes
MERGE: entity work.block_merge_20x8 
port map (
   -- Input port D - 20 word data, low freq
   RST_D => RESET_PCS,
   CLK_D => RXCLK,
   CE_D  => rxd_aligned_ce,
   D     => rxd_aligned,
   -- Output port Q - 8 word data, hi freq
   RST_Q => RESET_INT,
   CLK_Q => INT_CLK,
   CE_Q  => rxd_merged_ce,
   Q     => rxd_merged
);


GEN_SCR_DATA: for i in 0 to MAC_LANES-1 generate
   data_to_scr((i+1)*64-1 downto i*64) <= rxd_merged(66*(i+1)-1 downto i*66+2); -- Exclude sync bits
end generate;
-- 

DESCRAMBLE: entity work.descrambler_gen
generic map (
   WIDTH => MAC_LANES*64
)
port map (
   RESET => RESET_INT,
   CLK   => INT_CLK,
   EN    => rxd_merged_ce,
   BYPASS => SCR_BYPASS,
   SEED  => zeros(57 downto 0),
   D     => data_to_scr,
   Q     => data_from_scr
);

-- 
GEN_FIFO_DATA: for i in 0 to MAC_LANES-1 generate
   data_to_fifo((i+1)*66-1 downto i*66+2) <= data_from_scr(64*(i+1)-1 downto i*64);
   data_to_fifo(i*66+1 downto i*66) <= rxd_merged(66*i+1 downto i*66); -- Add sync bits
end generate;

-- FIFO for PCS/MAC bit rate compensation
FIFO: entity work.pcs_rx_fifo 
generic map (
   NUM_LANES => MAC_LANES
)
port map (
   RESET_D => RESET_INT,
   CLK_D   => INT_CLK,
   WE      => rxd_merged_ce,
   D       => data_to_fifo,
   -- 
   RESET_Q => RESET_MII,
   CLK_Q   => CGMII_CLK,
   Q       => data_from_fifo
);

-- 
-- -- Reclock control/status signals between clock domains
STATUS_RECLOCK: process(CGMII_CLK)
variable tmp : std_logic;
begin
   if CGMII_CLK'event and CGMII_CLK = '1' then
      reasync_blk_lock <= BLK_LOCK;
      blk_lock_xclk    <= reasync_blk_lock;
      
      reasync_hi_ber <= hi_ber_i;
      hi_ber_xclk    <= reasync_hi_ber;
      
      reasync_align_lock <= align_locked_i;
      align_locked_xclk  <= reasync_align_lock;
      
      tmp := '0';
      -- All lanes are locked
      for i in 0 to blk_lock_xclk'high loop
         tmp := tmp and blk_lock_xclk(i);
      end loop;
      -- blk_lock_all <= AND_REDUCE(blk_lock_xclk) and reasync_align_lock;
      --blk_lock_all <= tmp and reasync_align_lock;
      
      blk_lock_all <= (blk_lock_xclk(0) and blk_lock_xclk(1) and blk_lock_xclk(2) and blk_lock_xclk(3) and blk_lock_xclk(4)) and
                      (blk_lock_xclk(5) and blk_lock_xclk(6) and blk_lock_xclk(7) and blk_lock_xclk(8) and blk_lock_xclk(9)) and
                      (blk_lock_xclk(10) and blk_lock_xclk(11) and blk_lock_xclk(12) and blk_lock_xclk(13) and blk_lock_xclk(14)) and
                      (blk_lock_xclk(15) and blk_lock_xclk(16) and blk_lock_xclk(17) and blk_lock_xclk(18) and blk_lock_xclk(19)) and 
                       align_locked_i;

--       blk_lock_all <= (blk_lock_xclk(0) and blk_lock_xclk(1) and blk_lock_xclk(2) and blk_lock_xclk(3) and blk_lock_xclk(4)) and
--                       (blk_lock_xclk(5) and blk_lock_xclk(6) and blk_lock_xclk(7) and blk_lock_xclk(8) and blk_lock_xclk(9)) and
--                       (blk_lock_xclk(10) and blk_lock_xclk(11) and blk_lock_xclk(12) and blk_lock_xclk(13) and blk_lock_xclk(14)) and
--                       (blk_lock_xclk(15) and blk_lock_xclk(16) and blk_lock_xclk(17) and blk_lock_xclk(18) and blk_lock_xclk(19));
   end if;
end process;

DECODE: entity work.block_decode
generic map (
   NUM_LANES   => 8
)
port map (
   RESET        => RESET_MII,
   CLK          => CGMII_CLK,
   -- Encoded data input
   D            => data_from_fifo,
   --  XL(C)GMII output
   RXD          => cgmii_rxd_i,
   RXC          => cgmii_rxc_i,
   -- Control & status
   BYPASS       => '0',
   BLK_LOCK     => blk_lock_all,
   LOOPBACK     => '0',
   HI_BER       => hi_ber_xclk,
   BLK_ERR_CNTR => BLK_ERR_CNTR,
   BLK_ERR_CLR  => BLK_ERR_CLR,
   --
   DEC_ERROR   => dec_error,
   SEQ_ERROR   => seq_error
);

CGMII_RXD <= cgmii_rxd_i;
CGMII_RXC <= cgmii_rxc_i;

linkstatus_i <= align_locked_i and (not hi_ber_i);

-- --
HI_BER       <= hi_ber_i;
LINKSTATUS   <= linkstatus_i;
ALIGN_LOCKED <= align_locked_i;

-- ----------------------------------------------------------------------------
-- Monitor decode errors and assert the "DEC_HI_BER" when the line is 
-- erroneous
-- ----------------------------------------------------------------------------

GEN_ERR_FLAG: for i in 0 to MAC_LANES-1 generate
   dec_err_flg(i*2+1 downto i*2) <= "00" when (dec_error(i) = '1') or (seq_error(i) = '1') else -- generate invalid flag
                                    "01"; -- 01 or 10 means "valid", "00" and "11" are invalid flags
end generate;

DECODE_MON: entity work.ber_mon 
generic map (
   NUM_LANES       => 8
)
port map (
   RESET         => RESET_MII,
   CLK           => CGMII_CLK,
   CE            => '1',
   SH            => dec_err_flg,
   -- 
   BER_CNT       => open,
   BER_COUNT_CLR => '0',
   BER_COUNT     => open,
   HI_BER        => DEC_HI_BER
);

-- RXD_O      <= data_to_fifo;
-- RXD_CE     <= rxd_aligned_ce;

-- --------------------------------------------------------------------------------
-- Debug stuff, Chipscope ILA core, etc. 
-- --------------------------------------------------------------------------------

--GEN_SCR_DATA_DBG: for i in 0 to 19 generate
--   data_to_scr_dbg((i+1)*64-1 downto i*64) <= rxd_aligned(66*(i+1)-1 downto i*66+2); -- Exclude sync bits
--end generate;
---- 

--DESCRAMBLE_DBG: entity work.descrambler_gen
--generic map (
--   WIDTH => 20*64
--)
--port map (
--   RESET => RESET_PCS,
--   CLK   => RXCLK,
--   EN    => rxd_aligned_ce,
--   BYPASS => SCR_BYPASS,
--   SEED  => zeros(57 downto 0),
--   D     => data_to_scr_dbg,
--   Q     => data_from_scr_dbg
--);


--GEN_BS: for i in 0 to 63 generate
--   rxd_bitswap(i+2) <= rxd(65-i);
--end generate;
--rxd_bitswap(1 downto 0) <= rxd(1 downto 0);

--rxd_wswap(1 downto 0)   <= rxd(1 downto 0);  
--rxd_wswap(33 downto 2)  <= rxd(65 downto 34);
--rxd_wswap(65 downto 34) <= rxd(33 downto 2);

--rxd_wswap_bswap(1 downto 0)   <= rxd_bitswap(1 downto 0);  
--rxd_wswap_bswap(33 downto 2)  <= rxd_bitswap(65 downto 34);
--rxd_wswap_bswap(65 downto 34) <= rxd_bitswap(33 downto 2);

---- Debug: find Align markers in various configurations
--DEBUG_AM_LOOKUP0: entity work.am_check_20
--port map (
--   RESET     => reset_pcs,
--   CLK       => RXCLK,
--   EN        => pwr,
--   D         => rxd( 65 downto   0),
--   MATCH     => match0, 
--   BIP_MATCH => open
--);

--DEBUG_AM_LOOKUP1: entity work.am_check_20
--port map (
--   RESET     => reset_pcs,
--   CLK       => RXCLK,
--   EN        => pwr,
--   D         => rxd_bitswap(65 downto 0),
--   MATCH     => match1, 
--   BIP_MATCH => open
--);

--DEBUG_AM_LOOKUP2: entity work.am_check_20
--port map (
--   RESET     => reset_pcs,
--   CLK       => RXCLK,
--   EN        => pwr,
--   D         => rxd_wswap(65 downto 0),
--   MATCH     => match2, 
--   BIP_MATCH => open
--);

--DEBUG_AM_LOOKUP3: entity work.am_check_20
--port map (
--   RESET     => reset_pcs,
--   CLK       => RXCLK,
--   EN        => pwr,
--   D         => rxd_wswap_bswap(65 downto 0),
--   MATCH     => match3, 
--   BIP_MATCH => open
--);

--FOUND_FFS: process(RXCLK)
--begin
--   if RXCLK'event and RXCLK = '1' then
--      found0 <= (match0(0)  or match0(1)  or match0(2)  or match0(3)  or match0(4)) or 
--                (match0(5)  or match0(6)  or match0(7)  or match0(8)  or match0(9)) or 
--                (match0(10) or match0(11) or match0(12) or match0(13) or match0(14)) or 
--                (match0(15) or match0(16) or match0(17) or match0(18) or match0(19));
      
--      found1 <= (match1(0)  or match1(1)  or match1(2)  or match1(3)  or match1(4)) or 
--                (match1(5)  or match1(6)  or match1(7)  or match1(8)  or match1(9)) or 
--                (match1(10) or match1(11) or match1(12) or match1(13) or match1(14)) or 
--                (match1(15) or match1(16) or match1(17) or match1(18) or match1(19));
                
--      found2 <= (match2(0)  or match2(1)  or match2(2)  or match2(3)  or match3(4)) or 
--                (match2(5)  or match2(6)  or match2(7)  or match2(8)  or match3(9)) or 
--                (match2(10) or match2(11) or match2(12) or match2(13) or match3(14)) or 
--                (match2(15) or match2(16) or match2(17) or match2(18) or match3(19));
                
--      found3 <= (match3(0)  or match3(1)  or match3(2)  or match3(3)  or match3(4)) or 
--                (match3(5)  or match3(6)  or match3(7)  or match3(8)  or match3(9)) or 
--                (match3(10) or match3(11) or match3(12) or match3(13) or match3(14)) or 
--                (match3(15) or match3(16) or match3(17) or match3(18) or match3(19));
--   end if;
--end process;

--   ILA : cgmii_ila
--   PORT MAP (
--     clk    => ila_clk, --INT_CLK, -- CGMII_CLK, -- CGMII_CLK, -- RXCLK, --  
--     probe0 => probe0,
--     probe1 => probe1,
--     probe2 => probe2
--   );
  
----   probe0 <= cgmii_rxd_i;
----   probe1 <= cgmii_rxc_i;

--    probe0 <= data_from_scr_dbg(64*8-1 downto 64*7) & data_from_scr_dbg(64*7-1 downto 64*6) & data_from_scr_dbg(64*6-1 downto 64*5) & data_from_scr_dbg(64*5-1 downto 64*4) & -- 
--              data_from_scr_dbg(255 downto 192) & data_from_scr_dbg(191 downto 128) & data_from_scr_dbg(127 downto  64) & data_from_scr_dbg( 63 downto   0);  -- Data blocks
   
--    probe1<= rxd_aligned_ce & "000" & -- 63:60
--             X"0000" &                -- 59:44
--             am_found_dbg &           -- 43:24
--             X"00" &                  -- 23:16
--             rxd_aligned(463 downto 462) & rxd_aligned(397 downto 396) & rxd_aligned(331 downto 330) & rxd_aligned(265 downto 264) & -- Sync headers 15:7
--             rxd_aligned(199 downto 198) & rxd_aligned(133 downto 132) & rxd_aligned( 67 downto  66) & rxd_aligned(  1 downto   0);  -- 7:0
--    ila_clk <= RXCLK;
--
--
--    probe0 <= rxd_aligned(527 downto 464) & rxd_aligned(461 downto 398) & rxd_aligned(395 downto 332) & rxd_aligned(329 downto 266) & --  
--              rxd_aligned(263 downto 200) & rxd_aligned(197 downto 134) & rxd_aligned(131 downto  68) & rxd_aligned( 65 downto   2);  -- Data blocks
              
--    probe1<= rxd_aligned_ce & "000" & -- 63:60
--             X"0000"  &               -- 59:44
--             am_found_dbg &,          -- 43:24
--             X"00" &    -- 23:16
--             rxd_aligned(463 downto 462) & rxd_aligned(397 downto 396) & rxd_aligned(331 downto 330) & rxd_aligned(265 downto 264) & -- Sync headers 15:7
--             rxd_aligned(199 downto 198) & rxd_aligned(133 downto 132) & rxd_aligned( 67 downto  66) & rxd_aligned(  1 downto   0);  -- 7:0
--    ila_clk <= RXCLK;

     
--    probe0 <= rxd_merged(527 downto 464) & rxd_merged(461 downto 398) & rxd_merged(395 downto 332) & rxd_merged(329 downto 266) & -- 
--              rxd_merged(263 downto 200) & rxd_merged(197 downto 134) & rxd_merged(131 downto  68) & rxd_merged( 65 downto   2);  -- Data blocks
--              
--    probe1<= rxd_merged_ce & "000" & -- 63:60
--             X"0000" &               -- 59:44
--             X"00000" & -- 43:24
--             X"00" &        -- 23:16
--             rxd_merged(463 downto 462) & rxd_merged(397 downto 396) & rxd_merged(331 downto 330) & rxd_merged(265 downto 264) & -- Sync headers 15:7
--             rxd_merged(199 downto 198) & rxd_merged(133 downto 132) & rxd_merged( 67 downto  66) & rxd_merged(  1 downto   0);  -- 7:0
--    ila_clk <= INT_CLK;

                          
    probe0 <= data_to_fifo(527 downto 464) & data_to_fifo(461 downto 398) & data_to_fifo(395 downto 332) & data_to_fifo(329 downto 266) & -- 
              data_to_fifo(263 downto 200) & data_to_fifo(197 downto 134) & data_to_fifo(131 downto  68) & data_to_fifo( 65 downto   2);  -- Data blocks
              
    probe1<= rxd_merged_ce & "000" & -- 63:60
             X"0000" &               -- 59:44
             X"00000" & -- 43:24
             X"00" &        -- 23:16
             data_to_fifo(463 downto 462) & data_to_fifo(397 downto 396) & data_to_fifo(331 downto 330) & data_to_fifo(265 downto 264) & -- Sync headers 15:7
             data_to_fifo(199 downto 198) & data_to_fifo(133 downto 132) & data_to_fifo( 67 downto  66) & data_to_fifo(  1 downto   0);  -- 7:0
    ila_clk <= INT_CLK;
    
-- 
--    probe1<= "0000" & -- 63:60
--             X"0000" &               -- 59:44
--             X"00000" & -- 43:24
--             X"00" &        -- 23:16
--             data_from_fifo(463 downto 462) & data_from_fifo(397 downto 396) & data_from_fifo(331 downto 330) & data_from_fifo(265 downto 264) & -- Sync headers 15:7
--             data_from_fifo(199 downto 198) & data_from_fifo(133 downto 132) & data_from_fifo( 67 downto  66) & data_from_fifo(  1 downto   0);  -- 7:0

             
   
--   probe0 <= rxd(527 downto 464) & rxd(461 downto 398) & rxd(395 downto 332) & rxd(329 downto 266) & -- 
--             rxd(263 downto 200) & rxd(197 downto 134) & rxd(131 downto  68) & rxd( 65 downto   2);  -- Data blocks
--   probe1<= RXCLK_EN & -- 63:44
--            am_found_dbg & -- 43:24
--            X"00" &        -- 23:20
--            --found3 & found2 & found1 & found0 & -- 19:16
--            rxd(463 downto 462) & rxd(397 downto 396) & rxd(331 downto 330) & rxd(265 downto 264) & -- Sync headers 15:7
--            rxd(199 downto 198) & rxd(133 downto 132) & rxd( 67 downto  66) & rxd(  1 downto   0);  -- 7:0
--   ila_clk <= RXCLK; 
            
--   -- probe2 <= RESET_MII & RESET_INT & RESET_PCS & X"000" & align_locked_i & BLK_LOCK(7 downto 0) & RXCLK_EN(7 downto 0);
--   -- probe2 <= RESET_MII & RESET_INT & RESET_PCS & X"00" & "00" & blk_lock_all & hi_ber_xclk & align_locked_i & BLK_LOCK(15 downto 0);
   
   
   probe2 <= (others => '0');
  
end structural;

