-- process_arch.vhd: CGMII_PROCESS full architecture.
-- Copyright (C) 2012 CESNET
-- Author(s): Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of cgmii_obuf_process is
   --Constants --------------------------------------------
   --! Shift register depth. FLU_REORD unit has two pipeline stage
   --! CRC_DV delay is 4 clocks after EOP. We need to hold data for 2 clocks (BLOCK_REORD + OUTPUT_PIPE).
   --! Perform two more clock to be absolutly sure that CRC is in FIFO.
   constant SHREG_DEPTH    : integer := 7;

   -- Component signals -----------------------------------
   --FLU_BLOCK_REORD interface
   --! FLU_REORD data
   signal flu_rx_reord_data      : std_logic_vector(511 downto 0);
   --! FLU_REORD SOP
   signal flu_rx_reord_sop       : std_logic;
   --! FLU_REORD SOP_POS
   signal flu_rx_reord_sop_pos   : std_logic_vector(2 downto 0);
   --! FLU_REORD EOP         
   signal flu_rx_reord_eop       : std_logic;
   --! FLU_REORD EOP_POS      
   signal flu_rx_reord_eop_pos   : std_logic_vector(5 downto 0);
   --! FLU_REORD destination redy    
   signal flu_rx_reord_dst_rdy   : std_logic;
   --! FLU_REORD source ready 
   signal flu_rx_reord_src_rdy   : std_logic;

   --! FLU_REORD data output
   signal reord_tx_data          : std_logic_vector(511 downto 0);
   --! FLU_REORD sop_v output
   signal reord_tx_sop_v         : std_logic_vector(7 downto 0);
   --! FLU_REORD eop_v     
   signal reord_tx_eop_v         : std_logic_vector(7 downto 0);
   --! FLU_REORD eop_pos in block
   signal reord_tx_eop_pos       : std_logic_vector(2 downto 0);
      
   --CRC32 & CRC32 FIFO & Shift register ------------------
   --! CRC32 data input
   signal crc32_di         : std_logic_vector(511 downto 0);
   --! CRC32 data input valid
   signal crc32_di_dv      : std_logic;
   --! CRC32 EOP_POS signal for CRC32 computation unit
   signal crc32_mask       : std_logic_vector(5 downto 0);
   --! CRC32 EOP valid signal
   signal crc32_eop        : std_logic;
   --! CRC32 SOP_POS signal
   signal crc32_start      : std_logic_vector(5 downto 0);
   signal crc32_start_n    : std_logic_vector(5 downto 0);
   --! CRC32 SOP signal valid
   signal crc32_sop        : std_logic;
   --! CRC32 data output
   signal crc32_data       : std_logic_vector(31 downto 0);
   --! Reordered CRC32 value
   signal crc32_data_reord : std_logic_vector(31 downto 0); 
   --! CRC32 data output valid
   signal crc32_data_vld  : std_logic;
   --! Shift register input
   signal shreg_in   : std_logic_vector(530 downto 0) := (others => '0');
   --! Shift register output
   signal shreg_out  : std_logic_vector(530 downto 0);
   --! Shift enable signal
   signal shreg_en   : std_logic;
   
   --! Transfer of packet is running
   signal crc32_di_dv_reg  : std_logic := '0';
   --! EOP_POS in WORD
   signal crc32_eop_pos    : std_logic_vector(5 downto 0);
   --! Repaired EOP_POS 
   signal crc32_eop_pos_rep: std_logic_vector(5 downto 0);      

begin
   -- Input signal map. Temporal connection, payload filling component have to be used here.
   flu_rx_reord_data    <= RX_DATA;
   flu_rx_reord_sop     <= RX_SOP;
   flu_rx_reord_sop_pos <= RX_SOP_POS;
   flu_rx_reord_eop     <= RX_EOP;
   flu_rx_reord_eop_pos <= RX_EOP_POS;
   flu_rx_reord_src_rdy <= RX_SRC_RDY;
   RX_DST_RDY <= flu_rx_reord_dst_rdy;

   --! FLU block(64 bit) reordering component for CGMII OBUF. With respect to PREAMBLE, CRC, IFG (+DIC)
   --! One input pipeline stage.

   FLU_BLOCK_REORD_I:entity work.FLU_BLOCK_REORD
   port map(
      -----------------------------------------------------
      --! \ name Common interface
      -----------------------------------------------------
      --! Input clock
      CLK         => CLK,
      --! Synchronous reset
      RESET       => RESET,

      -----------------------------------------------------
      --! \name FLU RX interface
      -----------------------------------------------------
      --! Write side data
      RX_DATA       	=> flu_rx_reord_data,
      --! Start of packet position
      RX_SOP_POS    	=> flu_rx_reord_sop_pos,
      --! End of packet position
      RX_EOP_POS    	=> flu_rx_reord_eop_pos,
      --! Start of packet valid
      RX_SOP        	=> flu_rx_reord_sop,
      --! End of packet valid
      RX_EOP        	=> flu_rx_reord_eop,
      --! Source ready
      RX_SRC_RDY     => flu_rx_reord_src_rdy,
      --! Destination ready
      RX_DST_RDY    	=> flu_rx_reord_dst_rdy,
      
      -----------------------------------------------------
      --! \name TX output interface 
      -----------------------------------------------------
      --! Read side ready
      TX_DATA        => reord_tx_data,
      --! SOP block indication (one-hot)
      TX_SOP_V       => reord_tx_sop_v,
      --! EOP block indication (one-hot)
      TX_EOP_V       => reord_tx_eop_v,
      --! End in block position
      TX_EOP_POS     => reord_tx_eop_pos
   );

   --! CRC data input valid register   
   transf_vld_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            crc32_di_dv_reg <= '0';
         elsif(reord_tx_eop_v /= "00000000" and reord_tx_sop_v = "00000000")then
            crc32_di_dv_reg <= '0';
         elsif(reord_tx_sop_v /= "00000000")then
            crc32_di_dv_reg <= '1';
         end if;
      end if;
   end process;

   --! Repaider EOP_POS value
   crc32_eop_pos_rep <= "000" & reord_tx_eop_pos;

   --! EOP_POS in block
   crc32_eop_pos <= crc32_eop_pos_rep + 8  when(reord_tx_eop_v = "00000010")else
                    crc32_eop_pos_rep + 16 when(reord_tx_eop_v = "00000100")else
                    crc32_eop_pos_rep + 24 when(reord_tx_eop_v = "00001000")else
                    crc32_eop_pos_rep + 32 when(reord_tx_eop_v = "00010000")else
                    crc32_eop_pos_rep + 40 when(reord_tx_eop_v = "00100000")else
                    crc32_eop_pos_rep + 48 when(reord_tx_eop_v = "01000000")else
                    crc32_eop_pos_rep + 56 when(reord_tx_eop_v = "10000000")else
                    crc32_eop_pos_rep;

   --! CRC32 data in
   crc32_di    <= reord_tx_data;
   --! CRC32 data input valid
   crc32_di_dv <= '1' when(crc32_di_dv_reg = '1' or reord_tx_sop_v /= "00000000") else '0';
   --! CRC32 EOP_POS signal
   crc32_mask  <= not(crc32_eop_pos) when(crc32_eop = '1') else (others=>'0'); 
   --! CRC32 EOP signal
   crc32_eop   <= '1' when(reord_tx_eop_v /= "00000000") else '0';
   --! CRC32 SOP_POS signal (SOP_POS times 8)
   crc32_start <=  "001000" when(reord_tx_sop_v = "00000010") else
                   "010000" when(reord_tx_sop_v = "00000100") else
                   "011000" when(reord_tx_sop_v = "00001000") else
                   "100000" when(reord_tx_sop_v = "00010000") else
                   "101000" when(reord_tx_sop_v = "00100000") else
                   "110000" when(reord_tx_sop_v = "01000000") else
                   "111000" when(reord_tx_sop_v = "10000000") else
                   "000000";
                   
   --! CRC32 SOP signal
   crc32_sop   <= '1' when(reord_tx_sop_v /= "00000000") else '0';

   crc32_start_n <= not crc32_start;

   --! CRC32 unit
   CRC32_I:entity work.crc32_gen_start
   generic map(
      DATA_WIDTH => 512,
      REG_BITMAP => 31 --11111 => CRC32 value is asserted 4 clocks after EOP
   )
   port map(
      DI    => crc32_di,
      DI_DV => crc32_di_dv,
      EOP   => crc32_eop,
      SOP   => crc32_sop,
      MASK  => crc32_mask,
      START => crc32_start_n,
      CLK   => CLK,
      RESET => RESET,
      CRC   => crc32_data,
      DO_DV => crc32_data_vld
   );

   --! Reorder byte positon of CRC32
   crc32_data_reord(7 downto 0)     <= crc32_data(31 downto 24);
   crc32_data_reord(15 downto 8)    <= crc32_data(23 downto 16);
   crc32_data_reord(23 downto 16)   <= crc32_data(15 downto 8);
   crc32_data_reord(31 downto 24)   <= crc32_data(7 downto 0);

   --! CRC32 FIFO component
   CRC32_FIFO_I:entity work.FIFO 
   generic map(
      -- Set data width here
      DATA_WIDTH     => 32,
      -- Distributed RAM type, only 16, 32, 64 bits
      DISTMEM_TYPE   => CRC32_FIFO_DISTMEM_TYPE,
      -- Set number of items in FIFO here
      ITEMS          => CRC32_FIFO_ITEMS,
      -- for last block identification
      BLOCK_SIZE     => 0
   )
   port map(
      RESET    => RESET,  -- Global reset signa
      CLK      => CLK,  -- Global clock signal

      -- Write interface
      DATA_IN  => crc32_data_reord,-- Data input
      WRITE_REQ=> crc32_data_vld,  -- Write request
      FULL     => open, -- FIFO is full
      LSTBLK   => open, -- Last block identifier

      -- Read interface
      DATA_OUT => CRC32_DATA_OUT,   -- Data output
      READ_REQ => CRC32_READ_REQ,   -- Read request
      EMPTY    => open       -- FIFO is empty
   );
   
   --Shift register input pipeline ----

   shiftreg_pipe_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            shreg_in(527 downto 512) <= (others => '0');
         else
            --! Shift register data map
            shreg_in(511 downto 0)     <= reord_tx_data;
            --! Shift register sop_v map
            shreg_in(519 downto 512)   <= reord_tx_sop_v;
            --! Shift register eop_v map
            shreg_in(527 downto 520)   <= reord_tx_eop_v;
            --! Shift register eop_pos map
            shreg_in(530 downto 528)   <= reord_tx_eop_pos;
         end if;
      end if;
   end process;

   --! Shifting is enabled by default
   shreg_en <= '1';

   --! Output shift register for DATA & SOP_V & EOP_V & EOP_POS signals
   SHREG_I:entity work.sh_reg_bus
   generic map(
      NUM_BITS    => SHREG_DEPTH,
      --INIT        : std_logic_vector(15 downto 0) := X"0000";
      --INIT_EXT00  : std_logic_vector(63 downto 0) := X"0000000000000000";      
      DATA_WIDTH  => 531 
   )
   port map(
      CLK      => CLK,

      DIN      => shreg_in,
      CE       => shreg_en,
      DOUT     => shreg_out
   );

   --------------------------------------------------------
   --Output signal map
   --------------------------------------------------------
   --! TX_DATA bus map
   TX_DATA     <= shreg_out(511 downto 0);
   --! TX_SOP_V bus map
   TX_SOP_V    <= shreg_out(519 downto 512);
   --! TX_EOP_V bus map
   TX_EOP_V    <= shreg_out(527 downto 520);
   --! TX_EOP_POS bus map
   TX_EOP_POS  <= shreg_out(530 downto 528);

end architecture full;
