-- ibuf_cgmii_ent.vhd: CGMII Input buffer - entity declaration
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz> 
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------

entity ibuf_cgmii is
   generic(
      -- Number of items in Data FIFO
      DFIFO_SIZE          : integer := 512;
      -- Number of items in Header FIFO
      HFIFO_SIZE          : integer := 256;
      -- Width of header buses in bits
      HDR_WIDTH           : integer := 128;
      -- Number of MAC addresses that can be placed into CAM (at most 16)
      MAC_COUNT           : integer := 16;
      -- Remove FCS from the packet (false -> remove, true -> don't remove)
      INBANDFCS           : boolean := true;
   	-- Length of link error timeout counter (number of counter bits)
      CNT_ERROR_LENGTH    : integer := 5;
      -- Synthesise design without resets, default values of registers are set
      -- by initialization of signals
      RESET_BY_INIT       : boolean := false;
      CNT_DSP             : boolean := true
   );
   port(
      -- CGMII interface -----------------------------------------------------
      -- CGMII Receive Clock
      CGMII_RXCLK        : in  std_logic;
      -- CGMII Receive Reset
      CGMII_RXRESET      : in  std_logic;
      -- CGMII Receive Data
      CGMII_RXD          : in  std_logic_vector(511 downto 0);
      -- CGMII Receive Control
      CGMII_RXC          : in  std_logic_vector(63 downto 0);
     
      -- FrameLinkUnaligned interface (data payload output) -------------------
      -- Output clock
      TX_CLK              : in  std_logic;
      -- Output reset
      TX_RESET            : in  std_logic;
      -- Payload data
      TX_DATA             : out std_logic_vector(511 downto 0);
      -- Position of the start of the payload
      -- valid only if TX_SOP is set to '1'
      TX_SOP_POS          : out std_logic_vector(2 downto 0);
      -- Position of the end of the payload
      -- valid only if TX_EOP is set to '1'
      TX_EOP_POS          : out std_logic_vector(5 downto 0);
      -- Start of the payload
      TX_SOP              : out std_logic;
      -- End of the payload
      TX_EOP              : out std_logic;
      -- Source is ready
      TX_SRC_RDY          : out std_logic;
      -- Destination is ready
      TX_DST_RDY          : in  std_logic;

      -- FrameLink interface (data header output) -----------------------------
      -- Header data
      TX_HDATA            : out std_logic_vector(HDR_WIDTH-1 downto 0);
      -- Start of the header, active in '0'
      TX_HSOP_N           : out std_logic;
      -- End of the header, active in '0'
      TX_HEOP_N           : out std_logic;
      -- Posiiton of the end of the header
      -- valid only if TX_HEOP_N is set to '0'
      TX_HREM             : out std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
      -- Source is ready, active in '0'
      TX_HSRC_RDY_N       : out std_logic;
      -- Destination is ready, active in '0'
      TX_HDST_RDY_N       : in std_logic;
      
      -- MI32 interface -------------------------------------------------------
      -- Clock signal
      MI_CLK              : in  std_logic;
      -- Reset signal
      MI_RESET            : in  std_logic;
      -- Input Data
      MI_DWR      	     : in  std_logic_vector(31 downto 0);
      -- Address
      MI_ADDR     	     : in  std_logic_vector(31 downto 0);
      -- Read Request
      MI_RD       	     : in  std_logic;
      -- Write Request
      MI_WR          	  : in  std_logic;
      -- Byte Enable
      MI_BE       	     : in  std_logic_vector(3  downto 0);
      -- Output Data
      MI_DRD      	     : out std_logic_vector(31 downto 0);
      -- Address Ready
      MI_ARDY     	     : out std_logic;
      -- Data Ready
      MI_DRDY     	     : out std_logic;

      -- Sampling unit interface ----------------------------------------------
      -- Clock for SAU component
      SAU_CLK             : out std_logic;
      -- Reset for SAU component
      SAU_RESET           : out std_logic;
      -- Request for sampling information
      SAU_REQ             : out std_logic;
      -- Accept incoming frame
      SAU_ACCEPT          : in std_logic;
      -- Sampling information valid
      SAU_DV              : in std_logic;
     
      -- Statistics for PACODAG -----------------------------------------------
      -- MAC address is not accepted
      CTRL_STAT_MAC_ERR    : out std_logic;
      -- Frame does not have minimal length
      CTRL_STAT_MINTU_ERR  : out std_logic;
      -- Frame is longer than maximal length
      CTRL_STAT_MTU_ERR    : out std_logic;
      -- Frame has bad CRC
      CTRL_STAT_CRC_ERR    : out std_logic;
      -- Incoming protocol error
      CTRL_STAT_FRAME_ERR  : out std_logic;
      -- Frame length
      CTRL_STAT_FRAME_LEN  : out std_logic_vector(15 downto 0);
      
      -- PACODAG interface ----------------------------------------------------
      -- Control CLK
      CTRL_CLK       : out std_logic;
      -- Control reset
      CTRL_RESET     : out std_logic;
      -- Control data
      CTRL_DATA      : in std_logic_vector(HDR_WIDTH-1 downto 0);
      -- Start of control data, active in '0'
      CTRL_SOP_N     : in std_logic;
      -- End of control data, active in '0'
      CTRL_EOP_N     : in std_logic;
      -- Position of the end of control data,
      -- valid only if CTRL_EOP_N is set to '0'.
      CTRL_REM       : in std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
      -- Asserted when the input signals from control data generator are valid
      CTRL_SRC_RDY_N : in std_logic;
      -- Asserted when data from control data generator will be accepted
      CTRL_DST_RDY_N : out std_logic;
      -- New frame is being received; prepare to generate new control data
      CTRL_SOP       : out std_logic;
      -- Statistics for PACODAG are valid
      CTRL_STAT_DV   : out std_logic;
      -- Control data generator is ready to receive new request
      CTRL_RDY       : in std_logic;

      -- State of the link interface ------------------------------------------
      -- Active when link is up
      LINK_UP             : out std_logic;
      -- Active when a packet is being received
      INCOMING_PACKET     : out std_logic
   );
end entity ibuf_cgmii;

