--
-- crc32_gen_start.vhd: 32-bit CRC module processing generic number of bits in parallel
-- Copyright (C) 2011 CESNET
-- Author(s): Lukas Kekely <kekely@cesnet.cz> 
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;

entity crc32_gen_start is
   generic(
      DATA_WIDTH : integer := 64;   
      REG_BITMAP : integer := 0
   );
   port(
      DI    : in std_logic_vector(DATA_WIDTH-1 downto 0);
      DI_DV : in std_logic;
      EOP   : in std_logic;
      SOP   : in std_logic;
      MASK  : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      START : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      CLK   : in std_logic;
      RESET : in std_logic;
      CRC   : out std_logic_vector(31 downto 0);
      DO_DV : out std_logic
   );
end entity crc32_gen_start;

architecture crc32_gen_arch of crc32_gen_start is
   constant MW    : integer := log2(DATA_WIDTH/8); 
   
   signal crc_reg, crc_end, crc_start: std_logic_vector(DATA_WIDTH-1 downto 0) := (others => '0');
   signal crc_reg_input: std_logic_vector(DATA_WIDTH-1 downto 0);
   signal deop, dsop : std_logic := '1';
   signal reg_low, reg_low_data, crctab_do, crctab_tree_do : std_logic_vector(31 downto 0);
   signal mx_di, do_reg : std_logic_vector(31 downto 0);

   signal reg_mask, reg_start : std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0) := (others => '0');
   signal reg_di : std_logic_vector(DATA_WIDTH-1 downto 0);
   signal reg_di_input : std_logic_vector(DATA_WIDTH-1 downto 0);
   signal reg_eop_in : std_logic;
   signal reg_mask_in : std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0) := (others => '0');
   signal reg_di_dv : std_logic;
   
   signal tree_sel    : std_logic;
   signal tree_vld    : std_logic;
    
begin 
   assert DATA_WIDTH = 64 or DATA_WIDTH = 128 or DATA_WIDTH = 256 or DATA_WIDTH = 512 report "CRC32: Wrong DATA_WIDTH set! DATA_WIDTH must be 64, 128, 256 or 512." severity error;
   
   crc32_gen_tab_instance: entity work.crc32_fast_tab
      generic map(
         DATA_WIDTH => DATA_WIDTH
      )
      port map(
         DI   => crc_start,
         DO   => crctab_do
      );
      
   crc32_gen_tab_tree_instance: entity work.crc32_gen_tab_tree
      generic map(
         DATA_WIDTH => DATA_WIDTH,
         REG_BITMAP => REG_BITMAP
      )
      port map(
         CLK   => CLK,
         DI    => crc_end,
         DI_DV => deop,
         MASK  => reg_mask,
         DO    => crctab_tree_do,
         DO_DV => tree_vld
      );
   
   deop_reg : process(CLK)
   begin
      if CLK = '1' AND CLK'event then
         if RESET = '1' then
            deop      <= '0';
         else
            deop      <= EOP and DI_DV;
         end if;
      end if;
   end process;
   
   in_reg : process(CLK)
   begin
      if CLK = '1' AND CLK'event then
         if DI_DV='1' then
            crc_reg   <= crc_reg_input;
            dsop      <= SOP;
            reg_mask  <= MASK;
            reg_start <= START;
         end if;
      end if;
   end process;

   out_reg : process(CLK)
   begin
      if CLK = '1' AND CLK'event then
         if RESET = '1' then
            DO_DV  <= '0';
         else
            DO_DV  <= tree_vld;
         end if;
      end if;
   end process;
   process(CLK)
   begin
      if CLK = '1' AND CLK'event then
         if RESET = '1' then
            do_reg <= (others => '0');
         else
            do_reg <= crctab_tree_do;
         end if;
      end if;
   end process;

   -- mx_di multiplexor - handles special situation when MASK > DATA_WIDTH/8 - 4
   mx_di <=     X"00" & DI(23 downto 0) when MASK = conv_std_logic_vector(DATA_WIDTH/8-3,MW) else
              X"0000" & DI(15 downto 0) when MASK = conv_std_logic_vector(DATA_WIDTH/8-2,MW) else
            X"000000" & DI( 7 downto 0) when MASK = conv_std_logic_vector(DATA_WIDTH/8-1,MW) else
            DI(31 downto 0);

   reg_low <= crctab_do XOR mx_di(31 downto 0) when SOP='0' or START /= (MW-1 downto 0 => '1') else NOT mx_di(31 downto 0);
   crc_reg_input <= DI(DATA_WIDTH-1 downto 32) & reg_low;
   
   starter512 : if DATA_WIDTH = 512 generate
      process(crc_reg,dsop,reg_start)
      begin
         crc_start <= crc_reg;
         if (dsop = '1') then
            case reg_start(5 downto 3) is
               when "110" => crc_start <= crc_reg(DATA_WIDTH-1 downto 64)  & (63  downto 32 => '0') & X"EFE4D0AF";
               when "101" => crc_start <= crc_reg(DATA_WIDTH-1 downto 128) & (127 downto 32 => '0') & X"38a70E28";
               when "100" => crc_start <= crc_reg(DATA_WIDTH-1 downto 192) & (191 downto 32 => '0') & X"B216BC71";
               when "011" => crc_start <= crc_reg(DATA_WIDTH-1 downto 256) & (255 downto 32 => '0') & X"CE6670b0";
               when "010" => crc_start <= crc_reg(DATA_WIDTH-1 downto 320) & (319 downto 32 => '0') & X"9DBA6447";
               when "001" => crc_start <= crc_reg(DATA_WIDTH-1 downto 384) & (383 downto 32 => '0') & X"C1F58982";
               when "000" => crc_start <= crc_reg(DATA_WIDTH-1 downto 448) & (447 downto 32 => '0') & X"CAC1238E";
               when others => null; 
            end case;         
         end if;
      end process;
   end generate;
   starter256 : if DATA_WIDTH = 256 generate
      process(crc_reg,dsop,reg_start)
      begin
         crc_start <= crc_reg;
         if (dsop = '1') then
            case reg_start(4 downto 3) is
               when "10" => crc_start <= crc_reg(DATA_WIDTH-1 downto 64)  & (63  downto 32 => '0') & X"EFE4D0AF";
               when "01" => crc_start <= crc_reg(DATA_WIDTH-1 downto 128) & (127 downto 32 => '0') & X"38a70E28";
               when "00" => crc_start <= crc_reg(DATA_WIDTH-1 downto 192) & (191 downto 32 => '0') & X"B216BC71";
               when others => null; 
            end case;         
         end if;
      end process;
   end generate;
   starter128 : if DATA_WIDTH = 128 generate
      process(crc_reg,dsop,reg_start)
      begin
         crc_start <= crc_reg;
         if (dsop = '1') then
            case reg_start(3 downto 3) is
               when "0" => crc_start <= crc_reg(DATA_WIDTH-1 downto 64)  & (63  downto 32 => '0') & X"EFE4D0AF";
               when others => null; 
            end case;         
         end if;
      end process;
   end generate;
   starter64 : if DATA_WIDTH = 64 generate
      crc_start <= crc_reg;
   end generate;
   
   
   crc_end   <= crc_start when dsop='1' and deop='1' and reg_mask<=reg_start else crc_reg;
                
   CRC <= NOT (do_reg(7 downto 0) & do_reg(15 downto 8) & do_reg(23 downto 16) & do_reg(31 downto 24));

end architecture;
