-- buf_fsm.vhd: FSM for Buf module of CGMII IBUF
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                            Entity declaration
-- ----------------------------------------------------------------------------

entity fsm is
   generic(
      -- Synthesise design without resets, default values of registers are set
      -- by initialization of signals
      RESET_BY_INIT       : boolean := false
   );
   port (
      -- Clock signal
      CLK                     : in std_logic;
      -- Reset signal
      RESET                   : in std_logic;

      -- IBUF is enabled
      IBUF_EN                 : in std_logic;

      -- Start of incoming packet
      RX_SOP                  : in std_logic;
      -- Position of the start of incoming packet
      -- valid only if RX_SOP is set to '1'
      RX_SOP_POS              : in std_logic_vector(2 downto 0);
      -- End of incoming packet
      RX_EOP                  : in std_logic;
      -- Position of the end of incoming packet
      -- valid only if RX_EOP is set to '1'
      RX_EOP_POS              : in std_logic_vector(5 downto 0);
      -- Discard the packet because of some error
      DISCARD                 : in std_logic;

      -- Data FIFO is full
      DFIFO_FULL              : in std_logic;
      -- Header FIFO is full
      HFIFO_FULL              : in std_logic;
      -- PACODAG is ready
      CTRL_RDY                : in std_logic;

      -- Release actual packet from Data FIFO
      DFIFO_RELEASE           : out std_logic;
      -- DFIFO overflow occured
      DFIFO_OVF               : out std_logic;

      -- Request to PACODAG component
      CTRL_SOP                : out std_logic;
      -- Statistics are valid - start generate control data
      CTRL_STAT_DV            : out std_logic;
      -- PACODAG overflow occured
      CTRL_OVF                : out std_logic;

      -- Status for debug purposes
      STATUS                  : out std_logic_vector(1 downto 0);

      -- Active once per each received frame
      FRAME_RECEIVED          : out std_logic;
      -- Active once per each discarded frame
      FRAME_DISCARDED         : out std_logic;
      -- Active once per each frame discarded due to buffer overflow
      FRAME_DISCARDED_BUF_OVF : out std_logic
   );
end entity fsm;


-- ----------------------------------------------------------------------------
--                               Architecture
-- ----------------------------------------------------------------------------

architecture fsm_arch of fsm is

   -- Type definition ---------------------------------------------------------
   type fsm_states is (st_wait, st_packet, st_error);


   -- Attribute declaration ---------------------------------------------------
   attribute SAFE_FSM : boolean;
   attribute SAFE_FSM of fsm_states : type is true;


   -- Signal declaration ------------------------------------------------------

   -- FSM state signals
   signal curr_state          : fsm_states := st_wait;
   signal next_state          : fsm_states;

   -- Auxiliary processing flags
   signal sop_pos_lt_eop_pos  : std_logic;
   signal sop_eop             : std_logic;
   signal eop_sop             : std_logic;
   signal sop                 : std_logic;
   signal eop                 : std_logic;


begin

   -- Auxiliary frame flags ---------------------------------------------------

   sop_pos_lt_eop_pos <= '1' when ((RX_SOP_POS & "000") < RX_EOP_POS) else '0'; 

   sop_eop <= '1' when (RX_SOP = '1') and (RX_EOP = '1') and 
                       (sop_pos_lt_eop_pos = '1') else '0';
   
   eop_sop <= '1' when (RX_SOP = '1') and (RX_EOP = '1') and 
                       (sop_pos_lt_eop_pos = '0') else '0';

   sop <= RX_SOP;
        
   eop <= RX_EOP;


   -- -------------------------------------------------------------------------
   --                              FSM description
   -- -------------------------------------------------------------------------
   
   -- Current state logic -----------------------------------------------------
   current_state_logic : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            curr_state <= st_wait;
         else
            curr_state <= next_state;
         end if;
      end if;
   end process current_state_logic;


   -- Next state logic --------------------------------------------------------
   next_state_logic: process(curr_state, IBUF_EN, DISCARD, DFIFO_FULL, sop, eop, sop_eop, eop_sop)
   begin

      case (curr_state) is

         -- st_wait -----------------------------------------------------------
         when st_wait =>
            next_state <= st_wait; 

            if (IBUF_EN = '1') then
               if (sop_eop = '1') then
                  next_state <= st_wait; 
               elsif (sop = '1') then
                  if (DFIFO_FULL = '1') then
                     next_state <= st_error;
                  else
                     next_state <= st_packet;
                  end if;
               end if;
            end if;


         -- st_packet ---------------------------------------------------------
         when st_packet =>
            next_state <= st_packet;

            if (IBUF_EN = '1') then
               if (eop_sop = '1') then
                  if (DFIFO_FULL = '1') then
                     next_state <= st_error;
                  else
                     next_state <= st_packet;
                  end if;
               elsif (eop = '1') then
                  next_state <= st_wait;
               elsif (DFIFO_FULL = '1') then
                  next_state <= st_error;
               end if;
            else
               next_state <= st_wait;
            end if;

         -- st_error ----------------------------------------------------------
         when st_error =>
            next_state <= st_error;

            if (IBUF_EN = '1') then
               if (eop_sop = '1') then
                  if (DFIFO_FULL = '0') then
                     next_state <= st_packet;
                  else
                     next_state <= st_error;
                  end if;
               elsif (eop = '1') then
                  next_state <= st_wait;
               end if;
            else
               next_state <= st_wait;
            end if;

         -- others ------------------------------------------------------------
         when others =>
            next_state <= st_wait;

      end case;

   end process next_state_logic;


   -- Output logic ------------------------------------------------------------
   output_logic: process(curr_state, IBUF_EN, DISCARD, DFIFO_FULL, HFIFO_FULL, CTRL_RDY, sop, eop, sop_eop, eop_sop)
   begin
            
      DFIFO_OVF      <= '0';

      CTRL_SOP       <= '0';
      CTRL_STAT_DV   <= '0';
      CTRL_OVF       <= '0';

      FRAME_RECEIVED          <= '0';
      FRAME_DISCARDED         <= '0';
      FRAME_DISCARDED_BUF_OVF <= '0';   


      case (curr_state) is

         -- st_wait -----------------------------------------------------------
         when st_wait =>
            STATUS         <= "00";
            DFIFO_RELEASE  <= '1';   

            if (IBUF_EN = '1') then
               if (sop_eop = '1') then
                  if (DISCARD = '1') then
                     FRAME_DISCARDED <= '1';
                  elsif (DFIFO_FULL = '1') then
                     DFIFO_OVF <= '1';
                     FRAME_DISCARDED         <= '1';
                     FRAME_DISCARDED_BUF_OVF <= '1';   
                  elsif (CTRL_RDY = '0') then
                     CTRL_OVF <= '1';
                     FRAME_DISCARDED         <= '1';
                     FRAME_DISCARDED_BUF_OVF <= '1'; 
                  elsif (HFIFO_FULL = '1') then
                     FRAME_DISCARDED         <= '1';
                     FRAME_DISCARDED_BUF_OVF <= '1'; 
                  else
                     DFIFO_RELEASE <= '0'; 
                     CTRL_SOP <= '1';
                     CTRL_STAT_DV <= '1';
                     FRAME_RECEIVED <= '1';
                  end if;
               elsif (sop = '1') then
                  if (DISCARD = '0' and DFIFO_FULL = '0') then
                     DFIFO_RELEASE <= '0'; 
                  end if;
               end if;
            end if;


         -- st_packet ---------------------------------------------------------
         when st_packet =>
            STATUS         <= "01";
            DFIFO_RELEASE  <= '0';

            if (IBUF_EN = '1') then
               if (eop = '1') then
                  if (DISCARD = '1') then
                     DFIFO_RELEASE <= '1'; 
                     FRAME_DISCARDED <= '1';
                  elsif (DFIFO_FULL = '1') then
                     DFIFO_RELEASE <= '1'; 
                     DFIFO_OVF <= '1';
                     FRAME_DISCARDED         <= '1';
                     FRAME_DISCARDED_BUF_OVF <= '1';   
                  elsif (CTRL_RDY = '0') then
                     DFIFO_RELEASE <= '1'; 
                     CTRL_OVF <= '1';
                     FRAME_DISCARDED         <= '1';
                     FRAME_DISCARDED_BUF_OVF <= '1'; 
                  elsif (HFIFO_FULL = '1') then
                     DFIFO_RELEASE <= '1'; 
                     FRAME_DISCARDED         <= '1';
                     FRAME_DISCARDED_BUF_OVF <= '1'; 
                  else
                     CTRL_SOP <= '1';
                     CTRL_STAT_DV <= '1';
                     FRAME_RECEIVED <= '1';
                  end if;
               end if;
            else
               DFIFO_RELEASE  <= '1';
            end if;


         -- st_error ----------------------------------------------------------
         when st_error =>
            STATUS         <= "11";
            DFIFO_RELEASE  <= '1'; 

            if (IBUF_EN = '1') then
               if (eop = '1') then
                  DFIFO_OVF <= not DISCARD; 
                  FRAME_DISCARDED         <= '1';
                  FRAME_DISCARDED_BUF_OVF <= not DISCARD;                           
               end if;
            end if;

         -- others ------------------------------------------------------------
         when others =>
            STATUS         <= "00";
            DFIFO_RELEASE  <= '0';

      end case;

   end process output_logic;


end architecture fsm_arch;
