#include <limits.h>
#include <assert.h>
#include <math.h>

// k! (mod p)
int factorial_mod(int k, int p)
{
	int m = 1;

	for(int i = 1; i <= k; i++)
	{
		m *= i;
		m %= p;
	}

	return m;
}

// b^k (mod p)
int power_mod(int b, int k, int p)
{
	int m = 1;

	for(int i = 1; i <= k; i++)
	{
		m *= b;
		m %= p;
	}

	return m;
}

// b^p (mod p)
int power_p_mod(int b, int p)
{
	int m = 1;

	for(int s = sizeof(p)*CHAR_BIT-1; s >= 0; s--)
	{
		m *= m;
		if( (p>>s)&1 )
		{
			m *= b;
		}
		m %= p;
	}

	return m;
}

// b^k = g (mod N), smallest k > 0
int dlog(int p, int b, int g)
{
	// b^k
	int m = 1;

	int k = 0;

	do
	{
		// b^k = b^k * b = b^(k+1)
		m *= b;
		m %= p;

		// k = k + 1
		k++;

		// b^k == g
		if( g == m )
			return k;
	}
	while( k < p );

	// fallback
	return 0;
}

// lambda
int dlog_max(int p, int g)
{
	int max = 0;

	for(int b = 1; b <= p; b++)
	{
		int t = dlog(p, b, g);

		if(t > max)
			max = t;
	}

	return max;
}

// psi
int dlog_sum(int p, int g)
{
	int sum = 0;

	for(int b = 1; b <= p; b++)
	{
		int t = dlog(p, b, g);

		if(t > 0)
			sum++;
	}

	return sum;
}

int ceil_sqrt(int x)
{
	return (int)ceil(sqrt((double)x));
}

int is_prime_trial_division(int p)
{
	assert( p >= 0 );

	if( p < 2 )
		return 0;

	if( p == 2 )
		// prime
		return 1;

	const int sqrt_p = ceil_sqrt(p);

	for(int factor = 2; factor <= sqrt_p; factor++)
	{
		if( p % factor == 0 )
			// composite
			return 0;
	}

	// prime
	return 1;
}

// root in half-totient tree
int get_tree_root(int p)
{
	int t = p;

	do
	{
		t = dlog_sum(t, 1)/2;
	}
	while(t > 2);

	return t;
}

// p^k
int power(int p, int k)
{
	int m = 1;

	for(int i = 1; i <= k; i++)
	{
		m *= p;
	}

	return m;
}

// x % p
int mod(int x, int p)
{
	// init
	int m = 0;

	for(int s = sizeof(x)*CHAR_BIT-1; s >= 0; s--)
	{
		// m = 2*m + x(s)
		m *= 2;
		m += (x>>s) & 1;

		// if( m >= p ): m -= p
		if( m >= p )
			m -= p;
	}

	return m;
}

int popcnt(
	unsigned x
)
{
	x -= x >> 1 & (unsigned)~(unsigned)0/3;
	x = (x & (unsigned)~(unsigned)0/15*3) + (x >> 2 & (unsigned)~(unsigned)0/15*3);
	x = (x + (x >> 4)) & (unsigned)~(unsigned)0/255*15;
	return (x * ((unsigned)~(unsigned)0/255)) >> (sizeof(unsigned) - 1) * CHAR_BIT;
}

int is_pow2(int x)
{
	return (x & (x - 1)) == 0;
}

int pow2_ceil_log2(
	int x
)
{
	assert( x > 0 );

	x--;

	unsigned shift = 1;

	while(shift < sizeof(int) * CHAR_BIT)
	{
		x |= x >> shift;
		shift <<= 1;
	}

	x++;

	return x;
}

int ceil_log2(
	int x
)
{
	return popcnt(pow2_ceil_log2(x) - 1);
}

int floor_log2(int x)
{
	return ceil_log2(x+1) - 1;
}
