/*****************************************************\
| File: packet_capture.c                              | 
| Author: Jan Drazil (xdrazi00)                       |
| Desc: Utility to capture data from uProbe exporter  |
|                                                     |
| Copyright (c) 2014 Brno University of Technology    |
\*****************************************************/

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdint.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <linux/if_packet.h>
#include <net/ethernet.h> /* the L2 protocols */
#include <netdb.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <time.h>
#include <signal.h>

#include <pcap/pcap.h>

#include "packet_capture.h"

/**
 * Recv TCP connection from uProbe
 * port - listening port
 * Return socket ready for listening
 **/
int tcp_connection(char *port);
int udp_connection(char *port);

/**
 * Prepare pcap_handle for receiven from hardware uProbe export
 * interface - listening interface
 * Return pcap_handle of capturing interface
 **/
pcap_t *direct_connection(char *interface);

/**
 * Capture packets form export, parse INI3 and save in pcap
 * socket - socket where comes captured packets
 * output_file - ouptut pcap file open in write and binary mode
 * Return 0 if connection id normaly closed
 **/
int capture_Net(int socket);

/**
 * Capture packets form hw export, parse INI3MAC and save in pcap
 * pcap - pcapt_handle with prepared promiscuid interface
 * output_file - ouptut pcap file open in write and binary mode
 * Return 0 if connection id normaly closed
 **/
int capture_DC(pcap_t *pcap);

int set_timer(unsigned sec);
int packet_write(struct pcap_pkthdr *h, const u_char *packet);

int verbose = 0;
int stop = 0;
uint64_t ring_size = 0;
int alarm_proceed = 0;
int skip_ini3 = 0;

char filename[1024] = "dump";
char filename_act[1050];
char filename_dump[1050];

pcap_t *pcap_handle = NULL;
pcap_dumper_t *dumper = NULL;

void app_exit(int sig, siginfo_t *si, void *uc)
{
	VERBOSE("Exiting...\n");
	stop = 1;
}

void timeout_handler(int sig, siginfo_t *si, void *uc)
{
	VERBOSE("Timer expired\n");
	alarm_proceed = 1;
}

int main(int argc, char **argv)
{

	struct sigaction act;
	memset(&act, 0, sizeof(struct sigaction));
	act.sa_sigaction = app_exit;
	act.sa_flags = SA_SIGINFO;
	sigemptyset(&act.sa_mask);
	sigaction(SIGINT, &act, NULL);

	int print_help = 0;
	int direct_export = -1;
	char source[128];
	int opt;
	int udp = 0;
	
	uint64_t timeout = 0;


	while ((opt = getopt(argc, argv, "vmuf:d:p:r:t:h")) != -1) {
		switch(opt) {
			case 'd':
				if(direct_export == 0) {
					fprintf(stderr, "Parametr d and p cannot be used at the same time!\n");
					exit(EXIT_FAILURE);
				}
				direct_export = 1;
				if(strlen(optarg) > 127) {
					fprintf(stderr, "Interface name can be long only 127 symbols.\n");
					exit(EXIT_FAILURE);
				}
				strcpy(source, optarg);
				break;
			case 'p':
				if(direct_export == 1) {
					fprintf(stderr, "Parametr d and p cannot be used at the same time!\n");
					exit(EXIT_FAILURE);
				}
				direct_export = 0;
				if(strlen(optarg) > 5) {
					fprintf(stderr, "Port can be 5 chars long only!\n");
					exit(EXIT_FAILURE);
				}
				strcpy(source, optarg);
				break;
			case 'v':
				#ifdef ALLOW_VERBOSE
					verbose = 1;
				#else
					fprintf(stderr, "Verbose mode must be enabled at compilation time!\n");
				#endif
				break;
			case'f':
				if(strlen(optarg) > 1000) {
					fprintf(stderr, "Output file can be long only 1000 symbols.\n");
					exit(EXIT_FAILURE);
				}
				strcpy(filename, optarg);
				break;
			case 'r':
				ring_size = atoi(optarg);
				if(ring_size == 0) {
					fprintf(stderr, "Ring size must be number bigger than zero!\n");
					exit(EXIT_FAILURE);
				}
				ring_size *= 1024*1024; //Convert to MB
				break;
			case 't':
				timeout = atoi(optarg);
				if(timeout == 0) {
					fprintf(stderr, "Timeout must be number bigger than zero!\n");
					exit(EXIT_FAILURE);
				}
				break;
			case 'u':
				udp = 1;
				break;
			case 'm':
				skip_ini3 = 1;
				break;
			case 'h':
			default:
				print_help = 1;
		}
	}

	if(direct_export < 0) {
		fprintf(stderr, "Expected parameter p or q!\n");
		print_help = 1;
	}
	
	if(timeout > 0 && ring_size > 0) {
		fprintf(stderr, "Is not possible use parameters -r and -t at the same time!\n");
		print_help = 1;
	}

	if(print_help || argc == 1) {
		printf("Usage: %s [-h] "
		#ifdef ALLOW_VERBOSE
		"[-v] "
		#endif
		"[-f filename] [-u] [-m] {-d interface | -p port}\n", argv[0]);
		
		printf("\t-h             - Show this help message and exit\n");
		
		#ifdef ALLOW_VERBOSE
		printf("\t-v             - Enable verbose mode\n");
		#endif
		
		printf("\t-u             - Activate listening on UDP port instead of TCP\n"
		       "\t                 If -d is set then this option has no effect\n");
		
		printf("\t-m             - Create regular pcap file without INI3 headers\n"
		       "\t                 If -d is set then this option has no effect\n");
		
		printf("\t-p port        - Save communication from uProbe when export\n"
		       "\t                 is set to HDD export.\n"
		       "\t                 Connection from uProbe is expected on given TCP port\n"
		       "\t                 If -u is set then will be used UDP port\n");
		       
		printf("\t-d interface   - Save communication from uProbe when export\n"
		       "\t                 is set to Direct export.\n"
		       "\t                 Captured packets are expected on given interface.\n");
		
		printf("\t-t timeout     - Set timeout to specified number of second.\n"
		       "\t                 Defaults to 0 (infinity).\n");
		
		printf("\t-r size        - Create 2 output files with max size 'size'MB.\n"
		       "\t                 File with suffix _act is currentlz used for saving.\n"
		       "\t                 File with suffix _dump is ready to download.\n");
		
		printf("\t-f filename    - Set output filename (default dump).\n"
		       "\t                 Extension pcap will be added automatically\n");
		exit(EXIT_SUCCESS);
	}
	
	int linktype = DLT_EN10MB;
	
	if(!skip_ini3 && !direct_export)
		linktype = DLT_USER4;
	else if(direct_export)
		linktype = DLT_USER3;
	
	pcap_handle = pcap_open_dead(linktype, 64*1024);
	
	if(ring_size == 0) {
		if(timeout > 0) {
			char filename_ts[300]; 		
			time_t t = time(NULL);
			struct tm tm = *localtime(&t);		
			sprintf(filename_ts, "%s_%04d%02d%02d_%02d%02d%02d.pcap", filename, tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);
			VERBOSE("Opening file %s\n", filename_ts);
			dumper = pcap_dump_open(pcap_handle, filename_ts);
		} else {
			strcpy(filename_act, filename);
			strcpy(filename_act+strlen(filename), ".pcap");
			VERBOSE("Opening file %s\n", filename_act);
			dumper = pcap_dump_open(pcap_handle, filename_act);
		}
	} else {
		strcpy(filename_act, filename);
		strcpy(filename_act+strlen(filename), "_act.pcap");
		strcpy(filename_dump, filename);
		strcpy(filename_dump+strlen(filename), "_dump.pcap");
		VERBOSE("Opening file %s\n", filename_act);
		dumper = pcap_dump_open(pcap_handle, filename_act);
	}
	
	if(dumper == NULL) {
		fprintf(stderr, "%s\n", pcap_geterr(pcap_handle));
		exit(EXIT_FAILURE);
	}

	if(direct_export == 0) {
		int socket;
		if(udp) {
			VERBOSE("Activating capturing on UDP\n");
			socket = udp_connection(source);
		} else {
			VERBOSE("Activating capturing on TCP\n");
			socket = tcp_connection(source);
		}
		if(timeout > 0)
			set_timer(timeout);
		capture_Net(socket);
		close(socket);
	} else {
		VERBOSE("Activating direct connect capturing\n");
		pcap_t *handle = direct_connection(source);
		capture_DC(handle);
		pcap_close(handle);
	}
	
	pcap_dump_close(dumper);
	pcap_close(pcap_handle);

	return 0;
}

int set_timer(unsigned sec)
{
	timer_t timerid;
	struct sigevent sev;
	struct itimerspec its;
	struct sigaction sa;
	
	/* Establish handler for timer signal */
	memset(&sa, 0, sizeof(struct sigaction));
	sa.sa_sigaction = timeout_handler;
	sa.sa_flags = SA_SIGINFO;
	sigemptyset(&sa.sa_mask);
	if (sigaction(SIGALRM, &sa, NULL) == -1)
	   return -1;
	
	/* Create the timer */
	sev.sigev_notify = SIGEV_SIGNAL;
	sev.sigev_signo = SIGALRM;
	if (timer_create(CLOCK_REALTIME, &sev, &timerid) == -1)
	   return -1;

	/* Start the timer */
	its.it_value.tv_sec = sec;
	its.it_value.tv_nsec = 0;
	its.it_interval.tv_sec = sec;
	its.it_interval.tv_nsec = 0;

	VERBOSE("Activating timer - interval %us\n", sec);
	if (timer_settime(timerid, 0, &its, NULL) == -1)
		return -1;
	return 0;
}

int packet_write(struct pcap_pkthdr *h, const u_char *packet)
{
	static uint64_t written_size = 0;
	
	written_size += h->caplen + PCAP_HEADER_ITEM_SIZE;
	if(ring_size > 0 && written_size > ring_size) {
		pcap_dump_close(dumper);
		VERBOSE("Moving %s to %s\n", filename_act, filename_dump);
		rename(filename_act, filename_dump);
		VERBOSE("Opening %s\n", filename_act);
		
		dumper = pcap_dump_open(pcap_handle, filename_act);
		if(dumper == NULL) {
			fprintf(stderr, "%s\n", pcap_geterr(pcap_handle));
			exit(EXIT_FAILURE);
		}
		
		written_size = h->caplen + PCAP_HEADER_ITEM_SIZE;
	} else if(alarm_proceed) {
		char filename_ts[300]; 
		time_t t = time(NULL);
		struct tm tm = *localtime(&t);
		sprintf(filename_ts, "%s_%04d%02d%02d_%02d%02d%02d.pcap", filename, tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);
		pcap_dump_close(dumper);
		VERBOSE("Opening %s\n", filename_ts);
		
		dumper = pcap_dump_open(pcap_handle, filename_ts);
		if(dumper == NULL) {
			fprintf(stderr, "%s\n", pcap_geterr(pcap_handle));
			exit(EXIT_FAILURE);
		}
		
		alarm_proceed = 0;
	}
	
	VERBOSE("Writing packet - length: %dB\n", h->caplen);
	pcap_dump((u_char *)dumper, h, packet);
	
	return 0;
}

int udp_connection(char *port)
{
	int sockfd;
	struct addrinfo hints, *servinfo, *p;
	int rv;

	memset(&hints, 0, sizeof hints);
	hints.ai_family = AF_INET6; // set to AF_INET to force IPv4
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_flags = AI_PASSIVE; // use my IP

	if ((rv = getaddrinfo(NULL, port, &hints, &servinfo)) != 0) {
		fprintf(stderr, "getaddrinfo: %s\n", gai_strerror(rv));
		return -1;
	}

	// loop through all the results and bind to the first we can
	for(p = servinfo; p != NULL; p = p->ai_next) {
			if ((sockfd = socket(p->ai_family, p->ai_socktype,
				p->ai_protocol)) == -1) {
			continue;
		}

		if (bind(sockfd, p->ai_addr, p->ai_addrlen) == -1) {
			close(sockfd);
			continue;
		}

		break;
	}

	if (p == NULL) {
		fprintf(stderr, "ERROR: Failed to bind\n");
		return -1;
	}

	freeaddrinfo(servinfo);

	int optval = 1;
	setsockopt(sockfd, IPPROTO_IPV6, IPV6_V6ONLY, &optval, sizeof(int));

	return sockfd;
}

/**
 * Recv TCP connection from uProbe
 * port - listening port
 * Return socket ready for listening
 **/
int tcp_connection(char *port)
{
	int sockfd, new_fd;  // listen on sock_fd, new connection on new_fd
	struct addrinfo hints, *servinfo, *p;
	struct sockaddr_storage their_addr; // connector's address information
	socklen_t sin_size;
	int reuse_addr=1;
	int rv;

	memset(&hints, 0, sizeof hints);
	hints.ai_family = AF_INET6;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_flags = AI_PASSIVE; // use my IP

	if ((rv = getaddrinfo(NULL, port, &hints, &servinfo)) != 0) {
		fprintf(stderr, "getaddrinfo: %s\n", gai_strerror(rv));
		return -1;
	}

	// loop through all the results and bind to the first we can
	for(p = servinfo; p != NULL; p = p->ai_next) {
		if ((sockfd = socket(p->ai_family, p->ai_socktype,
				p->ai_protocol)) == -1) {
			continue;
		}

		if (setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &reuse_addr,
				sizeof(int)) == -1) {
			return -1;
		}

		if (bind(sockfd, p->ai_addr, p->ai_addrlen) == -1) {
			close(sockfd);
			continue;
		}

		break;
	}

	if (p == NULL)  {
		fprintf(stderr, "ERROR: Failed to bind\n");
		return -1;
	}

	freeaddrinfo(servinfo); // all done with this structure

	if (listen(sockfd, 5) == -1) {
		fprintf(stderr, "ERROR: (listen) %s\n", strerror(errno));
		return -1;
	}

	VERBOSE("Waiting for connection...\n");

	sin_size = sizeof their_addr;
	new_fd = accept(sockfd, (struct sockaddr *)&their_addr, &sin_size);
	if (new_fd == -1) {
		fprintf(stderr, "ERROR (accept): %s\n", strerror(errno));
		return -1;
	}

	VERBOSE("New connection established.\n");
	close(sockfd);

	int optval = 1;
	setsockopt(new_fd, IPPROTO_IPV6, IPV6_V6ONLY, &optval, sizeof(int));
	
	return new_fd;
}

/**
 * Capture packets form export, parse INI3 and save in pcap
 * socket - socket where comes captured packets
 * output_file - ouptut pcap file open in write and binary mode
 * Return 0 if connection id normaly closed
 **/
int capture_Net(int socket)
{	
	u_char buff[BUFF_SIZE];
	u_char pcap_item[BUFF_SIZE];

	int buffer_last_read;
	int buffer_read = 0;
	int pcap_item_seized = 0;
	struct ini3_header *ini3;
	struct pcap_pkthdr pcap_h;

	int ini3_load_left = INI3_HEADER_SIZE;
	int data_load_left;

	int read_next = 1;
	
	u_char *packet_start = pcap_item;
	if(skip_ini3)
		packet_start += INI3_HEADER_SIZE;

	while(!stop) {

		if(read_next) {
			//Whole recv buffer was proccess, get new data
			buffer_last_read = recv(socket, buff, BUFF_SIZE, 0);
			if(buffer_last_read == -1) {
				if(errno != EINTR) {
					fprintf(stderr, "Read error: %s\n", strerror(errno));
					break;
				} else {
					continue;
				}

			} else if(buffer_last_read == 0) {
				VERBOSE("Connection closed.\n");
				return 0;
			}
			buffer_read = 0;
		}

		if(ini3_load_left) {
			//INI3 is load partially or not at all
			if(buffer_last_read-buffer_read >= ini3_load_left) {
				//All neccessary bytes are in actual recv buffer 
				
				//Copy all neccessary bytes of INI3 into buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, ini3_load_left);
				
				//Set pointer on position of unproccessed data in recv buffer
				buffer_read += ini3_load_left; 
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += ini3_load_left;
				
				//Mark INI3 as read
				ini3_load_left = 0;

				//Nasty "zero-copy" hack
				ini3 = (struct ini3_header*) pcap_item;
				
				//Prepare for loading captured packet
				data_load_left = ini3->blob_size;
			} else {
				//In recv buffer is not enought bytes for INI3
				
				//Copy all unproccessed bytes from recv buffer to buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, buffer_last_read-buffer_read);
				
				//Set count of bytes which are need for read rest of INI3
				ini3_load_left -= buffer_last_read-buffer_read;
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += buffer_last_read-buffer_read;
				
				//Mark recv buffer as fully proccessed
				read_next = 1;
				continue;
			}
		}

	   if(data_load_left) {
		   //Loading captured data
		   
			if(buffer_last_read-buffer_read == data_load_left) {
				//in unproccessed part of recv buff are only and exact captured packet bytes
				
				//Copy packet data into buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, data_load_left);
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += data_load_left;
				
				//Mark recv buffer as fully proccessed
				read_next = 1;				
			} else if(buffer_last_read-buffer_read > data_load_left) {
				//in unproccessed part of recv buff are exact captured packet bytes, but not only
				
				//Copy packet data into buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, data_load_left);
				
				//Set pointer on position of unproccessed data in recv buffer
				buffer_read += data_load_left;
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += data_load_left;
				
				//In next cycle DO NOT overwrite recv buffer
				read_next = 0;
			} else {
				//in unproccessed part of recv buff isn't whole captured packet
				
				//Copy packet data into buffer for active packet proccessing
				memcpy(pcap_item+pcap_item_seized, buff+buffer_read, buffer_last_read-buffer_read);
				
				//Set count of bytes which are need for read rest of packet
				data_load_left -= buffer_last_read-buffer_read;
				
				//Mark seized data in buffer for active packet proccessing
				pcap_item_seized += buffer_last_read-buffer_read;
				
				//Mark recv buffer as fully proccessed
				read_next = 1;
				continue;
			}
		}

		//Extraction meta-data from INI3 for save packet into pcap with valid params
		pcap_h.ts.tv_sec = ini3->timestamp_s;
		pcap_h.ts.tv_usec = ini3->timestamp_ns/1000;
		pcap_h.caplen = ini3->blob_size;
		pcap_h.len = ini3->blob_size;
		
		if(!skip_ini3) {
			pcap_h.caplen += INI3_HEADER_SIZE;
			pcap_h.len += INI3_HEADER_SIZE;
		}

		//Write packet into pcap
		packet_write(&pcap_h, packet_start);

		//Initialize for process of next packet
		ini3_load_left = INI3_HEADER_SIZE;
		pcap_item_seized = 0;

	}
	return 0;
}

/**
 * Prepare pcap_handle for receiven from hardware uProbe export
 * interface - listening interface
 * Return pcap_handle of capturing interface
 **/
pcap_t* direct_connection(char *interface)
{
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_t *handle = pcap_open_live(interface, BUFSIZ, 1, 0, errbuf);
	if(handle == NULL) {
		fprintf(stderr, "Open interface %s error: %s", interface, errbuf);
		return NULL;
	}

	if (pcap_datalink(handle) != DLT_EN10MB) {
		fprintf(stderr, "Device %s doesn't provide Ethernet headers - not supported\n", interface);
		return NULL;
	}

	return handle;
}

/**
 * Capture packets form hw export, parse INI3MAC and save in pcap
 * pcap - pcapt_handle with prepared promiscuid interface
 * output_file - ouptut pcap file open in write and binary mode
 * Return 0 if connection id normaly closed
 **/
int capture_DC(pcap_t *pcap) {
	struct pcap_pkthdr pkt_h;
	const u_char *packet;
	while(!stop) {
		packet = pcap_next(pcap, &pkt_h);

		//Timestamp extraction from INI3
		pkt_h.ts.tv_sec = *((uint32_t *)(packet+4));
		pkt_h.ts.tv_usec = *((uint32_t *)(packet+8))/1000;

		//Save paket into pcap
		packet_write(&pkt_h, packet);
	}
	return 0;

}
