/* packet-ini3.c
 * Routines for PROTONAME dissection
 * Copyright 201x, YOUR_NAME <YOUR_EMAIL_ADDRESS>
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"

#if 0
/* Include only as needed */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#endif

#include <glib.h>

#include <epan/packet.h>
#include <epan/expert.h>
#include <epan/prefs.h>

/* Forward declaration that is needed below if using the
 * proto_reg_handoff_ini3 function as a callback for when protocol
 * preferences get changed. */
void proto_reg_handoff_ini3(void);
void proto_register_ini3(void);

/* Callbacks for range_foreach */
static void range_add_tcp_port_callback(guint32 port);
static void range_add_udp_port_callback(guint32 port);
static void range_delete_tcp_port_callback(guint32 port);
static void range_delete_udp_port_callback(guint32 port);

/* Initialize the protocol and registered fields */
static int proto_ini3 = -1;
static int hf_ini3_bs = -1;
static int hf_ini3_if = -1;
static int hf_ini3_res = -1;
static int hf_ini3_ts = -1;
static int hf_ini3_rid0 = -1;
static int hf_ini3_rid1 = -1;

/* Global port preference */
static range_t *tcp_ports = NULL;
static range_t *udp_ports = NULL;

/* Initialize the subtree pointers */
static gint ett_ini3 = -1;

/* Handler for ethernet dissector */
static dissector_handle_t eth_handle; 

/* Handler for this protocol */
static dissector_handle_t ini3_handle;

/* Minimum length (in bytes) of the protocol data.
 * If data is received with fewer than this many bytes it is rejected by
 * the current dissector. */
#define INI3_MIN_LENGTH 16

/* Code to actually dissect the packets */
static int
dissect_ini3(tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree,
        void *data _U_)
{
    /* Set up structures needed to add the protocol subtree and manage it */
    proto_item *ti;
    proto_tree *ini3_tree;
    /* Other misc. local variables. */
    guint offset = 0;

    /* Buffer for BLOB from INI3 */
    tvbuff_t *next_tvb;

    /*** HEURISTICS ***/

    /* Check that the packet is long enough for it to belong to us. */
    if (tvb_length(tvb) < INI3_MIN_LENGTH)
        return 0;



    /*** PROTOCOL TREE ***/

    /* create display subtree for the protocol */
    ti = proto_tree_add_item(tree, proto_ini3, tvb, 0, 16, ENC_NA);

    ini3_tree = proto_item_add_subtree(ti, ett_ini3);

    /* Add an item to the subtree, see section 1.5 of README.dissector for more
     * information. */
    proto_tree_add_item(ini3_tree, hf_ini3_bs, tvb,
            offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
	proto_tree_add_item(ini3_tree, hf_ini3_if, tvb,
            offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
	proto_tree_add_item(ini3_tree, hf_ini3_res, tvb,
            offset, 1, ENC_LITTLE_ENDIAN);
    offset += 1;
	proto_tree_add_item(ini3_tree, hf_ini3_ts, tvb,
            offset, 8, ENC_TIME_TIMESPEC | ENC_LITTLE_ENDIAN);
    offset += 8;
	proto_tree_add_item(ini3_tree, hf_ini3_rid0, tvb,
            offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;
	proto_tree_add_item(ini3_tree, hf_ini3_rid1, tvb,
            offset, 2, ENC_LITTLE_ENDIAN);
    offset += 2;

    /*** CALL SUBDISSECTORS ***/
     next_tvb = tvb_new_subset_remaining(tvb, offset);
     call_dissector(eth_handle, next_tvb, pinfo, tree);
    
    /*** COLUMN DATA ***/
    col_clear(pinfo->cinfo,COL_PROTOCOL);
    col_set_str(pinfo->cinfo, COL_PROTOCOL, "INI3");
    
    
    /* Return the amount of data this dissector was able to dissect (which may
     * or may not be the total captured packet as we return here). */
    return tvb_length(tvb);
}

/* Register the protocol with Wireshark.*/
void
proto_register_ini3(void)
{
    module_t *ini3_module;
    
    /* Setup list of header fields  See Section 1.5 of README.dissector for
     * details. */
    static hf_register_info hf[] = {
        { &hf_ini3_bs,
            { "Blob size", "ini3.bs",
               FT_UINT16,  BASE_DEC, NULL, 0,
              NULL, HFILL }
        },
		{ &hf_ini3_if,
            { "Interface", "ini3.if",
               FT_UINT8,  BASE_DEC, NULL, 0,
              NULL, HFILL }
        },
		{ &hf_ini3_res,
            { "Reserved", "ini3.res",
               FT_UINT8,  BASE_HEX, NULL, 0,
              NULL, HFILL }
        },
		{ &hf_ini3_ts,
            { "Timestamp (unix)", "ini3.ts",
               FT_ABSOLUTE_TIME,  ABSOLUTE_TIME_UTC, NULL, 0,
              NULL, HFILL }
        },
		{ &hf_ini3_rid0,
            { "RID 0", "ini3.rid0",
               FT_UINT16,  BASE_HEX, NULL, 0,
              NULL, HFILL }
        },
		{ &hf_ini3_rid1,
            { "RID 1", "ini3.rid1",
               FT_UINT16,  BASE_HEX, NULL, 0,
              NULL, HFILL }
        }
    };

    /* Setup protocol subtree array */
    static gint *ett[] = {
        &ett_ini3
    };

    /* Create default range of ports for INI3 */
    range_convert_str(&tcp_ports, "21103,", 65535);
    range_convert_str(&udp_ports, "21171,", 65535);

    /* Register the protocol name and description */
    proto_ini3 = proto_register_protocol("Internal Network Interface 3",
            "INI3", "ini3");

    /* Required function calls to register the header fields and subtrees */
    proto_register_field_array(proto_ini3, hf, array_length(hf));
    proto_register_subtree_array(ett, array_length(ett));
    
    /* Register a preferences module */
    ini3_module = prefs_register_protocol(proto_ini3,
            proto_reg_handoff_ini3);

    /* Register a port preference */
    prefs_register_range_preference(ini3_module, "tcp.port", "INI3 TCP Ports", "TCP Ports range",
        &tcp_ports, 65535);
    prefs_register_range_preference(ini3_module, "udp.port", "INI3 UDP Ports", "UDP Ports range",
        &udp_ports, 65535);
}

 /*
 * This function is registered as a prefs callback (see
 * prefs_register_protocol above) this function is also called by Wireshark's
 * preferences manager whenever "Apply" or "OK" are pressed.
 */
void
proto_reg_handoff_ini3(void)
{
    static gboolean initialized = FALSE;
    static range_t *tcp_ports_bckup = NULL;
    static range_t *udp_ports_bckup = NULL;

	eth_handle = find_dissector("eth");
	
    if (!initialized) {
        /* Use new_create_dissector_handle() to indicate that
         * dissect_ini3() returns the number of bytes it dissected (or 0
         * if it thinks the packet does not belong to INI3).
         */
        ini3_handle = new_create_dissector_handle(dissect_ini3,
                proto_ini3);

        initialized = TRUE;

    } else {
        /* Unregister TCP and UDP ports connected with INI3 */
        range_foreach(tcp_ports_bckup, range_delete_tcp_port_callback);
        range_foreach(udp_ports_bckup, range_delete_udp_port_callback);        
    }
    /* Register TCP and UDP ports with INI3 */
    range_foreach(tcp_ports, range_add_tcp_port_callback);
    range_foreach(udp_ports, range_add_udp_port_callback);        
    
    g_free(tcp_ports_bckup);
    g_free(udp_ports_bckup);
    
    tcp_ports_bckup = range_copy(tcp_ports);
    udp_ports_bckup = range_copy(udp_ports);
}

/* Helper functions for changing assigned TCP/UDP ports to INI3 */
static void
range_add_tcp_port_callback(guint32 port)
{
    dissector_add_uint("tcp.port", port, ini3_handle);
}

static void
range_add_udp_port_callback(guint32 port)
{
    dissector_add_uint("udp.port", port, ini3_handle);
}

    static void
range_delete_tcp_port_callback(guint32 port)
{
    dissector_delete_uint("tcp.port", port, ini3_handle);
}

static void
range_delete_udp_port_callback(guint32 port)
{
    dissector_delete_uint("udp.port", port, ini3_handle);
}
/*
 * Editor modelines  -  http://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 4
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=4 tabstop=8 expandtab:
 * :indentSize=4:tabSize=8:noTabs=true:
 */
