﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;   

using MySql.Data.MySqlClient;
using System.Data;

using CommonBridge;

namespace DatabaseModule
{

    // 
    // an ordinary database class which communicate with database
    // 
    public class DatabaseModuleBaseClass
    {

        // ------------------------------------------------------------------------------
        //
        //     TRANSLATOR MODULE
        //
        // ------------------------------------------------------------------------------
        // VARIABLE 
        internal TranslateModule.TranslateModuleClass Translator = null;

        /// <summary>
        /// Register translator to this module
        /// </summary>
        /// <param name="_Translator"></param>
        /// <returns></returns>
        public bool RegisterTranslator(TranslateModule.TranslateModuleClass _Translator)
        {
            Translator = _Translator;
            return true;
        }

        /// <summary>
        /// Wrapped translator function
        /// </summary>
        /// <param name="str">String for translation</param>
        /// <returns>translated text</returns>
        public string i18n(string str)
        {
            // default set output string
            string trans = "X:" + str;

            // if Translator exist, call i18n function
            if (Translator != null)
                trans = Translator.i18n(str);
            return trans;
        }


        // ------------------------------------------------------------------------------
        //
        //      VARIABLES
        //
        // -----------------------------------------------------------------------------
        internal MySqlConnection db_connection = null;
        //private MySqlDataReader db_reader = null;

        private MySqlCommand db_command = new MySqlCommand();         

        internal String dbsetup_server = null;
        private Int32  dbsetup_port = 0;
        private String dbsetup_username = null;
        private String dbsetup_pswhash = null;
        private String dbsetup_psw = null;
        internal String dbsetup_database = null;

        internal Object mLockReader = new Object();

        // LOG
        internal String ModuleName = "";
        public LoggerModule.LoggerClass LogUnit = new LoggerModule.LoggerClass();

        public enum EnumOrderByDirection { NONE, DESC, ASC };

        // ------------------------------------------------------------------------------
        //
        //      DELEGATES
        //
        // -----------------------------------------------------------------------------
        public delegate void ConnectionState_CallbackHandle();
        public event ConnectionState_CallbackHandle ConnectionState_Callback = null;


        // ------------------------------------------------------------------------------
        //
        //      FUNCTIONS
        //
        // -----------------------------------------------------------------------------

        /// <summary>
        /// 
        /// </summary>
        public virtual void Init()
        {
            
        }

        public DatabaseModuleBaseClass()
        {
            ModuleName = this.ToString();   
        }

        ~DatabaseModuleBaseClass()
        {
            if (LogUnit != null)
            {
                LogUnit.LogDebug(ModuleName, "Destructor");
                LogUnit = null;
            }
        }

        /// <summary>
        /// Load module configuration
        /// </summary>
        /// <param name="ini_filename"></param>
        /// <returns></returns>
        public bool LoadConfiguration(String ini_filename)
        {
            bool iniExist = false;
            INIFile ini = null;

            // try to open ini file (create new)
            try
            {
                ini = new INIFile(ini_filename, out iniExist);
            }
            catch (Exception exc)
            {
                System.Windows.Forms.MessageBox.Show(exc.Message, i18n("Exception") + "_" + this.ToString());
                return false;
            }

            if (iniExist)
            {
                try
                {
                    String str = "";
                    Boolean tf = true;

                    String SectionName = "DATABASE";

                    // get server name
                    dbsetup_server = ini.GetValue(SectionName, "Server", dbsetup_server);

                    // get port number
                    dbsetup_port = ini.GetValue(SectionName, "Port", dbsetup_port);

                    dbsetup_database = ini.GetValue(SectionName, "Database", dbsetup_database);
                    dbsetup_username = ini.GetValue(SectionName, "User", dbsetup_username);  

                    // get password                                    
                    str = ini.GetValue(SectionName, "Psw","");
                                      
                    // recognize, that password is crypted or encrypted or need to be crypted
                    bool change = false;
                    tf &= CommonBridge.CryptoClass.RecognizePassword(str, out dbsetup_psw, out dbsetup_pswhash, out change);

                    // change == true, password loaded with symbol # at the begining
                    if (change)
                    {
                        // set and save encrypted (hashed) password
                        ini.SetValue(SectionName, "Psw", dbsetup_pswhash);
                        ini.Flush();
                    } 

                    if (tf == false)
                    {
                        System.Windows.Forms.MessageBox.Show(i18n("Failed configuration") + ": " + this.ToString());
                        LogUnit.LogError(ModuleName, "Load configuration - wrong or missing value");
                        return false;
                    }
                }
                catch (Exception exc)
                {
                    System.Windows.Forms.MessageBox.Show(exc.ToString());

                    LogUnit.LogError(ModuleName, "Load configuration: " + exc.ToString());
                    return false;
                }
                
                LogUnit.LogDebug(ModuleName, "Load configuration - OK");
                return true;
            }
            else
            {
                return false;
            }
        }


        /// <summary>
        /// Database connection establishing.
        /// </summary>
        /// <returns></returns>
        public bool Connect()
        {
            if (CheckConnection() == false)
            {
                //db_button.Image = db_sync;
                try
                {
                    String connection_string = "";
                    connection_string += "Server=" + dbsetup_server + ";";
                    connection_string += "Port=" + dbsetup_port.ToString() + ";";
                    connection_string += "User Id=" + dbsetup_username + ";";
                    connection_string += "Password=" + dbsetup_psw + ";";
                    //connection_string += "Persist Security Info=True;";
                    connection_string += "Database=" + dbsetup_database + ";";
                    connection_string += "Convert Zero Datetime=True;Allow Zero Datetime=True;";   // without that throw exception: Unable to convert MySQL date/time value to System.DateTime

                    db_connection = new MySqlConnection(connection_string);
                    try
                    {

                        // try to open
                        db_connection.Open();

                        // check if dtb opened
                        if (CheckConnection())
                        {
                            Console.WriteLine("Connection Established"); 
                            db_command.Connection = db_connection;

                            // call callback
                            if (ConnectionState_Callback != null)
                                ConnectionState_Callback();

                            return true;
                        }
                        else 
                        {
                            Console.WriteLine("Connection Not opened");
                        }

                        return false;

                    }
                    catch (Exception e)
                    {
                        Console.WriteLine(e.ToString());
                        Console.WriteLine("Connection Failed");
                        //db_button.Image = db_disconnect;

                        return false;
                    }
                }
                catch (System.Data.SqlClient.SqlException sqlException)
                {
                    Console.WriteLine(sqlException.Message);
                    //db_button.Image = db_disconnect;
                    return false;
                }
            }
            else 
            { 
                return false;
            }
        }

        /// <summary>
        /// Disconnect from database
        /// </summary>
        public void Diconnect()
        {
            if (CheckConnection() == true)
            {  
                try
                {
                    db_connection.Close();
                    //db_button.Image = db_disconnect;

                    // check if dtb opened
                    if (!CheckConnection())
                    {
                        // call callback
                        if (ConnectionState_Callback != null)
                            ConnectionState_Callback();
                    }  
                }
                catch (Exception exc)
                {
                    Console.WriteLine(exc.Message);
                    //db_button.Image = db_disconnect;                    
                }
            }
        }


        /// <summary>
        /// Return true if connection is established and open. Elsewhere it returns false.
        /// </summary>
        public bool CheckConnection()
        {
            return (db_connection != null && db_connection.State.Equals(ConnectionState.Open));
        }
        
        /// <summary>
        /// Execute the query command and return result. Can be used for arbitrary query.
        /// </summary>
        ///  <param name="command"> 
        ///  Query to database.
        ///  </param>
        /// <returns> 
        /// Return results as MySqlDataReader.
        /// </returns>
        public MySqlDataReader GetQueryReader(String query)
        {
            MySqlDataReader db_read = null;
            if (!CheckConnection()) return null;
            if (query == null) return null;

            try
            {
                Console.WriteLine("SQL QUERY: " + query);
                MySqlCommand db_cmd = new MySqlCommand();
                db_cmd.Connection = db_connection;
                db_cmd.CommandText = query;
                db_read = db_cmd.ExecuteReader(); 
            }
            catch (MySqlException ex)
            {
                LogUnit.LogError(ModuleName, "GetQueryReader ERROR: " + query + ". Message: " + ex.ToString());
                return null;
            }

            return db_read;
        }

        /// <summary>
        /// Execute the query command which returns one integer value. Can be used for queries such as "SELECT COUNT FROM ...".
        /// </summary>
        ///  <param name="command"> 
        ///  Query to database.
        ///  </param>
        /// <returns> 
        /// Return integer value as result of the query.
        /// </returns>
        public int GetQueryIntValue(String command)
        {
            if (!CheckConnection()) return -1; 

            int return_value = -1;             
                              
                lock(mLockReader)
                {
                   MySqlDataReader reader = null;
                   try
                   {

                    // get data
                    reader = GetQueryReader(command);

                    if (reader == null)
                        return return_value;

                    if (reader.Read())
                    {
                        if (reader.FieldCount > 0)
                        {
                            try
                            {
                                // try to convert
                                return_value = Convert.ToInt32(reader[0]);
                            }
                            catch (Exception)
                            {

                            }
                        }
                    }
                    }
                   catch (MySqlException ex)
                   {
                       LogUnit.LogError(ModuleName, "GetQueryIntValue " + ex.ToString());
                   }
                   finally
                   {
                       reader.Close();
                   }
                }
            

            return return_value;
        }

        public Int32 GetRowsCount(String command)
        {
            if (!CheckConnection()) return -1;
            Int32 count = -1;

            lock (mLockReader) 
            {            
                MySqlDataReader reader = null; 
                try
                {
                    reader = GetQueryReader(command); 

                    if (reader != null)
                    {
                        if (reader.Read())
                        {
                            if (reader.FieldCount > 0)
                            {
                                count = reader.FieldCount;
                            }
                        }
                    }
                }
                catch(Exception exc)
                {

                }
                finally
                {
                    reader.Close();
                }
            }
            return count;
        }
        
        protected Int32 ExecuteNonQuery(String query)
        {
            if (!CheckConnection()) return -1;
            if (query == null) return -1;

            try
            {
                Console.WriteLine("SQL QUERY: " + query);
                MySqlCommand db_cmd = new MySqlCommand();
                db_cmd.Connection = db_connection;
                db_cmd.CommandText = query;
                return db_cmd.ExecuteNonQuery();

            }
            catch (MySqlException ex)
            {
                Console.WriteLine("ExecuteNonQuery Error: {0}", ex.ToString());
                return -1;
            }
        }

        public Boolean ExistTable(String table_name)
        {  
            String query = @"SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_NAME = '" + table_name + "'";
            Int32 count = GetQueryIntValue(query);
            return Convert.ToBoolean(count);
        }

        protected Boolean DropTable(String table_name)
        {
            String query = "DROP TABLE " + table_name + "";
            Int32 count = ExecuteNonQuery(query);
            return (count >= 0) ? true : false;
        }

        protected Boolean CreateTable(String query)
        {              
            Int32 count = ExecuteNonQuery(query);
            return (count >= 0 ) ? true : false;
        }

        protected Boolean ClearTable(String table_name, String where = null)
        {  
            String query = "DELETE FROM " + table_name + "";

            if (where != null && where.Length > 0) 
            {
                query += " WHERE " + where;
            }
            Int32 count = ExecuteNonQuery(query);
            return (count >= 0) ? true : false;
        }

        public List<int> GetQueryIntValues(String command)
        {
            MySqlDataReader reader = GetQueryReader(command);
            
            List<int> ret_values = new List<int>();

            if (reader == null) 
            { 
                return ret_values; 
            }

            while (reader.Read())
                ret_values.Add(Convert.ToInt16(reader[0]));

            reader.Close();

            return ret_values;
        }

        /// <summary>
        /// Delete rows from "table_name" which correspond to expression "where".
        /// </summary>
        /// <param name="table_name"></param>
        /// <param name="where"></param>
        /// <returns></returns>
        public Int32 DeleteRows(String table_name, String where)
        {
            Int32 rows_count = 0;
            if (!CheckConnection()) return 0;
            if (table_name == null || table_name == "" || where == null || where == "") return 0;

            // create query
            String query = "";
            query += "DELETE FROM " + table_name + " WHERE " + where;

            lock (mLockReader)
            {
                MySqlDataReader reader = null;

                try
                {
                    // call query and get reader
                    reader = GetQueryReader(query);

                    // get number of affected rows
                    if (reader != null) rows_count = reader.RecordsAffected;

                    LogUnit.LogDebug(ModuleName, "DeleteRows: " + rows_count + ", table: " + table_name + ", where: " + where);

                }
                catch (MySqlException ex)
                {
                    LogUnit.LogError(ModuleName, "DeleteRows ERROR: " + ex.ToString() + ", Query: " + query);
                    return 0;
                }
                finally
                {
                    if (reader != null) reader.Close();
                }
            }
            return rows_count;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="table_name"></param>
        /// <param name="where"></param>
        /// <param name="values"></param>
        /// <returns></returns>
        public Int32 UpdateRows(String table_name, String where, String values)
        {
            if (!CheckConnection()) return 0;
            if (table_name == null || table_name == "" || where == null || where == "" || values == null || values == "") return 0;

            int rows_count = 0;               
           
            // create query
            String query = "";
            query += "UPDATE " + table_name;
            query += " SET " + values;
            query += " WHERE " + where;
                     
            lock (mLockReader)
            {  
                MySqlDataReader reader = null;
                try
                {                    
                    // call query and get reader
                    reader = GetQueryReader(query);

                    // get number of affected rows
                    if (reader != null) 
                        rows_count = reader.RecordsAffected;
                    
                    LogUnit.LogDebug(ModuleName, "UpdateRows: " + rows_count + " rows, table: " + table_name + ", where: " + where + ", values: " + values);
                   
                }
                catch (MySqlException ex)
                {
                    LogUnit.LogError(ModuleName, "UpdateRows ERROR: " + ex.ToString() + ", Query: " + query);
                    return 0;
                }
                finally
                {
                    if (reader != null) reader.Close();
                }
            }
            return rows_count;
        }


    }
}
