function bbs = lbpDetect(I, detector, fileName, threads)
% Detect objects using LBP detector
%
% BBS = LBPDETECT(I, D, fileNmae, threads);
%
% Inputs:
%   I        Image, RGB or grayscale, single or uint8
%   D        Detector structure or 1xK cell array of detectors trained with
%            the same feature channels.
%   fileNmae File to save results
%   threads  Max number of threads to use
%
% Outputs:
%   BBS     Nx5 matrix with object bounding boxes
%
% By default, threads are disabled so the detection core scans images
% in single thread. This is requirement of many matlab installations.
% When multiple images are given, threads are automatically disabled. To
% scan images in parallel, start parpool.
%
% Piotr's Computer Vision Matlab Toolbox      Version NEW
% Copyright 2014 Piotr Dollar.  [pdollar-at-gmail.com]
% Licensed under the Simplified BSD License [see external/bsd.txt]
%
% Modified by Roman Juranek. [ijuranek-at-fit.vutbr.cz]
% - threads parameter to explicitly set threads in mex func
% - mex scans all pyramid levels at once
%
% See also: lbpTrain

% TODO:
% - Output ns, nf
% - Set correct stride

if nargin<4, threads=0; end;
if nargin<3, fileName=''; end;
multiple=iscell(I);
if ~isempty(fileName) && exist(fileName,'file'), bbs=1; return; end
if ~multiple
    bbs=lbpDetectImg(I,detector,threads);
else
    n=length(I); bbs=cell(n,1);
    parfor i=1:n, bbs{i}=lbpDetectImg(I{i},detector,0); end
end

% write results to disk if fileName specified
if isempty(fileName), return; end;
d=fileparts(fileName);
if ~isempty(d) && ~exist(d,'dir'), mkdir(d); end;
if multiple % add image index to each bb and flatten result
    for i=1:n, bbs{i}=[ones(size(bbs{i},1),1)*i bbs{i}]; end
    bbs=vertcat(bbs{:});
end
dlmwrite(fileName,bbs,'precision',7);
bbs=1;

end

function bbs = lbpDetectImg(I, detector, threads)
if nargin == 2, threads=0; end;
Ds=detector; if(~iscell(Ds)), Ds={Ds}; end; nDs=length(Ds);
opts=Ds{1}.opts; pPyramid=opts.pPyramid; pNms=opts.pNms;
imreadf=opts.imreadf; imreadp=opts.imreadp;
shrink=pPyramid.pChns.shrink; pad=pPyramid.pad;
separate=nDs>1 && isfield(pNms,'separate') && pNms.separate;
% read image and compute features (including optionally applying filters)
try
    % This prevents from throwing errors when corrupted images have to be loaded.
    % Empty bbs is returned instead.
    if(all(ischar(I))), I=feval(imreadf,I,imreadp{:}); end
catch
    warning(I);
    bbs = [];
    return;
end
P=chnsPyramid(I,pPyramid); bbs=cell(1,nDs);
if(isfield(opts,'filters') && ~isempty(opts.filters)), shrink=shrink*2;
    for i=1:P.nScales, fs=opts.filters; C=repmat(P.data{i},[1 1 size(fs,4)]);
        for j=1:size(C,3), C(:,:,j)=conv2(C(:,:,j),fs(:,:,j),'same'); end
        P.data{i}=imResample(C,.5);
    end
end
% Modified by R.Juranek
for j=1:nDs,
    opts=Ds{j}.opts;
    modelDsPad=opts.modelDsPad;
    modelDs=opts.modelDs;
    bb = mexScalarDetect(P.data, single(1:P.nScales), Ds{j}.clf.ftr, Ds{j}.clf.hs, Ds{j}.clf.thr, Ds{j}.clf.size, 1, threads);
    if ~isempty(bb)
        bb = bb';
        s = bb(:,6);
        bb = bb(:,1:5);
        bb(:,1:4) = bb(:,1:4) * shrink;
        shift=(modelDsPad-modelDs)/2-pad;
        bb(:,1)=(bb(:,1)+shift(2))./P.scaleshw(s,2);
        bb(:,2)=(bb(:,2)+shift(1))./P.scaleshw(s,1);
        bb(:,3)=modelDs(2)./P.scales(s);
        bb(:,4)=modelDs(1)./P.scales(s);
        if(separate), bb(:,6)=j; end; bbs{j}=bb;
    end
end
bbs=cat(1,bbs{:});
if ~isempty(pNms), bbs=bbNms(bbs,pNms); end;
end
