#include "detector.h"
#include "detector.pb.h"
#include <fstream>
#include <iostream>
#include <google/protobuf/repeated_field.h>

using namespace std;
using namespace LBPDetector;

DetectorAndChns loadDetectorPB(const string fileName)
{
    // Parse protocol buffer with detector structure
    LBPDetector::DetectorProto detector;
    fstream input(fileName, ios::in | ios::binary);
    if (!detector.ParseFromIstream(&input))
    {
        return {0,{}};
    }

    int T = detector.t();

    // Expected sizes of arrays
    const int ftrs_size = 5*T;
    const int thr_size = T;
    const int hs_size = 256*T;

    // Check sizes
    if (detector.ftrs_size() != ftrs_size || detector.thr_size() != thr_size || detector.hs_size() != hs_size)
    {
        return {0,{}};
    }

    Detector * D = new Detector;
    D->wdim0 = detector.dim0();
    D->wdim1 = detector.dim1();
    D->T = T;

    if (detector.has_label())
    {
        D->label = detector.label();
    }

    // Alloc buffers for parameters
    D->ftr = new unsigned char[ftrs_size];
    D->thr = new float[thr_size];
    D->alpha = new float[hs_size];
    // Copy data

    copy_n(detector.ftrs().begin(), ftrs_size, D->ftr);
    copy_n(detector.thr().begin(), thr_size, D->thr);
    copy_n(detector.hs().begin(), hs_size, D->alpha);
    
    D->dynamic = true;
    if (!initDetector(*D))
    {
        destroyDetector(&D);
        D = 0;
    }

    ChnsParams C;
    if (detector.has_shrink()) C.shrink = detector.shrink();
    if (detector.has_colorsmooth()) C.colorSmooth = detector.colorsmooth();
    if (detector.has_pyrsmooth()) C.pyrSmooth = detector.pyrsmooth();
    if (detector.has_colorspace())
    {
        switch (detector.colorspace())
        {
            case LBPDetector::DetectorProto::RGB:
                C.colorSpace = LBPDetector::ChnsParams::CS_RGB;
                break;
            default:
                break;
        }
    }
    
    DetectorAndChns DC = {D, C};

    return DC;
}