
/*
 * File:   Utils.c
 */
#include "WirelessProtocols/MCHP_API.h"
#include <rtcc.h>
#include "iot/Utils.h"
#include "iot/LEDs.h"
#include "iot/Protocol.h"
#include "WirelessProtocols/NVM.h"
#include "Transceivers/Security.h"
#include "debug.h"

#define NUM_BATT_SAMPLES 4

extern BOOL timeout;
//int sec = 0;

void StartTimer(WORD seconds)
{
    BYTE sec, min, hour;
    // max time is 18 hours
    if (seconds > HOURS_18)
        seconds = HOURS_18;

    // get seconds, minutes, hours
    sec = seconds % 60;
    min = (seconds/60) % 60;
    hour = (seconds/(60*60)) % 24;

    // decimal to BCD
    sec = (sec/10)*16 + sec%10;
    min = (min/10)*16 + min%10;
    hour = (hour/10)*16 + hour%10;

    RtccWrOn();

    // set start time to "zero" (begining of RTC)
    RTCCFGbits.RTCPTR1 = 1;
    RTCCFGbits.RTCPTR0 = 1;
    RTCVALL = 0;
    RTCVALH = 0;

    RTCVALL = 0;
    RTCVALH = 0;

    RTCVALL = 0;
    RTCVALH = 0;

    RTCVALL = 0;
    RTCVALH = 0;

    mRtccAlrmDisable();
    // set alarm
    ALRMCFGbits.ALRMPTR0 = 1;
    ALRMCFGbits.ALRMPTR1 = 1;
    ALRMVALL = 0;
    ALRMVALH = 0;

    ALRMVALL = 0;
    ALRMVALH = 0;

    ALRMVALL = hour;
    ALRMVALH = 0;
    ALRMVALL = sec;
    ALRMVALH = min;
    
    // set mask for alarm
    if (seconds == 1)
        RtccSetAlarmRpt(RTCC_RPT_SEC, FALSE);
    else if (seconds < 10)
        RtccSetAlarmRpt(RTCC_RPT_TEN_SEC, FALSE);
    else if (seconds < 60)
        RtccSetAlarmRpt(RTCC_RPT_MIN, FALSE);
    else if (seconds < 600)
        RtccSetAlarmRpt(RTCC_RPT_TEN_MIN, FALSE);
    else if (seconds < 3600)
        RtccSetAlarmRpt(RTCC_RPT_HOUR, FALSE);
    else
        RtccSetAlarmRpt(RTCC_RPT_DAY, FALSE);

    mRtccAlrmEnable();
    timeout = FALSE;
    PIE3bits.RTCCIE = 1;
//
    mRtccOn();
    mRtccWrOff();
}

/*********************************************************************
* Function:         WORD Read_VBGVoltage(void)
*
* PreCondition:     none
*
* Input:		    none
*
* Output:		    ADRES
*
* Side Effects:	    none
*
* Overview:		    Reads the band gap voltage and compares with reference voltage
*					to arrive at the current voltage level
*
* Note:
**********************************************************************/
WORD Read_VBGVoltage(void)
{
    ADCON0 = 0x3D;				// Configures the channel as VBG
    ADCON1 = 0xBD;				// Program the acquisition time
    ANCON1bits.VBGEN = 1;		// Enable Band gap reference voltage

    Delay10us(1000);			//Wait for the Band Gap Settling time

    PIR1bits.ADIF = 0;
    PIE1bits.ADIE = 0;			//Disable ADC interrupts
    							//This routine uses the polling based mechanism
    ADCON0bits.GO = 1;		    //Start A/D conversion
    while(ADCON0bits.DONE);

    ADCON0bits.ADON = 0;	    // Turn ADC OFF
    ANCON1bits.VBGEN = 0;	    // Disable Bandgap

    return ADRES;
}


/*********************************************************************
* Function:         BYTE ReadTempSensor(WORD VBGResult)
*
* PreCondition:     Proper reference voltage value has been determined.
*
* Input:		    WORD VBGResult - Reference voltage for temp calculation.
*
* Output:		    BYTE temp
*
* Side Effects:	    none
*
* Overview:		    Following routine reads the on board Tempature Sensor and
*                   calculates the temp value.
*
* Note:
**********************************************************************/
WORD ReadBattery(WORD VBGResult)
{
    WORD battValue;
    double batt;
    WORD battHere;
    BYTE i = 0;
    float battAverage = 0;
    WORD battArray[NUM_BATT_SAMPLES];


    // Configure the ADC register settings
    ADCON0 = 0x00;
    ADCON1 = 0xBD;

    PIR1bits.ADIF = 0;
    PIE1bits.ADIE = 0;

    do {
        ADCON0bits.ADON = 1;
        Delay10us(10); // Wait Acquisition time

        ADCON0bits.GO = 1;
        while (ADCON0bits.DONE);

        batt = (1200.0 / VBGResult);
        battValue = ADRES;
        batt = (batt * battValue);
      //  batt = (batt - 500.0) / 10.0;

        battArray[i] = (WORD) batt;

        ADCON0bits.ADON = 0;
        Delay10us(1);
        i++;
    } while (i < NUM_BATT_SAMPLES);


    for(i = 0; i<NUM_BATT_SAMPLES; i++)
    {
        battAverage = (battAverage + battArray[i]);
    }
    battAverage = (battAverage/NUM_BATT_SAMPLES);
    battHere = (WORD) battAverage;
    battAverage = (battAverage - battHere) * 10;

    if(battAverage >= 5)
        battHere = battHere + 1;

    return (WORD)battHere;


}

WORD Battery(void)
{
    WORD VBGResult, battery;

    VBGResult = Read_VBGVoltage();
    MiWiPROTasks();
    battery = ReadBattery(VBGResult);
    MiWiPROTasks();

#if MY_PAN_ID == 0xFFFF  // coordinator or end device
    if (battery < BATTERY_CRIT)
    {
        LED1_PWM = 1;
        LED2_PWM = 0;
    }
    else if (battery < BATTERY_LOW)
        LED1_PWM = LED2_PWM = 1;
    else
    {
        LED1_PWM = 0;
        LED2_PWM = 1;
    }
    LED_blink_once();
#endif
    return battery;
}

void PrintConnTable()
{
    BYTE i, j;
    for (i = 0; i < CONNECTION_SIZE; i++)
    {
        if (ConnectionTable[i].status.bits.isValid)
        {
            PRINT("Zaznam id: %d\n", i);
            PRINT("AltAddr: 0x%04X\n", ConnectionTable[i].AltAddress.Val);
            PRINT("PANID: 0x%04X\n", ConnectionTable[i].PANID.Val);
            PRINT("Addr: 0x");
            for (j = 0; j < MY_ADDRESS_LENGTH; j++)
                PRINT("%X", ConnectionTable[i].Address[j]);
            PRINT("\n");
        }		
    }

}

void ChangePANIDs(WORD panid)
{
    BYTE i;
    for ( i = 0; i < CONNECTION_SIZE; i++)
    {
        if (ConnectionTable[i].status.bits.isValid)
        {
            ConnectionTable[i].PANID.Val = panid;
        }
    }
}

// zpracovava zpravy dokud nenarazi na zpravu s kodem == code nebo neprijde timeout
// pouziti pro koncove senzory kdy cekaji na specifickou zpravu (napr pri joinovani)
void WaitForMessage(BYTE code)
{
	MIWI_TICK delayTick = MiWi_TickGet();
	while(!timeout)
	{

#ifdef NWK_ROLE_END_DEVICE		
		//DelayMs(100);	// delayMs is interfering with RTC
		while(MiWi_TickGetDiff(MiWi_TickGet(),delayTick) < (300 * ONE_MILI_SECOND))
        {
            MiWiPROTasks();
        }
        MiApp_TransceiverPowerState(POWER_STATE_WAKEUP_DR);
#endif
		if(MiApp_MessageAvailable())
		{
            PRINT("Prijata sprava\n");
			if(rxMessage.Payload[0] == code)
            {
                //timeout = TRUE;
				break;
            }
   			else
				HandleMessage(rxMessage.Payload[0]);
			MiApp_DiscardMessage();
		}
		delayTick = MiWi_TickGet();
	}
}


void factoryReset(void)
{
	BYTE tmp;
	WORD_VAL tmpv;
	tmp = 0;
	nvmPutMyParent(&tmp);
	nvmPutConnMode(&tmp);
	
	#ifdef ENABLE_SLEEP
	tmp = DEFAULT_SLEEP;
	nvmPutSleepSeconds(&tmp);
	#endif
	
	tmp = 15;
	nvmPutCurrentChannel(&tmp);
	tmpv.Val = 0xFFFF;
	nvmPutMyPANID(tmpv.v);
	nvmPutMyShortAddress(tmpv.v);

	setDefaultKey();
	#if MY_PAN_ID == 0xFFFF
        nvmPutSecurityKey( securityKey );
	#endif
}

void setKey(unsigned char *key)
{
    BYTE i;
    for(i = 0; i < KEY_SIZE; i++)
    {
        securityKey[i] = key[i];
    }
}

void setDefaultKey(void)
{
    BYTE i;
    for(i = 0; i < KEY_SIZE; i++)
    {
        securityKey[i] = defaultSecurityKey[i];
    }
}

#if MY_PAN_ID != 0xFFFF
void setMyKey(void)
{
    BYTE i;
    for(i = 0; i < KEY_SIZE; i++)
    {
        securityKey[i] = mySecurityKey[i];
    }
}
#endif

void printKey(void)
{
    BYTE i;
    for(i = 0; i < KEY_SIZE; i++)
    {
        PRINT("Key[%d]: 0x%02X\n", i, securityKey[i]);
    }
}


BYTE findInTable()
{
    BYTE i;
    for (i = 0; i < CONNECTION_SIZE; i++)
    {
        if ((ConnectionTable[i].status.bits.isValid) &&
                ((rxMessage.Payload[ADDR_INDEX] == ConnectionTable[i].Address[0]) &&
                (rxMessage.Payload[ADDR_INDEX + 1] == ConnectionTable[i].Address[1]) &&
                (rxMessage.Payload[ADDR_INDEX + 2] == ConnectionTable[i].Address[2]) &&
                (rxMessage.Payload[ADDR_INDEX + 3] == ConnectionTable[i].Address[3])))
        {
            return i;
        }
    }
    return 0xFF;
}
