﻿//
// Author: Ladislav Pospech, xpospe01
// Email: xpospe01@stud.fit.vutbr.cz, ladislav.pospech@gmail.com
//
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Xml;
using System.Collections;
using System.IO;
using System.Xml.Schema;

namespace XMLDabster
{
  public partial class MainForm : Form
  {
    public MainForm()
    {
      InitializeComponent();
    }
    #region Global variables
    int mouseX, mouseY; 
    const int DgvWidthOffset = 2;
    const string DefNS = "xDabster";
    string path = "";
    int columnIndex = 0;
    int depth = 1;

    XmlDataDocument doc = new XmlDataDocument();
    XmlNode xmlRoot;
    TreeNode treeRoot;
    DataSet extDs = new DataSet();
    XmlNamespaceManager nsMan;
    #endregion
    
    
    /// <summary>
    /// Obsluha tlacika pro otevreni XML souboru. Vola metody pro nacteni XML souboru a vytvoreni stromoveho
    /// zobrazeni v TreeView
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    private void openToolStripMenuItem_Click(object sender, EventArgs e)
    {
      //string path = "";
      if (openXmlDialog.ShowDialog() == DialogResult.OK)
      {
        this.Cursor = Cursors.WaitCursor;
        //mazani predesleho obsahu
        tcTables.Visible = false;
        xmlTree.Nodes.Clear();
        foreach (TabPage tabPage in tcTables.TabPages)
        {
          tcTables.TabPages.Remove(tabPage);
        }
  
        path = openXmlDialog.FileName.ToString();
        doc = new XmlDataDocument();
        extDs = new DataSet();
        try
        {
          doc.Load(path);

          nsMan = new XmlNamespaceManager(doc.NameTable);
          LoadNamespaces(nsMan);
          LoadRoots();
          CreateTreeView(xmlRoot, treeRoot);
        }
        catch (Exception ex)
        {
          MessageBox.Show("XML Document can't be opened. Probably it is not valid! \n\n" + ex.Message, "Open Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        }
        this.Cursor = Cursors.Default;
        
      }
    }

    /// <summary>
    /// Metoda nacitajici z korenoveho elementu XML souboru pouzite jmenne prostory
    /// a pridava je do manazeru jmennych prostor.
    /// </summary>
    /// <param name="nsMan">manazar jmenych prostor dokumentu</param>
    void LoadNamespaces(XmlNamespaceManager nsMan)
    {
      XmlNode rootNode = doc.DocumentElement;
      foreach (XmlAttribute attribute in rootNode.Attributes)
      {
        if (attribute.Name == "xmlns")
        {
          nsMan.AddNamespace(DefNS, attribute.Value);
        }
        else if (attribute.Name.StartsWith("xmlns:"))
        {
          nsMan.AddNamespace(attribute.Name.Substring(attribute.Name.IndexOf(":") + 1), attribute.Value);
        }
      }
    }

    /// <summary>
    /// Nacte korenovy element z xml souboru a z nej vytvori korenovy uzel TreeView
    /// </summary>
    public void LoadRoots()
    {
      xmlRoot = doc.DocumentElement;
      if (xmlRoot.Attributes.Count > 0)
      {
        string attributes = "";
        for (int i = 0; i < xmlRoot.Attributes.Count; i++)
        {
          attributes += " [";
          attributes += xmlRoot.Attributes[i].Name;
          attributes += "]";
        }
        treeRoot = xmlTree.Nodes.Add(xmlRoot.Name, xmlRoot.Name + attributes);
      }
      else
      {
        treeRoot = xmlTree.Nodes.Add(xmlRoot.Name, xmlRoot.Name);
      }
    }

    /// <summary>
    /// Funkce prochazi XML dokumentem a hleda uzly. Pokud najde vice uzlu stejneho jmena, pocita je a nasledne
    /// zavola funci AddTreeNode() aby zapsala novy uzel do TreeView. Funkce pracuje rekurzivne pro prohledavani
    /// detskych uzlu.
    /// </summary>
    /// <param name="xmlRoot">Korenovy element XML soubotu</param>
    /// <param name="treeRoot">Korenovy uzel TreeView</param>
    public void CreateTreeView(XmlNode xmlRoot, TreeNode treeRoot)
    {
      int count = 1;
      TreeNode newTreeNode;
      XmlNodeList xmlRootChilds = xmlRoot.ChildNodes;
      for (int i = 0; i < xmlRootChilds.Count; i++)
      {
        if (i == xmlRootChilds.Count-1)
        {
          if (xmlRootChilds[i].Name == "#text")
          {
            break;
          }
          newTreeNode = AddTreeNode(xmlRootChilds[i], treeRoot, count, HasReservedAttributes(xmlRootChilds[i]));
          if (count > 1)
          {
            newTreeNode.Checked = true;
          }
          if (xmlRootChilds[i].HasChildNodes && (xmlRootChilds[i].FirstChild.NodeType == XmlNodeType.Element))
          {
            CreateTreeView(xmlRootChilds[i], newTreeNode);
          }
          continue;
        }
        
        if (xmlRootChilds[i].Name == xmlRootChilds[i + 1].Name)
        {
          if (xmlRootChilds[i].Attributes.Count > 0)
          {
            if (!HasReservedAttributes(xmlRootChilds[i]))
            {
              count++;
            }
            else
            {
              newTreeNode = AddTreeNode(xmlRootChilds[i], treeRoot, count, true);
              if (count > 1)
              {
                newTreeNode.Checked = true;
              }
              count = 1;
              if (xmlRootChilds[i].HasChildNodes && (xmlRootChilds[i].FirstChild.NodeType == XmlNodeType.Element))
              {
                CreateTreeView(xmlRootChilds[i], newTreeNode);
              }
            }
          }
          else
          {
            count++;
          }
        }
        else
        {
          newTreeNode = AddTreeNode(xmlRootChilds[i], treeRoot, count, false);
          if (count > 1)
          {
            newTreeNode.Checked = true;
          }
          count = 1;
          if (xmlRootChilds[i].HasChildNodes && (xmlRootChilds[i].FirstChild.NodeType == XmlNodeType.Element))
          {
            CreateTreeView(xmlRootChilds[i], newTreeNode);
          }
        }
      }
    }

    /// <summary>
    /// Kontroluje zda neni mezi atributy elementu nejaky rezervovany("xmlns" nebo nejaky zacinajici "xsi:")
    /// </summary>
    /// <param name="xmlNode"></param>
    /// <returns></returns>
    private bool HasReservedAttributes(XmlNode xmlNode)
    {
      if (xmlNode.NodeType == XmlNodeType.Element)
      {
        for (int i = 0; i < xmlNode.Attributes.Count; i++)
        {
          if (xmlNode.Attributes[i].Name.StartsWith("xsi:", true, null))
          {
            return true;
          }
          else if (xmlNode.Attributes[i].Name == "xmlns")
          {
            return true;
          }
        }
      }
      return false;
    }

    /// <summary>
    /// metoda tvorici retezec pro XPath dotaz, ktery vyhleda vsechny atributy, 
    /// ktere se u daneho jmena elementu v dane urovni dokumentu vyskytuji
    /// </summary>
    /// <param name="node">Uzel, jehoz atributy chceme ziskat</param>
    /// <returns>retezec pro XPath dotaz</returns>
    public string BuildXPathStringForAttributes(XmlNode node)
    {
      XmlNodeList nodes;
      string xpath = "/";
      if (node.Name.Contains(":") || nsMan.LookupNamespace(DefNS) == null)
      {
        xpath += node.Name + "/@*";
      }
      else
      {
        xpath += DefNS + ":" + node.Name + "/@*";
      }
      while (true)
      {
        if (node.ParentNode != null && node.ParentNode.NodeType != XmlNodeType.Document)
        {
          if (node.ParentNode.Name.Contains(":") || nsMan.LookupNamespace(DefNS) == null)
          {
            xpath = xpath.Insert(0, "/" + node.ParentNode.Name);
          }
          else
          {
            xpath = xpath.Insert(0, "/" + DefNS + ":" + node.ParentNode.Name);
          }
          node = node.ParentNode;
        }
        else
          break;
      }
      if (nsMan.LookupNamespace(DefNS) == null)
      {
        nodes = doc.SelectNodes(xpath);
      }
      else
      {
        nodes = doc.SelectNodes(xpath, nsMan);
      }

      List<String> attributes = new List<String>();
      string attributeString = "";

      foreach (XmlNode atNode in nodes)
      {
        if (atNode.NodeType == XmlNodeType.Attribute)
        {
          if (!attributes.Contains(atNode.Name))
          {
            attributes.Add(atNode.Name);
            attributeString += " [" + atNode.Name + "]";
          }
        }
      }
      //MessageBox.Show(attributeString);
      return attributeString;

    }

    /// <summary>
    /// Funkce pridava novy uzel do TreeView podle xml elementu. K nazvu pridava do hranatych zavorek [] jmena 
    /// atributu a do kulatych zavorek () pocet vyskytu elementu v XML.
    /// </summary>
    /// <param name="xmlNode">XML element pro ktery se ma novy uzel vytvorit</param>
    /// <param name="treeParent">Rodic uzlu v TreeView</param>
    /// <param name="count">Pocet vyskytu elementu/uzlu v XML souboru</param>
    /// <returns>Nove vytvoreny TreeNode</returns>
    public TreeNode AddTreeNode(XmlNode xmlNode, TreeNode treeParent, int count, bool reservedAtt)
    {
      string attributes = BuildXPathStringForAttributes(xmlNode);
      //if (xmlNode.NodeType == XmlNodeType.Element && xmlNode.Attributes.Count > 0)
      //{
      //  ////////////////////////////////////
      //  string attributes = "";
      //  for (int i = 0; i < xmlNode.Attributes.Count; i++)
      //  {
      //    attributes += " [";
      //    if (reservedAtt)
      //    {
      //      attributes += xmlNode.Attributes[i].Name;
      //      attributes += "=\"";
      //      attributes += xmlNode.Attributes[i].Value;
      //      attributes += "\"";
      //    }
      //    else
      //    {
      //      attributes += xmlNode.Attributes[i].Name;
      //    }
      //    attributes += "]";
      //  }
      //  if (count > 1)
      //  {
      //    return treeParent.Nodes.Add(xmlNode.Name, xmlNode.Name + attributes + " (" + count.ToString() + ")");
      //  }
      //  return treeParent.Nodes.Add(xmlNode.Name, xmlNode.Name + attributes);
      //}
      if (count > 1)
      {
        return treeParent.Nodes.Add(xmlNode.Name, xmlNode.Name + attributes + " (" + count.ToString() + ")");
      }
      return treeParent.Nodes.Add(xmlNode.Name, xmlNode.Name + attributes);
    }

    /// <summary>
    /// metoda vyrvori novy prvek DataGrisView a priradi mu zakladni parametry
    /// </summary>
    /// <param name="name">jmeno vysledneho DataGridView</param>
    /// <param name="tPage">TabPage, na ktere se prvek vytvori</param>
    /// <returns>Vraci odkaz na nove vytvorene DataGridView</returns>
    DataGridView CreateNewDGV(string name, TabPage tPage)
    {
      //vytvoreni
      DataGridView newDGV = new DataGridView();
      newDGV.Name = name;
      //obsluha udalosti
      newDGV.ColumnHeaderMouseClick += new System.Windows.Forms.DataGridViewCellMouseEventHandler(this.dataGridView1_ColumnHeaderMouseClick);
      //nastaveni vlastnosti
      newDGV.Dock = DockStyle.Fill;
      newDGV.BackgroundColor = Color.LightSkyBlue;
      newDGV.AllowUserToAddRows = false;
      newDGV.AllowUserToOrderColumns = true;
      newDGV.EditMode = DataGridViewEditMode.EditProgrammatically;
      //pridani do controlu
      tPage.Controls.Add(newDGV);

      return newDGV;
    }

    /// <summary>
    /// Vytvori novy panel se zalozkou (tabPage) na prvku TabControl
    /// </summary>
    /// <param name="treeNode">Uzel zjednoduseneho stromu, podle ktereho vytvarime TabPage</param>
    /// <returns>Odkaz na novy TabPage</returns>
    TabPage AddTabPage(TreeNode treeNode)
    {
      if (!tcTables.Visible)
      {
        tcTables.Visible = true;
      }
      TabPage newTabPage = new TabPage(treeNode.Name);
      newTabPage.Name = treeNode.Name;
      newTabPage.Tag = GetNodePath(treeNode);
      tcTables.TabPages.Add(newTabPage);
      //pridani kontextoveho menu
      tcTables.SelectedTab = tcTables.TabPages[tcTables.TabPages.Count - 1];
      return newTabPage;
    }

    /// <summary>
    /// Metoda pro nalezeni cesty prvku stromu od korene
    /// </summary>
    /// <param name="treeNode">Uzel stromu, kteremu chceme vytvorit cestu</param>
    /// <returns>Retezec posloupnosti uzlu od korene</returns>
    string GetNodePath(TreeNode treeNode)
    {
      string path = treeNode.Name;
      while (true)
      {
        if (treeNode.Parent != null)
        {
          path = path.Insert(0, treeNode.Parent.Name + ".");
          treeNode = treeNode.Parent;
        }
        else
          return path;
      }
    }
    
    /// <summary>
    /// Identifikuje pozadovany uzel ve stromu a sestroji XPath vyraz pro jeho nalezeni
    /// </summary>
    /// <param name="treeNode">Vybrany uzel stromu</param>
    /// <returns>Seznam vsech nalezenych uzlu</returns>
    XmlNodeList IdentificateXmlNodes(TreeNode treeNode)
    {
      XmlNodeList nodes;
      string xpath = "/";
      if (treeNode.Name.Contains(":") || nsMan.LookupNamespace(DefNS) == null)
      {
        xpath += treeNode.Name;
      }
      else
      {
        xpath += DefNS + ":" + treeNode.Name;
      }
      while (true)
      {
        if (treeNode.Parent != null)
        {
          if (treeNode.Parent.Name.Contains(":") || nsMan.LookupNamespace(DefNS) == null)
          {
            xpath = xpath.Insert(0, "/" + treeNode.Parent.Name);
          }
          else
          {
            xpath = xpath.Insert(0, "/" + DefNS + ":" + treeNode.Parent.Name);
          }
          treeNode = treeNode.Parent;
        }
        else
          break;
      }
      if (nsMan.LookupNamespace(DefNS) == null)
      {
        nodes = doc.SelectNodes(xpath);
      }
      else
      {
        nodes = doc.SelectNodes(xpath, nsMan);
      }
      return nodes;
    }

    /// <summary>
    /// Metoda zakladajici novou tabulku se specifickym jmenem
    /// </summary>
    /// <param name="mainNode">Uzel XML, podle ktereho je tabulka pojmenovana</param>
    /// <returns>Nove vytvorena tabulka prirazena k datasetu</returns>
    DataTable CreateExtendedTable(XmlNode mainNode)
    {
      try
      {
        DataTable newTable = extDs.Tables.Add(mainNode.Name);
        return newTable;
      }
      catch
      {
        return null;
      }
      
      //return extDs.Tables.Add("newTable");
    }

    /// <summary>
    /// Metoda zakladajici novy radek tabulky
    /// </summary>
    /// <param name="table">Rodicovska tabulka radku</param>
    /// <returns>Novy radek prirazeny tabulce</returns>
    DataRow CreateExtendedRow(DataTable table)
    {
      DataRow newRow = table.NewRow();
      return newRow;
    }

    /// <summary>
    /// Metoda plnici radek rozsirene tabulky. Rekurzivne prochazi child elementy
    /// </summary>
    /// <param name="node">Element jehoz deti se budou prochazet</param>
    /// <param name="extTable">Tabulka, do ktere radek patri</param>
    /// <param name="extRow">Radek do ktereho se nacitaji data</param>
    /// <returns></returns>
    DataRow FillDataRow(XmlNode node, DataTable extTable, DataRow extRow, bool firstLevel, string sibMarker)
    {
      #region Local variables
      bool sib = false;
      int sibCount = 1;
      string innerSibMarker = "";
      string prefix = "@";
      string sibMarkers = innerSibMarker + sibMarker;
      #endregion
      depth++;
      if (node.Attributes.Count > 0 && firstLevel)
      {//uzel ma atributy, je treba zalozit jim sloupec(pokud jiz neni) a vlozit je do radku
        foreach (XmlAttribute nodeAttr in node.Attributes)
        {
          if (!extTable.Columns.Contains(prefix + nodeAttr.Name + sibMarkers))
          {
            extTable.Columns.Add(prefix + nodeAttr.Name + sibMarkers);
          }
          extRow[prefix + nodeAttr.Name + sibMarkers] = nodeAttr.Value;
        }
      }
      if (node.Value != null)
      {//uzel ma nejakou hodnotu(taxt)
        if (!extTable.Columns.Contains(node.Name + sibMarkers))
        {
          extTable.Columns.Add(node.Name + sibMarkers);
        }
        extRow[node.Name + sibMarkers] = node.Value;
      }
      if (node.ParentNode == null || node.ParentNode.NodeType == XmlNodeType.Document)
      {//uzel je nejvyssim uzlem stromu, korenovym uzlem XML dokumentu. dalsi zpracovani se neprovadi
        return extRow;
      }
      //Obsluha vsech detskych uzlu
      foreach (XmlNode chNode in node.ChildNodes)
      {
        if (chNode.NextSibling != null && chNode.Name == chNode.NextSibling.Name && chNode.NodeType == chNode.NextSibling.NodeType)
        {//uzel ma sourozence stejneho jmena a typu, musime zavest pocitadlo
          sib = true;
          innerSibMarker = CreateSibMarker(sibCount);
          sibCount++;
        }
        else if (chNode.PreviousSibling != null &&  chNode.Name == chNode.PreviousSibling.Name && chNode.NodeType == chNode.PreviousSibling.NodeType)
        {
          innerSibMarker = CreateSibMarker(sibCount);
          sib = false;
        }
        sibMarkers = innerSibMarker + sibMarker;
        if (chNode.NodeType == XmlNodeType.Element && chNode.Attributes.Count > 0)
        {//uzel ma nejake atributy
          prefix = chNode.Name + "@";
          foreach (XmlAttribute nodeAttr in chNode.Attributes)
          {//obsluha vsech atributu
            if (!extTable.Columns.Contains(prefix + nodeAttr.Name + sibMarkers))
            {//sloupec daneho jmena v tabulce jeste neexistuje, musi se vytvorit
              extTable.Columns.Add(prefix + nodeAttr.Name + sibMarkers);
            }
            extRow[prefix + nodeAttr.Name + sibMarkers] = nodeAttr.Value;
          }
        }
        if (chNode.HasChildNodes && chNode.FirstChild.NodeType == XmlNodeType.Element)
        {//zkoumany uzel ma potomky typu element
          if (nudMaxDepth.Value == 0 || nudMaxDepth.Value >= depth)
          {
            extRow = FillDataRow(chNode, extTable, extRow, false, sibMarkers);
          }
        }
        else
        {//uzel nema deti typu element
          prefix = "";
          if (!firstLevel)
          {
            prefix = chNode.ParentNode.Name + ".";
          }
          if (!extTable.Columns.Contains(prefix + chNode.Name + sibMarkers))
          {//sloupec daneho jmena v tabulce jeste neexistuje, musi se vytvorit
            extTable.Columns.Add(prefix + chNode.Name + sibMarkers);
          }
          if (chNode.HasChildNodes)
            extRow[prefix + chNode.Name + sibMarkers] = chNode.FirstChild.Value;
          else
            extRow[prefix + chNode.Name + sibMarkers] = chNode.Value;
        }
        if (!sib)
        {
          sibCount = 1;
          innerSibMarker = "";
        }
      }
      depth--;
      //MessageBox.Show(depth.ToString());
      return extRow;
    }

    /// <summary>
    /// Vytvari znacku pridavanou za stejne se jmenujici sourozence
    /// </summary>
    /// <param name="sibCount">poradove cislo sourozence</param>
    /// <returns>retezcovou hodnotu vyjadrujici poradove cislo sourozence</returns>
    string CreateSibMarker(int sibCount)
    {
      return (" (#" + sibCount.ToString() + ")");
    }

    /// <summary>
    /// metoda provadi zapis do souboru. Funguje jak pro XML, tak XSL konverzi
    /// </summary>
    /// <param name="dgv">DataGridView, ktere chceme ukladat</param>
    /// <param name="path">cesta pro ulozeni souboru</param>
    void WriteTableToXLSorXML(DataGridView dgv, string path)
    {
      DataTable newDT = CreateEditedTable(dgv);
      newDT.WriteXml(path);
    }

    /// <summary>
    /// Metoda zapisujici soubor s kodem HTML. Pred zapisem vola zpracovani dat do HTML
    /// </summary>
    /// <param name="dgv">DataGridView, ktere chceme exportovat</param>
    /// <param name="path">cesta k souboru pro ulozeni</param>
    void WriteTableToHtml(DataGridView dgv, string path)
    {
      string HTML = HTMLProc.CreateHTML(CreateEditedTable(dgv), tcTables.TabPages[tcTables.SelectedIndex].Tag.ToString());

      if (!File.Exists(path))
      {
        Stream lStream = File.Create(path);
        lStream.Close();
      }
      StreamWriter lWriter = File.AppendText(path);
      lWriter.Write(HTML);
      lWriter.Close();
    }

    /// <summary>
    /// Vytvari upravenou tabulku pro ulozeni, napriklad pro pripad ze by v tabulce byly nektere sloupce odstraneny
    /// </summary>
    /// <param name="dgv">DataGridView pro ktere chceme tabylky vytvorit</param>
    /// <returns>nove vytvorena datova tabulka</returns>
    DataTable CreateEditedTable(DataGridView dgv)
    {
      try
      {
        extDs.Tables.Remove(dgv.Name + ".e");
      }
      catch { }
      DataTable newDT = extDs.Tables.Add(dgv.Name + ".e");
      foreach (DataGridViewColumn dgvCol in dgv.Columns)
      {
        newDT.Columns.Add(dgvCol.Name);
      }

      foreach (DataGridViewRow dgvRow in dgv.Rows)
      {
        DataRow newDR = newDT.NewRow();
        foreach (DataGridViewColumn dgvCol in dgv.Columns)
        {
          newDR[dgvCol.Name] = dgvRow.Cells[dgvCol.Name].Value;
        }
        newDT.Rows.Add(newDR);
      }
      extDs.Tables.Remove(newDT);
      return newDT;
    }

    #region eventHandlers
    private void xLSFileToolStripMenuItem_Click(object sender, EventArgs e)
    {
      if (tcTables.TabPages.Count == 0)
      {
        MessageBox.Show("No table is opened! Nothing to export.", "Export Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }
      if (saveXlsDialog.ShowDialog() == DialogResult.OK)
      {
        try
        {
          WriteTableToXLSorXML((DataGridView)cmsDataGV.SourceControl, saveXlsDialog.FileName);
        }
        catch
        {
          int index = tcTables.SelectedIndex;
          string name = tcTables.TabPages[index].Name;
          DataGridView dgv = (DataGridView)tcTables.TabPages[index].Controls[name];
          WriteTableToXLSorXML(dgv, saveXlsDialog.FileName);
        }
      }
    }

    private void hTMLFileToolStripMenuItem_Click(object sender, EventArgs e)
    {
      if (tcTables.TabPages.Count == 0)
      {
        MessageBox.Show("No table is opened! Nothing to export.", "Export Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }
      if (saveHtmlDialog.ShowDialog() == DialogResult.OK)
      {
        try
        {
          WriteTableToHtml((DataGridView)cmsDataGV.SourceControl, saveHtmlDialog.FileName);
        }
        catch
        {
          int index = tcTables.SelectedIndex;
          string name = tcTables.TabPages[index].Name;
          DataGridView dgv = (DataGridView)tcTables.TabPages[index].Controls[name];
          WriteTableToHtml(dgv, saveHtmlDialog.FileName);
        }
      }
    }

    private void newXMLFileToolStripMenuItem_Click(object sender, EventArgs e)
    {
      if (tcTables.TabPages.Count == 0)
      {
        MessageBox.Show("No table is opened! Nothing to export.", "Export Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }
      if (saveXmlDialog.ShowDialog() == DialogResult.OK)
      {
        try
        {
          WriteTableToXLSorXML((DataGridView)cmsDataGV.SourceControl, saveXmlDialog.FileName);
        }
        catch
        {
          int index = tcTables.SelectedIndex;
          string name = tcTables.TabPages[index].Name;
          DataGridView dgv = (DataGridView)tcTables.TabPages[index].Controls[name];
          WriteTableToXLSorXML(dgv, saveXmlDialog.FileName);
        }
      }
    }

    private void toolStripButton_Print_Click(object sender, EventArgs e)
    {
      if (tcTables.TabPages.Count == 0)
      {
        MessageBox.Show("No table is opened! Nothing to print.", "Print Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }
      int index = tcTables.SelectedIndex;
      string name = tcTables.TabPages[index].Name;
      DataGridView dgv = (DataGridView)tcTables.TabPages[index].Controls[name];
      PrintDataGrid.PrintDGV.Print_DataGridView(dgv);

    }

    private void closeToolStripMenuItem_Click(object sender, EventArgs e)
    {
      Application.Exit();
    }

    private void closeTabToolStripMenuItem_Click(object sender, EventArgs e)
    {
      if (tcTables.TabPages.Count == 0)
      {
        //MessageBox.Show("No tab is opened or selected!\n", "Closing tab", MessageBoxButtons.OK, MessageBoxIcon.Error);
        return;
      }
      try
      {
        tcTables.TabPages.Remove(tcTables.TabPages[tcTables.SelectedIndex]);
        if (tcTables.TabPages.Count == 0)
        {
          tcTables.Visible = false;
        }
      }
      catch (Exception ex)
      {
        MessageBox.Show("Selected tab can't be closed for this reason:\n" + ex.Message, 
                        "Closing tab", MessageBoxButtons.OK, MessageBoxIcon.Error);
      }
    }

    private void closeAllTabsToolStripMenuItem_Click(object sender, EventArgs e)
    {
      if (tcTables.TabPages.Count == 0)
      {
        return;
      }
      try
      {
        tcTables.TabPages.Clear();
        tcTables.Visible = false;
      }
      catch (Exception ex)
      {
        MessageBox.Show("Tab/tabs can't be closed for this reason:\n" + ex.Message,
                        "Closing tabs", MessageBoxButtons.OK, MessageBoxIcon.Error);
      }
    }

    private void toolStripButton_Help_Click(object sender, EventArgs e)
    {
      AboutBox about = new AboutBox();
      about.ShowDialog();
    }

    private void toolStripButton4_Click(object sender, EventArgs e)
    {
      if (openXmlDialog.ShowDialog() == DialogResult.OK)
      {
        if (openXsdDialog.ShowDialog() == DialogResult.OK)
        {
          //ValidatingXML(openXmlDialog.FileName.ToString(), openXsdDialog.FileName.ToString());
          Validator myValidator = new Validator(openXsdDialog.FileName.ToString());
          ArrayList myResultList = myValidator.ValidateIt(openXmlDialog.FileName.ToString());
          if (myResultList.Count > 0)
          {
            string resultMessage = "";
            foreach (String myResStr in myResultList)
            {
              resultMessage += myResStr + "\n\n";
            }
            MessageBox.Show(resultMessage, "Validation error!", MessageBoxButtons.OK, MessageBoxIcon.Error);
          }
          else
            MessageBox.Show("Everything is OK, XML file is valid!", "Validation complete", MessageBoxButtons.OK, MessageBoxIcon.Information);
        }
      }
    }
    
    private void xmlTree_DoubleClick(object sender, EventArgs e)
    {
      try
      {
        //((TreeView)sender).SelectedNode.Expand();
        xmlTree.SelectedNode.Expand();
      }
      catch
      {
        return;
      }
      if (xmlTree.SelectedNode != null)
      {
        if (extDs.Tables.Contains(xmlTree.SelectedNode.Name))
        {
          if (tcTables.TabPages[xmlTree.SelectedNode.Name] != null)
          {
            tcTables.SelectedTab = tcTables.TabPages[xmlTree.SelectedNode.Name];
          }
          else
          {
            DataGridView newDGV = CreateNewDGV(xmlTree.SelectedNode.Name, AddTabPage(xmlTree.SelectedNode));
            newDGV.DataSource = extDs;
            newDGV.DataMember = xmlTree.SelectedNode.Name;
          }
        }
        else
        {
          try
          {

            {
              DataGridView newDGV = CreateNewDGV(xmlTree.SelectedNode.Name, AddTabPage(xmlTree.SelectedNode));

              XmlNodeList extNodes = IdentificateXmlNodes(xmlTree.SelectedNode);
              DataTable extTable = CreateExtendedTable(extNodes.Item(0));
              List<DataRow> extRows = new List<DataRow>();

              if (extTable != null)
              {
                foreach (XmlNode node in extNodes)
                {
                  extRows.Add(CreateExtendedRow(extTable));
                  extRows[extRows.Count - 1] = FillDataRow(node, extTable, extRows[extRows.Count - 1], true, "");
                  extTable.Rows.Add(extRows[extRows.Count - 1]);
                }
                newDGV.DataSource = extDs;
                newDGV.DataMember = extNodes.Item(0).Name;
              }
            }
          }
          catch
          {
            MessageBox.Show("Počet různých uzlů podstromu je příliš velký, zvolte menší podstrom dokumentu.", "Chyba!", MessageBoxButtons.OK, MessageBoxIcon.Error);

          }
        }
      }
    }

    private void ctsDGV_Close_Click(object sender, EventArgs e)
    {
      Point p = new Point(mouseX, mouseY);
      int tabID = -1;
      for (int i = 0; i < tcTables.TabCount; i++)
      {
        Rectangle r = tcTables.GetTabRect(i);
        if (r.Contains(p))
        {
          tabID = i;
          break;
        }
      }
      if (tabID >= 0)
      {
        extDs = new DataSet();
        //extDs.Tables.Remove(tcTables.TabPages[tabID].Name);
        tcTables.TabPages[tabID].Controls.RemoveByKey(tcTables.TabPages[tabID].Name);
        tcTables.TabPages.Remove(tcTables.TabPages[tabID]);
        if (tcTables.TabPages.Count == 0)
        {
          tcTables.Visible = false;
        }
        //tcTables.SelectedTab = tcTables.TabPages[tabID];
      }
    }

    private void tcTables_MouseDown(object sender, MouseEventArgs e)
    {
      if (e.Button == MouseButtons.Right)
      {
        mouseX = e.X;
        mouseY = e.Y;
        Point p = new Point(mouseX, mouseY);
        int tabID = -1;
        for (int i = 0; i < tcTables.TabCount; i++)
        {
          Rectangle r = tcTables.GetTabRect(i);
          if (r.Contains(p))
          {
            tabID = i;
            cmsTabs.Show(tcTables, p);
            break;
          }
        }
      }
    }

    private void dataGridView1_ColumnHeaderMouseClick(object sender, DataGridViewCellMouseEventArgs e)
    {
      if (e.Button == MouseButtons.Right)
      {
        int x = ((DataGridView)sender).RowHeadersWidth;
        for (int i = 0; i < e.ColumnIndex; i++)
        {
          if (((DataGridView)sender).Columns[i].Visible)
          {
            x += ((DataGridView)sender).Columns[i].Width;
          }
        }
        x += e.X;
        Point p = new Point(x, e.Y);
        columnIndex = e.ColumnIndex;
        cmsDataGV.Show(((DataGridView)sender), p);
      }

    }

    private void closeColumnToolStripMenuItem_Click(object sender, EventArgs e)
    {
      ((DataGridView)cmsDataGV.SourceControl).Columns.Remove(((DataGridView)cmsDataGV.SourceControl).Columns[columnIndex]);
    }

    private void hideColumnToolStripMenuItem_Click(object sender, EventArgs e)
    {
      ((DataGridView)cmsDataGV.SourceControl).Columns[((DataGridView)cmsDataGV.SourceControl).Columns[columnIndex].Name].Visible = false;
    }

    private void revealColumnsToolStripMenuItem_Click(object sender, EventArgs e)
    {
      foreach (DataGridViewColumn dc in ((DataGridView)cmsDataGV.SourceControl).Columns)
      {
        dc.Visible = true;
      }
    }

    private void reloadColumnsToolStripMenuItem_Click(object sender, EventArgs e)
    {
      string name = ((DataGridView)cmsDataGV.SourceControl).Name;
      TabPage tabPege = ((TabPage)((DataGridView)cmsDataGV.SourceControl).Parent);

      ((DataGridView)cmsDataGV.SourceControl).Parent.Controls.Remove(((DataGridView)cmsDataGV.SourceControl));

      DataGridView newDGV = CreateNewDGV(name, tabPege);
      newDGV.DataSource = extDs;
      newDGV.DataMember = name;
    }
    #endregion
  }
}