#include "pyramid.h"

#include "scale.h"

const int LEVELS_PER_OCTAVE = 4;

// Level

Pyramid::Level::Level()
{
	memset(this, 0, sizeof(*this));
}

void Pyramid::Level::Init(const CvSize & size, const TClassifier & classifier)
{
	Kill();
	image = cvCreateImage(size, IPL_DEPTH_8U, 1);
	convolutions = createConvolutions(size);
	xAddressTable = prepareColAddressTable(*convolutions);
	yAddressTable = prepareRowAddressTable(*convolutions);
	ranks = new Rank[classifier.stageCount];
	calculateClassifierRanks(ranks, &classifier, *convolutions);
}

void Pyramid::Level::Kill()
{
	cvReleaseImage(&image);
	releaseConvolutions(&convolutions);
	delete [] xAddressTable;
	delete [] yAddressTable;
	delete [] ranks;
	memset(this, 0, sizeof(*this));
}

void Pyramid::Level::InitInstance(const Pyramid::Level & level)
{
	KillInstance();
	CvSize size = cvGetSize(level.image);
	//alloc image and convolutions
	image = cvCreateImage(size, IPL_DEPTH_8U, 1);
	convolutions = createConvolutions(size);
	//remaining data are shared
	xAddressTable = level.xAddressTable;
	yAddressTable = level.yAddressTable;
	ranks = level.ranks;
	width = level.width;
	height = level.height;
}

void Pyramid::Level::KillInstance()
{
	cvReleaseImage(&image);
	releaseConvolutions(&convolutions);
	memset(this, 0, sizeof(*this));
}

void Pyramid::Level::Convolve()
{
	convolve12all(image, *convolutions);
}

unsigned Pyramid::Level::Scan(const TClassifier & classifier, TDetectionList &detections)
{
	return scanConvolvedImage(*convolutions, xAddressTable, yAddressTable, &classifier, ranks, detections.begin(), detections.end());
}

//Pyramid

Pyramid::Pyramid(unsigned instanceCount)
{
	if(instanceCount > 0)
	{
		instances.resize(instanceCount);
		instances[0] = this;
		for(unsigned i = 1; i < instances.size(); ++i)
		{
			instances[i] = new Pyramid(0);
		}
		//printf("%d pyramid instances\n", instances.size());
	}
}

Pyramid::~Pyramid()
{
	Kill();
	for(unsigned i = 1; i < instances.size(); ++i)
	{
		delete instances[i];
	}
}

unsigned Pyramid::getLevelCount(unsigned octaves, CvSize size, const TClassifier & classifier)
{
	float octaveWidth = float(size.width), octaveHeight = float(size.height);
	float width = octaveWidth, height = octaveHeight;
	unsigned levels = 0;
	while((octaves > 0) && (width >= classifier.width) && (height >= classifier.height))
	{
		++levels;
		if(levels%LEVELS_PER_OCTAVE == 0)
		{
			--octaves;
			octaveWidth  = width  = octaveWidth *0.5f;
			octaveHeight = height = octaveHeight*0.5f;
		}
		else
		{
			width  = width *0.875f;
			height = height*0.875f;
		}
	}
	return levels;
}

void Pyramid::Init(unsigned octaves, const CvSize & size, const TClassifier &classifier)
{
	unsigned levelCount = getLevelCount(octaves, size, classifier);
	bool rebuild = true;
/*	if((levelCount == 0) || (levelCount != levels.size()))
	{
		rebuild = true;
	}
	else
	{
		CvSize topSize = cvGetSize(levels[0].image);
		rebuild = (size.width != topSize.width) || (size.height != topSize.height);
	}*/
	if(rebuild)
	{
		Kill();
		levels.resize(levelCount);
		levels[0].Init(alignSize(size, cvSize(16, 8)), classifier);
		levels[0].width  = float(levels[0].image->width);
		levels[0].height = float(levels[0].image->height);
		for(unsigned i = 1; i < levels.size(); ++i)
		{
			CvSize s;
			float w, h;
			if(i%LEVELS_PER_OCTAVE == 0)
			{
				s = alignSize(getDstSize<ScaleHalf>(cvGetSize(levels[i-LEVELS_PER_OCTAVE].image)), cvSize(16, 8));
				w = levels[i-LEVELS_PER_OCTAVE].width *0.5f;
				h = levels[i-LEVELS_PER_OCTAVE].height*0.5f;
			}
			else
			{
				s = alignSize(getDstSize<Scale8to7>(cvGetSize(levels[i-1].image)), cvSize(16, 8));
				w = levels[i-1].width *0.875f;
				h = levels[i-1].height*0.875f;
			}
			levels[i].Init(s, classifier);
			levels[i].width  = w;
			levels[i].height = h;
		}
		//init instances
		for(unsigned i = 1; i < instances.size(); ++i)
		{
			instances[i]->InitInstance(*this);
		}
	}
}

void Pyramid::Kill()
{
	for(unsigned i = 0; i < levels.size(); ++i)
	{
		levels[i].Kill();
	}
	for(unsigned i = 1; i < instances.size(); ++i)
	{
		instances[i]->KillInstance();
	}
}

void Pyramid::InitInstance(const Pyramid & pyramid)
{
	levels.resize(pyramid.levels.size());
	for(unsigned i = 0; i < levels.size(); ++i)
	{
		levels[i].InitInstance(pyramid.levels[i]);
	}
}

void Pyramid::KillInstance()
{
	for(unsigned i = 0; i < levels.size(); ++i)
	{
		levels[i].KillInstance();
	}
}

void Pyramid::Build(IplImage *image)
{
	cvResize(image, levels[0].image, CV_INTER_LINEAR);
	for(unsigned i = 1; i < levels.size(); ++i)
	{
		if((i%LEVELS_PER_OCTAVE) == 0)
		{
			doBlockOp<ScaleHalf>(levels[i-LEVELS_PER_OCTAVE].image, levels[i].image);
		}
		else
		{
			doBlockOp<Scale8to7>(levels[i-1].image, levels[i].image);
		}
		//cvResize(levels[i-1].image, levels[i].image, CV_INTER_LINEAR);
	}
}

unsigned Pyramid::Scan(const TClassifier & classifier, const CvSize &originalSize, TDetectionList & detections)
{
	unsigned detection_count = 0;
	#pragma omp parallel
	{
		TDetectionList results(2048);
		#pragma omp for schedule(dynamic, 1)
		for(int l = 0; l < static_cast<int>(levels.size()); ++l)
		{
			levels[l].Convolve();
			unsigned n = levels[l].Scan(classifier, results);
			for(TDetectionList::iterator i = results.begin(); i != results.begin()+n; ++i)
			{
				float fx = float(originalSize.width )/float(levels[l].width);
				float fy = float(originalSize.height)/float(levels[l].height);
				i->area.x = int(i->area.x*fx);
				i->area.y = int(i->area.y*fy);
				i->area.width  = int(i->area.width *fx);
				i->area.height = int(i->area.height*fy);
			}
			#pragma omp critical(vysledky_paralelnich_scalu)
			{
				detections.insert(detections.end(), results.begin(), results.begin()+n);
				detection_count += n;
			}
		}
	}
	return detection_count;
}

Pyramid *Pyramid::getInstance(unsigned id)
{
	return (id < instances.size()) ? instances[id] : NULL;
}
