/*
 *  LRD evaluation demonstration
 *  $Id$
 *
 *  facedetect [options] filename [filename ...]
 *
 *
 *  The program uses WaldBoost [1] classifiers to detect objects in images.  It
 *  uses XML classifier generated by the Research Boosting Framework [1].The
 *  classifiers are restricted to contain an ensemble of LRD [2] features. The
 *  detection engine implements simple LRD evaluation algorithm and fast
 *  algorithm using SSE2 instruction set. See [3] for comparison of the
 *  approaches.
 *  For additional information about the classifier training algorithm see [Sch97] and [Vio01]
 *
 *  [Soc05] WaldBoost
 *  [1] Hradis, M.: Boosting Framework for Research on Detection Classifiers,
 *      In SCCG, 2008, Budmerice, Slovakia
 *  [2] Zemcik, P., Herout, A., Hradis, M.: Local Rank Differences - Novel Features...
 *  [3] Local Rank Differences Image Features Implemented Using SIMD instructions of CPU
 *  [Vio01] Rapid Object Detection
 *  [Sch97] Real AdaBoost
 */

// STL
#include <iostream>
#include <fstream>
#include <vector>
#include <utility>
#include <map>
#include <ctime>
#include <algorithm>

// OpenCV
#include <cv.h>
#include <cxcore.h>
#include <highgui.h>

#ifdef _OPENMP
#include <omp.h>
#endif

#include "pyramid.h"

// The WaldBoost Engine
#include "lrd_engine.h"      // The detection engine
#include "faces-a020-1000.h" // built-in face-detection classifier; alpha = 0.2

using namespace std;


// Distance of centers of two rectangles
float rectDist(CvRect & a, CvRect & b)
{
    CvPoint cA = cvPoint(a.x, a.y);
    CvPoint cB = cvPoint(b.x, b.y);
    return sqrt(pow(float(cA.x - cB.x), 2) + pow(float(cA.y - cB.y), 2));
}


// Clustering and non-maxima supression
int groupDetections(TDetectionList & list, int count, unsigned minConfidence, float threshold)
{
    if (count == 0) return 0;
    std::vector< pair<unsigned, float> > binMap(count, pair<unsigned, float>(0, 0.0f));

    for (int i = 0; i < count; ++i)
    {
        binMap[i].first = i;
    }

    // Assign a bin to each detection - actual clustering
    for (int i = 0; i < count-1; ++i)
    {
        CvRect U = list[i].area;
        for (int j = i; j < count; ++j)
        {
            CvRect V = list[j].area;

            float d = rectDist(U, V);

            int ru = (U.width + U.height);
            int rv = (V.width + V.height);

            float r1 = float(min(ru, rv)) / 2;
            float r2 = float(max(ru, rv)) / 2;

            if (d > r1+r2)
            {
                continue;
            }

			float overlap = (r1/r2) * (1.0f - (d / (r1+r2)));

            if (overlap > threshold && overlap > binMap[j].second)
            {
                unsigned k = i;

                while (binMap[k].first != k)
                    k = binMap[k].first; // transitive closure, lol

                binMap[j].first = k;
                binMap[j].second = overlap;
            }
        }
    }


    std::map< unsigned, std::pair<unsigned, unsigned> > bins;

    // Select datection with max response from each cluster
    for (int i = 0; i < count; ++i)
    {
        unsigned bin = binMap[i].first;
        if (bins.find(bin) != bins.end())
        {
            if (list[i].response > list[bins[bin].first].response)
            {
                bins[bin].first = i;
            }
            bins[bin].second++;
        }
        else
        {
            bins[bin] = pair<unsigned,unsigned>(i, 1);
        }
    }

    // Copy selected detections to temp list
    int total = 0;
    TDetectionList grouped(count);
    std::map<unsigned, pair<unsigned,unsigned> >::iterator b = bins.begin();
    while (b != bins.end())
    {
        if (b->second.second >= minConfidence)
        {
            grouped[total] = list[b->second.first];
            ++total;
        }
        ++b;
    }

    // cpy the detections to original list
    copy(grouped.begin(), grouped.begin()+total, list.begin());

    return total;
}


void drawDetections(IplImage * inputFrame, TDetectionList::iterator first, TDetectionList::iterator last,
        CvScalar color, bool outline, bool printResponse, CvFont * font)
{
    for (TDetectionList::iterator i = first; i != last; ++i)
    {
        TDetection & d = *i;
        CvRect & r = d.area;
        float angle = d.angle;
        float response = d.response;

        // Prepare rotation matrix
        float _negRot1M[6];
        CvMat negRot1M = cvMat(2, 3, CV_32FC1, _negRot1M); // Origin rotation matrix
        cv2DRotationMatrix( cvPoint2D32f(0,0), -angle, 1.0, &negRot1M);

        // Left and Down vectors
        float _right[3] = {float(r.width), 0, 1}, _right1[3];
        float _down[3] = {0, float(r.height), 1}, _down1[3];

        CvMat right = cvMat(3, 1, CV_32F, _right);
        CvMat right1 = cvMat(2, 1, CV_32F, _right1);
        CvMat down = cvMat(3, 1, CV_32F, _down);
        CvMat down1 = cvMat(2, 1, CV_32F, _down1);

        cvMatMul(&negRot1M, &right, &right1);
        cvMatMul(&negRot1M, &down, &down1);

        CvPoint TL = cvPoint(int(r.x - _right1[0]/2 - _down1[0]/2), int(r.y - _right1[1]/2 - _down1[1]/2));
        CvPoint TR = cvPoint(int(r.x + _right1[0]/2 - _down1[0]/2), int(r.y + _right1[1]/2 - _down1[1]/2));
        CvPoint BL = cvPoint(int(r.x - _right1[0]/2 + _down1[0]/2), int(r.y - _right1[1]/2 + _down1[1]/2));
        CvPoint BR = cvPoint(int(r.x + _right1[0]/2 + _down1[0]/2), int(r.y + _right1[1]/2 + _down1[1]/2));

        // Black outline
        if (outline)
        {
            cvLine(inputFrame, TL, TR, cvScalar(0,0,0), 5);
            cvLine(inputFrame, TR, BR, cvScalar(0,0,0), 5);
            cvLine(inputFrame, BR, BL, cvScalar(0,0,0), 5);
            cvLine(inputFrame, BL, TL, cvScalar(0,0,0), 5);
        }

        // line
        cvLine(inputFrame, TL, TR, color);
        cvLine(inputFrame, TR, BR, color);
        cvLine(inputFrame, BR, BL, color);
        cvLine(inputFrame, BL, TL, color);

        if (font && printResponse)
        {
            char responseStr[256];
            sprintf(responseStr, "%3.2f", response);
            CvPoint textPos = TL;
            textPos.y -= 5;
            cvPutText(inputFrame, responseStr, TL, font, color);
        }

    } // Result loop
}


// Print Graph@FIT tag
void tagImage(IplImage * image, const char * text, CvFont * font)
{
    CvSize sz;
    int baseline = 0;
    cvGetTextSize(text, font, &sz, &baseline);

    cvRectangle(image, cvPoint(0, image->height-(baseline+sz.height+5)), cvPoint(image->width, image->height), CV_RGB(20,20,20), CV_FILLED);

    cvPutText(image, text, cvPoint(8, image->height-(baseline)), font, CV_RGB(255,255,255));
}



struct TProcessSettings
{
 	float threshold;
//    float pyrFactor;
//    float pyrMaxFactor;
	float scaleBase;
//    int scanStep;
	int minGroup;
    float groupThreshold;
    float maxAngle;
    int angleSteps;
    bool drawAllDetections;
    bool printTag;
    int startFrame;
    int frames;
    int frameStep;
    int smooth;
	int rthreads;
	int sthreads;

};
/*
struct TPyramidLevel
{
    IplImage * image;
    vector<TConvolution*> * convolutions;
    int * xAddressTable;
    int * yAddressTable;

    TPyramidLevel(): image(0), convolutions(0), xAddressTable(0), yAddressTable(0) {};
    ~TPyramidLevel()
    {
        //if (image) cvReleaseImage(&image);
        //releaseConvolutions(convlutions);
    }
};

CvSize align8(CvSize sz)
{
    CvSize new_sz;
    new_sz.width = sz.width & 0xFFFFFFF8;
    new_sz.height = sz.height & 0xFFFFFFF8;
    return new_sz;
}

vector<TPyramidLevel> * createPyramid(CvSize sz, const TProcessSettings & settings, int levels)
{
    vector<TPyramidLevel> * pyramid = new vector<TPyramidLevel>(levels);

    for (unsigned i = 0; i < pyramid->size(); ++i) // prepare images in different resolutions
    {
        float factor = settings.scaleBase * pow(settings.pyrFactor, float(i));
        CvSize levelSize = align8(cvSize(int(sz.width/factor), int(sz.height/factor)));
        // cout << "level " << i << ": " << levelSize.width << "x" << levelSize.height << endl;
        (*pyramid)[i].image = cvCreateImage(levelSize, IPL_DEPTH_8U, 1);
        (*pyramid)[i].convolutions = createConvolutions(levelSize);
        (*pyramid)[i].xAddressTable = prepareColAddressTable(*((*pyramid)[i].convolutions));
        (*pyramid)[i].yAddressTable = prepareRowAddressTable(*((*pyramid)[i].convolutions));

    }
    return pyramid;
}

void releasePyramid(vector<TPyramidLevel> ** pyramid)
{
    if (pyramid && *pyramid)
    {
        vector<TPyramidLevel> & p = **pyramid;
        for (unsigned i = 0; i < p.size(); ++i)
        {
            if (p[i].image) cvReleaseImage(&(p[i].image));
            releaseConvolutions(&p[i].convolutions);
            if (p[i].xAddressTable) delete [] p[i].xAddressTable;
            if (p[i].yAddressTable) delete [] p[i].yAddressTable;
        }
        delete *pyramid;
        *pyramid = 0;
    }
}
*/
int processImage(IplImage *inputFrame, Pyramid & pyramid, const TClassifier & classifier,
				  const TProcessSettings & settings, CvFont & font)
{
    // Flip image if needed
    if (inputFrame->origin)
    {
        inputFrame->origin = 0;
        cvFlip(inputFrame);
    }

    // FIXME: remove alloc - use preallocated images
    IplImage * inputGrayFrame = cvCreateImage(cvSize(inputFrame->width, inputFrame->height), IPL_DEPTH_8U, 1);

    cvCvtColor(inputFrame, inputGrayFrame, CV_RGB2GRAY);

    if (settings.smooth > 0)
    {
        cvSmooth(inputGrayFrame, inputGrayFrame, CV_GAUSSIAN, 2*settings.smooth+1);
    }

	float angleStep = (2 * settings.maxAngle) / (settings.angleSteps-1);
	TDetectionList detections;

	#pragma omp parallel num_threads(settings.rthreads)
	{
#ifdef _OPENMP
		Pyramid *instance = pyramid.getInstance(omp_get_thread_num());
		omp_set_num_threads(settings.sthreads);
#else
		Pyramid *instance = pyramid.getInstance(0);
#endif

		IplImage * grayFrame = cvCreateImage(cvSize(inputFrame->width, inputFrame->height), IPL_DEPTH_8U, 1);
		TDetectionList results;

		#pragma omp for schedule(dynamic, 1)
		for (int angle_index = 0; angle_index < settings.angleSteps; ++angle_index)
		{
			float angle = -settings.maxAngle + angle_index*angleStep;
			// Prepare rotation matrices for the angle
			float _rotM[6], _negRotM[6];
			CvMat rotM = cvMat(2, 3, CV_32FC1, _rotM);
			CvMat negRotM = cvMat(2, 3, CV_32FC1, _negRotM);
			cv2DRotationMatrix( cvPoint2D32f(grayFrame->width/2, grayFrame->height/2), angle, 1.0, &rotM);
			cv2DRotationMatrix( cvPoint2D32f(grayFrame->width/2, grayFrame->height/2), -angle, 1.0, &negRotM);

			cvWarpAffine(inputGrayFrame, grayFrame, &rotM);

			instance->Build(grayFrame);
			unsigned n = instance->Scan(classifier, cvGetSize(grayFrame), results);

			for(TDetectionList::iterator i = results.begin(); i != results.end(); ++i)
			{
				i->angle = angle;

				float _center[3]={float(i->area.x+i->area.width/2), float(i->area.y+i->area.height/2), 1}, _center1[3];
				CvMat center = cvMat(3,1,CV_32F,_center);
				CvMat center1 = cvMat(2,1,CV_32F,_center1);
				cvMatMul(&negRotM, &center, &center1);
				i->area.x = int(_center1[0]);
				i->area.y = int(_center1[1]);
			}
			#pragma omp critical(vysledky_paralelnich_rotaci)
			{
				detections.insert(detections.end(), results.begin(), results.end());
			}
			results.clear();
		}//for
		cvReleaseImage(&grayFrame);
	}//parallel block

	if (settings.drawAllDetections)
    {
        drawDetections(inputFrame, detections.begin(), detections.end(), cvScalar(255,255,255, 128), false, false, 0);
    }

    // Non-maxima suppression
    int groupedCount = groupDetections(detections, detections.size(), settings.minGroup, settings.groupThreshold);

    drawDetections(inputFrame, detections.begin(), detections.begin() + groupedCount, cvScalar(0,255,0), true, false, &font);

    cvReleaseImage(&inputGrayFrame);

    return groupedCount;
}

/*
int pyramidLevels(CvSize sz, float scaleBase, float factor, float maxFactor, int minSize)
{
    int levels = 0;
    float f = 1.0;
    while ((sz.width / f > minSize) && (sz.height / f > minSize) && f < maxFactor)
	{
		++levels;
		f = scaleBase * pow(factor, float(levels));
	}
    return levels;
}
*/

int main(int argc, char ** argv)
{
    // Global parameters
    TProcessSettings settings;

	settings.threshold = 0.0f;
//    settings.pyrFactor = pow(2.0f, 0.25f); //1.1f;
//    settings.pyrMaxFactor = 100.0f;
	settings.scaleBase = 1.0f;
//    settings.scanStep = 1;
	settings.minGroup = 2;
    settings.groupThreshold = 0.4;
    settings.maxAngle = 20.0f;
    settings.angleSteps = 5;
    settings.drawAllDetections = false;
    settings.printTag = true;
    settings.startFrame = 0;
    settings.frames = 0; // whole video
    settings.frameStep = 1;
    settings.smooth = 0;
	settings.rthreads = 1;
	settings.sthreads = 1;

#ifdef _OPENMP
	settings.rthreads = omp_get_num_procs();
	settings.sthreads = 1;
#endif

    bool quiet = false;
    bool showOutput = false;

    enum { I_IMAGE, I_VIDEO, I_CAMERA } input;

    char * classifierFile = 0;
    char * outFile = 0;
    const char * fourcc = "XVID";

    TClassifier * cls = &classifier;

    vector<char*> files;

    // prepare font
    CvFont font;
    cvInitFont(&font, CV_FONT_HERSHEY_PLAIN, 0.7, 0.8);

    ////
    // Process command line options
    // Override default parameters
    ////
    for (int a = 1; a < argc; ++a)
    {
		if (strcmp(argv[a], "-rthreads") == 0)
		{
			settings.rthreads = atoi(argv[++a]);
			continue;
		}
		if (strcmp(argv[a], "-sthreads") == 0)
		{
			settings.sthreads = atoi(argv[++a]);
			continue;
		}
        if (strcmp(argv[a], "-t") == 0)
        {
            settings.threshold = atof(argv[++a]);
            continue;
        }
/*        if (strcmp(argv[a], "-p") == 0)
        {
            settings.pyrFactor = atof(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-mf") == 0)
        {
            settings.pyrMaxFactor = atof(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-k") == 0)
        {
            settings.scanStep = atoi(argv[++a]);
            continue;
        }*/
        if (strcmp(argv[a], "-v") == 0)
        {
            settings.groupThreshold = atof(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-n") == 0)
        {
            settings.minGroup = atoi(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-a") == 0)
        {
            settings.maxAngle = atof(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-s") == 0)
        {
            settings.angleSteps = atoi(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-b") == 0)
        {
            settings.scaleBase = atof(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-smooth") == 0)
        {
            settings.smooth = atoi(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-start") == 0)
        {
            settings.startFrame = atoi(argv[++a]);
            continue;
        }
        if (strcmp(argv[a], "-frames") == 0)
        {
            settings.frames = max(0, atoi(argv[++a]));
            continue;
        }
        if (strcmp(argv[a], "-frame-step") == 0)
        {
            settings.frameStep = max(0, atoi(argv[++a]));
            continue;
        }
        if (strcmp(argv[a], "-o") == 0)
        {
            outFile = argv[++a];
            continue;
        }
        if (strcmp(argv[a], "-c") == 0)
        {
            classifierFile = argv[++a];
            continue;
        }
        if (strcmp(argv[a], "-fourcc") == 0)
        {
            fourcc = argv[++a];
            continue;
        }
        if (strcmp(argv[a], "-d") == 0)
        {
            settings.drawAllDetections = true;
            continue;
        }
        if (strcmp(argv[a], "-q") == 0)
        {
            quiet = true;
            continue;
        }
        if (strcmp(argv[a], "-show") == 0)
        {
            showOutput = true;
            continue;
        }
        files.push_back(argv[a]);
    }

#ifdef _OPENMP

	printf("threads num %d max %d\n", omp_get_num_threads(), omp_get_max_threads());
	printf("processors %d\n", omp_get_num_procs());
	printf("dynamic %d nested %d\n", omp_get_dynamic(), omp_get_nested());

	//omp_set_num_threads(settings.rthreads);
#else
	if(settings.rthreads > 1)
	{
		printf("multithreaded detection not suported\n");
		settings.rthreads = 1;
	}
#endif

	////
	// Image pyramid
	////
	Pyramid pyramid(settings.rthreads);

	////
    // Prepare classifier
    ////
    if (classifierFile)
    {
        //cls = loadClassifierXML(classifierFile);
        cerr << "External classifiers are not supported!" << endl;
        return -1;
    }

    if (!cls)
    {
        cerr << "Error: Cannot load '" << classifierFile << "'" << endl;
        return -1;
    }

    initClassifier(cls); // Initialization of a classifier
    cls->threshold = settings.threshold;

	////
    // Prepare results
    ////
    TDetectionList results(10000);

    ////
    // Process all files in the list
    ////
    int outId = 0; // The ID of file (for output file naming)
	vector<char*>::iterator file;

	for (file = files.begin(); file != files.end(); ++file, ++outId) // file loop
    {
		// Determine type of input
        if (strstr(*file, ".avi") != 0 || strstr(*file, ".mpg") != 0) // Video
        {
            input = I_VIDEO;
        }
        else if (strcmp(*file, "CAM") == 0)
        {
            input = I_CAMERA;
        }
        else
        {
            input = I_IMAGE;
        }

        if (input == I_VIDEO || input == I_CAMERA)
        {
            // The input is assumed to be video or camera

            // Open the camera or video
			CvCapture * inputVideo = (input == I_CAMERA) ? cvCaptureFromCAM(0) : cvCaptureFromFile(*file);

            // If the stream is opened init and process it
            if (!inputVideo)
            {
                cerr << "Error: Cannot load '" << *file << "'" << endl;
            }
            else
            {
                IplImage * inputFrame = cvQueryFrame(inputVideo);

				double fps = cvGetCaptureProperty(inputVideo, CV_CAP_PROP_FPS);
                //double width = cvGetCaptureProperty(inputVideo, CV_CAP_PROP_FRAME_WIDTH);
                //double height = cvGetCaptureProperty(inputVideo, CV_CAP_PROP_FRAME_HEIGHT);
                double width = inputFrame->width;
                double height = inputFrame->height;

				if (fps == 0) fps = 25;
                double outfps = fps/settings.frameStep; // Output video should have same speed as the input

                CvVideoWriter * outputVideo = 0;
                char outFileName[256];
                ////
                // Prepare output video writer
                ////
                if (outFile)
                {
                    if (files.size() == 1)
                    {
                        sprintf(outFileName, "%s.avi", outFile);
                    }
                    else
                    {
                        sprintf(outFileName, "%s%04d.avi", outFile, outId);
                    }
                    unsigned FOURCC = CV_FOURCC(fourcc[0], fourcc[1], fourcc[2], fourcc[3]);
                    outputVideo = cvCreateVideoWriter(outFileName, FOURCC, outfps, cvSize(int(width), int(height)));
                    if (!outputVideo)
                    {
                        cerr << "Warning: Cannot write to '" << outFileName << "'" << endl;
                    }
                } // file output

				//Prepare image pyramid
				pyramid.Init(256, cvSize(int(width/settings.scaleBase), int(height/settings.scaleBase)), *cls);

				//Process video
				for (int k = 0; k < settings.startFrame; ++k)
				{
					if (!cvGrabFrame(inputVideo)) break;
				}

                if (settings.frames == 0)
                    settings.frames = 100000;

				int f = 0;
                int totalFaces = 0;
				clock_t start = clock();
				float calcTime = 0;
				while (f < settings.frames)
				{
                    ////
					// Skip frameStep frames
					////
                    bool fuckup = false;
                    for (int k = 0; k < settings.frameStep; ++k)
                    {
                        if (!cvGrabFrame(inputVideo))
                        {
                            fuckup = true;
                            break;
                        }
                    }
                    if (fuckup) break;

                    ////
                    // Get a frame from the capture
                    ////
                    inputFrame = cvRetrieveFrame(inputVideo);
                    if (!inputFrame) break;

					clock_t start = clock();
					unsigned faces = processImage(inputFrame, pyramid, *cls, settings, font);
					totalFaces += faces;
					float time = static_cast<float>(clock()-start)/CLOCKS_PER_SEC;
					//printf("%f\n", time);
					calcTime += time;

                    ////
                    // Print tag
                    ////
                    if (settings.printTag)
                    {
                        char tag[256];
                        sprintf(tag, "Graph@FIT | WaldBoost | %d faces found | %d total faces |", faces, totalFaces);
                        tagImage(inputFrame, tag, &font);
                    }

                    ////
                    // Write processed frame
                    ////
                    if (outputVideo)
                    {
                        cvWriteFrame(outputVideo, inputFrame);
                    }

					if (showOutput)
                    {
                        cvNamedWindow("output", 1);
                        cvShowImage("output", inputFrame);
                        int c = cvWaitKey(10);
                        if (c == 'q') break;
                    }
                    cout << "." << flush;
                    ++f;
				}
				cvReleaseCapture(&inputVideo);
                cvReleaseVideoWriter(&outputVideo);

				float time = static_cast<float>(clock()-start)/CLOCKS_PER_SEC;
				printf("\nFrames %d, total time %f, per frame %f, avg calc %f\n", f, time, time/f, calcTime/f);
			}
		}
		else
		{
            assert(input == I_IMAGE);
            // The file is assumed to be image
            // Load the image
            IplImage * inputFrame = cvLoadImage(*file);
            if (!inputFrame)
            {
                cerr << "Error: Cannot load " << *file << "'" << endl;
            }
            else
            { // The image is loaded all right
                ////
                // Prepare the image pyramid
                ////
				pyramid.Init(256, cvSize(int(inputFrame->width/settings.scaleBase), int(inputFrame->height/settings.scaleBase)), *cls);

                ////
                // Process image
                ////
				unsigned faces = processImage(inputFrame, pyramid, *cls, settings, font);

                ////
                // write results
                ////
/*                cout << *file << "\t";
                for (TDetectionList::iterator i = results.begin(); i != results.begin() + detections; ++i)
                {
                    TDetection & d = *i;
                    CvRect & r = d.area;
                    float angle = d.angle;
                    float response = d.response;
                    cout << r.x << " " << r.y << " " << r.width << " " << r.height << " " << angle << " " << response << "  ";

                } // Result loop
                cout << endl;
*/
                ////
                // Print tag
                ////
                if (settings.printTag)
                {
                    char tag[256];
                    sprintf(tag, "Graph@FIT | WaldBoost | %d faces found", faces);
                    tagImage(inputFrame, tag, &font);
                }

                ////
                // Write output file
                // known bugs:
                // - The output is always JPEG
                ////
                if (outFile)
                {
                    char outFileName[256];
                    if (files.size() == 1)
                    {
                        // only one file given - save directly to output file
                        sprintf(outFileName, "%s.jpg", outFile);
                    }
                    else
                    {
                        // generate different output files for each input file
                        // {out}{x}.jpg, where out is given name and x order of the image in input list
                        sprintf(outFileName, "%s%04d.jpg", outFile, outId);
                    }
                    cvSaveImage(outFileName, inputFrame);
                } // save file
				cvReleaseImage(&inputFrame);
            } // image loaded
		}
    } // file loop

    /*
    if (classifierFile) // classifier was loaded from file - release it
        releaseClassifier(&cls);
    */
}
    /*
    if (!quiet)
    {
        cout << "# image size = " << inputFrame->width << "x" << inputFrame->height << endl;
        cout << "# detection threshold = " << threshold << endl;
        cout << "# pyramid factor = " << pyrFactor << endl;
        cout << "# max factor = " << pyrMaxFactor << endl;
        cout << "# pyramid levels = " << levels << endl;
        cout << "# scan step = " << scanStep << endl;
        cout << "# scale base = " << scaleBase << endl;
        cout << "# max angle = " << maxAngle << endl;
        cout << "# angle steps = " << angleSteps << endl;
        cout << "# min detection overlap = " << groupThreshold << endl;
        cout << "# min group = " << minGroup << endl;
    }

    */




	//cvNamedWindow("Image", 1);

    // Init images

    // flip if needed
   //cvSmooth(grayFrame, grayFrame, CV_GAUSSIAN, 11);

    /*
    resetStats();
    time_t tp0, tp1;

    tp0 = clock();

    unsigned detections = 0;
    scanTime = 0;





    tp1 = clock();
    totalTime = tp1 - tp0;

    float scanTimeMS = (1000.0f * float(scanTime)) / CLOCKS_PER_SEC;
    float totalTimeMS = (1000.0f * float(totalTime)) / CLOCKS_PER_SEC;

    if (!quiet)
    {
        cout << "# hypotheses evaluated = " << hypothesesEvaluated << endl;
        cout << "# windows evaluated = " << windowsEvaluated << endl;
        cout << "# average hypotheses per window = " << double(hypothesesEvaluated) / double(windowsEvaluated) << endl;
        cout << "# search time = " << scanTimeMS << " ms" << endl;
        cout << "# total time = " << totalTimeMS << " ms" << endl;
    }




    //cvShowImage("Image", inputFrame);


    } // file loop


    return 0;
}
*/
