/******************************************************************************
* TASTE (Testability Analysis SuiTE) version 1.00
* Copyright (c) 2008 Josef Strnadel, all rights reserved
* Brno University of Technology, Faculty of Information Technology
* -----------------------------------------------------------------------------
* This is a free software: you can redistribute it and/or modify it 
* under the terms of the latest version of the GNU Lesser General Public License 
* as published by the Free Software Foundation.
* 
* This software is distributed in the hope that it will be useful, 
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
* See the GNU General Public License and the GNU Lesser General Public License 
* for more details.
* 
* You should have received a copy of the GNU General Public License
* and GNU Lesser General Public License along with this software. 
* If not, see <http://www.gnu.org/licenses/>.
* -----------------------------------------------------------------------------
* filename:     model.cc
* version:      1.00
* started:      25. 4. 2007
* last revised: 21. 7. 2008
* language:     ANSI C++
* author:       Josef Strnadel
*               web:      http://www.fit.vutbr.cz/~strnadel/index.php.en
*               e-mail:   strnadel@fit.vutbr.cz
*               address:  Bozetechova 2, 61266 Brno, Czech Republic
*               phone:    +420 54114-1211
*               fax:      +420 54114-1270
* -----------------------------------------------------------------------------
* description:	circuit structure model
* -----------------------------------------------------------------------------
* notes:        -
* -----------------------------------------------------------------------------
* history of modifications:
*               -
* -----------------------------------------------------------------------------
* known bugs:   -
******************************************************************************/
//! \file model.cc module with circuit structure model and functions

#include <fstream>
#include <iostream>
#include <map>
#include <set>
#include <string>
#include <utility>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "misc.h"
#include "model.h"

using namespace std;

extern clsDesign *dsgnPtr;
extern bool debug_mode;
extern bool scan_mode;
extern tProPen mark_operation;
extern tSeqEst clk_cnt;
extern unsigned long nerrors;

extern string freg, fscan;

/******************************************************************************
*	class clsModuleType - implementation
******************************************************************************/
bool clsModuleType::addTranis(string s)
{
  bool b=false;
  if(debug_mode) cout<< "\t addTranis(): '" << s << "'" << endl; 

  string tempStr = s;
  StringTokenizer strtok = StringTokenizer(tempStr,"|");
  int ntokens = strtok.countTokens();

  if(ntokens==3) 
  {
      mTypeData data;
      string s_res;
      if(debug_mode) cout<< "\t\t" << ntokens << " token(s):" << endl; 
      for(int i = 0; i < ntokens; i++)
      {
        tempStr=strtok.nextToken();   
        if(debug_mode) cout<< "\t\ttoken " << i << ": '" << tempStr << "'" << endl; 
        switch(i)
        {
          case 0: data.from_name = tempStr; break;
          case 1: data.to_name = tempStr; break;
          case 2: data.cond = tempStr; break;
        }
      }
      data.composed = s;
      data.orig = 1;
      if(!(b=setOfTRANIS.insert(data).second)) {warn("Attempt to insert TRANIS '%s' for module-type '%s' failed!\n", s.c_str(), name.c_str()); return(false);}
  }
  else warn("Some information missing in TRANIS %s for module %s! Skipping insertion of this TRANIS.\n", s.c_str(), name.c_str());

  return(b);
}

bool clsModuleType::addPort(string s)
{
  bool b=false;
  string s_res = s;
  
  if(s.find("[") != string::npos)
  {
    string tempStr = s;
    StringTokenizer strtok = StringTokenizer(tempStr,"[");
    int ntokens = strtok.countTokens();
    
    if(ntokens!=2) {warn("Unsupported format for '%s' port in module-type '%s'! Insertion of the port aborted.\n", s_res.c_str(), name.c_str()); return(false);}
    else
    {
      s_res = strtok.nextToken(); 
      tempStr=strtok.nextToken(); 
      if(tempStr.find("]", tempStr.size()-1) == string::npos) {warn("Unsupported format for '%s' port in module-type '%s'! Insertion of the port aborted.\n", s_res.c_str(), name.c_str()); return(false);}
      else 
      {
        stringstream oss1, oss2;
        int i;

        tempStr.erase(tempStr.size()-1); 
        oss1 << tempStr;
        oss1 >> i;

        if(isInt(tempStr))
        {
          if(i>1) oss2 << "(" << i-1 << ":" << 0 << ")";
          else oss2 << "(" << 0 << ")";
        }
        else { oss2 << "(" << tempStr << "-1:0)";}
        s_res += oss2.str();
      }
    }
  }
  
  if(debug_mode) cout<< "\t addPort(): '" << s_res << "'" << endl; 
  if(!(b=setOfPorts.insert(s_res).second)) warn("Attempt to insert port '%s' for module-type '%s'!\n", s_res.c_str(), name.c_str());
  return(b);
}

/******************************************************************************
*	class clsWire - implementation
******************************************************************************/
clsWire::clsWire(string fm, string fp, int fb, string tm, string tp, int tb, bool orig)
{
  setOrig(orig);
  from_module=fm; from_port=fp; from_bit=fb; to_module=tm; to_port=tp; to_bit=tb;
  string _s1, _s2; 
  ostringstream _strm1, _strm2;
  
  if((from_module.size())>0)  // module port
    _strm1 << from_module << NAME_SEPARATOR << from_port << "(" << from_bit << ")";
  else                        // circuit/design... port (from_module name not needed)
    _strm1 << from_port << "(" << from_bit << ")";
  _s1 = _strm1.str();

  if((to_module.size())>0)  // module port
    _strm2 << to_module << NAME_SEPARATOR << to_port << "(" << to_bit << ")";
  else                        // circuit/design... port (to_module name not needed)
    _strm2 << to_port << "(" << to_bit << ")";
  _s2 = _strm2.str();
  
  from_composed = _s1;
  to_composed = _s2;
  composed = _s1 + "-" + _s2;
}

/******************************************************************************
*	class clsBit - implementation
******************************************************************************/
clsBit::clsBit(int bn, int nt, bool orig)
{
  setOrig(orig);
  bit_number = bn;
  node_type = nt;
  if(debug_mode) cout << "clsBit constructor for bit #" << bit_number << " called" << endl;
}

clsBit::~clsBit()
{
  if(debug_mode) cout << "clsBit destructor called for bit #" << bit_number << endl;
}

/******************************************************************************
*	class clsPort - implementation
******************************************************************************/
clsPort::clsPort(string nm, int nt, int w, bool orig)
{
  setOrig(orig);
  name = string(nm);
  node_type = nt;
  lbit=0;
  hbit=w-1;
  addBits(orig);
  if(debug_mode) cout << "clsPort constructor for object named '" << name << "' called" << endl;
}

clsPort::clsPort(string nm, int nt, int hb, int lb, bool orig)
{
  setOrig(orig);
  int i;
  name = string(nm);
  node_type = nt;

  if(lbit>hbit){ i=hbit; hbit=lbit; lbit=i; }
  
  lbit=lb;
  hbit=hb;
  if(hbit<lbit){i=lbit; lbit=hbit; hbit=i;}
  for(i=lbit; i<=hbit; i++) addBit(i, node_type, orig);
  if(debug_mode) cout << "clsPort constructor for object named '" << name << "' called" << endl;
}

clsPort::~clsPort()
{
  if(debug_mode) cout << "clsPort destructor called for object named '" << name << "'" << endl;
}

bool clsPort::addBit(int bn, int nt, bool orig)
{
  if(bn<0) {warn("Attempt to insert bit '%s(%d)' into mapBit failed because of bit-number < 0!\n", name.c_str(), bn); return(false);}
  bool b=mapBit.insert(vt_bit(bn, clsBit(bn, nt))).second;
  if(debug_mode) cout<< "addBit(): " << bn << endl; 
  if(!b) warn("Attempt to insert bit '%s(%d)' into mapBit failed!\n", name.c_str(), bn); 
  return(b);
}

int clsPort::getNCon() 
{
  map<int, clsBit>::iterator it; 
  int n=0; 
  clsBit *b=NULL; 
  for(it=mapBit.begin(); it!=mapBit.end(); it++)
  {
    if((b=&(*it).second)) 
      if(b->getTAdata().con>0) n++;
  }
  return(n);
}

int clsPort::getNObs()
{
  map<int, clsBit>::iterator it; 
  int n=0; 
  clsBit *b=NULL; 
  for(it=mapBit.begin(); it!=mapBit.end(); it++)
  {
    if((b=&(*it).second)) 
      if(b->getTAdata().obs>0) n++;
  }
  return(n);
}

int clsPort::getNTst()
{
  map<int, clsBit>::iterator it; 
  int n=0; 
  clsBit *b=NULL; 
  for(it=mapBit.begin(); it!=mapBit.end(); it++)
  {
    if((b=&(*it).second)) 
      if((b->getTAdata().con>0) && (b->getTAdata().obs>0)) n++;
  }
  return(n);
}

void clsPort::evalTst()
{
  map<int, clsBit>::iterator it; 
  double sumC=0.0, sumO=0.0; 
  clsBit *b=NULL; 
  t_TAdata data = getTAdata(), bit_data;

  for(it=mapBit.begin(); it!=mapBit.end(); it++) 
  { if((b=&(*it).second)) 
    { bit_data=b->getTAdata();
      sumC+=bit_data.con; sumO+=bit_data.obs; } }
  
  int w = getWidth();
  w = w*w;

  data.con = sumC*getNCon()/w;
  data.obs = sumO*getNObs()/w;
  data.tst = data.con*data.obs; 
  
  setTAdata(data);
}

/******************************************************************************
*	class clsModule - implementation
******************************************************************************/
clsModule::clsModule(string nm, string t, bool orig)
{
  setOrig(orig);
  name = string(nm);
  type = string(t);
  if(debug_mode) cout << "clsModule constructor for object named '" << name << "' called" << endl;
}

clsModule::~clsModule()
{
  if(debug_mode) cout << "clsModule destructor called for object named '" << name << "'" << endl;
}

/*! @param[in] s_names constant names to be added to the module
 *  @param[in] s_values constant values to be assigned to constant names
*/
bool clsModule::addConst(string s_names, string s_values)
{
  bool b=true;
  StringTokenizer strtok1 = StringTokenizer(s_names, "_");
  StringTokenizer strtok2 = StringTokenizer(s_values, "_");
  int i1=strtok1.countTokens();
  int i2=strtok2.countTokens();
  if(i1!=i2) return(false);
  string str1, str2;
  for(int i=0; i<i1; i++)
  {
    str1 = strtok1.nextToken();
    str2 = strtok2.nextToken();
    if(isInt(str2))
    {
      int val;              
      stringstream sstrm;
      sstrm << str2;
      sstrm >> val;          // convert string to int
      b = b&&mapConsts.insert(vt_strint(str1, val)).second;
    }
    else {b=false; warn("Attempt to insert const '%s'=%s into mapConsts of module '%s' failed. The value assigned to the constant should be of int type!\n", str1.c_str(), str2.c_str(), name.c_str());}
  }
  if(debug_mode) cout<< "addConst(): " << s_names << ", value(s): " << s_values << endl; 
  return(b);
}

int clsModule::evalExpr(string nm)
{
  if(isInt(nm)) {return(str2int(nm));}
  map<string, int>::iterator it; long res=-1; string nm_orig = nm;
  for(it=mapConsts.begin(); it!=mapConsts.end(); it++)
  {
    unsigned int i, val;
    string str = (*it).first, str2;
    while((i=nm.find(str)) != string::npos) // replace each constant-name by its constant-value
    {
      val = (*it).second;
      str2 = int2str(val);
      nm.replace(i, str.size(), str2);
      if(debug_mode) cout << "result-string: " << nm << endl;
    }
  }
  if(debug_mode) cout << "result-evaluated: " << nm;
  if (ExpressionEvaluator::calculateLong(nm, res) != ExpressionEvaluator::eval_ok){res=-1;}
  if(debug_mode) cout << " = " << res << endl;

  return(res);
}

/*!
 *  @param[in] nm name of the module port is to be added to
 *  @param[in] nt port type (see ::tNode)
 *  @param[in] w port width
 *  @param[in] orig originality flag (implicitly set to true)
 */
bool clsModule::addPort(string nm, int nt, int w, bool orig)
{
  bool b=mapPort.insert(vt_port(nm, clsPort(nm, nt, w, orig))).second;
  if(debug_mode) cout<< "module.addPort(): " << nm << ", orig: " << orig << endl; 
  if(!b) warn("Attempt to insert port '%s' into mapPort of module '%s' failed!\n", nm.c_str(), name.c_str()); 
  return(b);
}

/*!
 *  @param[in] nm name of the module port is to be added to
 *  @param[in] nt port type (see ::tNode)
 *  @param[in] hbit highest bit in the port
 *  @param[in] lbit lowest bit in the port
 *  @param[in] orig originality flag (implicitly set to true)
 */
bool clsModule::addPort(string nm, int nt, int hbit, int lbit, bool orig)
{
  bool b=mapPort.insert(vt_port(nm, clsPort(nm, nt, hbit, lbit, orig))).second;
  if(debug_mode) cout<< "module.addPort(): " << nm << ", orig: " << orig << endl;  
  if(!b) warn("Attempt to insert port '%s' into mapPort of module '%s' failed!\n", nm.c_str(), name.c_str()); 
  return(b);
}

/*!
 *  @param[in] mtptr pointer to module-type module's interface will be synchronized with
 *  @param[in] orig originality flag (implicitly set to true)
 */
bool clsModule::addInterface(clsModuleType *mtptr, bool orig)
{
  bool b;
  
  if(mtptr)
  {  
    if(debug_mode){cout << "addInterface() for module '" << name << "' by means of module-type '" << mtptr->getName() << "'" << endl;}
    set<string>::iterator it_sports;
    if(debug_mode) cout << "\tInterface loaded:" << (((mtptr->getSPortsBegin()) == (mtptr->getSPortsEnd()))?" none":"") << endl;
    for(it_sports = mtptr->getSPortsBegin(); it_sports != mtptr->getSPortsEnd(); it_sports++)
    {
      if(debug_mode) cout << "\tport: '" << *it_sports << "'" << endl;
      
      StringTokenizer strtok = StringTokenizer(*it_sports, "@");
      int ntokens = strtok.countTokens();
      
      if(ntokens==2)
      {
        string ptyp = strtok.nextToken();
        if(debug_mode) cout << "token1 = " << ptyp << endl;
        int ptype = 0;
        if(ptyp=="in") {ptype=NODE_IN;}
        if(ptyp=="out") {ptype=NODE_OUT;}
        if(ptyp=="inout") {ptype=NODE_IN|NODE_OUT;}
        if(ptyp=="sel") {ptype=NODE_CTRL;}
        if(ptyp=="clk") {ptype=NODE_CLK;}
  
        string pw = strtok.nextToken(), port_name, port_range;
        if(debug_mode) cout << "token2 = " << pw << endl;
        port_name = str_before(pw, "(");
        if(debug_mode) cout << "token3 = " <<  port_name << endl;
        port_range = str_between(pw, "(", ")");
        if(debug_mode) cout << "token4 = " <<  port_range << endl;
        if(debug_mode) cout << "ptype = " <<  ptype << endl;

        int hbit=-1, lbit=-1;
        stringstream ss1, ss2;
        if(port_range.find(":") != string::npos)  // range detected
        {
          string high = str_before(port_range, ":");
          string low = str_after(port_range, ":");
          if(isInt(high)) // highbit-number given
            {ss1 << high; ss1 >> hbit;}
          else // identifier given -> find its value
            { hbit = evalExpr(high);}  // eval expressions like 'n-1' etc.
          
          if(isInt(low)) // lowbit-number given
            {ss2 << low; ss2 >> lbit;}
          else // identifier given -> find its value
            {lbit = evalExpr(low); }  // eval expressions like 'n-1' etc.
          
          if(debug_mode) {cout << "range " << hbit << ":" << lbit << endl;}
        }else{                                      // not range, i.e., 1 bit only
          if(isInt(port_range)) // bit-number given
          {
            ss1 << port_range;
            ss1 >> hbit;
            lbit = hbit;            
          }else{  // identifier given -> find its value
            hbit=evalExpr(port_range);
            lbit=hbit;
          }
          if(debug_mode) {cout << "single " << hbit << ":" << lbit << endl;}
        }

      	unsigned int nbits = abs(hbit-lbit)+1;
      	stringstream sstrm;
        sstrm << pw;
      	sstrm >> nbits;
      	
      	string nm;
        if((hbit>=0)&&(lbit>=0)&&ptype) { addPort(port_name, ptype, hbit, lbit, orig); /*addPort(port_name, ptype, nbits);*/}
        else warn("During addInterface(module '%s'): Type '%s' for port '%s' not supported (it should be one of: in, out, inout, sel, clk) - please check data for module-type '%s' in its library! Skipping insertion of the port.\n", name.c_str(), ptyp.c_str(), port_name.c_str(), (mtptr->getName()).c_str());
      }
      else {warn("Something wrong with format of '%s' port!\n", (*it_sports).c_str());}
    }
  }
  else warn("Attempt to insert interface to module type '%s' failed!\n", name.c_str());

  return(b);
}

bool clsModule::bitsExist(string bits)
{
  bool b=false;
  string s_lbit, s_hbit;
  int lbit, hbit;
  string port_nm, range;
  clsPort *p = NULL;
  if(debug_mode){cout << "Checking existence of bits from '" << bits <<"'" << endl;}
  port_nm = str_before(bits, "(");  range = str_between(bits, "(", ")");
  if((p = findPort(port_nm)))
  {
    if(range.find("^") != string::npos) // ^ found
    {
      string s1=str_before(range, "^");
      string s2=str_after(range, "^");

      if(s1.find(":") == string::npos) // range in 'i' or '4' format
      {
        if(s2 == "i"){ b = ((p->findBit(p->getLBit())) != NULL); } 
        else {if(isInt(s1)){b = ((p->findBit(str2int(s1))) != NULL); }}
      }
    }
    else  // ^ not found
    {
      if(range.find(":") == string::npos) // range in 'i' or '4' format
      {
        if(range == "i"){b = ((p->findBit(p->getLBit())) != NULL); } 
        else {if(isInt(range)){b = ((p->findBit(str2int(range))) != NULL); }}
      }
      else  // range in '0:5' format
      {
        b=true;   int i;
        s_lbit = str_before(range, ":");  s_hbit = str_after(range, ":");
        lbit = str2int(s_lbit);           hbit = str2int(s_hbit);
        if(lbit>hbit) {i=lbit; lbit=hbit; hbit=i;}
        for(i=lbit; i<=hbit; i++){b &= ((p->findBit(i)) != NULL); };
      }    
    }
  }
  return(b);
}

/*! @param[in] s tranis information
 *  @param[in] orig originality flag (implicitly set to true)
*/
bool clsModule::addTranis(string s, bool orig)
{
  bool b=false;
  if(debug_mode) cout<< "\t module.addTranis(): '" << s << "'" << endl; 

  string tempStr = s;
  StringTokenizer strtok = StringTokenizer(tempStr,"|");
  int ntokens = strtok.countTokens();

  if(debug_mode) cout<< "\t ntokens: '" << ntokens << endl; 

  if(ntokens==3) 
  {
      mTypeData data;
      string s_res;
      if(debug_mode) cout<< "\t\t" << ntokens << " token(s):" << endl; 
      for(int i = 0; i < ntokens; i++)
      {
        tempStr=strtok.nextToken();   
        if(debug_mode) cout<< "\t\ttoken " << i << ": '" << tempStr << "'" << endl; 
        switch(i)
        {
          case 0: data.from_name = tempStr; break;
          case 1: data.to_name = tempStr; break;
          case 2: data.cond = tempStr; break;
        }
      }
      data.composed = s;
      data.orig = orig;
      if(!(b=setOfTRANIS.insert(data).second)) {warn("Attempt to insert TRANIS '%s' for module-type '%s' failed!\n", s.c_str(), name.c_str()); return(false);}
  }
  else warn("Some information missing in TRANIS %s for module %s! Skipping insertion of this TRANIS.\n", s.c_str(), name.c_str());

  return(b);
}

/*! @param[in] mtptr poiter to module-type tranises are to be synchronized with
 *  @param[in] orig originality flag (implicitly set to true)
*/
bool clsModule::addTranspData(clsModuleType *mtptr, bool orig)
{
  bool b=false;
  bool i_set=false;
  int i_low=-1, i_high=-1, i_val=-1;
  
  if(mtptr)
  {  
    string s_tmp, s_range, s_frombit, s_tobit, s_res;
    string s_TRANISfrom, s_TRANISto, s_TRANIScond, s_TRANIS;
    int from_bit, to_bit;
    
    if(debug_mode)
    {cout << "addTranspData() for module '" << name << "' by means of module-type '" << mtptr->getName() << "'" << endl;}
    
    set<tMTypeData, lt_mtdata>::iterator it_stranis;
    if(debug_mode) {cout << "###\tTRANIS loaded:" << (((mtptr->getSTranisBegin()) == (mtptr->getSTranisEnd()))?" none":"") << endl;}
    bool checkBitsOK;
    for(it_stranis = mtptr->getSTranisBegin(); it_stranis != mtptr->getSTranisEnd(); it_stranis++)
    { 
      i_set = false;
      if(debug_mode){
        cout << "###\t-from: '" << (it_stranis->from_name) <<"'" << endl;
        cout << "###\t-to: '" << (it_stranis->to_name) << "'" << endl;
        cout << "###\t-cond: '" << (it_stranis->cond) << "'" << endl;
        cout << "###\t-composed: '" << (it_stranis->composed) << "'" << endl;}

      // (1) synchronize from-name
      s_tmp = (it_stranis->from_name);
      StringTokenizer strtok = StringTokenizer(s_tmp, ")");
      int ntokens = strtok.countTokens(); bool fromOK = true; checkBitsOK = true; s_TRANISfrom = "";
      for(int i=0; i<ntokens; i++)
      { string s1=strtok.nextToken();   s1 = s1+")";    s_res = s1;
        if((s_range = str_between(s1, "(", ")")).size())
        { if(s_range.find(":") != string::npos)
          { s1 = str_before(s1, "(");
            s_frombit = str_before(s_range, ":");   from_bit = evalExpr(s_frombit);
            s_tobit = str_after(s_range, ":");      to_bit = evalExpr(s_tobit);
            if((from_bit==-1)||(to_bit==-1)){warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) cannot be evaluated! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); fromOK=false;}
            if(from_bit==to_bit) s_res = s1+"(" + int2str(from_bit) + ")";
            else s_res = s1+"(" + int2str(from_bit) + ":" + int2str(to_bit) + ")";
          }else { 
            s1 = str_before(s1, "(");   int res = evalExpr(s_range);
            if(s_range != "i"){
              if(res==-1) {warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) cannot be evaluated! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); fromOK=false;}
              s_res = s1+"(" + int2str(res) + ")"; 
            }
            else {
              clsPort *p=NULL;
              if(debug_mode) {cout << "##### solving i_val problem: " << s1 << endl;}

              p=findPort(s1);
              if(p && !i_set)
              {
                if(debug_mode) {cout << "##### " << s1  << " found" << endl;}
                i_low=p->getLBit();
                i_high=p->getHBit();
                i_set = true;

                if(debug_mode) {cout << "##### i_range set to " << i_low <<":" <<i_high << endl;}
              }
            }
          }
        } s_TRANISfrom += s_res;  checkBitsOK &= bitsExist(s_res);
        if(!checkBitsOK){ warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) does not exist(s)! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); }
      }
      if(debug_mode) {cout << "... TRANISfrom = '" << s_TRANISfrom << "'" << endl;}
      
      // (2) synchronize to-name
      s_tmp = (it_stranis->to_name);
      strtok = StringTokenizer(s_tmp, ")");
      ntokens = strtok.countTokens(); bool toOK = true; s_TRANISto = "";
      for(int i=0; i<ntokens; i++)
      { string s1=strtok.nextToken();   s1 = s1+")";    s_res = s1;
        if((s_range = str_between(s1, "(", ")")).size())
        { if(s_range.find(":") != string::npos)
          { s1 = str_before(s1, "(");
            s_frombit = str_before(s_range, ":");   from_bit = evalExpr(s_frombit);
            s_tobit = str_after(s_range, ":");      to_bit = evalExpr(s_tobit);
            if((from_bit==-1)||(to_bit==-1)){warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) cannot be evaluated! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); toOK=false;}
            if(from_bit==to_bit) s_res = s1+"(" + int2str(from_bit) + ")";
            else s_res = s1+"(" + int2str(from_bit) + ":" + int2str(to_bit) + ")";
          }else { 
            s1 = str_before(s1, "(");   int res = evalExpr(s_range);
            if(s_range != "i"){
              if(res==-1) {warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) cannot be evaluated! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); toOK=false;}
              s_res = s1+"(" + int2str(res) + ")"; 
            }
            else {
              clsPort *p=NULL;
              if(debug_mode) {cout << "##### solving i_val problem: " << s1 << endl;}

              p=findPort(s1);
              if(p && !i_set)
              {
                if(debug_mode) {cout << "##### " << s1  << " found" << endl;}
                i_low=p->getLBit();
                i_high=p->getHBit();
                i_set = true;

                if(debug_mode) {cout << "##### i_range set to " << i_low <<":" <<i_high << endl;}
              }
            }
          }
        } s_TRANISto += s_res;  checkBitsOK &= bitsExist(s_res);
        if(!checkBitsOK){ warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) does not exist(s)! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); }
      }
      if(debug_mode) {cout << "... TRANISto = '" << s_TRANISto << "'" << endl;}

      // (3) synchronize cond
      s_tmp = (it_stranis->cond);
      strtok = StringTokenizer(s_tmp, ")");
      ntokens = strtok.countTokens(); bool condOK = true; s_TRANIScond = "";
      for(int i=0; i<ntokens; i++)
      { string s1=strtok.nextToken();   
        if(s1=="-") {s1="-";} else {if(s1.find("(") != string::npos) s1 = s1+")"; } s_res = s1;
        if((s_range = str_between(s1, "(", ")")).size())
        { if(s_range.find(":") != string::npos)
          { s1 = str_before(s1, "(");
            s_frombit = str_before(s_range, ":");   from_bit = evalExpr(s_frombit);
            s_tobit = str_after(s_range, ":");      to_bit = evalExpr(s_tobit);
            if((from_bit==-1)||(to_bit==-1)){warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) cannot be evaluated! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); condOK=false;}
            if(from_bit==to_bit) s_res = s1+"(" + int2str(from_bit) + ")";
            else s_res = s1+"(" + int2str(from_bit) + ":" + int2str(to_bit) + ")";
          }else { 
          
            if(s_range.find("^") != string::npos) // ^ found
            {
              string ss1 = str_before(s_range, "^");
              string ss2 = str_after(s_range, "^");
              int res = evalExpr(ss2);

              s1 = str_before(s1, "(");   

              if(ss2 != "i")
              {              
                if(debug_mode) cout << "in " << s1 << ": ^ found (" << ss1 << " powered to " << ss2 << "=" << res <<")"<< endl;
                s_res = s1+"(" + ss1 + "^" + int2str(res) + ")";
                if(debug_mode) cout << "TRANIS to be inserted: " << s_res << endl;
              }
              else {
                clsPort *p=NULL;
                if(debug_mode) {cout << "##### solving i_val problem: " << s1 << endl;}
  
                p=findPort(s1);
                if(p && !i_set)
                {
                  if(debug_mode) {cout << "##### " << s1  << " found" << endl;}
                  i_low=p->getLBit();
                  i_high=p->getHBit();
                  i_set = true;
  
                  if(debug_mode) {cout << "##### i_range set to " << i_low <<":" <<i_high << endl;}
                }
              }
            }
            else  // without ^
            {             
              s1 = str_before(s1, "(");   int res = evalExpr(s_range);
              if(s_range != "i"){
                if(res==-1) {warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) cannot be evaluated! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); condOK=false;}
                s_res = s1+"(" + int2str(res) + ")"; 
              }
              else {
                clsPort *p=NULL;
                if(debug_mode) {cout << "##### solving i_val problem: " << s1 << endl;}
    
                p=findPort(s1);
                if(p && !i_set)
                {
                  if(debug_mode) {cout << "##### " << s1  << " found" << endl;}
                  i_low=p->getLBit();
                  i_high=p->getHBit();
                  i_set = true;
    
                  if(debug_mode) {cout << "##### i_range set to " << i_low <<":" <<i_high << endl;}
                }
              }
            }
          }
        } s_TRANIScond += s_res;  if(s_res!="-") checkBitsOK &= bitsExist(s_res);
        if(!checkBitsOK){ warn("Attempt to insert TRANIS'%s'-part '%s' for module '%s' failed because range-bit(s) does not exist(s)! TRANIS insertion skipped.\n", s_tmp.c_str(), s_res.c_str(), name.c_str()); }
      }
      if(debug_mode) {cout << "... TRANIScond = '" << s_TRANIScond << "'" << endl;
                      cout << "TRANIS-bitsExist = " << (checkBitsOK?"true":"false") << endl;}
      
      //add TRANIS to the module
      if(checkBitsOK && fromOK && toOK && condOK)
      {
          s_TRANIS = s_TRANISfrom+"|"+s_TRANISto+"|"+s_TRANIScond;
          
          if(i_set)
          {
            for(i_val=i_low; i_val<=i_high; i_val++)
            {
              string s_tmp = s_TRANIS;

              if(debug_mode) {cout << "setting i to: " << i_val << endl;}
              s_tmp = replaceSubstrWithVal(s_tmp, "i)", int2str(i_val)+")");
              addTranis(s_tmp, orig); 
              if(debug_mode) {cout << "final TRANIS*: '" << s_tmp << "'" << endl;}
            }
          }
          else
          {
            addTranis(s_TRANIS, orig); 
            if(debug_mode) {cout << "final TRANIS**: '" << s_TRANIS << "'" << endl;}
          }
      }
    }
  }

  return(b);
}

bool clsModule::changeTypeTo(string type)
{
  bool res=true;

  setOrig(false); // module-modified TAG

  if(debug_mode) cout << "'" << getName() << "'.changeTypeTo() '" << type << "' started..." << endl;

  string tmp = str_between(type, "<", ">");  
  
  StringTokenizer strtok = StringTokenizer(tmp,"_");
  int ntokens = strtok.countTokens();

  for(int i=0; i<ntokens; i++)  // get numerical value of each template-parameter
  {
    map<string, int>::iterator it; 
    string s = strtok.nextToken();

    if((it = mapConsts.find(s)) != mapConsts.end())
    {
      res &= (isInt(int2str((*it).second)));
      
      tmp = replaceSubstrWithVal(tmp, (*it).first, int2str((*it).second));      
    }
    else { warn("Compatibility problem occured during conversion '%s' to '%s': constant '%s' not defined in '%s' scope. Effect: conversion aborted.\n", 
    getName().c_str(), type.c_str(), s.c_str(), getName().c_str()); return(false);}
  }

  type = str_before(type, "<") + tmp + str_after(type, ">");  // pass valued-arguments to make an template instance
  clsModule tmp_sreg("tmp_sreg", type, false);

  dsgnPtr->syncModWithType(&tmp_sreg);  // create tmp sReg object to be used for Reg modification

  // synchronize Reg and Sreg area, pwr, interface
  if(debug_mode) cout << "... synchronizing areas" << endl;
  setAreaOrig(getArea());       // store original area-value
  setArea(tmp_sreg.getArea());  // set new area-value

  if(debug_mode) cout << "... synchronizing pwr" << endl;
  setPowerOrig(getPower());        // store original pwr-value
  setPower(tmp_sreg.getPower());  // set new pwr-value

  if(debug_mode) cout << "... synchronizing interfaces" << endl;

  map<string, clsPort>::iterator it_p;
  clsPort *p=NULL;
  for(it_p = tmp_sreg.getMapPortBegin(); it_p != tmp_sreg.getMapPortEnd(); it_p++)  
  { 
    if((p = &((*it_p).second)))
    {
      if(debug_mode) cout << "\t... port '" << p->getName() << "'" << endl;
      mapPort.insert(vt_port(p->getName(), clsPort(p->getName(), p->getType(), p->getWidth(), false))).second;
    }
    else { warn("'%s'.changeTypeTo(%s): Port '%s' not found - synchronization of the port aborted.\n", getName().c_str(), type.c_str(), (*it_p).first.c_str()); }
  }

  // synchronize Reg and Sreg TRANISes
  if(debug_mode) cout << "... synchronizing TRANISes" << endl;

  set<tMTypeData, lt_mtdata>::iterator it_tr;
  tMTypeData data;
  for(it_tr = tmp_sreg.getSTranisBegin(); it_tr != tmp_sreg.getSTranisEnd(); it_tr++)  
  { 
      if(debug_mode) cout << "\t... TRANIS '" << it_tr->composed << "'" << endl;
      data = *it_tr;
      data.orig = false;
      setOfTRANIS.insert(data).second;
  }

  if(debug_mode) cout << "'" << getName() << "'.changeTypeTo() '" << type << "' ended with result: " << (res?"OK":"ABORTED") << endl;
  
  return(res);
}

bool clsModule::typeBack()
{
  bool res=false;

  setOrig(true); // module-modified TAG

  // remove all non-original (orig=0) ports and TRANISes:

  // ports
  if(debug_mode) cout << "... ports back" << endl;

  map<string, clsPort>::iterator it_p;
  clsPort *p=NULL;
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)  
  { 
    if((p = &((*it_p).second)))
    {
      if(debug_mode) cout << "\t... port '" << p->getName() << "', remove: " << ((p->getOrig())?"NO":"YES") << endl;
    }
  }

  // TRANISes
  if(debug_mode) cout << "... TRANISes back" << endl;

  set<tMTypeData, lt_mtdata>::iterator it_tr;
  tMTypeData data;
  for(it_tr = getSTranisBegin(); it_tr != getSTranisEnd(); it_tr++)  
  { 
      if(debug_mode) cout << "\t... TRANIS '" << it_tr->composed << "', remove: " << ((it_tr->orig)?"NO":"YES") << endl;
  }
   
  // restore original area and power
  setArea(getAreaOrig());
  setPower(getPowerOrig());
  
  return(res);
}

/******************************************************************************
*	class clsCircuit - implementation
******************************************************************************/
clsCircuit::clsCircuit(string nm)
{
  name = string(nm);
  if(debug_mode) cout << "clsCircuit constructor for object named '" << name << "' called" << endl;
}

clsCircuit::~clsCircuit()
{
  if(debug_mode) cout << "clsCircuit destructor called for object named '" << name << "'" << endl;
}

int clsCircuit::getNBits()
{
  int nbits=0;

  clsModule* m = NULL;
  clsPort* p = NULL;
  map<string, clsPort>::iterator it_p;
  map<string, clsModule>::iterator it_m;
  map<int, clsBit>::iterator it_b;
  string str_tmp;

  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)  // count PI bits
  { p = &((*it_p).second);
    if(p) if((p->getType()) & (NODE_IN|NODE_CTRL)) nbits+=p->getWidth(); }

  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)  // count in-module bits
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)  
    { p = &((*it_p).second);
      if(p) if((p->getType()) & (NODE_IN|NODE_CTRL)) nbits+=p->getWidth(); }
  }
  
  return(nbits);
}

int clsCircuit::getNAllBits()
{
  int nbits=0;

  clsModule* m = NULL;
  clsPort* p = NULL;
  map<string, clsPort>::iterator it_p;
  map<string, clsModule>::iterator it_m;
  map<int, clsBit>::iterator it_b;
  string str_tmp;

  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)  // count PI bits
  { p = &((*it_p).second);
    if(p) if((p->getType())) nbits+=p->getWidth(); }

  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)  // count in-module bits
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)  
    { p = &((*it_p).second);
      if(p) if((p->getType())) nbits+=p->getWidth(); }
  }
  
  return(nbits);
}

int clsCircuit::getNClks()
{
  int nbits=0;
  int npbits=0;

  clsModule* m = NULL;
  clsPort* p = NULL;
  map<string, clsPort>::iterator it_p;
  map<string, clsModule>::iterator it_m;
  map<int, clsBit>::iterator it_b;
  string str_tmp;

  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)  // count PI bits
  { p = &((*it_p).second);
    if(p) if((p->getType()) & (NODE_CLK)) { nbits+=p->getWidth(); npbits+=p->getWidth(); }
  }

  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)  // count in-module bits
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)  
    { p = &((*it_p).second);
      if(p) if((p->getType()) & (NODE_CLK)) nbits+=p->getWidth(); }
  }

  set<tMTypeData, lt_mtdata>::iterator it_stranis;
  int nbits_from_tranises = npbits;
  string pnm, exp;
  int exp_val;
 
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  {
    m = &((*it_m).second);
    
    for(it_stranis = m->getSTranisBegin(); it_stranis != m->getSTranisEnd(); it_stranis++)
    {
      string tmp;

      StringTokenizer strtok(it_stranis->cond, ")");
      int i, ntokens = strtok.countTokens();
      
      for(i=0; i<ntokens; i++)
      {
        tmp = strtok.nextToken();
        tmp += ")";
        pnm = str_before(tmp, "(");
        exp = str_between(tmp, "^", ")");
        if(exp=="") exp="0";
        exp_val = str2int(exp);

        p = m->findPort(pnm);
        if(p)
        {
          if((p->getType()) & (NODE_CLK))
          {
            exp_val++;
          }
        }
        nbits_from_tranises += exp_val;
      }
    }
  }

  if(clk_cnt == SE_PRECISE) return(nbits_from_tranises);
  else return(nbits);
}

/*! @param[in] nm port name
 *  @param[in] nt port type (see ::tNode)
 *  @param[in] w port width
 *  @param[in] orig port originality flag (set implicitly to true for all original ports)
 */
bool clsCircuit::addPort(string nm, int nt, int w, bool orig)
{
  bool b=mapPort.insert(vt_port(nm, clsPort(nm, nt, w, orig))).second;
  if(debug_mode) cout<< "circuit.addPort(): " << nm << ", orig: " << orig << endl; 
  if(!b) warn("Attempt to insert port '%s' into mapPort of circuit '%s' failed!\n", name.c_str(), nm.c_str()); 
  return(b);
}

int clsCircuit::getNallPorts()
{
  int nports=0;

  clsModule* m = NULL;
  map<string, clsPort>::iterator it_p;
  map<string, clsModule>::iterator it_m;
  map<int, clsBit>::iterator it_b;
  string str_tmp;

  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)  // count PI bits
  { nports++; }

  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)  // count in-module bits
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)  
    { nports++; }
  }
  
  return(nports);
}

double clsCircuit::getAvgPortWidth()
{
  return(getNAllBits()/getNallPorts());
}

/*! @param[in] nm module name
 *  @param[in] t module type 
 *  @param[in] orig module originality flag (set implicitly to true for all original modules)
 */
bool clsCircuit::addMod(string nm, string t, bool orig)
{
  bool b=mapMod.insert(vt_mod(nm, clsModule(nm, t, orig))).second;
  if(debug_mode) cout<< "addMod(): '" << nm << "'"<< endl; 
  if(!b) warn("Attempt to insert module '%s' into mapMod failed!\n", nm.c_str()); 
  return(b);
}


bool clsCircuit::addWire(clsWire w)
{
  bool b;
  unsigned long _nerrors;
  clsModule *m = NULL;
  clsPort *p = NULL;

  _nerrors = nerrors;

  if(debug_mode) cout<< "addWire(): " << w.composed << ", orig: " << w.getOrig() << endl; 

  if(w.from_module.size()>0) // modules and module-ports
  if((m = findMod(w.from_module)))
  {
    if((p = m->findPort(w.from_port)))
    {
      if(p->findBit(w.from_bit))
      {
      }
      else warn("addWire(): '%s' not inserted because SRC-bit %ld does not exist!\n", (w.composed).c_str(), w.from_bit);
    }
    else warn("addWire(): '%s' not inserted because SRC-port '%s' does not exist!\n", w.composed.c_str(), w.from_port.c_str());
  }
  else warn("addWire(): '%s' not inserted because SRC-module '%s' does not exist!\n", w.composed.c_str(), w.from_module.c_str());

  if(w.to_module.size()>0) // modules and module-ports
  if((m = findMod(w.to_module)))
  {
    if((p = m->findPort(w.to_port)))
    {
      if(p->findBit(w.to_bit))
      {
      }
      else warn("addWire(): '%s' not inserted because DST-bit %ld does not exist!\n", (w.composed).c_str(), w.to_bit);
    }
    else warn("addWire(): '%s' not inserted because DST-port '%s' does not exist!\n", w.composed.c_str(), w.to_port.c_str());
  }
  else warn("addWire(): '%s' not inserted because DST-module '%s' does not exist!\n", w.composed.c_str(), w.to_module.c_str());

  // ********************************************

  if(w.from_module.size()==0) // circuit and circuit-ports
  if((p = findPort(w.from_port)))
  {
    if(p->findBit(w.from_bit))
    {
    }
    else warn("addWire(): '%s' not inserted because SRC-bit %ld does not exist in circuit '%s' interface!\n", (w.composed).c_str(), w.from_bit, name.c_str());
  }
  else warn("addWire(): '%s' not inserted because SRC-port '%s' does not exist in circuit '%s' interface!\n", w.composed.c_str(), w.from_port.c_str(), name.c_str());

  if(w.to_module.size()==0) // circuit and circuit-ports
  if((p = findPort(w.to_port)))
  {
    if(p->findBit(w.to_bit))
    {
    }
    else warn("addWire(): '%s' not inserted because DST-bit %ld does not exist in circuit '%s' interface!\n", (w.composed).c_str(), w.from_bit, name.c_str());
  }
  else warn("addWire(): '%s' not inserted because DST-port '%s' does not exist in circuit '%s' interface!\n", w.composed.c_str(), w.from_port.c_str(), name.c_str());

  b = (_nerrors == nerrors);
  if(b) setOfWires.insert(w).second; 

  return(b);
}

/*! @param[in] sm name of source module 
 *  @param[in] sp name of source port (belonging to the source module)
 *  @param[in] dm name of destination module 
 *  @param[in] dp name of destination port (belonging to the destination module)
 *  @param[in] reverse reversability flag (false to connect ports in given direction, true to connoect them in the oposit direction)
 *  @param[in] orig connection originality flag (set implicitly to true for all original interconnections)
 */
bool clsCircuit::connectPorts(string sm, string sp, string dm, string dp, bool reverse, bool orig)
{
  bool b=false;
  int wf, wt;
  clsModule *fm = NULL, *tm = NULL;
  clsPort *fp = NULL, *tp = NULL;
  string s_fh="", s_fl="", s_frange = "", 
         s_th="", s_tl="", s_trange = "";
  int fh=-1, fl=-1, th=-1, tl=-1;

  if(debug_mode) cout << "connecting ports " << sm+"."+sp << " and " << dm+"."+dp << endl;
  if(sp.find("(") != string::npos)
  {
    if(debug_mode) cout << "range detected" << endl;
    s_frange=str_between(sp, "(", ")");
    sp = str_before(sp, "(");
    if(s_frange.size())
    {
      s_fh = str_before(s_frange, ":");
      s_fl = str_after(s_frange, ":");
      
      stringstream ss1, ss2;
      ss1 << s_fh;  ss2 << s_fl;
      ss1 >> fh;    ss2 >> fl;

      if(debug_mode) {cout << "from: " << fh << endl; cout << "to: " << fl << endl;}
    }
  }
  if(dp.find("(") != string::npos)
  {
    if(debug_mode) cout << "range detected" << endl;
    s_trange=str_between(dp, "(", ")");
    dp = str_before(dp, "(");
    if(s_trange.size())
    {
      s_th = str_before(s_trange, ":");
      s_tl = str_after(s_trange, ":");
      
      stringstream ss1, ss2;
      ss1 << s_th;  ss2 << s_tl;
      ss1 >> th;    ss2 >> tl;

      if(debug_mode) {cout << "from: " << th << endl; cout << "to: " << tl << endl;}
    }
  }

  // check existence of from-port
  if(sm.size()) // module-port
  {
    fm=findMod(sm);
    if(fm){ fp = fm->findPort(sp); 
      if(fp){wf=fp->getWidth();}
      else { warn("connectPorts(): Attempt to connect ports '%s' and '%s' failed because SRC-port does not exist!\n", (sm+"."+sp).c_str(), (dm+"."+dp).c_str()); }
    }  
    else { warn("connectPorts(): Attempt to connect ports '%s' and '%s' failed because SRC-module does not exist!\n", (sm+"."+sp).c_str(), (dm+"."+dp).c_str()); }
  }
  else  // circuit-port
  {
    fp = findPort(sp); 
    if(fp){wf=fp->getWidth();}
    else { warn("connectPorts(): Attempt to connect ports '%s' and '%s' failed because SRC-port does not exist!\n", (sm+"."+sp).c_str(), (dm+"."+dp).c_str()); }
  }

  // check existence of to-port
  if(dm.size()) // module-port
  {
    tm=findMod(dm);
    if(tm){ tp = tm->findPort(dp); 
      if(tp){wt=tp->getWidth();}
      else { warn("connectPorts(): Attempt to connect ports '%s' and '%s' failed because DST-port does not exist!\n", (sm+"."+sp).c_str(), (dm+"."+dp).c_str()); }
    }  
    else { warn("connectPorts(): Attempt to connect ports '%s' and '%s' failed because DST-module does not exist!\n", (sm+"."+sp).c_str(), (dm+"."+dp).c_str()); }
  }
  else  // circuit-port
  {
    tp = findPort(dp); 
    if(tp){wt=tp->getWidth();}
    else { warn("connectPorts(): Attempt to connect ports '%s' and '%s' failed because DST-port does not exist!\n", (sm+"."+sp).c_str(), (dm+"."+dp).c_str()); }
  }
  
  if((fp!=NULL)&&(tp!=NULL))  // connect bits of the ports
  {
    b=true;
    int if_l=(fl>=0)?fl:fp->getLBit(), 
        if_h=(fh>=0)?fh:fp->getHBit(), 
        it_l=(tl>=0)?tl:tp->getLBit(), 
        it_h=(th>=0)?th:tp->getHBit();
    if(debug_mode) {cout << "fl: " << if_l << ", fh: " << if_h << ", tl: " << it_l << ", th: " << it_h << endl;}
    if((abs(if_h-if_l))!=(abs(it_h-it_l))){ warn("connectPorts(): Attempt to connect ports '%s' and '%s' failed because they are of various bit-widths! There are many ways how to connect them!\n", (sm+"."+sp).c_str(), (dm+"."+dp).c_str()); return(false);}

    bool fcnt_up = (if_l<if_h),  // for '0:3 -> xxx' COUNT-SRC-UP
         tcnt_up = (it_l<it_h);  // for 'xxx -> 0:3' COUNT-DST-UP
    
    int w = (abs(if_h-if_l)+1);
    
    for(int i=0; i<w; i++)
    { 
      if((!fcnt_up)&&(!tcnt_up))  // SRC--, DST--
      {
        if(!reverse) b = b && addWire(clsWire(sm, sp, if_l--, dm, dp, it_l--, orig));  
        else b = b && addWire(clsWire(sm, sp, if_l--, dm, dp, it_h++, orig)); 
      }
      else if((fcnt_up)&&(!tcnt_up)) // SRC++, DST--
      {
        if(!reverse) b = b && addWire(clsWire(sm, sp, if_l++, dm, dp, it_l--, orig));  
        else b = b && addWire(clsWire(sm, sp, if_l++, dm, dp, it_h++, orig)); 
      }
      else if((!fcnt_up)&&(tcnt_up)) // SRC--, DST++
      {
        if(!reverse) b = b && addWire(clsWire(sm, sp, if_l--, dm, dp, it_l++, orig));  
        else b = b && addWire(clsWire(sm, sp, if_l--, dm, dp, it_h++, orig)); 
      }
      else  // SRC++, DST++
      {
        if(!reverse) b = b && addWire(clsWire(sm, sp, if_l++, dm, dp, it_l++, orig));  // connect lower-bits with lower bits + higher bits with higher bits
        else b = b && addWire(clsWire(sm, sp, if_l++, dm, dp, it_h--, orig)); // connect lower-bits with higher bits + higher bits with lower bits
      }
    }
  }
  
  return(b);
}

//! clears values stored in all testability-data blocks within the circuit
void clrTAdata(clsCircuit *c)
{
  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;

  // clear PI-port(s) data
  for(it_p = c->getMapPortBegin(); it_p != c->getMapPortEnd(); it_p++)
  { p = &((*it_p).second);
    if(p) {p->clrTAdata(); // clr port-data
    for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
    { b = &((*it_b).second);
      if(b) b->clrTAdata();}}  // clr bit-data
  }
  // clear inner-port(s) data
  for(it_m = c->getMapModBegin(); it_m != c->getMapModEnd(); it_m++)
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
    { p = &((*it_p).second); 
      if(p) {p->clrTAdata(); // clr port-data
      for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
      { b = &((*it_b).second);
        if(b) b->clrTAdata();}}  //clr bit-data
    }
  }
}

//! dumps data stored in given set of string
void dump_string_set(set<string> sset, string id)
{
  set<string>::iterator it;
  cout << "Dumping '" << id << "' members: " << ((sset.begin() == sset.end())?"none":"");
  
  for(it = sset.begin(); it != sset.end(); it++) {cout << *it << " ";}
  cout << endl;
}

void clrTAdata(t_TAdata *ta_data);
void setTAdata(t_TAdata *ta_data);
void dumpTAdata(t_TAdata *ta_data);

//! transports mark through wire
/*! @param[in] c pointer to clsCircuit object the wire belongs to
 * @param[in] wire interconnection that will be utilized to transport the mark
 * @param[in] Cmode direction of transportation process: during controllability (observability) analysis, marks are transported in direction from PIs to POs (POs to PIs)
*/
void transport_wireMark(clsCircuit *c, clsWire wire, bool Cmode)
{
  clsBit *fbit = NULL, *tbit = NULL;
  clsPort *p = NULL;
  clsModule *m = NULL;

  //1) find from-bit
  if(!wire.from_module.size()) // circuit-bit
  { if((p = c->findPort(wire.from_port))){ fbit = p->findBit(wire.from_bit); } }
  else // module-bit
  { if((m = c->findMod(wire.from_module))) { if((p = m->findPort(wire.from_port))){ fbit = p->findBit(wire.from_bit); } } }
  if(!fbit) {warn("Attempt to transport mark throught '%s' connection failed because of from-bit problem!", (wire.composed).c_str()); return;}
  
  //2) find to-bit
  if(!wire.to_module.size()) // circuit-bit
  { if((p = c->findPort(wire.to_port))){ tbit = p->findBit(wire.to_bit); } }
  else // module-bit
  { if((m = c->findMod(wire.to_module))) { if((p = m->findPort(wire.to_port))){ tbit = p->findBit(wire.to_bit); } } }
  if(!tbit) {warn("Attempt to transport mark throught '%s' connection failed because of to-bit problem!", (wire.composed).c_str()); return;}
  
  //3) transport mark
  if(Cmode)                                       // controllability-analysis mode
  { 
    t_TAdata dta_t = tbit->getTAdata(), dta_f = fbit->getTAdata();
    if(dta_f.con > dta_t.con) tbit->setTAdata(dta_f);
  }  
  else                                           // observability-analysis mode
  {
    t_TAdata dta_t = tbit->getTAdata(), dta_f = fbit->getTAdata();
    
    // remain fbit's controllability results unchanged ...
    dta_t.nbitsC = dta_f.nbitsC; 
    dta_t.nclksC = dta_f.nclksC; 
    dta_t.multCprev = dta_f.multCprev; 
    dta_t.multCctrl = dta_f.multCctrl; 
    dta_t.con = dta_f.con; 
    
    if(dta_t.obs > dta_f.obs) fbit->setTAdata(dta_t);   // ... transfer only O-mark

    //dumpTAdata(&dta_t);
  }
}

//! clears values stored in testability-data block 
void clrTAdata(t_TAdata *ta_data){ta_data->nbitsC=ta_data->nclksC=ta_data->nbitsO=ta_data->nclksO=0; 
  ta_data->multCprev=ta_data->multCctrl=ta_data->multOprev=ta_data->multOctrl=ta_data->con=ta_data->obs=ta_data->tst=0.0;}

//! sets values to testability-data block 
void setTAdata(t_TAdata *ta_data){ta_data->nbitsC=ta_data->nclksC=ta_data->nbitsO=ta_data->nclksO=1; 
  ta_data->multCprev=ta_data->multCctrl=ta_data->multOprev=ta_data->multOctrl=ta_data->con=ta_data->obs=ta_data->tst=1.0;}

//! dumps data stored in given testability-data block
void dumpTAdata(t_TAdata *ta_data)
{
  cout << "\t\t\tTA-data info:" << endl;
  cout << "\t\t\tnbitsC: " << ta_data->nbitsC;
  cout << ", nclksC: " << ta_data->nclksC;
  cout << ", nbitsO: " << ta_data->nbitsO;
  cout << ", nclksO: " << ta_data->nclksO;
  cout << ", multCprev: " << ta_data->multCprev;
  cout << ", multCctrl: " << ta_data->multCctrl;
  cout << ", multOprev: " << ta_data->multOprev;
  cout << ", multOctrl: " << ta_data->multOctrl;
  cout << ", con: " << ta_data->con;
  cout << ", obs: " << ta_data->obs;
  cout << ", tst: " << ta_data->tst << endl;
}
  
//! computes and returns average of values stored in testability-data blocks of specified module bits 
/*! @param[in] m pointer to clsModule object
 *  @param[in] bits string of bits for computing average 
*/
t_TAdata getAvgOfTAdata(clsModule *m, string bits)
{
  clsPort *p = NULL;
  clsBit *b = NULL;
  set<string>::iterator s_it;
  string pname, tmp;
  int bit, nbits=0, ntokens;
  t_TAdata sumTAdata, bitTAdata, res;

  clrTAdata(&sumTAdata);
 
  if(m)
  {
      StringTokenizer strtok = StringTokenizer(bits, ")");
      ntokens = strtok.countTokens();
      for(int i=0; i<ntokens; i++)
      {
        nbits++;
        tmp = strtok.nextToken() + ")";
        tmp = str_after(tmp, ".");  // remove module-name
        pname = str_before(tmp, "("); // get port-name
        if((p = (m->findPort(pname))))
        {
          bit = str2int(str_between(tmp, "(", ")"));  // get bit-name
          if((b = (p->findBit(bit))))
          {
            bitTAdata = b->getTAdata();
            
            sumTAdata.nbitsC+=bitTAdata.nbitsC;
            sumTAdata.nclksC+=bitTAdata.nclksC;
            sumTAdata.nbitsO+=bitTAdata.nbitsO;
            sumTAdata.nclksO+=bitTAdata.nclksO;
            sumTAdata.multCprev+=bitTAdata.multCprev;
            sumTAdata.multCctrl+=bitTAdata.multCctrl;
            sumTAdata.multOprev+=bitTAdata.multOprev;
            sumTAdata.multOctrl+=bitTAdata.multOctrl;
            sumTAdata.con+=bitTAdata.con;
            sumTAdata.obs+=bitTAdata.obs;
            sumTAdata.tst+=bitTAdata.tst;

          } else {res.tst = -1.0; return(res);}
        } else {res.tst = -1.0; return(res);}
      }
  } else {res.tst = -1.0; return(res);}

  res.nbitsC=sumTAdata.nbitsC/nbits;
  res.nclksC=sumTAdata.nclksC/nbits;
  res.nbitsO=sumTAdata.nbitsO/nbits;
  res.nclksO=sumTAdata.nclksO/nbits;
  res.multCprev=sumTAdata.multCprev/nbits;
  res.multCctrl=sumTAdata.multCctrl/nbits;
  res.multOprev=sumTAdata.multOprev/nbits;
  res.multOctrl=sumTAdata.multOctrl/nbits;
  res.con=sumTAdata.con/nbits;
  res.obs=sumTAdata.obs/nbits;
  res.tst=sumTAdata.tst/nbits;

  return(res);
}

//! computes and returns product of values stored in testability-data blocks of specified module bits 
/*! @param[in] m pointer to clsModule object
 *  @param[in] bits string of bits for computing product
 *  @param[out] nbits number of control-bits to be added to actual mark
 *  @param[out] nclks number of clock-bits to be added to actual mark
*/
t_TAdata getProdOfTAdata(clsModule *m, string bits, int *nbits, int *nclks)
{
  clsPort *p = NULL;
  clsBit *b = NULL;
  set<string>::iterator s_it;
  string pname, tmp;
  int bit, ntokens;
  t_TAdata res, bitTAdata;

  setTAdata(&res);
 
  if(m)
  {
      StringTokenizer strtok = StringTokenizer(bits, ")");
      ntokens = strtok.countTokens();
      for(int i=0; i<ntokens; i++)
      {
        //nbits++;
        tmp = strtok.nextToken() + ")";
        tmp = str_after(tmp, ".");  // remove module-name
        pname = str_before(tmp, "("); // get port-name
        if((p = (m->findPort(pname))))
        {
          bit = str2int(str_between(tmp, "(", ")"));  // get bit-name
          if((b = (p->findBit(bit))))
          {
            // clk-bit and data-bit counters
            if((b->getType()) & NODE_CLK){ (*nclks)++; } 
            else { (*nbits)++; }

            bitTAdata = b->getTAdata();
            
            res.nbitsC*=bitTAdata.nbitsC;
            res.nclksC*=bitTAdata.nclksC;
            res.nbitsO*=bitTAdata.nbitsO;
            res.nclksO*=bitTAdata.nclksO;
            res.multCprev*=bitTAdata.multCprev;
            res.multCctrl*=bitTAdata.multCctrl;
            res.multOprev*=bitTAdata.multOprev;
            res.multOctrl*=bitTAdata.multOctrl;
            res.con*=bitTAdata.con;
            res.obs*=bitTAdata.obs;
            res.tst*=bitTAdata.tst;

          } else {res.tst = -1.0; return(res);}
        } else {res.tst = -1.0; return(res);}
      }
  } else {res.tst = -1.0; return(res);}

  return(res);
}

//! transports controllability mark through module structure
/*! @param[in] c pointer to clsCircuit object module belongs to
 *  @param[in] module name of the module
 *  @param[in] olds set set of bits having controllability mark assigned
 *  @param[out] news set of bits marked during transportation through module structure
*/
void transport_modCmark(clsCircuit *c, string module, set<string> *olds, set<string> *news)
{
  set<tMTypeData, lt_mtdata>::iterator it_stranis, it_stranis2;
  clsModule *mptr = NULL;
  string str_tmp, str_tmp2;
  int ntk1, ntk2, lbit, rbit, n;

  t_TAdata ta_data_avg, ta_data_prod;   
  string s_fbits, s_cbits;
  int addBits, addClks;

  bool bits_ready;

  if(debug_mode) cout << "Transporting mark(s) for module: '" << module << "'" << endl;

  if((mptr = c->findMod(module)))
  {
      for(it_stranis = mptr->getSTranisBegin(); it_stranis != mptr->getSTranisEnd(); it_stranis++)
      {
        bool fbits_ready = true;
        s_fbits = "";
        if(debug_mode) cout << "\t... checking in-module TRANIS'" << (it_stranis->composed) << "'";// << endl;
        StringTokenizer stk1 = StringTokenizer(it_stranis->from_name, ")");
        ntk1=stk1.countTokens();
        for(int i=0; i<ntk1; i++) // check if from-bits are in ready list 
        {
          str_tmp = stk1.nextToken() + ")";

          StringTokenizer stk2 = StringTokenizer(str_tmp, ":");
          ntk2=stk2.countTokens();
          if(ntk2==1) // port_name(n) format
          {
            lbit = (str2int(str_between(str_tmp, "(", ")")));  
            rbit = lbit;  
          }
          else if(ntk2==2)  // port_name(n:m) format
          {
            str_tmp2 = str_between(str_tmp, "(", ")");
            lbit = (str2int(str_before(str_tmp2, ":")));  
            rbit = (str2int(str_after(str_tmp2, ":")));  
          }
          if(lbit<rbit) {n=rbit; rbit=lbit; lbit=n;}
          
          for(int i=lbit; i>=rbit; i--) // check readyness of particular from-bits
          {
            str_tmp2 = module + "." + str_before(str_tmp, "(") + "(" + int2str(i) + ")";
            s_fbits += str_tmp2;
            fbits_ready &= (olds->find(str_tmp2) != olds->end());
          }
        }
        if(debug_mode) cout << ", from-part " << (fbits_ready?"READY":"NOT READY");
        
        bool cbits_ready = true;
        addBits = addClks = 0;
        s_cbits = "";
        stk1 = StringTokenizer(it_stranis->cond, ")");
        ntk1=stk1.countTokens();
        int power_to = 1;
        for(int i=0; i<ntk1; i++) // check if cond-bits are in ready list 
        {
          str_tmp = stk1.nextToken() + ")";

          if(str_tmp.find("^") != string::npos) // ^ found
          {
            string ss = str_before(str_after(str_tmp, "^"), ")");
            power_to = str2int(ss)+1;
            str_tmp = str_before(str_tmp, "^") + ")";
          }

          StringTokenizer stk2 = StringTokenizer(str_tmp, ":");
          ntk2=stk2.countTokens();
          if(ntk2==1) // port_name(n) format
          {
            lbit = (str2int(str_between(str_tmp, "(", ")")));  
            rbit = lbit;  
          }
          else if(ntk2==2)  // port_name(n:m) format
          {
            str_tmp2 = str_between(str_tmp, "(", ")");
            lbit = (str2int(str_before(str_tmp2, ":")));  
            rbit = (str2int(str_after(str_tmp2, ":")));  
          }
          if(lbit<rbit) {n=rbit; rbit=lbit; lbit=n;}

          for(int i=lbit; i>=rbit; i--) // check readyness of particular cond-bits
          {
            str_tmp2 = module + "." + str_before(str_tmp, "(") + "(" + int2str(i) + ")";
            s_cbits += str_tmp2;
            cbits_ready &= (olds->find(str_tmp2) != olds->end());
          }
        }
        if(debug_mode) cout << ", cond-part " << (cbits_ready?"READY":"NOT READY");
        bits_ready = fbits_ready & cbits_ready;
        if(debug_mode) cout << "... TRANIS src+cond bits " << (bits_ready?"READY":"NOT READY") << endl;

        if(bits_ready)  // mark transportation
        {
          {
              ta_data_avg = getAvgOfTAdata(mptr, s_fbits); // averages of from-data
              ta_data_prod = getProdOfTAdata(mptr, s_cbits, &addBits, &addClks); // product of cond-data
              
              if((ta_data_avg.tst != -1.0) && (ta_data_prod.tst != -1.0))  // all-bits-data OK
              {
                if(debug_mode) cout << "\t\t### to-be-transported mark data:" << endl;
      
                if(clk_cnt == SE_PRECISE) addClks *= power_to;
      
                int nbits = c->getNBits();
                int nclks = c->getNClks();
                int nb = ta_data_avg.nbitsC + addBits;
                int nc = ta_data_avg.nclksC + addClks;
                double multc_;
                double multc;
                
                if(mark_operation==PP_LIN)
                {
                  multc_ = ta_data_avg.multCprev / power_to;
                  multc = (multc_ * ta_data_prod.multCctrl) / power_to;
                }
                else
                {
                  multc_ = pow(ta_data_avg.multCprev, power_to);
                  multc = pow(multc_ * ta_data_prod.multCctrl, power_to);
                }
                
                double con = (1-(nb/(double)(nbits+1)))*(1-(nc/(double)(nclks+1)))*multc;

                if(debug_mode)
                {    
                    cout << "### power_to: " << power_to << endl; 

                    cout << "### avgData: " <<endl; dumpTAdata(&ta_data_avg); 
                    cout << "### productData: " <<endl; dumpTAdata(&ta_data_prod);

                    cout << "\t\t\tadding " << addBits << " nbitsC, sum = " << nb << endl;
                    cout << "\t\t\tadding " << addClks << " nclksC, sum = " << nc << endl;
                    cout << "\t\t\tsetting multc- to " << multc_ << endl;
                    cout << "\t\t\tsetting multc to " << multc << endl;
                    cout << "\t\t\tgetting # of in-circuit in+sel+... bits: " << nbits << endl; 
                    cout << "\t\t\tgetting # of in-circuit clk bits: " << nclks << endl; 
                    cout << "\t\t\tbracket1: " << (1-(nb/(double)(nbits+1))) << endl; 
                    cout << "\t\t\tbracket2: " << (1-(nc/(double)(nclks+1))) << endl; 
                    cout << "\t\t\tsetting con to " << con << endl << endl;
                }
      
                t_TAdata propag_data; // data to be set to to-bits

                if(debug_mode) cout << "\t\t###### ...power_to="<<power_to << endl;
                
                propag_data.nbitsC=nb;
                propag_data.nclksC=nc;
                propag_data.multCprev=multc_;
                propag_data.multCctrl=multc;
                propag_data.multOprev=0.0;
                propag_data.multOctrl=0.0;
                propag_data.con=con;
                propag_data.obs=0.0;
                propag_data.tst=0.0;

                if(debug_mode) dumpTAdata(&propag_data);
                
                // propagate mark to to-bits of actual TRANIS
                StringTokenizer stk1 = StringTokenizer(it_stranis->to_name, ")");
                  if(debug_mode) cout << "\t\t### propagating mark to '" << (it_stranis->to_name) << "' bit(s)" << endl;

                int ntk1=stk1.countTokens(), lbit, rbit;

                for(int i=0; i<ntk1; i++) 
                {
                  str_tmp = stk1.nextToken() + ")";

                  StringTokenizer stk2 = StringTokenizer(str_tmp, ":");
                  int ntk2=stk2.countTokens();
                  if(ntk2==1) // port_name(n) format
                  {
                    lbit = (str2int(str_between(str_tmp, "(", ")")));  
                    rbit = lbit;  
                  }
                  else if(ntk2==2)  // port_name(n:m) format
                  {
                    str_tmp2 = str_between(str_tmp, "(", ")");
                    lbit = (str2int(str_before(str_tmp2, ":")));  
                    rbit = (str2int(str_after(str_tmp2, ":")));  
                  }
                  if(lbit<rbit) {n=rbit; rbit=lbit; lbit=n;}
                  
                  for(int i=lbit; i>=rbit; i--) // propagate mark
                  {
                    string pname = str_before(str_tmp, "(");
                    clsPort *pptr=NULL;
                    clsBit *bptr=NULL;
                    str_tmp2 = module + "." + pname + "(" + int2str(i) + ")";

                    if((pptr=(mptr->findPort(pname))))
                    {
                      if((bptr=(pptr->findBit(i))))
                      {
                        if(bptr->getTAdata().con < propag_data.con) // transport better-mark only
                        {
                          if(debug_mode) cout << "\t\t### ...transporting mark to '" << str_tmp2 << "' bit" << endl;
                          if(debug_mode) cout << "\t\t### ...power_to="<<power_to << endl;
                          bptr->setTAdata(propag_data);
                          news->insert(str_tmp2);
                        }
                        else {if(debug_mode) cout << "\t\t### ...mark to '" << str_tmp2 << "' bit NOT transported" << endl;}
                      }  
                    }
                  }
                }
              }
            }
        }
      }
  }
  else error("Attempt to transport mark through module %s failed! Because module-ptr is NULL!\n", module.c_str());
}

//! transports observability mark through module structure
/*! @param[in] c pointer to clsCircuit object module belongs to
 *  @param[in] module name of the module
 *  @param[in] olds set of bits having observability mark assigned
 *  @param[in] markedCbits set of bits having controllability mark assigned
 *  @param[out] news set of bits marked during transportation through module structure
*/
void transport_modOmark(clsCircuit *c, string module, set<string> *olds, set<string> *markedCbits, set<string> *news)
{
  set<tMTypeData, lt_mtdata>::iterator it_stranis, it_stranis2;
  clsModule *mptr = NULL;
  string str_tmp, str_tmp2;
  int ntk1, ntk2, lbit, rbit, n;
 
  t_TAdata ta_data_avg, ta_data_prod;   
  string s_fbits, s_cbits;
  int addBits, addClks;

  bool bits_ready;

  if(debug_mode) cout << "Transporting mark(s) for module: '" << module << "'" << endl;
  if((mptr = c->findMod(module)))
  {
      for(it_stranis = mptr->getSTranisBegin(); it_stranis != mptr->getSTranisEnd(); it_stranis++)
      {
        bool fbits_ready = true;
        s_fbits = "";
        if(debug_mode) cout << "\t... checking in-module TRANIS'" << (it_stranis->composed) << "'";// << endl;
        StringTokenizer stk1 = StringTokenizer(it_stranis->to_name, ")");
        ntk1=stk1.countTokens();

        for(int i=0; i<ntk1; i++) // check if to-bits are in ready list 
        {
          str_tmp = stk1.nextToken() + ")";

          StringTokenizer stk2 = StringTokenizer(str_tmp, ":");
          ntk2=stk2.countTokens();
          if(ntk2==1) // port_name(n) format
          {
            lbit = (str2int(str_between(str_tmp, "(", ")")));  
            rbit = lbit;  
          }
          else if(ntk2==2)  // port_name(n:m) format
          {
            str_tmp2 = str_between(str_tmp, "(", ")");
            lbit = (str2int(str_before(str_tmp2, ":")));  
            rbit = (str2int(str_after(str_tmp2, ":")));  
          }
          if(lbit<rbit) {n=rbit; rbit=lbit; lbit=n;}
          
          for(int i=lbit; i>=rbit; i--) // check readyness of particular from-bits
          {
            str_tmp2 = module + "." + str_before(str_tmp, "(") + "(" + int2str(i) + ")";
            s_fbits += str_tmp2;
            fbits_ready &= (olds->find(str_tmp2) != olds->end());
          }
        }
        if(debug_mode) cout << ", from-part " << (fbits_ready?"READY":"NOT READY");
        
        bool cbits_ready = true;
        addBits = addClks = 0;
        s_cbits = "";
        stk1 = StringTokenizer(it_stranis->cond, ")");
        ntk1=stk1.countTokens();
        int power_to = 1;
        for(int i=0; i<ntk1; i++) // check if cond-bits are in ready list 
        {
          str_tmp = stk1.nextToken() + ")";

          if(str_tmp.find("^") != string::npos) // ^ found
          {
            string ss = str_before(str_after(str_tmp, "^"), ")");
            power_to = str2int(ss)+1;
            str_tmp = str_before(str_tmp, "^") + ")";
          }

          StringTokenizer stk2 = StringTokenizer(str_tmp, ":");
          ntk2=stk2.countTokens();
          if(ntk2==1) // port_name(n) format
          {
            lbit = (str2int(str_between(str_tmp, "(", ")")));  
            rbit = lbit;  
          }
          else if(ntk2==2)  // port_name(n:m) format
          {
            str_tmp2 = str_between(str_tmp, "(", ")");
            lbit = (str2int(str_before(str_tmp2, ":")));  
            rbit = (str2int(str_after(str_tmp2, ":")));  
          }
          if(lbit<rbit) {n=rbit; rbit=lbit; lbit=n;}
          
          for(int i=lbit; i>=rbit; i--) // check readyness of particular cond-bits
          {
            str_tmp2 = module + "." + str_before(str_tmp, "(") + "(" + int2str(i) + ")";
            s_cbits += str_tmp2;
            cbits_ready &= (markedCbits->find(str_tmp2) != markedCbits->end());
          }
        }
        if(debug_mode) cout << ", cond-part " << (cbits_ready?"READY":"NOT READY");
        bits_ready = fbits_ready & cbits_ready;
        if(debug_mode) cout << "... TRANIS src+cond bits " << (bits_ready?"READY":"NOT READY") << endl;
        
        if(bits_ready)  // mark transportation
        {
//        for() ???
          {
              ta_data_avg = getAvgOfTAdata(mptr, s_fbits); // averages of from-data
              ta_data_prod = getProdOfTAdata(mptr, s_cbits, &addBits, &addClks); // product of cond-data
              
              if((ta_data_avg.tst != -1.0) && (ta_data_prod.tst != -1.0))  // all-bits-data OK
              {
                if(debug_mode) cout << "\t\t### to-be-transported mark data:" << endl;
    
                if(clk_cnt == SE_PRECISE) addClks *= power_to;

                int nbits = c->getNBits();
                int nclks = c->getNClks();
                int nb = ta_data_avg.nbitsO + addBits;
                int nc = ta_data_avg.nclksO + addClks;
                double multo_;
                double multo;

                if(mark_operation==PP_LIN)
                {
                  multo_ = ta_data_avg.multOprev / power_to;
                  multo = (multo_ * ta_data_prod.multCctrl) / power_to;
                }
                else
                {
                  multo_ = pow(ta_data_avg.multOprev, power_to);
                  multo = pow(multo_ * ta_data_prod.multCctrl, power_to);
                }

                double obs = (1-(nb/(double)(nbits+1)))*(1-(nc/(double)(nclks+1)))*multo;

                if(debug_mode)
                {    
                    cout << "### power_to: " << power_to << endl; 

                    cout << "### avgData: " <<endl; dumpTAdata(&ta_data_avg); 
                    cout << "### productData: " <<endl; dumpTAdata(&ta_data_prod);
        
                    cout << "\t\t\tadding " << addBits << " nbitsO, sum = " << nb << endl;
                    cout << "\t\t\tadding " << addClks << " nclksO, sum = " << nc << endl;
                    cout << "\t\t\tsetting multo- to " << multo_ << endl;
                    cout << "\t\t\tsetting multo to " << multo << endl;
                    cout << "\t\t\tgetting # of in-circuit in+sel+... bits: " << nbits << endl; 
                    cout << "\t\t\tgetting # of in-circuit clk bits: " << nclks << endl; 
                    cout << "\t\t\tbracket1: " << (1-(nb/(double)(nbits+1))) << endl; 
                    cout << "\t\t\tbracket2: " << (1-(nc/(double)(nclks+1))) << endl; 
                    cout << "\t\t\tsetting obs to " << obs << endl << endl;
                }
         
                t_TAdata propag_data; // data to be set to to-bits
                
                propag_data.nbitsO=nb;
                propag_data.nclksO=nc;
                propag_data.multOprev=multo_;
                propag_data.multOctrl=multo;
                propag_data.obs=obs;
                propag_data.tst=0.0;
    
                if(debug_mode) dumpTAdata(&propag_data);
                
                // propagate mark to from-bits of actual TRANIS
                StringTokenizer stk1 = StringTokenizer(it_stranis->from_name, ")");
                int ntk1=stk1.countTokens(), lbit, rbit;
                for(int i=0; i<ntk1; i++) 
                {
                  str_tmp = stk1.nextToken() + ")";
        
                  StringTokenizer stk2 = StringTokenizer(str_tmp, ":");
                  int ntk2=stk2.countTokens();
                  if(ntk2==1) // port_name(n) format
                  {
                    lbit = (str2int(str_between(str_tmp, "(", ")")));  
                    rbit = lbit;  
    
                    if(str_between(str_tmp, "(", ")") == "i")
                    {
                    }            
                  }
                  else if(ntk2==2)  // port_name(n:m) format
                  {
                    str_tmp2 = str_between(str_tmp, "(", ")");
                    lbit = (str2int(str_before(str_tmp2, ":")));  
                    rbit = (str2int(str_after(str_tmp2, ":")));  
                  }
                  if(lbit<rbit) {n=rbit; rbit=lbit; lbit=n;}
                  
                  for(int i=lbit; i>=rbit; i--) // propagate mark
                  {
                    string pname = str_before(str_tmp, "(");
                    clsPort *pptr=NULL;
                    clsBit *bptr=NULL;
                    str_tmp2 = module + "." + pname + "(" + int2str(i) + ")";
                    
                    if((pptr=(mptr->findPort(pname))))
                    {
                      if((bptr=(pptr->findBit(i))))
                      {
                        if(bptr->getTAdata().obs < propag_data.obs) // transport better-mark only
                        {
                          if(debug_mode) cout << "\t\t### ...transporting mark to '" << str_tmp2 << "' bit" << endl;
                          if(debug_mode) cout << "\t\t### ...power_to="<<power_to << endl;
    
                          t_TAdata dta_t = propag_data, dta_f = bptr->getTAdata();
                          
                          // remain fbit's controllability results unchanged ...
                          dta_t.nbitsC = dta_f.nbitsC; 
                          dta_t.nclksC = dta_f.nclksC; 
                          dta_t.multCprev = dta_f.multCprev; 
                          dta_t.multCctrl = dta_f.multCctrl; 
                          dta_t.con = dta_f.con; 
    
                          bptr->setTAdata(dta_t);   // ... transfer only O-mark
    
                          news->insert(str_tmp2);
                        }
                        else {if(debug_mode) cout << "\t\t### ...mark to '" << str_tmp2 << "' bit NOT transported" << endl;}
                      }  
                    }
                  }
                }
              }
            }
        }
      }
  }
  else error("Attempt to transport mark through module %s failed! Because module-ptr is NULL!\n", module.c_str());
}

//! set-of-string union definition
set<string> set_union(set<string> s1, set<string> s2)
{
  set<string>::iterator s_it;
  set<string> s;
  
  for(s_it = s1.begin(); s_it != s1.end(); s_it++) s.insert(*s_it);
  for(s_it = s2.begin(); s_it != s2.end(); s_it++) s.insert(*s_it);
  
  return(s);
}

/******************************************************************************
*	class clsCircuit - implementation of controllability analysis method 
******************************************************************************/
/*!
 * @param[out] marked_bits set of bits with controllability marks assigned
 */
int clsCircuit::controlability_analysis(set<string> *marked_bits)
{
  set<string> newly_marked_bits;
  set<string> transported_to;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  set<clsWire>::iterator it_w;
  set<string>::iterator it_str;
  string str_tmp;

  if(debug_mode) cout << "\nControllability analysis process started..." << endl;

  // phase 1: marking of PIs
  if(debug_mode) cout << "\tPhase 1..." << endl;
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)
  {   
      p = &((*it_p).second);
      if((p->getType()) & (NODE_IN|NODE_CTRL|NODE_CLK))  // inputs only
      {
        if(debug_mode) cout << "\t\tMarking PI-bits " << p->getName() << "(" << p->getHBit() << ":" << p->getLBit() << ")..." << endl;
        for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
        {
            t_TAdata ta_data; b = &((*it_b).second);  ta_data=b->getTAdata();
            ta_data.nbitsC=ta_data.nclksC=0;
            ta_data.multCprev=ta_data.multCctrl=ta_data.con=1.0;
            b->setTAdata(ta_data);

            str_tmp = p->getName() + "(" + int2str(b->getNumber()) + ")";
            newly_marked_bits.insert(str_tmp);
        }
      }
  }

  (*marked_bits) = newly_marked_bits;

  if(debug_mode) dump_string_set(newly_marked_bits, "newly-marked = transported-to = marked");

  // phase 2+3:
  while(newly_marked_bits.size())
  { // phase 2: transport marks through connections
    transported_to.clear();
    if(debug_mode) cout << "\tPhase 2..." << endl;
    
    for(it_str = newly_marked_bits.begin(); it_str != newly_marked_bits.end(); it_str++)
    { // find connections, which begin with newly-marked bits
      for(it_w = getSetOfWiresBegin(); it_w != getSetOfWiresEnd(); it_w++)  
      { // such a connection found -> transport mark through it
        if((*it_w).from_composed == *it_str)  
        { transport_wireMark(this, *it_w, 1); transported_to.insert((*it_w).to_composed); }
      }
    }
    if(debug_mode) {dump_string_set(transported_to, "transported-to");}
    
    newly_marked_bits = transported_to;
    (*marked_bits) = set_union((*marked_bits), newly_marked_bits);
    if(debug_mode) dump_string_set((*marked_bits), "marked");

    // phase 3: transport marks through modules
    set<string> s_modNames;
    s_modNames.clear();
    transported_to.clear();
    
    if(debug_mode) cout << "\tPhase 3..." << endl;

    for(it_str = newly_marked_bits.begin(); it_str != newly_marked_bits.end(); it_str++)
    { StringTokenizer strtok = StringTokenizer(*it_str, ".");
      if(strtok.countTokens() == 2){ s_modNames.insert(strtok.nextToken()); }
    }   
      // dump set of modules to be propagated
    if(debug_mode) dump_string_set(s_modNames, "modules for propagation");
    for(it_str = s_modNames.begin(); it_str != s_modNames.end(); it_str++)
    {
      transport_modCmark(this, *it_str, marked_bits, &transported_to);
    }

    if(debug_mode) dump_string_set(transported_to, "transported-to");
    newly_marked_bits = transported_to;
    (*marked_bits) = set_union((*marked_bits), newly_marked_bits);
    if(debug_mode) dump_string_set((*marked_bits), "marked");
  }
  return(0);
}

/******************************************************************************
*	class clsCircuit - implementation of observability analysis method 
******************************************************************************/
/*!
 * @param[in] marked_Cbits set of bits with controllability marks assigned
 */
int clsCircuit::observability_analysis(set<string> *marked_Cbits)
{
  set<string> marked_Obits;
  set<string> newly_marked_bits;
  set<string> transported_to;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  set<clsWire>::iterator it_w;
  set<string>::iterator it_str;
  string str_tmp;

  if(debug_mode) cout << "\nObservability analysis process started..." << endl;

  // phase 1: marking of POs
  if(debug_mode) cout << "\tPhase 1..." << endl;
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)
  {   
      p = &((*it_p).second);
      if((p->getType()) & NODE_OUT)  // outputs only
      {
        if(debug_mode) cout << "\t\tMarking PO-bits " << p->getName() << "(" << p->getHBit() << ":" << p->getLBit() << ")..." << endl;
        for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
        {
            t_TAdata ta_data; b = &((*it_b).second);  ta_data=b->getTAdata();
            ta_data.nbitsO=ta_data.nclksO=0;
            ta_data.multOprev=ta_data.multOctrl=ta_data.obs=1.0;
            b->setTAdata(ta_data);

            str_tmp = p->getName() + "(" + int2str(b->getNumber()) + ")";
            newly_marked_bits.insert(str_tmp);
        }
      }
  }

  marked_Obits = newly_marked_bits;
  if(debug_mode) dump_string_set(newly_marked_bits, "newly-marked = transported-to = marked");

  // phase 2+3:
  while(newly_marked_bits.size())
  { // phase 2: transport marks through connections
    transported_to.clear();
    if(debug_mode) cout << "\tPhase 2..." << endl;

    for(it_str = newly_marked_bits.begin(); it_str != newly_marked_bits.end(); it_str++)
    { // find connections, which begin with newly-marked bits
      for(it_w = getSetOfWiresBegin(); it_w != getSetOfWiresEnd(); it_w++)  
      { // such a connection found -> transport mark through it
        if((*it_w).to_composed == *it_str)  
        { transport_wireMark(this, *it_w, 0); transported_to.insert((*it_w).from_composed); }
      }
    }
    
    if(debug_mode) dump_string_set(transported_to, "transported-to");
    newly_marked_bits = transported_to;
    marked_Obits = set_union(marked_Obits, newly_marked_bits);
    if(debug_mode) dump_string_set(marked_Obits, "marked");

    // phase 3: transport marks through modules
    set<string> s_modNames;
    s_modNames.clear();
    transported_to.clear();

    if(debug_mode) cout << "\tPhase 3..." << endl;

    for(it_str = newly_marked_bits.begin(); it_str != newly_marked_bits.end(); it_str++)
    { StringTokenizer strtok = StringTokenizer(*it_str, ".");
      if(strtok.countTokens() == 2){ s_modNames.insert(strtok.nextToken()); }
    }   
      // dump set of modules to be propagated
    if(debug_mode) dump_string_set(s_modNames, "modules for propagation");
    
    for(it_str = s_modNames.begin(); it_str != s_modNames.end(); it_str++)
    {
      transport_modOmark(this, *it_str, &marked_Obits, marked_Cbits, &transported_to);
    }

    if(debug_mode) dump_string_set(transported_to, "transported-to");
    newly_marked_bits = transported_to;
    marked_Obits = set_union(marked_Obits, newly_marked_bits);
    if(debug_mode) dump_string_set(marked_Obits, "marked");
  }

  return(0);
}

void clsCircuit::evalTst()
{
  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  t_TAdata data;

  int nbits=getNAllBits(), ncon=0, nobs=0, ntst=0;   
  con=0.0, obs=0.0, tst=0.0;                  

  // set tst-values to bits and con,obs,tst-values to ports
  //    (a) PI-port(s) data
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)
  { p = &((*it_p).second);
    if(p) {// port-data
    for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
    { b = &((*it_b).second);  // bit-data
      if(b) b->evalTst();} 
      p->evalTst(); data = p->getTAdata();
      con += (data.con * p->getWidth()); obs += (data.obs * p->getWidth()); tst += (data.tst * p->getWidth());
      ncon += p->getNCon(); nobs += p->getNObs(); ntst += p->getNTst(); 
      } }
  //    (b) inner-port(s) data
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
    { p = &((*it_p).second); 
      if(p) {// port-data
      for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
      { b = &((*it_b).second); //bit-data
      if(b) b->evalTst();} 
      p->evalTst(); data = p->getTAdata();
      con += (data.con * p->getWidth()); obs += (data.obs * p->getWidth()); tst += (data.tst * p->getWidth());
      ncon += p->getNCon(); nobs += p->getNObs(); ntst += p->getNTst(); 
      } }
  }
  
  rcon = 100.0*ncon/nbits;
  robs = 100.0*nobs/nbits;
  rtst = 100.0*ntst/nbits;

  con=con/nbits;
  obs=obs/nbits;
  tst=tst/nbits;
}

/******************************************************************************
*	class clsCircuit - implementation of testability analysis method 
******************************************************************************/
int clsCircuit::testability_analysis()
{
  set<string> s_markedCBits;

  //  controllability+observability processes
  clrTAdata(this);
  controlability_analysis(&s_markedCBits);
  observability_analysis(&s_markedCBits);
  evalTst();

  return(0);
}

set<string> clsCircuit::get_untestableBits(tmodeSel mode)
{
  set<string> s;
  set<string>::iterator s_it;

  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  t_TAdata data;

  // set tst-values to bits and con,obs,tst-values to ports
  //    (a) PI-port(s) data
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)
  { p = &((*it_p).second);
    if(p) {// port-data
    for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
    { b = &((*it_b).second);  // bit-data
      if(b) 
      {
        data=b->getTAdata();
        string s_tmp = p->getName() + "(" + int2str(b->getNumber()) + ")";
        
        switch(mode)
        {
          case CON_MODE: if(!data.con) {/*cout << s_tmp << endl;*/ s.insert(s_tmp); } break;
          case OBS_MODE: if(!data.obs) {/*cout << s_tmp << endl;*/ s.insert(s_tmp); } break;
          case TST_MODE: if(!data.tst) {/*cout << s_tmp << endl;*/ s.insert(s_tmp); } break;
          default: ;
        }
      }
        } } }
  //    (b) inner-port(s) data
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
    { p = &((*it_p).second); 
      if(p) {// port-data
      for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
      { b = &((*it_b).second); //bit-data
      if(b) 
      {
        data=b->getTAdata();
        string s_tmp = m->getName() + "." + p->getName() + "(" + int2str(b->getNumber()) + ")";
        
        switch(mode)
        {
          case CON_MODE: if(!data.con) {/*cout << s_tmp << endl;*/ s.insert(s_tmp);} break;
          case OBS_MODE: if(!data.obs) {/*cout << s_tmp << endl;*/ s.insert(s_tmp);} break;
          case TST_MODE: if(!data.tst) {/*cout << s_tmp << endl;*/ s.insert(s_tmp);} break;
          default: ;
        }
      }
        } } }
  }
  
  return(s);
}

set<string> clsCircuit::get_badTsourceBits(int type)
{
  set<string> s;
  set<string>::iterator s_it;

  return(s);
}

set<string> clsCircuit::get_regNames(string fname)
{
  set<string> s;
  set<string>::iterator s_it;
  char data[MAX_STR_LEN];
  const char REG_DELIM = ' ';
  size_t endc;

  string read_from = fname;

  ifstream fin;
  fin.open(read_from.c_str());
  
  if(!fin) { warn("Unable to determine registers in '%s'. Cause: file '%s' containing list of register names missing! Effect: information about scan candidates cannot be given.\n", fname.c_str(), read_from.c_str()); return(s); }

  while(!fin.eof())
  {
    fin.getline(data, MAX_STR_LEN, REG_DELIM);

    string tmp = string(data);
    endc = tmp.find_first_of('\n');
    if(endc != string::npos) {if(endc>0) tmp.resize(endc-1); } 
    s.insert(tmp);
  }
  
  if(!fin) {error("Attempt to read from file '%s' failed! Read aborted.\n", read_from.c_str()); return(s);}
  
  fin.close();

  return(s);
}

set<string> clsCircuit::get_scanRegCandidateBits(set<string> regs)
{
  set<string> s;
  set<string>::iterator it_s;

  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  t_TAdata data;

  // set tst-values to bits and con,obs,tst-values to ports
  // inner-port(s) data
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  { m = &((*it_m).second);
    if(m)
    { 
      if(regs.find(m->getName()) != regs.end())
      {
          for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
          { p = &((*it_p).second); 
            if(p) {// port-data
            for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
            { b = &((*it_b).second); //bit-data
            if(b){ 
            if((b->getType() == NODE_IN) || (b->getType() == NODE_OUT)) // only I/O data bits
            {
              data=b->getTAdata();
              string s_tmp = m->getName() + "." + p->getName() + "(" + int2str(b->getNumber()) + ")";
              
              if(!data.tst) {/*cout << "sReg candidate: " << s_tmp << endl;*/ s.insert(s_tmp); }
            }}
              } } }
      }
    }    
  }
  
  return(s);
}

set<string> clsCircuit::get_scanRegCandidates(set<string> regs)
{
  set<string> s;
  set<string>::iterator it_s;

  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  t_TAdata data;

  // set tst-values to bits and con,obs,tst-values to ports
  // inner-port(s) data
  unsigned int cnt, w;
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  { m = &((*it_m).second);
    if(m)
    { 
      string s_tmp = m->getName();
      if(regs.find(s_tmp) != regs.end())
      {
          cnt=0;
          for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
          { p = &((*it_p).second); 
            if(p) { if(p->getType() == NODE_IN) {// port-data
            cnt=0;
            w=p->getWidth();
            for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
            { b = &((*it_b).second); //bit-data
           if(b)
            {
              data=b->getTAdata();
              
              if(!data.tst) 
              {
                cnt++;
              }
            }
              } } } }
        ostringstream oss;
        oss << hex << (unsigned int)(255*cnt/w);
        s_tmp = s_tmp + "/" + oss.str();
        if(oss.str()=="0") s_tmp += "0";
        if(cnt>0) s.insert(s_tmp);
      }
    }    
  }
  
  return(s);
}
  
void clsCircuit::export_TAresults_txt(string fname)
{
  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  t_TAdata data;

  string save_to = fname+"_TA.txt";
  cout << "Saving testability information of circuit '" << name << "' into file '" << save_to << "'" << endl;
  ofstream fout;
  fout.open(save_to.c_str());
  fout.setf(ios::fixed);
  
  if(!fout) {error("Attempt to write to file '%s' failed! Save aborted.\n", save_to.c_str()); return;}

  fout << "Testability information for circuit '" << name << "'" << endl;
  fout << "-parameters: ";
  fout << "sequential length estimation used (" << (clk_cnt==1?"basic":"precise") << "), ";
  fout << "function used to evaluate propagation penalty (" << (mark_operation==1?"linear":"exponential") << ")" << endl;

  for(int i=0; i<80; i++) fout << "="; fout << endl;
  fout << "port_name [width]\tcon\t\tobs\t\ttst" << endl;
  for(int i=0; i<80; i++) fout << "-"; fout << endl;

  // get tst-values 
  //    (a) PI-port(s) data
  fout << "primary" << endl;
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)
  { p = &((*it_p).second);
    if(p) {// port-data
    for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
    { b = &((*it_b).second);  // bit-data
      if(b) b->evalTst();} 
      p->evalTst(); data = p->getTAdata();
      fout << p->getName() << " [" << p->getWidth() << "]" << "\t\t" << data.con << "\t" << data.obs << "\t" << data.tst << endl; } }
  for(int i=0; i<80; i++) fout << "-"; fout << endl;
  
  //    (b) inner-port(s) data
  fout << "inner" << endl;
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
    { p = &((*it_p).second); 
      if(p) {// port-data
      for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
      { b = &((*it_b).second); //bit-data
      if(b) b->evalTst();} 
      p->evalTst(); data = p->getTAdata();
      fout << m->getName() << "." << p->getName() << " [" << p->getWidth() << "]" << "\t\t" << data.con << "\t" << data.obs << "\t" << data.tst << endl; 
      } } }
  for(int i=0; i<80; i++) fout << "="; fout << endl;
  fout << "Summary:" << endl;
  for(int i=0; i<20; i++) fout << "-"; fout << endl;
  fout << "Controllable 1-ports/average controllability: " << rcon << " %/" << con<< endl;
  fout << "Observable 1-ports/average observability: " << robs << " %/" << obs<<endl;
  fout << "Testable 1-ports/average testability: " << rtst << " %/" << tst<<endl<<endl;
  fout << "Estimated worst-case sequential depth of data path: " << getNClks() << endl;
  fout << "Total number of in-circuit:" << endl;
  fout << "\t- modules: " << getNmodules() << endl;
  fout << "\t- multi-bit ports (n-ports)/average n-port width: " << getNallPorts() << "/" << getAvgPortWidth() << endl;
  fout << "\t- port bits (1-ports): " << getNAllBits() << endl;
  fout << "\t- wires: " << getNwires() << endl;

  for(int i=0; i<80; i++) fout << "="; fout << endl;
  fout << "Details:" << endl;
  for(int i=0; i<20; i++) fout << "-"; fout << endl;

  set<string>::iterator s_it;
  set<string> s;
  fout << "Uncontrollable bits: ";
  s = get_untestableBits(CON_MODE);
  if(s.begin() == s.end()) {fout << "-";}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "Unobservable bits: ";
  s = get_untestableBits(OBS_MODE);
  if(s.begin() == s.end()) {fout << "-";}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "Untestable bits: ";
  s = get_untestableBits(TST_MODE);
  if(s.begin() == s.end()) {fout << "-";}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "Candidate bits for scan: ";
  s = get_scanRegCandidateBits(get_regNames(freg));
  if(s.begin() == s.end()) {fout << "-";}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "Candidate regs for scan/suitability: ";
  s = get_scanRegCandidates(get_regNames(freg));
  if(s.begin() == s.end()) {fout << "-";}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "\nInformation was automatically generated by TASTE system\nCopyright (c) 2008 Josef Strnadel, strnadel@fit.vutbr.cz\nhttp://www.fit.vutbr.cz, Brno University of Technology, Czech Republic\n";
  
  fout.close();
}

void clsCircuit::export_TAresults_htm(string fname)
{
  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  t_TAdata data;

  string save_to = fname+"_TA.html";
  cout << "Saving testability information of circuit '" << name << "' into file '" << save_to << "'" << endl;
  ofstream fout;
  fout.open(save_to.c_str());
  fout.setf(ios::fixed);
  
  if(!fout) {error("Attempt to write to file '%s' failed! Save aborted.\n", save_to.c_str()); return;}

  fout << "<html>\n<head>\n<title>";
  fout << "Testability information for circuit '" << name << "' </title>" << endl;
  fout << "<style>"; 
  fout << ".TABLE\n{\n"<<"font-family: verdana, sans-serif;\n"<<"font-size: 8pt;\n}\n\n";
  fout << ".TITLE\n{\n"<<"padding: 10pt;\n"<<"font-family: verdana, sans-serif;\n"<<"color: black;\n"<<"font-weight: normal;\n"<<"font-size: 12pt;\n"<<"text-align: center;\n"<<"background: lightblue;\n}\n";
  fout << ".HEADER\n{\n"<<"font-family: verdana, sans-serif;\n"<<"color: black;\n"<<"font-weight: bold;\n"<<"font-size: 10pt;\n"<<"text-align: center;\n"<<"background: #DDDDDD;\n}\n\n";
  fout << ".SUMMARY\n{\n"<<"font-family: verdana, sans-serif;\n"<<"color: black;\n"<<"font-size: 8pt;\n"<<"text-align: left;\n"<<"background: #DDDDDD;\n}\n\n";
  fout << ".PORTNAME\n{\n"<<"font-family: verdana, sans-serif;\n"<<"color: black;\n"<<"font-weight: bold;\n"<<"font-size: 8pt;\n"<<"text-align: center;\n"<<"}\n\n";
	fout << ".NONTESTPORT\n{\n"<<"font-family: verdana, sans-serif;\n"<<"color: red;\n"<<"font-size: 8pt;\n"<<"text-align: center;\n"<<"}\n\n";
	fout << ".TESTPORT\n{\n"<<"font-family: verdana, sans-serif;\n"<<"color: black;\n"<<"font-size: 8pt;\n"<<"text-align: center;\n"<<"background: #BBFF77;\n"<<"}\n\n";
	fout << ".COMMENT\n{\n"<<"font-family: verdana, sans-serif;\n"<<"color: black;\n"<<"font-size: 7pt;\n"<<"text-align: center;\n"<<"}\n\n";
 fout << "</style>\n<head>\n\n\n<body>\n\n";

  fout << "<TABLE frame=border border=1 rules=all>\n";

  fout << "<TR><TD class=title colspan=4>Testability information for circuit '" << name << "'";
  fout << "<p class=comment>sequential length estimation used (<b>" << (clk_cnt==1?"basic":"precise") << "</b>), ";
  fout << "function used to evaluate propagation penalty (<b>" << (mark_operation==1?"linear":"exponential") << "</b>)" << endl;
  fout << "</p></TD></TR>\n\n";
    fout << "<tr class = header><TD>Port name [width]</TD>\n"<<"<TD>Con.</TD>\n"<<"<TD>Obs.</TD>\n"<<"<TD>Tst.</TD\n</TR>\n\n";
  
  fout << "<TBODY class=table align=center valign=middle>\n";

  // get tst-values 
  //    (a) PI-port(s) data
  fout << "<TR><TD class=title colspan=4><p class=comment><b>primary</b></p><TR>";
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)
  { p = &((*it_p).second);
    if(p) {// port-data
    for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
    { b = &((*it_b).second);  // bit-data
      if(b) b->evalTst();} 
      p->evalTst(); data = p->getTAdata();
      fout << "<tr><td class=portname>" << p->getName() << " [" << p->getWidth() << "]" << "</td>"<< (data.con?"<td class=testport>":"<td class=nontestport>") << data.con << "</td>"<< (data.obs?"<td class=testport>":"<td class=nontestport>") << data.obs << "</td>"<< (data.tst?"<td class=testport>":"<td class=nontestport>") << data.tst << "</td>" << endl; } }
  
  //    (b) inner-port(s) data
  fout << "<TR><TD class=title colspan=4><p class=comment><b>inner</b></p><TR>";
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
    { p = &((*it_p).second); 
      if(p) {// port-data
      for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
      { b = &((*it_b).second); //bit-data
      if(b) b->evalTst();} 
      p->evalTst(); data = p->getTAdata();
      fout << "<tr><td class=portname>" << m->getName() << "." << p->getName() << " [" << p->getWidth() << "]" << "</td>"<< (data.con?"<td class=testport>":"<td class=nontestport>") << data.con << "</td>"<< (data.obs?"<td class=testport>":"<td class=nontestport>") << data.obs << "</td>"<< (data.tst?"<td class=testport>":"<td class=nontestport>") << data.tst << "</td>" << endl; 
      } } }
  fout << "<tr><td class=summary colspan=4><h4>Summary:</h4>";
  fout << "<b>Controllable</b> 1-ports/average controllability: <b>" << rcon << " %</b>/" << con<< "<br>"<< endl;
  fout << "<b>Observable</b> 1-ports/average observability: <b>" << robs << " %</b>/" << obs<< "<br>"<<endl;
  fout << "<b>Testable</b> 1-ports/average testability: <b>" << rtst << " %</b>/" << tst<<"<br><br>"<< endl;
  fout << "Estimated worst-case <b>sequential depth</b> of data path: " << getNClks() << "<br>" << endl;
  fout << "<b>Total number</b> of in-circuit:<br>" << endl;
  fout << "\t- <b>modules</b>: " << getNmodules() << "<br>" << endl;
  fout << "\t- multi-bit <b>ports</b> (n-ports)/average n-port width: " << getNallPorts() << "/" << getAvgPortWidth() << "<br>" << endl;
  fout << "\t- port <b>bits</b> (1-ports): " << getNAllBits() << "<br>"<< endl;
  fout << "\t- <b>wires</b>: " << getNwires() << "<br>"<< endl;

  fout << "<hr>";
  fout << "<h4>Details:</h4>";
  fout << "<ul>";
  set<string>::iterator s_it;
  set<string> s;
  fout << "<li><b>Uncontrollable bits:</b> ";
  s = get_untestableBits(CON_MODE);
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "<li><b>Unobservable bits:</b> ";
  s = get_untestableBits(OBS_MODE);
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "<li><b>Untestable bits:</b> ";
  s = get_untestableBits(TST_MODE);
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "<li><b>Candidate bits for scan:</b> ";
  s = get_scanRegCandidateBits(get_regNames(freg));
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "<li><b>Candidate regs for scan/suitability:</b> ";
  s = get_scanRegCandidates(get_regNames(freg));
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "</ul>";

  fout << "</td><tr><td class=comment colspan=4>Information was automatically generated by TASTE system<BR>Copyright &copy 2008 <A HREF=\"http://www.fit.vutbr.cz/~strnadel\">Josef Strnadel</A><A HREF=mailto:strnadel@fit.vutbr.cz> (strnadel@fit.vutbr.cz)</A><BR><A HREF=\"http://www.fit.vutbr.cz\">Faculty of Information Technology</A><BR><A HREF=\"http://www.vutbr.cz\">Brno University of Technology</A><BR>Czech Republic</TD></TR>\n";
  fout << "\n</tbody>\n</table>\n</body>\n</html>\n";
  
  fout.close();
}

void clsCircuit::export_TAresults_tex(string fname)
{
  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  t_TAdata data;

  string save_to = fname+"_TA.tex";
  cout << "Saving testability information of circuit '" << name << "' into file '" << save_to << "'" << endl;
  ofstream fout;
  fout.open(save_to.c_str());
  fout.setf(ios::fixed);
  
  if(!fout) {error("Attempt to write to file '%s' failed! Save aborted.\n", save_to.c_str()); return;}

 fout << "\\documentclass[6pt]{report}\n" << "\\usepackage{longtable}\n" <<"\\begin{document}\n";
 fout <<"\n\\begin{center}\n"<< "Testability information for circuit '" << name << "'\n" << "\\end{center}\n\n";

 fout << "\n\\begin{center}\n \\tiny \\emph{parameters:} \n \n";
 fout << "sequential length estimation used (" << (clk_cnt==1?"basic":"precise") << ") \n \n";
 fout << "function used to evaluate propagation penalty (" << (mark_operation==1?"linear":"exponential") << ")";
 fout << "\n" << "\\end{center}\n\n";

 fout << "\\begin{center}\n" << "\\begin{longtable}{|c||c|c|c|}\n" << "\\hline\n";
 fout << "\\bfseries Port name [width]&\n" <<"\\bfseries Con. &\n" <<"\\bfseries Obs. &\n" <<"\\bfseries Tst. \\\\ \n" <<"\\hline\n";

  // get tst-values 
  //    (a) PI-port(s) data
  fout << "\\multicolumn{4}{|c|}{\\it primary} \\\\ \\hline\n" << endl;
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)
  { p = &((*it_p).second);
    if(p) {// port-data
    for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
    { b = &((*it_b).second);  // bit-data
      if(b) b->evalTst();} 
      p->evalTst(); data = p->getTAdata();
      fout << p->getName() << " [" << p->getWidth() << "]" << " & " << data.con << " & " << data.obs << " & " << data.tst << "\\\\" << endl<< "\\hline" << endl; } }
  
  //    (b) inner-port(s) data
  fout << "\\multicolumn{4}{|c|}{\\it inner} \\\\ \\hline\n" << endl;
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  { m = &((*it_m).second);
    if(m) for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
    { p = &((*it_p).second); 
      if(p) {// port-data
      for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
      { b = &((*it_b).second); //bit-data
      if(b) b->evalTst();} 
      p->evalTst(); data = p->getTAdata();
      fout << m->getName() << "." << p->getName() << " [" << p->getWidth() << "]" << " & " << data.con << " & " << data.obs << " & " << data.tst << "\\\\" << endl << "\\hline" << endl; 
      } } }

  fout << "\\end{longtable}\n\\end{center}\n\n";

  fout << "{\\bfseries Summary:}" << endl;
  fout << "\n{\\itshape Controllable} 1-ports/average controllability: " << rcon << " \\%/" << con<< endl;
  fout << "\n{\\itshape Observable} 1-ports/average observability: " << robs << " \\%/" << obs<<endl;
  fout << "\n\n{\\itshape Testable} 1-ports/average testability: " << rtst << " \\%/" << tst<<endl<<endl;
  fout << "\n\\vspace{5mm} Estimated worst-case \\emph{sequential depth} of data path: " << getNClks() << endl;
  fout << "\n\\vspace{5mm}{\\itshape Total number} of in-circuit:" << endl;
  fout << "\\begin{itemize}" << endl;
  fout << "\\item modules: " << getNmodules() << endl;
  fout << "\\item multi-bit ports (n-ports)/average n-port width: " << getNallPorts() << "/" << getAvgPortWidth() << endl;
  fout << "\\item port bits (1-ports): " << getNAllBits() << endl;
  fout << "\\item wires: " << getNwires() << endl;
  fout << "\\end{itemize}" << endl;

  set<string>::iterator s_it;
  set<string> s;
  fout << "\n\\vspace{5mm}{\\bfseries Details:}" << endl;
  fout << "\\begin{itemize}" << endl;
  fout << "\\item\n{\\itshape Uncontrollable bits}: ";
  s = get_untestableBits(CON_MODE);
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "\\item\n{\\itshape Unobservable bits}: ";
  s = get_untestableBits(OBS_MODE);
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "\\item\n{\\itshape Untestable bits}: ";
  s = get_untestableBits(TST_MODE);
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "\\item\n{\\itshape Candidate bits for scan:}: ";
  s = get_scanRegCandidateBits(get_regNames(freg));
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "\\item\n{\\itshape Candidate regs for scan/suitability:}: ";
  s = get_scanRegCandidates(get_regNames(freg));
  if(s.begin() == s.end()) {fout << "-" << endl;}
  for(s_it=s.begin(); s_it != s.end(); s_it++) fout << *s_it << " "; 
  fout << endl;

  fout << "\\end{itemize}" << endl;

  fout << "\\begin{center}\\itshape{" << endl;
  fout << "\n\\vspace{5mm}Information was automatically generated by TASTE system \n\nCopyright (c) 2008 Josef Strnadel, strnadel@fit.vutbr.cz\n\n http://www.fit.vutbr.cz, Brno University of Technology, Czech Republic\n";
  fout << "}\\end{center}" << endl;

  fout << "\\end{document}\n";
		
  fout.close();
}

set<string> clsCircuit::get_scan_notation(string fname)
{
  set<string> s;
  set<string>::iterator s_it;
  const char REG_DELIM = '.';
  string data;
  char c;

  string read_from = fname;

  ifstream fin;
  fin.open(read_from.c_str());
  
  if(!fin) { warn("Unable to load scan notation from '%s'. Cause: file '%s' is missing! Effect: scan will not be implemented.\n", fname.c_str(), read_from.c_str()); return(s); }

  int cnt=0;
  while(!fin.eof()){ if((c = fin.get()) == '\n') cnt++; if(cnt == 8) break; } // notation starts on 7nd+ line

  while(!fin.eof())
  {
    c = fin.get();
    if((c != REG_DELIM) && (c != '\n') && (c != 13)){ data += c; }
    else
    {
      if(data.size()) s.insert(data);
      data = "";
    }
  }
  
  fin.close();

  return(s);
}

/*!
 * @param[in] fsname name of a file containing scan-related information 
 * @param[in] frname name of a file containing names of registers within circuit/design
 */
bool clsCircuit::implement_scan_notation(string fsname, string frname)
{
  bool res=false;
  set<string>::iterator s_it;
  set<string> reg_set = get_regNames(frname); //get_regNames(fname);
  set<string> notation = get_scan_notation(fsname);  // get scan notation

  unsigned int sc=0, sr=0;    // counter of included  
  unsigned int nsc=0, nsr=0;  // counter of not-included

  clsModule *mp = NULL;
  string connect_to;

  string PSI="scan_in", PSO="scan_out", PSM="scan_mode";
  string ISI="s_in", ISO="s_out", ISM="mode";
  string TYPE_TO="SREG_<n>"; 

  // --- try to read parameters from .scan file
  ifstream fin;
  fin.open(fsname.c_str());
  if(!fin) { warn("Unable to load scan data from '%s'. Cause: file '%s' is missing! Effect: trying defaults.\n", fsname.c_str(), fsname.c_str()); }

  char c;
  int line=0, pos=0;
  string data;
  while(!fin.eof()){ if((c = fin.get()) == '\n') line++; if(line == 1) break; } // data start on 4nd+ line
  while(!fin.eof())
  {
    c = fin.get();
    if(c == '\n') {line++; pos=0;}
    if(isspace(c)) {pos++; c=' ';}
    switch(line)
    {
      case 4: 
        if((pos==1) && (!isspace(c))) data += c; /*cout << data << endl; */
        if(isspace(c) && data.size()) {TYPE_TO = data; data="";}
        break;
      case 5: 
        if(!isspace(c)) data += c; /*cout << data << endl; */
        if(isspace(c) && data.size()) 
        {
          switch(pos)
          {
            case 2: PSI = data; break;
            case 3: PSO = data; break;
            case 4: PSM = data; break;
          }
          data="";
        }
        break;
      case 6: 
        if(!isspace(c)) data += c; /*cout << data << endl; */
        if(isspace(c) && data.size()) 
        {
          switch(pos)
          {
            case 2: ISI = data; break;
            case 3: ISO = data; break;
            case 4: ISM = data; break;
          }
          data="";
        }
        break;
    }
  } 
  fin.close();

  if(debug_mode) cout << "SCAN implementation started..." << endl;
  for(s_it = notation.begin(); s_it != notation.end(); s_it++) 
  {
    sc++; sr=0;
    if(debug_mode) cout << "Processing scan chain: '" << *s_it << "' (chain #" << sc << ")" << endl;
    string tmp = *s_it, sreg;
    
    StringTokenizer strtok = StringTokenizer(tmp," ");  
    unsigned int ntokens = strtok.countTokens();
    bool started=false;
    string last="";

    for(unsigned int i = 0; i < ntokens; i++)
    {
      string s = strtok.nextToken();
      
      if(reg_set.find(s) == reg_set.end())  // unknown reg? -> warn
      {
        warn("Register named '%s' is not specified in '*.reg' file. Effect: Inclusion of the register to scan will be skipped.\n", s.c_str());
        nsr++;
      }
      else  // known reg -> include into scan
      {
        sr++;

        if(!started)  // start chain
        {
          last = s;
          started = true;
          
          if(sc==1) addPort(PSM, NODE_CTRL, 1, false);  // common for all chains
          connect_to = PSI+int2str(sc);
          addPort(connect_to, NODE_IN, 1, false);

          if((mp = findMod(s)))
          {
            mp->changeTypeTo(TYPE_TO);
            connectPorts("",PSM,s,ISM, false, false);

            if(connect_to.find(".") != string::npos) connectPorts(str_before(connect_to, "."),str_after(connect_to, "."), s, ISI, false, false);
            else connectPorts("",connect_to, s, "s_in", false, false);

            connect_to = s+"."+ISO;
          }
        }
        else
        {
          if((mp = findMod(s)))
          {
            mp->changeTypeTo(TYPE_TO);
            connectPorts("",PSM,s,ISM, false, false);

            if(connect_to.find(".") != string::npos) connectPorts(str_before(connect_to, "."),str_after(connect_to, "."), s, ISI, false, false);
            else connectPorts("",connect_to, s, ISI, false, false);

            connect_to = s+"."+ISO;
            last=s;
          }
        }
      }
    }
    
    // finish chain
    if(started)
    {
      addPort(PSO+int2str(sc), NODE_OUT, 1, false);
      connectPorts(str_before(connect_to, "."),str_after(connect_to, "."), "", PSO+int2str(sc), false, false);
    }
    else
    {
      nsc++;
    }
    
    connect_to="";
  }
  
  string msg="";
  if(nsr) msg = (int2str(nsr) + " register(s) could not be inserted into scan ");
  if(nsr && nsc) msg += "and ";
  if(nsc) msg += (int2str(nsc) + " scan chain(s) could not be formed.");

  if(nsr || nsc) warn("As a result of previous warning(s), %s.\n", msg.c_str());

  return(res);
}

bool clsCircuit::deimplement_changes()
{
  bool res=true;

  map<string, clsModuleType>::iterator it_mtyp;
  set<tMTypeData, lt_mtdata>::iterator it_stranis;
  set<string>::iterator it_sports;
  map<string, int>::iterator it_si;
  
  clsModule* m = NULL;
  clsPort* p = NULL;
  map<string, clsCircuit>::iterator it_c;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  set<clsWire>::iterator it_w;

  string str_tmp;

  if(debug_mode) cout << "Deimplementing changes made in '" << name << "' circuit. List of to-be-removed follows:" << endl;
  
  // remove all non-original parts
  for(it_p = getMapPortBegin(); it_p != getMapPortEnd(); it_p++)
  {
      p = &((*it_p).second);
      if(!p->getOrig()) 
      {
        mapPort.erase(it_p);
      }
  }
  
  for(it_m = getMapModBegin(); it_m != getMapModEnd(); it_m++)
  {
    m = &((*it_m).second);
    
    if(!(m->getOrig())) // module was modified (type changed)
    {
      m->typeBack();
    }
    
    //if(!m->getOrig()) 
    {
      for(it_stranis = m->getSTranisBegin(); it_stranis != m->getSTranisEnd(); it_stranis++)
      {
        if(!it_stranis->orig) 
        {
          m->eraseTranis(it_stranis);
        }
      }

      for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
      {
          p = &((*it_p).second);
          if(!p->getOrig()) 
          {
            m->erasePort(it_p);
          }
      }
    }
  }
  
  for(it_w = getSetOfWiresBegin(); it_w != getSetOfWiresEnd(); it_w++)
  {
      if(!it_w->getOrig()) 
      {
        eraseWire(it_w);
      }
  }
  
  return(res);
}

void clsCircuit::implementDft()
{
  if(scan_mode)
  {
    implement_scan_notation(fscan, freg);
  }
}

/******************************************************************************
*	class clsDesign - implementation
******************************************************************************/
clsDesign::clsDesign(string nm)
{
  name = string(nm);
  if(debug_mode) cout << "clsDesign constructor for object named '" << name << "' called" << endl;
}

clsDesign::~clsDesign()
{
  if(debug_mode) cout << "clsDesign destructor called for object named '" << name << "'" << endl;
}

bool clsDesign::addCir(string nm)
{  
  bool b=mapCir.insert(vt_cir(nm, clsCircuit(nm))).second; 
  if(debug_mode) cout<< "addCir(): " << nm << endl; 
  if(!b) warn("Attempt to insert circuit '%s' into mapCir failed!\n", nm.c_str()); return(b);
}

bool clsDesign::addModType(string nm)
{  
  bool b=mapModType.insert(vt_moduletype(nm, clsModuleType(nm))).second; 
  if(debug_mode) cout<< "addModType(): " << nm << endl; 
  if(!b) warn("Attempt to insert module type '%s' into mapModType failed! Probably, '%s' definition is stored more times in given library/ies.\n", nm.c_str(), nm.c_str()); return(b);
}

void clsDesign::dump_info()
{
  map<string, clsModuleType>::iterator it_mtyp;
  set<tMTypeData, lt_mtdata>::iterator it_stranis;
  set<string>::iterator it_sports;
  clsModuleType* mt = NULL;
  map<string, int>::iterator it_si;
  
  clsCircuit* c = NULL;
  clsModule* m = NULL;
  clsPort* p = NULL;
  clsBit* b = NULL;
  map<string, clsCircuit>::iterator it_c;
  map<string, clsModule>::iterator it_m;
  map<string, clsPort>::iterator it_p;
  map<int, clsBit>::iterator it_b;
  set<clsWire>::iterator it_w;

  string str_tmp;

  cout << "-------- module-types '" << name << "' info() start --------" << endl;
  cout << "Module-types loaded:" << ((mapModType.size()==0)?" none":"") << endl;
  for(it_mtyp = mapModType.begin(); it_mtyp != mapModType.end(); it_mtyp++)
  {
    cout << "'" << (str_tmp=(it_mtyp->second.getName())) << "'" << endl;
    mt = findModType(str_tmp);
    if(mt)
    {
      cout << "\tTRANIS loaded:" << (((mt->getSTranisBegin()) == (mt->getSTranisEnd()))?" none":"") << endl;
      for(it_stranis = mt->getSTranisBegin(); it_stranis != mt->getSTranisEnd(); it_stranis++)
      {
        cout << "\t\t'" << (it_stranis->composed) << "'" << endl;
      }
      cout << "\tInterface loaded:" << (((mt->getSPortsBegin()) == (mt->getSPortsEnd()))?" none":"") << endl;
      for(it_sports = mt->getSPortsBegin(); it_sports != mt->getSPortsEnd(); it_sports++)
      {
        cout << "\t\t'" << *it_sports << "'" << endl;
      }
      cout << "\tArea: '" << mt->getArea() << "'" << endl;
      cout << "\tPower: '" << mt->getPower() << "'" << endl;
    }
    else error("Module type %s not found in dump_info()!\n", str_tmp.c_str());
  }
  cout << "-------- module-types '" << name << "' info() end --------" << endl;

  cout << "-------- design '" << name << "' info() start --------" << endl;
  cout << "Area: '" << getArea() << "'" << endl;
  cout << "Power: '" << getPower() << "'" << endl;

  cout << "In-design circuits:" << (((getMapCirBegin()) == (getMapCirEnd()))?" none":"") << endl;
  for(it_c = getMapCirBegin(); it_c != getMapCirEnd(); it_c++)
  {
    c = &((*it_c).second);
    cout << "Circuit '" << c->getName() << "'" << endl;

    cout << "\tArea: '" << c->getArea() << "'" << endl;
    cout << "\tPower: '" << c->getPower() << "'" << endl;

    cout << "\tIn-circuit ports:" << (((c->getMapPortBegin()) == (c->getMapPortEnd()))?" none":"") << endl;
    for(it_p = c->getMapPortBegin(); it_p != c->getMapPortEnd(); it_p++)
    {
        t_TAdata ta_data;
        p = &((*it_p).second);
        cout << "\tPort '" << p->getName() << "', type: " << p->getType() << ", orig: " << p->getOrig() << endl;

        ta_data=p->getTAdata();
        cout << "\t\tTestability-info:" << endl;
        cout << "\t\tnbitsC: " << ta_data.nbitsC;
        cout << ", nclksC: " << ta_data.nclksC;
        cout << ", nbitsO: " << ta_data.nbitsO;
        cout << ", nclksO: " << ta_data.nclksO;
        cout << ", multC-: " << ta_data.multCprev;
        cout << ", multC: " << ta_data.multCctrl;
        cout << ", multO-: " << ta_data.multOprev;
        cout << ", multO: " << ta_data.multOctrl;
        cout << ", con: " << ta_data.con;
        cout << ", obs: " << ta_data.obs;
        cout << ", tst: " << ta_data.tst << endl;

        cout << "\t\tIn-port bits:" << (((p->getMapBitBegin()) == (p->getMapBitEnd()))?" none":"") << endl;
        for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
        {
            t_TAdata ta_data;
            b = &((*it_b).second);
            ta_data=b->getTAdata();
            cout << "\t\tBit #" << b->getNumber() << endl;
            
            cout << "\t\t\tTestability-info:" << endl;
            cout << "\t\t\tnbitsC: " << ta_data.nbitsC;
            cout << ", nclksC: " << ta_data.nclksC;
            cout << ", nbitsO: " << ta_data.nbitsO;
            cout << ", nclksO: " << ta_data.nclksO;
            cout << ", multCprev: " << ta_data.multCprev;
            cout << ", multCctrl: " << ta_data.multCctrl;
            cout << ", multO-: " << ta_data.multOprev;
            cout << ", multO: " << ta_data.multOctrl;
            cout << ", con: " << ta_data.con;
            cout << ", obs: " << ta_data.obs;
            cout << ", tst: " << ta_data.tst << endl;
        }
    }
    
    cout << "\tIn-circuit modules:" << (((c->getMapModBegin()) == (c->getMapModEnd()))?" none":"") << endl;
    for(it_m = c->getMapModBegin(); it_m != c->getMapModEnd(); it_m++)
    {
      m = &((*it_m).second);
      cout << "\tModule '" << m->getName() << "' of type '" << m->getType() << "'" << endl;
      
      cout << "\t\tTRANIS loaded:" << (((m->getSTranisBegin()) == (m->getSTranisEnd()))?" none":"") << endl;
      for(it_stranis = m->getSTranisBegin(); it_stranis != m->getSTranisEnd(); it_stranis++)
      {
        cout << "\t\t'" << (it_stranis->composed) << "'" << ", orig: " << it_stranis->orig << endl;
      }
      cout << "\tArea: '" << m->getArea() << "'" << endl;
      cout << "\tPower: '" << m->getPower() << "'" << endl;
      
      cout << "\t\tIn-module ports:" << (((m->getMapPortBegin()) == (m->getMapPortEnd()))?" none":"") << endl;
      for(it_p = m->getMapPortBegin(); it_p != m->getMapPortEnd(); it_p++)
      {
        t_TAdata ta_data;
        p = &((*it_p).second);
        cout << "\tPort '" << p->getName() << "', type: " << p->getType() << ", orig: " << p->getOrig() << endl;

        ta_data=p->getTAdata();
        cout << "\t\tTestability-info:" << endl;
        cout << "\t\tnbitsC: " << ta_data.nbitsC;
        cout << ", nclksC: " << ta_data.nclksC;
        cout << ", nbitsO: " << ta_data.nbitsO;
        cout << ", nclksO: " << ta_data.nclksO;
        cout << ", multC-: " << ta_data.multCprev;
        cout << ", multC: " << ta_data.multCctrl;
        cout << ", multO-: " << ta_data.multOprev;
        cout << ", multO: " << ta_data.multOctrl;
        cout << ", con: " << ta_data.con;
        cout << ", obs: " << ta_data.obs;
        cout << ", tst: " << ta_data.tst << endl;
        
        cout << "\t\t\tIn-port bits:" << (((p->getMapBitBegin()) == (p->getMapBitEnd()))?" none":"") << endl;
        for(it_b = p->getMapBitBegin(); it_b != p->getMapBitEnd(); it_b++)
        {
          t_TAdata ta_data;
          b = &((*it_b).second);
          ta_data=b->getTAdata();
          cout << "\t\tBit #" << b->getNumber() << endl;
          
          cout << "\t\t\tTestability-info:" << endl;
          cout << "\t\t\tnbitsC: " << ta_data.nbitsC;
          cout << ", nclksC: " << ta_data.nclksC;
          cout << ", nbitsO: " << ta_data.nbitsO;
          cout << ", nclksO: " << ta_data.nclksO;
          cout << ", multCprev: " << ta_data.multCprev;
          cout << ", multCctrl: " << ta_data.multCctrl;
          cout << ", multO-: " << ta_data.multOprev;
          cout << ", multO: " << ta_data.multOctrl;
          cout << ", con: " << ta_data.con;
          cout << ", obs: " << ta_data.obs;
          cout << ", tst: " << ta_data.tst << endl;
        }
      }

      cout << "\t\tIn-module constants:" << (((m->getMapConstsBegin()) == (m->getMapConstsEnd()))?" none":"") << endl;
      for(it_si = m->getMapConstsBegin(); it_si != m->getMapConstsEnd(); it_si++)
      {
        cout << "\t\t'" << (*it_si).first << "'=" << (*it_si).second<< endl;
      }
    }
    
    cout << "\tIn-cuircuit connections:" << (((c->getSetOfWiresBegin()) == (c->getSetOfWiresEnd()))?" none":"") << endl;
    for(it_w = c->getSetOfWiresBegin(); it_w != c->getSetOfWiresEnd(); it_w++)
    {
      cout << "\t" << (*it_w).composed << ", orig: " << (*it_w).getOrig() << endl;
    }
  }
  cout << "-------- design '" << name << "' info() end --------" << endl;
}

void clsDesign::export_TAresults_txt(string fname)
{
}

clsModuleType *clsDesign::findModType(string nm)
{
  map<string, clsModuleType>::iterator it=mapModType.find(nm); 
  if(it==mapModType.end()) 
  { 
    if(debug_mode) cout << "Module-type '" << nm << "' not found in a library. Trying to find a templatized module-type..." << endl;
    for(it = mapModType.begin(); it != mapModType.end(); it++)
    {
      string tempStr1 = (*it).first, tempStr2, tempStr3;
      string spart1, spart2, spart3;
      if(debug_mode) cout << "analyzing module-type " << tempStr1 << endl;

      if(tempStr1.find("<")==string::npos)
      {
        if(debug_mode) cout << "not a templatized module-type. Trying other one..." << endl;
      }
      else
      {
        spart1 = str_before(tempStr1, "<");         // a of a<n>b
        spart2 = str_between(tempStr1, "<", ">");   // n of a<n>b
        spart3 = str_after(tempStr1, ">");          // b of a<n>b

        if((spart1.size()==0) && (spart2.size()==0)) {warn("Wrong format of templatized module-type '%s'! Aborting insertion of the module type.\n", nm.c_str()); return(NULL);}

        string tmp=nm;
        unsigned int i;
        
        string nm1=nm;
        if(nm.size() > spart1.size()) nm1.erase(spart1.size());
        
        if((i=nm1.find(spart1)) != string::npos)
        {
          if(!spart3.size()){tmp.erase(i, spart1.size());}  // a<n> format... erase everything except value of n (e.g., REG_8 -> 8 for REG_<n> template)
          else // a<n>b format... e.g., ADD_16booth -> 16 for ADD_<n>booth template
          {
            tmp = str_between(tmp, spart1, spart3);
          }    
          if((i=nm.find(spart3)) != string::npos)
          {
            // number of tokens separated by '_' have to be the same in 'n' of 'a<n>b' and in 'nm'
            // e.g., add<n1_n2> ... add16_32 is OK, but not add<n1_n2> ... add16
            StringTokenizer strtok1 = StringTokenizer(spart2,"_");
            int ntokens1 = strtok1.countTokens();
            StringTokenizer strtok2 = StringTokenizer(tmp,"_");
            int ntokens2 = strtok2.countTokens();

            // number of parameters in the template and in the module-type must be equal 
            if(ntokens1 != ntokens2) {return(NULL);}

            bool ok=true;
            string s_tmp;
            for(int i=0; (i<ntokens2)&&ok; i++)
            {
              s_tmp = strtok2.nextToken();
              ok = ok && isInt(s_tmp);
            }
            if(!ok)
            {
              if(debug_mode) cout << "Templatized module-type detection aborted" << endl;
              return(NULL);
            }

            if(debug_mode) cout << "Templatized module-type " << spart1 << "<" << spart2 << ">" << spart3 << " with parameters " << tmp << " found for " << nm << endl;
            
            return(&(*it).second); 
          }
        }
      }
    } 
    return(NULL);
  } 
  else 
  {
    if(debug_mode) cout << "Module-type found for " << nm << endl;
    return(&(*it).second);
  }
}

void clsDesign::syncModWithType(clsModule *mptr)
{
    if(!mptr) return;

    string mtype = mptr->getType();
    if(debug_mode) cout << "Synchronizing module '" << mptr->getName() << "' with its type (" << mtype << ")" << endl;
    
    //add constants (according to module-type)
    clsModuleType *cmtptr = NULL;
    cmtptr = findModType(mtype);
    
    if(!cmtptr) return;
    
    if((cmtptr->getName()) != (mptr->getType())) // templatized module-type
    {
      unsigned int i1, i2;
      string tmpStr1=cmtptr->getName();
      i1 = tmpStr1.find("<"); i2 = tmpStr1.find(">");
      if((i1==string::npos) || (i2==string::npos)) {warn("syncModWithType(): module '%s' cannot be synchronized with its type!\n", (mptr->getName()).c_str()); return;}
      
      // get n from a<n>b (templatized module-type)
      tmpStr1 = str_between(tmpStr1, "<", ">");
      
      //get value of n from module-type
      string tmpStr2 = mptr->getType();
      string str1 = str_before(cmtptr->getName(), "<");
      string str2 = str_after(cmtptr->getName(), ">");
      
      if((str1.size()==0) && (str2.size()==0)) return;
      if(!str2.size()) tmpStr2 = str_after(tmpStr2, str1);
      else tmpStr2 = str_between(tmpStr2, str1, str2);
      
      // number of parameters in the template and in the module-type must be equal 
      string s_tmp;
      StringTokenizer strtok1 = StringTokenizer(tmpStr1,"_");
      int ntokens1 = strtok1.countTokens();
      StringTokenizer strtok2 = StringTokenizer(tmpStr2,"_");
      int ntokens2 = strtok2.countTokens();
      bool ok = (ntokens1 == ntokens2);
      for(int i=0; (i<ntokens2)&&ok; i++)
      {
        s_tmp = strtok2.nextToken();
        ok = ok && isInt(s_tmp);
      }
      if(!ok) {warn("Insertion of constant '%s' for module '%s' failed because non-int value(s) assigned to '%s'!\n",
                                  tmpStr1.c_str(), mptr->getName().c_str(), tmpStr1.c_str()); tmpStr2="";}
      
      // add constants if module-type is derived from templatized type
      mptr->addConst(tmpStr1, tmpStr2);
    }    

    int tmp_val;
    
    // set area (according to module-type)
    if((tmp_val = mptr->evalExpr(cmtptr->getArea())) == -1) 
        warn("Insertion of area for module '%s' failed because '%s' cannot be evaluated!\n", 
        mptr->getName().c_str(), cmtptr->getArea().c_str());
    mptr->setArea(tmp_val);

    // set power (according to module-type)
    if((tmp_val = mptr->evalExpr(cmtptr->getPower())) == -1) 
        warn("Insertion of power for module '%s' failed because '%s' cannot be evaluated!\n", 
        mptr->getName().c_str(), cmtptr->getPower().c_str());
    mptr->setPower(tmp_val);

    // add interface (according to module-type)
    mptr->addInterface(cmtptr);

    // add transparency data (according to module-type)
    mptr->addTranspData(cmtptr);
    
    if(debug_mode) {cout << "Synchronization successfully done." << endl;}
}

void clsDesign::implementDft(string cir)
{
    if(cir!="") // action over a circuit within the design
    {
      clsCircuit *cptr=NULL;
      if((cptr=findCir(cir)))
      {
        cptr->implementDft();
      }
    }
    else  // action over whole design
    {
    }
}

void clsDesign::deimplement_changes(string cir)
{
    if(cir!="") // action over a circuit within the design
    {
      clsCircuit *cptr=NULL;
      if((cptr=findCir(cir)))
      {
        cptr->deimplement_changes();
      }
    }
    else  // action over whole design
    {
    }
}
